//
// $Header: D:/ext2-os2/minifsd/RCS/fs_misc.c,v 1.1 1996/05/27 23:49:54 Willm Exp Willm $
//

// Linux ext2 file system driver for OS/2 2.x and WARP - Allows OS/2 to
// access your Linux ext2fs partitions as normal drive letters.
// Copyright (C) 1995, 1996 Matthieu WILLM
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>                // From the "Developer Connection Device Driver Kit" version 2.0
#include <dhcalls.h>            // From the "Developer Connection Device Driver Kit" version 2.0
#include <strat2.h>             // From the "Developer Connection Device Driver Kit" version 2.0


#include <string.h>
#include <stdlib.h>                // for atol() ...

#include <fsd.h>
#include <fsh.h>

/***********************************************************************************/
/*** Fichiers inclus locaux                                                      ***/
/***********************************************************************************/
#include <linux/stat.h>

#include <os2/ifsdbg.h>
#include <os2/filefind.h>
#include <os2/cdfsd.h>
#include <os2/errors.h>

#include <os2/log.h>         /* Prototypes des fonctions de log.c                      */
#include <os2/volume.h>      /* Prototypes des fonctions de volume.c                   */
#include <os2/files.h>       /* Prototypes des fonctions de files.c                    */

#include <os2/os2proto.h>
#include <os2/os2misc.h>
#include <os2/vfsapi.h>
#include <linux/fs.h>
#include <linux/fs_proto.h>
#include <linux/e2_fs.h>
#include <linux/e2_proto.h>
#include <linux/fcntl.h>
#include <linux/sched.h>
#include <linux/stat.h>

#include <os2/minifsd.h>

/***********************************************************************************/
/*** Nom et attributs du FSD                                                     ***/
/***********************************************************************************/


char FS_NAME[]             = "ext2";
unsigned long FS_ATTRIBUTE = FSA_LVL7;


/***********************************************************************************/
/*** Errors not in bseerr.h (IFS specific or ext2_os2 specific)                  ***/
/***********************************************************************************/

#define ERROR_VOLUME_NOT_MOUNTED 0xEE00                // IFS specific

/***********************************************************************************/
/*** Some useful defines for FS_OPENCREATE() ...                                 ***/
/***********************************************************************************/

   #define OPEN_ACCESS_MASK               0x0007  /* ---- ---- ---- -111 */
   #define OPEN_ACCESS_EXECUTE            0x0003  /* ---- ---- ---- -100 */

   #define OPEN_SHARE_MASK                0x0070  /* ---- ---- -111 ---- */
   #define OPEN_LOCALITY_MASK             0x0700  /* ---- -111 ---- ---- */
   #define OPEN_ACTION_EXIST_MASK         0x000F  /* ---- ---- ---- 1111 */
   #define OPEN_ACTION_NEW_MASK           0x00F0  /* ---- ---- 1111 ---- */

#define FILE_NONFAT     0x0040                // File is non 8.3 compliant

/***********************************************************************************/
/*** Internal volume table                                                       ***/
/***********************************************************************************/

volume_global_data volglobdat;

/***********************************************************************************/
/*** Device helper entry point                                                   ***/
/***********************************************************************************/

extern unsigned long Device_Help;

#define THISFILE FILE_TEST_C

/***********************************************************************************/
/*** Some external data ...                                                      ***/
/***********************************************************************************/


extern unsigned long event;                // To be moved somewhere in a .h file

/***********************************************************************************/
/*** Time zone so that time stamps shown in OS/2 are the same as in Linux        ***/
/***********************************************************************************/

long timezone                  = 0;     // Time zone (second shift from UTC)

/***********************************************************************************/
/*** FS_INIT()                                                                   ***/
/***********************************************************************************/
extern void stage2_ll_rw_block();
extern void (*__ll_rw_block)();

extern void (*wait_on_inode)();
extern void (*lock_inode)();
extern void (*unlock_inode)();

extern void stage2_wait_on_inode(struct inode * inode);
extern void stage2_lock_inode(struct inode * inode);
extern void stage2_unlock_inode(struct inode * inode);

extern void stage2_wake_up(void *wait);
extern void stage2_sleep_on(void *wait);
extern void (*sleep_on)();
extern void (*wake_up)();
extern int (far pascal *system_halt)();

extern void inode_stage1_to_stage2(void);

extern struct super_block *stage2_getvolume(unsigned short hVPB);
extern struct super_block *(*getvolume)(unsigned short);

_FS_RET _FS_ENTRY FS_INIT(
                          pchar                 szParm,
                          unsigned long         DevHelp,
                          unsigned long _FS_PTR pMiniFSD
                         )
{
    int i;

    printk("FS_INIT(DevHelp=%lu szParm=%lu pMiniFSD=%lu)", DevHelp, szParm, pMiniFSD);

    //
    // Saves the device helper routine entry point
    //
    Device_Help = DevHelp;

    //
    // Updates system halt entry point
    //
    system_halt = FSH_INTERR;

    //
    // Updates memory allocation entry points
    //
    G_malloc = stage2_G_malloc;
    G_free   = stage2_G_free;

    //
    // Updates ll_rw_block entry points
    //
    __ll_rw_block = stage2_ll_rw_block;

    //
    // Updates inode entry points
    //
    wait_on_inode  = stage2_wait_on_inode;
    lock_inode     = stage2_lock_inode;
    unlock_inode   = stage2_unlock_inode;

    //
    // Updates the sched entry points
    //
    sleep_on = stage2_sleep_on;
    wake_up  = stage2_wake_up;

    //
    // Updates the volume entry points
    //
    getvolume = stage2_getvolume;

    //
    // Initializes our hVPB array (used to keep a trace of mounted/unmounted volumes
    // and their status)
    //
    volglobdat.sem = 0;
    for (i = 0; i < NB_MAX_VOLS; i++) {
        volglobdat.listvol[i].hVPB   = 0;
        volglobdat.listvol[i].status = VOL_STATUS_FREE;
        volglobdat.listvol[i].sb     = 0;
    }

    //
    // Stage 1 to stage 2 transition for buffers and inodes.
    //
    inode_stage1_to_stage2();
    buffer_stage1_to_stage2();


    return NO_ERROR;

}


/***********************************************************************************/
/*** FS_CHGFILEPTR()                                                             ***/
/***********************************************************************************/
_FS_RET _FS_ENTRY FS_CHGFILEPTR(
                            struct sffsi _FS_PTR psffsi,
                            struct sffsd _FS_PTR psffsd,
                            long                 offset,
                            unsigned short       type,
                            unsigned short       IOflag
                           )
{
    off_t               newfileptr;
    struct super_block *sb;
    struct file        *p_file;

    sb = getvolume(psffsi->sfi_hVPB);

    if (!(p_file = ((_sffsd _FS_PTR)psffsd)->p_file)) {
        kernel_printf("FS_CHGFILEPTR() - p_file = NULL");
        return ERROR_INVALID_PARAMETER;
    }

    switch(type) {
        case CFP_RELBEGIN :
            newfileptr = offset;
            break;

        case CFP_RELCUR :
            newfileptr = psffsi->sfi_position + offset;
            break;

        case CFP_RELEND :
            newfileptr = p_file->f_inode->i_size + offset;
            break;

        default :
            return ERROR_INVALID_PARAMETER;
    }

    if (newfileptr < 0) {
        return ERROR_INVALID_PARAMETER;
    }

    p_file->f_pos        = newfileptr;
    p_file->f_reada      = 0;
    p_file->f_version    = ++event;

    psffsi->sfi_position = newfileptr;
    return NO_ERROR;
}


_FS_RET _FS_ENTRY FS_CLOSE(
                       unsigned short          type,
                       unsigned short          IOflag,
                       struct sffsi    _FS_PTR psffsi,
                       struct sffsd    _FS_PTR psffsd
                      )
{
    struct super_block * sb;
    struct file        * p_file;
    int                  rc;

    printk("FS_CLOSE(ino = %lu, type = %d)",  ((_sffsd _FS_PTR)psffsd)->p_file->f_inode->i_ino, type);

    //
    // Gets the superblock from psffsi
    //
    if (!(sb = getvolume(psffsi->sfi_hVPB))) {
       ext2_os2_panic(0, "FS_CLOSE - Unable to retrieve superblock");
    }

    //
    // Gets the file structure from psffsd
    //
    if (!(p_file = ((_sffsd _FS_PTR)psffsd)->p_file)) {
       ext2_os2_panic(0, "FS_CLOSE - p_file = NULL");
    }



    //
    // The doc isn't clear about the role of the 'type' parameter. It seems we must
    // only free the resources (file structure in sffsd) at FS_CL_FORSYS time. Otherwise
    // we'' receive an empty sffsd somewhere else !
    // For other 'type' values, maybe we could do a flush ...
    //
    if (type != FS_CL_FORSYS) {
#ifdef FS_TRACE
        kernel_printf("***** Non final system close **** - sffsi->sfi_type = %d - Type = %d", psffsi->sfi_type, type);
#endif
        return NO_ERROR;
    } /* endif */


    //
    // Closes the file
    //
    if ((rc = vfs_close(p_file)) != NO_ERROR) {
        fs_err(FUNC_FS_CLOSE, FUNC_CLOSE, rc, FILE_TEST_C, __LINE__);
        return rc;
    }

    //
    // Clean up of sffsd (safety purposes ...)
    //
    memset(psffsd, 0, sizeof(struct sffsd));

    return NO_ERROR;
}





void _FS_ENTRY FS_EXIT(
                       unsigned short uid,
                       unsigned short pid,
                       unsigned short pdb
                      )
{
    printk("FS_EXIT( uid = %u pid = %u pdb = %u )", uid, pid, pdb);
}



_FS_RET _FS_ENTRY FS_IOCTL(
                           struct sffsi   _FS_PTR psffsi,
                           struct sffsd   _FS_PTR psffsd,
                           unsigned short         cat,
                           unsigned short         func,
                           char           _FS_PTR pParm,
                           unsigned short         lenParm,
                           unsigned       _FS_PTR pParmLenInOut,
                           char           _FS_PTR pData,
                           unsigned short         lenData,
                           unsigned       _FS_PTR pDataLenInOut
                          )
{
    int           rc;
    struct vpfsi *pvpfsi;
    struct vpfsd *pvpfsd;

    printk("FS_IOCTL pre-invocation : cat = 0x%0X, func = 0x%0X", cat, func);

    if ((rc = FSH_GETVOLPARM(psffsi->sfi_hVPB, &pvpfsi, &pvpfsd)) == NO_ERROR) {
        if ((rc = FSH_DEVIOCTL(
                               0,
                               pvpfsi->vpi_hDEV,
                               psffsi->sfi_selfsfn, /* sfn */
                               cat,
                               func,
                               pParm,
                               lenParm,
                               pData,
                               lenData
                              )) == NO_ERROR) {
            /*
             * Nothing ...
             */
        } else {
            kernel_printf("FS_IOCTL - FSH_DEVIOCTL returned %d", rc);
        } /* FSH_DEVIOCTL failed */
    } else {
        kernel_printf("FS_IOCTL - FSH_GETVOLPARM returned %d", rc);
    } /* FSH_GETVOLPARM failed */

    if (pDataLenInOut) {
        *pDataLenInOut = lenData;
    }
    if (pParmLenInOut) {
        *pParmLenInOut = lenParm;
    }

    return rc;
}





_FS_RET _FS_ENTRY FS_MOUNT(
                           unsigned short         flag,
                           struct vpfsi   _FS_PTR pvpfsi,
                           struct vpfsd   _FS_PTR pvpfsd,
                           unsigned short         hVPB,
                           char           _FS_PTR pBoot
                          )
{
    int rc;
    int i;
    struct super_block *sb;
    unsigned short oldhVPB;


    switch(flag) {
        case MOUNT_MOUNT :
             printk("FS_MOUNT flg = MOUNT_MOUNT hVPB = 0x%04X", hVPB);


             //
             // We zero out pvpfsd ... for safety purposes !
             //
             memset(pvpfsd, 0, sizeof(struct vpfsd));

             //
             // First make a (small) test to see if it can be a ext2fs volume
             // This is NECESSARY since we could get a FS_MOUNT call for a volume
             // already mounted by another IFS. In this case we MUST know it's not
             // an ext2fs volume BEFORE calling FSH_FINDDUPHVPB (...if not we'll see
             // later a MOUNT_RELEASE for a non ext2fs volume !!  The test case is :
             // put cdfs.ifs after ext2-os2.ifs in CONFIG.SYS, and start WinOS2 file manager)
             //
             if (!Check_Ext2fs_magic(pvpfsi,  hVPB)) {
                 return ERROR_VOLUME_NOT_MOUNTED;
             }

             //
             // The volume serial number is a CRC checksum of the boot sector
             //
             pvpfsi->vpi_vid = updcrc((unsigned char *)pBoot, pvpfsi->vpi_bsize);

             //
             // The volume label is dummy for the moment ("ext2fs_<drive>")
             //
             sprintf(pvpfsi->vpi_text, "EXT2FS_%c", pvpfsi->vpi_unit + 'A');

             //
             // Is there another instance of the drive ?
             //     - Yes : update internal volume table and return silently
             //     - No  : continue the mount process
             //
             if ((rc = FSH_FINDDUPHVPB(hVPB, &oldhVPB)) == NO_ERROR) {
                 kernel_printf(" \tFSH_FINDDUPHVPB(0x%0X) - Found dup hVPB 0x%0X", hVPB, oldhVPB);
                 //
                 // We insert the new dup hVPB in the table so that it can be MOUNT_RELEASEd
                 //
                 i = 0;
                 while ((volglobdat.listvol[i].status != VOL_STATUS_FREE) && (i<NB_MAX_VOLS)) {
                     i++;
                 }
                 if (i == NB_MAX_VOLS) {
                     printk("ERROR : No more volumes ");
                     return ERROR_VOLUME_NOT_MOUNTED;
                 }
                 volglobdat.listvol[i].hVPB   = hVPB;
                 volglobdat.listvol[i].status = VOL_STATUS_MOUNTED;
                 volglobdat.listvol[i].sb     = 0;


                 //
                 // We update the status of the old hVPB if necessary
                 //
                 i = 0;
                 while ((volglobdat.listvol[i].hVPB != oldhVPB) && (i < NB_MAX_VOLS)) {
                     i++;
                 }
                 if (i == NB_MAX_VOLS) {
                     ext2_os2_panic(0, "ERROR : Cannot find dup volume in my internal tables !");
                 }
                 if (volglobdat.listvol[i].status == VOL_STATUS_REMOVED) {
                     kernel_printf("Remounting removed volume");
                     volglobdat.listvol[i].status == VOL_STATUS_MOUNTED;
                 }
                 return NO_ERROR;
             } else {
                 kernel_printf(" \tFSH_FINDDUPHVPB(0x%0X) - No dup hVPB", hVPB);
             }

             i = 0;
             while ((volglobdat.listvol[i].status != VOL_STATUS_FREE) && (i<NB_MAX_VOLS)) {
                 i++;
             }
             if (i == NB_MAX_VOLS) {
                 printk("ERROR : No more volumes");
                 return ERROR_VOLUME_NOT_MOUNTED;
             }

             if ((sb = openvolume(pvpfsi, pvpfsd, hVPB, pBoot)) == 0) {
                 return ERROR_VOLUME_NOT_MOUNTED;
             } /* end if */
             ((hvolume *)pvpfsd)->p_volume = sb;


             volglobdat.listvol[i].hVPB   = hVPB;
             volglobdat.listvol[i].status = VOL_STATUS_MOUNTED;
             volglobdat.listvol[i].sb     = sb;


             kernel_printf("Volume characteristics :");
             kernel_printf("\t volume id    : 0x%08X", pvpfsi->vpi_vid);
             kernel_printf("\t hDEV         : 0x%08X", pvpfsi->vpi_hDEV);
             kernel_printf("\t sector size  : %u", pvpfsi->vpi_bsize);
             kernel_printf("\t sector/track : %u", pvpfsi->vpi_trksec);
             kernel_printf("\t heads        : %u", pvpfsi->vpi_nhead);
             kernel_printf("\t tot sectors  : %lu", pvpfsi->vpi_totsec);
             kernel_printf("\t drive (0=A)  : %d", (int)(pvpfsi->vpi_drive));
             kernel_printf("\t unit code    : %d", (int)(pvpfsi->vpi_unit));
             kernel_printf("\t volume label : %s", pvpfsi->vpi_text);
             kernel_printf("\t hVPB         : 0x%08X", hVPB);

             return NO_ERROR;

        case MOUNT_VOL_REMOVED :
             kernel_printf("FS_MOUNT flg = MOUNT_VOL_REMOVED hVPB = 0x%04X", hVPB);
             return ERROR_NOT_SUPPORTED;

        case MOUNT_RELEASE :
             kernel_printf("FS_MOUNT flg = MOUNT_RELEASE hVPB = 0x%04X", hVPB);
             return ERROR_NOT_SUPPORTED;

        case MOUNT_ACCEPT :
             kernel_printf("FS_MOUNT flg = MOUNT_ACCEPT hVPB = 0x%04X", hVPB);
             return ERROR_NOT_SUPPORTED;

        default :
            kernel_printf("FS_MOUNT() invalid flag %d", flag);
            return ERROR_INVALID_PARAMETER;
    } /* end switch */

}






_FS_RET _FS_ENTRY FS_OPENCREATE(
                                struct cdfsi   _FS_PTR pcdfsi,
                                struct cdfsd   _FS_PTR pcdfsd,
                                char           _FS_PTR pName,
                                unsigned short         iCurDirEnd,
                                struct sffsi   _FS_PTR psffsi,
                                struct sffsd   _FS_PTR psffsd,
                                unsigned long          ulOpenMode,
                                unsigned short         openflag,
                                unsigned short _FS_PTR pAction,
                                unsigned short         attr,
                                char           _FS_PTR pEABuf,
                                unsigned short _FS_PTR pfgenFlag
                               )
{
    int rc;
    struct super_block * sb;
    pfile   p_file, dir;
    UINT32 openmode;
    UINT32 accessmode;
    UINT16 newflag, existflag;
    char component[CCHMAXPATH];
    char parent[CCHMAXPATH];
    struct inode *inode;
    struct inode *inode_parent;
    ino_t         ino_no;

    printk("FS_OPENCREATE(%s)", pName);

    //
    // Gets the superblock from psffsi
    //
    sb = getvolume(psffsi->sfi_hVPB);

#ifdef FS_TRACE
    if (ulOpenMode & OPEN_FLAGS_DASD) {
        printk("OPEN_FLAGS_DASD");
    }


    if (ulOpenMode & OPEN_FLAGS_WRITE_THROUGH) {
        printk("OPEN_FLAGS_WRITE_THROUGH");
    }
    if (ulOpenMode & OPEN_FLAGS_FAIL_ON_ERROR) {
        printk("OPEN_FLAGS_FAIL_ON_ERROR");
    }
    if (ulOpenMode & OPEN_FLAGS_NO_CACHE) {
        printk("OPEN_FLAGS_NO_CACHE");
    }
    if (ulOpenMode & OPEN_FLAGS_NOINHERIT) {
        printk("OPEN_FLAGS_NO_INHERIT");
    }
#endif
    accessmode = ulOpenMode & OPEN_ACCESS_MASK;

    if (accessmode == OPEN_ACCESS_READONLY) {
#ifdef FS_TRACE
        printk("OPEN_ACCESS_READONLY");
#endif
        openmode = OPENMODE_READONLY;
    }

    if (accessmode == OPEN_ACCESS_WRITEONLY) {
#ifdef FS_TRACE
        printk("OPEN_ACCESS_WRITEONLY");
#endif
        openmode = OPENMODE_WRITEONLY;
    }

    if (accessmode == OPEN_ACCESS_READWRITE) {
#ifdef FS_TRACE
        printk("OPEN_ACCESS_READWRITE");
#endif
        openmode = OPENMODE_READWRITE;
    }

#ifdef FS_TRACE
    if (accessmode == OPEN_ACCESS_EXECUTE) {
        printk("OPEN_ACCESS_EXECUTE");
    }
#endif

    newflag = openflag & OPEN_ACTION_NEW_MASK;

#ifdef FS_TRACE
    if (newflag == OPEN_ACTION_FAIL_IF_NEW) {
        printk("OPEN_ACTION_FAIL_IF_NEW");
    }
    if (newflag == OPEN_ACTION_CREATE_IF_NEW) {
        printk("OPEN_ACTION_CREATE_IF_NEW");
    }
#endif

    existflag = openflag & OPEN_ACTION_EXIST_MASK;

#ifdef FS_TRACE
    if (existflag == OPEN_ACTION_OPEN_IF_EXISTS) {
        printk("OPEN_ACTION_OPEN_IF_EXISTS");
    }
    if (existflag == OPEN_ACTION_FAIL_IF_EXISTS) {
        printk("OPEN_ACTION_FAIL_IF_EXISTS");
    }
    if (existflag == OPEN_ACTION_REPLACE_IF_EXISTS) {
        printk("OPEN_ACTION_REPLACE_IF_EXISTS");
    }
#endif

    if ((accessmode == OPEN_ACCESS_READWRITE) ||
        (accessmode == OPEN_ACCESS_WRITEONLY)) {
        return ERROR_NOT_SUPPORTED;
    }

    //
    // Direct access open of the whole device
    //
    if (ulOpenMode & OPEN_FLAGS_DASD) {
        kernel_printf("OPEN_FLAGS_DASD");
        if ((p_file = _open_by_inode(sb, INODE_DASD, openmode)) == 0) {
            kernel_printf("FS_OPENCREATE() - couldn't DASD open %s", pName);
            return ERROR_OPEN_FAILED;
        }
        ((_sffsd _FS_PTR)psffsd)->p_file = p_file;
        psffsi->sfi_tstamp   = ST_SCREAT | ST_PCREAT;
        psffsi->sfi_size     = p_file->f_inode->i_size;
        psffsi->sfi_position = p_file->f_pos;
        date_unix2dos(p_file->f_inode->i_ctime, &(psffsi->sfi_ctime), &(psffsi->sfi_cdate));
        date_unix2dos(p_file->f_inode->i_atime, &(psffsi->sfi_atime), &(psffsi->sfi_adate));
        date_unix2dos(p_file->f_inode->i_mtime, &(psffsi->sfi_mtime), &(psffsi->sfi_mdate));
        return NO_ERROR;

    }

    //
    // Now that we treated the OPEN_FLAGS_DASD special case, lets treat the general case :
    // Try to open the file readonly
    // Success : the file exists
    //     if !S_ISDIR && !S_ISREG => error
    //     if OPEN_ACTION_FAIL_IF_EXISTS set => error
    //     if OPEN_ACTION_OPEN_IF_EXISTS set
    //         <test file attrs>
    //         change the open mode and return OK
    //     if OPEN_ACTION_REPLACE_IF_EXISTS set
    //         OPEN_ACCESS_READONLY or OPEN_ACCESS_EXECUTE set => error
    //         OPEN_ACCESS_READWRITE or OPEN_ACCESS_WRITEONLY set
    //             truncate
    //             change openmode and return
    // Failure : the file does not exist
    //     OPEN_ACCESS_READONLY or OPEN_ACCESS_EXECUTE set => error
    //     OPEN_ACCESS_READWRITE or OPEN_ACCESS_WRITEONLY set
    //         if OPEN_ACTION_CREATE_IF_NEW set
    //             try to create the file
    //             open the file and return
    //         if OPEN_ACTION_FAIL_IF_NEW   set => error

    p_file = _open_by_name(sb, pName, openmode);
    if (p_file) {        // The file exists
        //
        // If it's not a regular file or a directory we cannot open
        //
        if (!S_ISREG(p_file->f_inode->i_mode)) {
            kernel_printf("Can't FS_OPENCREATE - %s is not a regular file", pName);
            if ((rc = vfs_close(p_file)) != NO_ERROR) {
                fs_err(FUNC_FS_OPENCREATE, FUNC_CLOSE, rc, FILE_TEST_C, __LINE__);
                return rc;
            }
            return ERROR_ACCESS_DENIED;
        }
        //
        // if OPEN_ACTION_FAIL_IF_EXISTS set => error
        //
        if (existflag == OPEN_ACTION_FAIL_IF_EXISTS) {
            printk("Can't FS_OPENCREATE() - File exists & OPEN_ACTION_FAIL_IF_EXISTS");
            if ((rc = vfs_close(p_file)) != NO_ERROR) {
                fs_err(FUNC_FS_OPENCREATE, FUNC_CLOSE, rc, FILE_TEST_C, __LINE__);
                return rc;
            }
            return ERROR_FILE_EXISTS;
        }

        //
        // if OPEN_ACTION_OPEN_IF_EXISTS : OK
        //
        if (existflag == OPEN_ACTION_OPEN_IF_EXISTS) {
    	    printk("\tFile %s OK", pName);
            *pAction        = FILE_EXISTED;
        }

    } else {                // The file doesn't exist
	printk("\tFile %s not found", pName);
        return ERROR_FILE_NOT_FOUND;
    }


    ((_sffsd _FS_PTR)psffsd)->p_file = p_file;
    /*
     * Time stamping is done by inode routines - Only propagate value.
     */
    psffsi->sfi_tstamp   |= ST_PREAD;
    psffsi->sfi_size      = p_file->f_inode->i_size;
    psffsi->sfi_position  = p_file->f_pos;


    date_unix2dos(p_file->f_inode->i_ctime, &(psffsi->sfi_ctime), &(psffsi->sfi_cdate));
    date_unix2dos(p_file->f_inode->i_atime, &(psffsi->sfi_atime), &(psffsi->sfi_adate));
    date_unix2dos(p_file->f_inode->i_mtime, &(psffsi->sfi_mtime), &(psffsi->sfi_mdate));

    psffsi->sfi_DOSattr =  (unsigned char)Linux_To_DOS_Attrs(p_file->f_inode, component);
    return NO_ERROR;

}





_FS_RET _FS_ENTRY FS_PROCESSNAME(
                                 pchar pNameBuf
                                )
{
    return NO_ERROR;
}

