package jere.applets.onthisday;

import java.util.*;

import org.xml.sax.*;

/**
 * SAX DocumentHandler class that prints the contents of
 * each "event" element of an input document.
 *
 * @author Jere Kpyaho
 * @revision $Revision$
 */
public class EventFinder extends HandlerBase {
    private Stack eventStack;
    private Stack topicStack;
    private boolean isEvent = false;
    private boolean isTopic = false;

    private Date filterDate;
    private Vector topics;
    private Hashtable events;      // contains vectors of events
    private Hashtable topicNames;  // contains descriptions of topics

    /**
     * Constructs the event finder. The methods of this class
     * are called by the SAX parser.
     *
     * @param filterDate The date to filter the events by
     */
    public EventFinder(Date filterDate) {
        super();

        eventStack = new Stack();
        topicStack = new Stack();

        this.filterDate = filterDate;

        topics = new Vector(10);
        events = new Hashtable();
        topicNames = new Hashtable();
    }

    /**
     * Marks the start of an event element.
     *
     * @param element The name of the element
     * @param attrlist The list of the element's attributes
     */
    public void startElement(String element, AttributeList attrlist) {
        if (element.equals("otdml")) {
            String lang = attrlist.getValue("lang");
            String syslang = Locale.getDefault().toString().substring(0, 2);
            if (!lang.equals(syslang)) {
                System.err.println("Warning: events language = " + lang + ", system language = " + syslang);
            }
        }
        else if (element.equals("event")) {
            isEvent = true;

            // process the attributes

            String dateString = attrlist.getValue("date");
            String topic = attrlist.getValue("topic");

            // Extract the date components
            int year = Integer.parseInt(dateString.substring(0, 4));
            int month = Integer.parseInt(dateString.substring(5, 7));
            int day = Integer.parseInt(dateString.substring(8, 10));
            Calendar calendar = new GregorianCalendar(year, month - 1, day, 0, 0, 0);

            // Create a new event object and push it on the stack
            Event event = new Event(calendar.getTime(), topic, "(none)");
            eventStack.push(event);
            //System.out.println("Pushed " + topic + " event to stack");

 
            //System.out.println(topics);
        }
        else if (element.equals("topic")) {
            isTopic = true;

            // Get the name of the topic
            String nameString = attrlist.getValue("name");

            topicStack.push(nameString);
        }

    }

    /**
     * Adds any text found inside an "event" element into the list.
     * 
     * @param chars The characters inside the element
     * @param start FIXME: what is this parameter?
     * @param length FIXME: what is this parameter?
     */
    public void characters(char[] chars, int start, int length) {
        if (isEvent) {
            String text = new String(chars, start, length);
            Event event;
            try {
                event = (Event) eventStack.peek();
                //System.out.println("Peeked at " + event.getTopic() + " event");
                event.setDescription(text);
            }
            catch (EmptyStackException ese) {
                System.err.println("No elements to peek");
                System.err.println(ese);
            }
        }
        else if (isTopic) {
            String text = new String(chars, start, length);

            try {
                String name = (String) topicStack.peek();
                topicStack.push(text);
            }
            catch (EmptyStackException ese) {
                System.err.println("No topics to peek");
                System.err.println(ese);
            }              
        }
    }

    /**
     * Marks the end of an event element.
     * 
     * @param element The element that ends
     */
    public void endElement(String element) {
        if (element.equals("event")) {
            isEvent = false;

            Event event = null;

            try {
                event = (Event) eventStack.pop();
                //System.err.println("Popped " + event.getTopic() + " event from stack");

                if (filterDate == null) {
                    System.err.println("Filter date is null!");
                }

                String topic = event.getTopic();

                // Filter out all other events except this day's:
                EventFilter filter = new DateFilter(filterDate);
                if (filter.accepts(event)) {
                    // Find the right topic vector from the events hashtable:
                    Vector topicVector = (Vector) events.get(topic);

                    // If there is no vector for this topic, create one
                    if (topicVector == null) {
                        topicVector = new Vector(10);
                        events.put(topic, topicVector);  // put the new vector in the hashtable
                        topics.addElement(topic);        // add to topic list
                    }

                    topicVector.addElement(event);
                    // FIXME: do we have to drag the topic around in the event object?

                    //System.out.println("Accepted event: " + event.toString());
                }
            }
            catch (EmptyStackException ese) {
                System.err.println("No elements to pop");
                System.err.println(ese);
            }
        }
        else if (element.equals("topic")) {
            isTopic = false;

            String description = (String) topicStack.pop();
            String name = (String) topicStack.pop();
            topicNames.put(name, description);
        }
    }

    /** 
     * Gets the event list of this event finder.
     *
     * @return The event list
     */
    public Hashtable getEvents() {
        return events;
    }

    /** 
     * Gets the topic list of this event finder.
     *
     * @return The topic list
     */
    public Vector getTopics() {
        return topics;
    }

    /** 
     * Gets the list of topic names for this event finder.
     *
     * @return The list of topic names
     */
    public Hashtable getTopicNames() {
        return topicNames;
    }
}

