unit Main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls, Buttons, ExtCtrls, RXShell, Menus, RxGrdCpt;

type
  // Enumerated values for the change interval type
  TIntervalType = (itHours, itMinutes, itSeconds);

const
  // timer interval multipliers, in milliseconds
  IntervalMultipliers : array [TIntervalType] of Longint =
    (60 * 60 * 1000, 60 * 1000, 1000);

type
  TFrmMain = class(TForm)
    LblWallpapers: TLabel;
    LstFiles: TListBox;
    ChkChangeEnabled: TCheckBox;
    LblChangeInterval: TLabel;
    TxtChangeInterval: TEdit;
    UpdChangeInterval: TUpDown;
    CbxIntervalType: TComboBox;
    DlgOpen: TOpenDialog;
    BtnMoveUp: TBitBtn;
    BtnMoveDown: TBitBtn;
    BtnAddWallpaper: TBitBtn;
    BtnDeleteWallpaper: TBitBtn;
    BtnApply: TBitBtn;
    BtnClose: TBitBtn;
    LblSample: TLabel;
    PnlSample: TPanel;
    ImgSample: TImage;
    TmrChanger: TTimer;
    StatusBar: TStatusBar;
    BtnChangeNow: TBitBtn;
    RxTrayIcon1: TRxTrayIcon;
    RxGradientCaption1: TRxGradientCaption;
    MnuMain: TPopupMenu;
    MnuAbout: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure ChkChangeEnabledClick(Sender: TObject);
    procedure BtnAddWallpaperClick(Sender: TObject);
    procedure BtnCloseClick(Sender: TObject);
    procedure LstFilesClick(Sender: TObject);
    procedure BtnDeleteWallpaperClick(Sender: TObject);
    procedure BtnApplyClick(Sender: TObject);
    procedure UpdChangeIntervalChanging(Sender: TObject;
      var AllowChange: Boolean);
    procedure TmrChangerTimer(Sender: TObject);
    procedure CbxIntervalTypeChange(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure BtnMoveUpClick(Sender: TObject);
    procedure BtnMoveDownClick(Sender: TObject);
    procedure BtnChangeNowClick(Sender: TObject);
    procedure RxTrayIcon1Click(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure MnuAboutClick(Sender: TObject);
  protected
    procedure WMDropFiles(var Msg: TMessage); message wm_DropFiles;
  private
    SettingsChanged: Boolean;    // True if settings changed
    CurrentImage: Integer;       // index number of the image to use
    SavedLeft: Integer;          // old left coordinate of form
  public
    procedure ChangeWallpaper(Filename: string);
    procedure SaveSettings;
    procedure RestoreSettings(var OK: Boolean);
    procedure MinimizeToTray(Sender: TObject);
  end;

var
  FrmMain: TFrmMain;

implementation

uses
  ShellAPI,  // DragXxx API calls
  Registry, About;  // TRegIniFile

{$R *.DFM}

procedure TFrmMain.FormCreate(Sender: TObject);
var
  OK: Boolean;
begin
  Application.OnMinimize := MinimizeToTray;

  MinimizeToTray(Self);
  
  // Read the settings from the registry.
  RestoreSettings(OK);
  if not OK then
  begin
    // Set the default values for change enabled and change interval.
    ChkChangeEnabled.Checked := False;
    UpdChangeInterval.Position := 1;
    CbxIntervalType.ItemIndex := Integer(itMinutes);
    CurrentImage := 0;  // select the first image
  end;
  SettingsChanged := False;
  BtnApply.Enabled := False;

  // Initially we have no picture files selected on the list,
  // so we disable the Delete button.
  BtnDeleteWallpaper.Enabled := False;

  DragAcceptFiles(Handle, True); // start accepting dropped files
end;

// User enabled or disabled the changes.
procedure TFrmMain.ChkChangeEnabledClick(Sender: TObject);
begin
  SettingsChanged := True;
  BtnApply.Enabled := True;
end;

// User clicked on the Add button. Show the File Open dialog
// and add the file(s) to the list box.
procedure TFrmMain.BtnAddWallpaperClick(Sender: TObject);
var
  DebugMessage: string;
  I: Integer;
begin
  // Show an Open... dialog with settings for a bitmap file
  if DlgOpen.Execute then
  begin
    // Add the selected files to the file list
    for I := 0 to DlgOpen.Files.Count - 1 do
    begin
      LstFiles.Items.Add(DlgOpen.Files[I]);
    end;
    CurrentImage := 0;
  end;
end;

// Close the application window.
procedure TFrmMain.BtnCloseClick(Sender: TObject);
begin
  TmrChanger.Enabled := False;
  Close;
end;

// User clicked on an item on the file list.
// Show the image in the sample control.
procedure TFrmMain.LstFilesClick(Sender: TObject);
var
  Item: Integer;
  Filename: string;
begin
  Item := LstFiles.ItemIndex;
  if Item <> -1 then
  begin
    BtnDeleteWallpaper.Enabled := True;
    if Item <> 0 then // can't move first up
      BtnMoveUp.Enabled := True;
    if Item <> LstFiles.Items.Count - 1 then // can't move last down
      BtnMoveDown.Enabled := True;
    BtnChangeNow.Enabled := True;
  end
  else
  begin
    BtnDeleteWallpaper.Enabled := False;
    BtnChangeNow.Enabled := False;
  end;

  // Load the selected wallpaper to the sample image control.
  Filename := LstFiles.Items[Item];
  try
    ImgSample.Picture.LoadFromFile(Filename);
  except
    on E: EInvalidGraphic do
    begin
      MessageDlg(Filename + 'ei ole kunnollinen BMP-tiedosto!',
                 mtError, [mbOK], 0);
    end;
  end;
end;

procedure TFrmMain.BtnDeleteWallpaperClick(Sender: TObject);
var
  Item: Integer;
begin
  Item := LstFiles.ItemIndex;
  if Item <> -1 then
  begin
    LstFiles.Items.Delete(Item);
    BtnDeleteWallpaper.Enabled := False;
    BtnMoveUp.Enabled := False;
    BtnMoveDown.Enabled := False;
  end;
  CurrentImage := 0;
end;

// Apply the changes to the settings
procedure TFrmMain.BtnApplyClick(Sender: TObject);
var
  SelectedMultiplier: TIntervalType;
begin
  // Get the selected interval multiplier (h/m/s)
  // and calculate the new timer interval based on the
  // updown control's position.
  SelectedMultiplier := TIntervalType(CbxIntervalType.ItemIndex);
  TmrChanger.Interval := UpdChangeInterval.Position *
    IntervalMultipliers[SelectedMultiplier];
  TmrChanger.Enabled := ChkChangeEnabled.Checked;

  SettingsChanged := False;   // indicate changes applied
  BtnApply.Enabled := False;  // so disable the Apply button
end;

procedure TFrmMain.UpdChangeIntervalChanging(Sender: TObject;
  var AllowChange: Boolean);
begin
  SettingsChanged := True;
  BtnApply.Enabled := True;
end;

procedure TFrmMain.TmrChangerTimer(Sender: TObject);
var
  Filename: string;
begin
  // If the list of image filenames is not empty,
  // change the wallpaper.
  if LstFiles.Items.Count > 0 then
  begin
    Filename := LstFiles.Items[CurrentImage];
    ChangeWallpaper(Filename);
    Inc(CurrentImage);
    if CurrentImage = LstFiles.Items.Count then
      CurrentImage := 0;
  end;
end;

procedure TFrmMain.ChangeWallpaper(Filename: string);
begin
  // Set the desktop wallpaper. The tiled/centered setting is made
  // in Control Panel / Desktop.
  if SystemParametersInfo(SPI_SETDESKWALLPAPER, 0, PChar(Filename), Ord(True)) then
  begin
    StatusBar.SimpleText := 'Taustakuvaksi vaihdettu ' + Filename;
  end
  else
  begin
    StatusBar.SimpleText := Format('Windows-virhe numero %.08x', [GetLastError]);
  end;
end;

procedure TFrmMain.CbxIntervalTypeChange(Sender: TObject);
begin
  SettingsChanged := True;
  BtnApply.Enabled := True;
end;

// Saves the settings to the registry, using an easier interface
// that resembles the use of an INI file.
// TRegIniFile is a Delphi helper class that uses the registry
// like an INI file.
procedure TFrmMain.SaveSettings;
var
  Settings: TRegIniFile;
  Item: Integer;
begin
  try
    try
      Settings := TRegIniFile.Create('Software\MikroBitti\Nice Wallpaper');

      with Settings do
      begin
        // We need to delete all the wallpaper names first
        EraseSection('Wallpapers');

        // Write the names of the image files.
        for Item := 0 to LstFiles.Items.Count - 1 do
        begin
          WriteString('Wallpapers',
                      Format('Bitmap%.3d', [Item]), LstFiles.Items[Item]);
        end;

        // Write the changing enabled/disabled flag, the change interval, and its type.
        WriteBool('ChangeInformation', 'Enabled', ChkChangeEnabled.Checked);
        WriteInteger('ChangeInformation', 'Interval', UpdChangeInterval.Position);
        WriteInteger('ChangeInformation', 'IntervalType', CbxIntervalType.ItemIndex);
        WriteInteger('ChangeInformation', 'CurrentImage', CurrentImage);
      end; // with Settings
    except
      on ERegistryException do
        MessageDlg('Virhe rekisteriin kirjoitettaessa', mtWarning, [mbOK], 0);
    end;
  finally
    Settings.Free;
  end;
end;

procedure TFrmMain.RestoreSettings(var OK: Boolean);
var
  Settings: TRegIniFile;
  Item: Integer;
  WallpaperFilename: string;
begin
  OK := False;
  try
    try
      // Create the TRegIniFile object and set the key.
      Settings := TRegIniFile.Create('Software\MikroBitti\Nice Wallpaper');
      with Settings do
      begin
        ReadSectionValues('Wallpapers', LstFiles.Items);

        // The ReadSectionValues call seems to get the key names, too.
        // We have to fix this. The actual filename starts at position 11.
        for Item := 0 to LstFiles.Items.Count - 1 do
        begin
          WallpaperFilename := LstFiles.Items[Item];
          LstFiles.Items[Item] := Copy(WallpaperFilename, 11, Length(WallpaperFilename));
        end;

        ChkChangeEnabled.Checked := ReadBool('ChangeInformation', 'Enabled', False);
        UpdChangeInterval.Position := SmallInt(ReadInteger('ChangeInformation', 'Interval', 1));
        TxtChangeInterval.Text := IntToStr(UpdChangeInterval.Position);
        CbxIntervalType.ItemIndex := ReadInteger('ChangeInformation', 'IntervalType', 1);
        CurrentImage := ReadInteger('ChangeInformation', 'CurrentImage', 0);
      end; // with Settings
    except
      on ERegistryException do
        MessageDlg('Virhe rekisteriin kirjoitettaessa', mtWarning, [mbOK], 0);
    end;
  finally
    Settings.Free;
  end;
  OK := True;
end;

procedure TFrmMain.FormDestroy(Sender: TObject);
begin
  SaveSettings;
  DragAcceptFiles(Handle, False); // stop accepting dropped files
end;

procedure TFrmMain.WMDropFiles(var Msg: TMessage);
var
  NumFiles: Integer;  // how many dropped files
  I: Integer;
  Filename: array [0..259] of Char;
begin
  // Find out the number of dropped files
  NumFiles := DragQueryFile(Msg.WParam, $FFFFFFFF, nil, 0);

  // Add the name of each file to the list box
  for I := 0 to NumFiles - 1 do
  begin
    DragQueryFile(Msg.WParam, I, @Filename, SizeOf(Filename));
    LstFiles.Items.Add(Filename);
  end;

  if LstFiles.Items.Count > 0 then
    BtnDeleteWallpaper.Enabled := True;

  inherited;
end;

// Move the selected item up one place.
// Would it be a good idea to handle EListError exceptions?
procedure TFrmMain.BtnMoveUpClick(Sender: TObject);
var
  ItemNumber: Integer;
begin
  ItemNumber := LstFiles.ItemIndex;
  if ItemNumber > 0 then
  begin
    //StatusBar.SimpleText := Format('ItemNumber = %d', [ItemNumber]);
    LstFiles.Items.Move(ItemNumber, ItemNumber - 1);
    LstFiles.ItemIndex := ItemNumber - 1;
    if LstFiles.ItemIndex = 0 then
      BtnMoveUp.Enabled := False;
    if LstFiles.ItemIndex < LstFiles.Items.Count - 1 then
      BtnMoveDown.Enabled := True;
  end;
end;

// Move the selected item down one place.
procedure TFrmMain.BtnMoveDownClick(Sender: TObject);
var
  ItemNumber: Integer;
begin
  ItemNumber := LstFiles.ItemIndex;
  if ItemNumber < LstFiles.Items.Count - 1 then
  begin
    //StatusBar.SimpleText := Format('ItemNumber = %d', [ItemNumber]);
    LstFiles.Items.Move(ItemNumber, ItemNumber + 1);
    LstFiles.ItemIndex := ItemNumber + 1;
    if LstFiles.ItemIndex = LstFiles.Items.Count - 1 then
      BtnMoveDown.Enabled := False;
    if LstFiles.ItemIndex > 0 then
      BtnMoveUp.Enabled := True;
  end;
end;

procedure TFrmMain.BtnChangeNowClick(Sender: TObject);
var
  Item: Integer;
  Filename: string;
begin
  Item := LstFiles.ItemIndex;
  CurrentImage := Item;
  Filename := LstFiles.Items[CurrentImage];
  ChangeWallpaper(Filename);
end;

procedure TFrmMain.RxTrayIcon1Click(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  // Restore the taskbar button
  SetWindowLong(Application.Handle, GWL_EXSTYLE, WS_EX_APPWINDOW);
  Left := SavedLeft;
  ShowWindowAsync(Handle, SW_SHOW);  // show the form
end;

procedure TFrmMain.MinimizeToTray(Sender: TObject);
begin
  SavedLeft := Left;
  // When the program starts, hide it. Only the tray icon is left.
  Left := -9999;  // move the form off-screen to avoid flashing
  ShowWindowAsync(Handle, SW_HIDE);  // hide the form
  // This is for Delphi 3 or later: hide the form's taskbar button.
  SetWindowLong(Application.Handle, GWL_EXSTYLE, WS_EX_TOOLWINDOW);
end;

procedure TFrmMain.MnuAboutClick(Sender: TObject);
begin
  FrmAbout.ShowModal;
end;

end.
