unit Engine;

interface

uses
  MMSystem, SysUtils, Windows;

var
  TrackCount: Integer;

type
  ECDPlayerException = class(Exception);

  TCDPlayer = class(TObject)
    private
      ErrorNumber: Longint;
      WindowHandle: HWND;
    public
      DeviceID: MCIDEVICEID;
      constructor Create(Window: HWND);
      destructor Destroy; override;
      procedure Load;
      procedure Eject;
      function MediaPresent: Boolean;
      procedure PlayTrack(StartTrack: Longint);
      procedure PlayDisc(StartTrack, EndTrack: Longint);
      procedure Stop;
      function Pause: Boolean;
      function Resume: Boolean;
      function GetPosition: Longint;
      function GetLength: Longint;
      function GetTrackCount: Longint;
      procedure GetTrackLength(Number: Longint; var Minutes, Seconds, Frames: Byte);
      function GetTrackPosition(Number: Longint): Longint; 
      function GetCurrentTrack: Longint;
      function GetCurrentPosition: Longint;
      function GetErrorNumber: Longint;
      function GetErrorMessage: string;
      procedure SetMSF;
      procedure SetTMSF;
      procedure SetMS;
      function GetMode: Longint;
  end;

implementation

constructor TCDPlayer.Create(Window: HWND);
var
  Flags: Longint;
  MCIOpen: TMCI_Open_Parms;
  MCISet: TMCI_Set_Parms;
begin
  inherited Create;

  WindowHandle := Window; // save the callback window handle

  FillChar(MCIOpen, SizeOf(MCIOpen), #0);
  MCIOpen.dwCallback := DWORD(WindowHandle);
  MCIOpen.lpstrDeviceType := PChar(MCI_DEVTYPE_CD_AUDIO);
  Flags := MCI_OPEN_TYPE or MCI_OPEN_TYPE_ID;
  ErrorNumber := mciSendCommand(0, MCI_OPEN, Flags, Longint(@MCIOpen));
  if ErrorNumber <> 0 then
  begin
    raise ECDPlayerException.Create('Error initializing player');
  end;

  DeviceID := MCIOpen.wDeviceID;

  FillChar(MCISet, SizeOf(MCISet), #0);
  MCISet.dwTimeFormat := MCI_FORMAT_TMSF;
  ErrorNumber := mciSendCommand(DeviceID, MCI_SET,
    MCI_SET_TIME_FORMAT, Longint(@MCISet));
  if ErrorNumber <> 0 then
  begin
    raise ECDPlayerException.Create('Error initializing player');
  end;
end;

destructor TCDPlayer.Destroy;
begin
  ErrorNumber := mciSendCommand(DeviceID, MCI_CLOSE, 0, Longint(nil));
  if ErrorNumber <> 0 then
  begin
    raise ECDPlayerException.Create('Error shutting down player');
  end;

  inherited Destroy;
end;

procedure TCDPlayer.Load;
var
  Info: TMCI_Set_Parms;
  Flags: Longint;
begin
  FillChar(Info, SizeOf(TMCI_Set_Parms), 0);
  Info.dwCallback := DWORD(WindowHandle);
  Flags := mci_Set_Door_Closed;
  ErrorNumber := mciSendCommand(DeviceID, MCI_SET, Flags, Longint(@Info));
end;

procedure TCDPlayer.Eject;
var
  Info: TMCI_Set_Parms;
  Flags: Longint;
begin
  FillChar(Info, SizeOf(TMCI_Set_Parms), 0);
  Info.dwCallback := DWORD(WindowHandle);
  Flags := mci_Set_Door_Open;
  ErrorNumber := mciSendCommand(DeviceID, MCI_SET, Flags, Longint(@Info));
end;

function TCDPlayer.MediaPresent: Boolean;
var
  MCIStatus: TMCI_Status_Parms;
begin
  FillChar(MCIStatus, SizeOf(TMCI_Status_Parms), 0);
  MCIStatus.dwItem := MCI_STATUS_MEDIA_PRESENT;
  mciSendCommand(DeviceID, MCI_STATUS,
    MCI_STATUS_ITEM or MCI_WAIT,
    Longint(@MCIStatus));
  Result := (MCIStatus.dwReturn <> 0);
end;

procedure TCDPlayer.PlayTrack(StartTrack: Longint);
var
  Info: TMCI_Play_Parms;
  Flags: Longint;
begin
  FillChar(Info, SizeOf(TMCI_Play_Parms), #0);
  SetTMSF;
  Info.dwCallback := DWORD(WindowHandle);
  Info.dwFrom := mci_Make_TMSF(StartTrack, 0, 0, 0);
  Flags       := MCI_FROM or MCI_NOTIFY;
  ErrorNumber := mciSendCommand(DeviceID, MCI_PLAY,
                 Flags, Longint(@Info));
end;

procedure TCDPlayer.PlayDisc(StartTrack, EndTrack: Longint);
var
  Info: TMCI_Play_Parms;
  Flags: Longint;
begin
  FillChar(Info, SizeOf(TMCI_Play_Parms), #0);
  SetTMSF;
  Info.dwFrom := mci_Make_TMSF(StartTrack, 0, 0, 0);
  Info.dwTo   := mci_Make_TMSF(EndTrack, 0, 0, 0);
  Info.dwCallback := DWORD(WindowHandle);
  Flags       := MCI_FROM or MCI_TO or MCI_NOTIFY;
  ErrorNumber := mciSendCommand(DeviceID, MCI_PLAY,
                 Flags, Longint(@Info));
end;

procedure TCDPlayer.Stop;
var
  Info: TMCI_Generic_Parms;
begin
  Info.dwCallback := 0;
  ErrorNumber := mciSendCommand(DeviceID, MCI_STOP, 0, Longint(@Info));
end;

function TCDPlayer.Pause: Boolean;
var
  Info: TMCI_Generic_Parms;
  Flags: Longint;
begin
  Info.dwCallback := 0;
  Flags := 0;

  ErrorNumber := mciSendCommand(DeviceID, MCI_PAUSE,
    Flags, Longint(@Info));
  Result := (ErrorNumber = 0);
end;

function TCDPlayer.Resume: Boolean;
var
  Info: TMCI_Generic_Parms;
  Flags: Longint;
begin
  Info.dwCallback := 0;
  Flags := 0;

  ErrorNumber := mciSendCommand(DeviceID, MCI_RESUME,
      Flags, Longint(@Info));
  Result := (ErrorNumber = 0);
end;

function TCDPlayer.GetPosition: Longint;
var
  Info: TMCI_Status_Parms;
begin
  FillChar(Info, SizeOf(TMCI_Status_Parms), #0);
  Info.dwItem := MCI_STATUS_POSITION;
  ErrorNumber := mciSendCommand(DeviceID, MCI_STATUS,
    MCI_STATUS_ITEM, Longint(@Info));
  Result := Info.dwReturn;
end;

function TCDPlayer.GetLength: Longint;
var
  Info: TMCI_Status_Parms;
begin
  FillChar(Info, SizeOf(TMCI_Status_Parms), #0);
  Info.dwItem := MCI_STATUS_LENGTH;
  ErrorNumber := mciSendCommand(DeviceID, MCI_STATUS,
    MCI_STATUS_ITEM, Longint(@Info));
  Result := Info.dwReturn;
end;

function TCDPlayer.GetTrackCount: Longint;
var
  Info: TMCI_Status_Parms;
begin
  FillChar(Info, SizeOf(TMCI_Status_Parms), #0);
  Info.dwItem := MCI_STATUS_NUMBER_OF_TRACKS;
  ErrorNumber := mciSendCommand(DeviceID, MCI_STATUS,
    MCI_STATUS_ITEM, Longint(@Info));
  Result := Info.dwReturn;
end;

procedure TCDPlayer.GetTrackLength(Number: Longint;
  var Minutes, Seconds, Frames: Byte);
var
  Info: TMCI_Status_Parms;
  MSF: Longint;
begin
  FillChar(Info, SizeOf(TMCI_Status_Parms), #0);
  Info.dwTrack := Number;
  Info.dwItem := MCI_STATUS_LENGTH;

  SetMSF;
  ErrorNumber := mciSendCommand(DeviceID, MCI_STATUS,
    MCI_STATUS_ITEM or MCI_TRACK, Longint(@Info));

  MSF := Info.dwReturn;
  Minutes := mci_MSF_Minute(MSF);
  Seconds := mci_MSF_Second(MSF);
  Frames := mci_MSF_Frame(MSF);
end;

procedure TCDPlayer.SetMSF;
var
  Info: TMCI_Set_Parms;
  Flags: Longint;
begin
  FillChar(Info, SizeOf(TMCI_Set_Parms), #0);
  Info.dwCallback := 0;
  Info.dwTimeFormat := MCI_FORMAT_MSF;
  Info.dwAudio := 0;

  ErrorNumber := mciSendCommand(DeviceID, MCI_SET,
    MCI_SET_TIME_FORMAT, Longint(@Info));
end;

procedure TCDPlayer.SetTMSF;
var
  Info: TMCI_Set_Parms;
  Flags: Longint;
begin
  FillChar(Info, SizeOf(TMCI_Set_Parms), #0);
  Info.dwCallback := 0;
  Info.dwTimeFormat := MCI_FORMAT_TMSF;
  Info.dwAudio := 0;

  ErrorNumber := mciSendCommand(DeviceID, MCI_SET,
    MCI_SET_TIME_FORMAT, Longint(@Info));
end;

procedure TCDPlayer.SetMS;
var
  Info: TMCI_Set_Parms;
  Flags: Longint;
begin
  FillChar(Info, SizeOf(TMCI_Set_Parms), #0);
  Info.dwCallback := 0;
  Info.dwTimeFormat := MCI_FORMAT_MILLISECONDS;
  Info.dwAudio := 0;

  ErrorNumber := mciSendCommand(DeviceID, MCI_SET,
    MCI_SET_TIME_FORMAT, Longint(@Info));
end;

function TCDPlayer.GetCurrentTrack: Longint;
var
  Info: TMCI_Status_Parms;
begin
  FillChar(Info, SizeOf(Info), #0);
  Info.dwItem := MCI_STATUS_CURRENT_TRACK;
  ErrorNumber := mciSendCommand(DeviceID, MCI_STATUS,
    MCI_STATUS_ITEM, Longint(@Info));
  if ErrorNumber <> 0 then
  begin
    raise ECDPlayerException.Create('Can''t get current track');
  end;
  Result := Info.dwReturn;
end;

function TCDPlayer.GetCurrentPosition: Longint;
var
  Info: TMCI_Status_Parms;
  Flags: Longint;
begin
  Info.dwItem := MCI_STATUS_POSITION;
  Flags := MCI_STATUS_ITEM;
  ErrorNumber := mciSendCommand(DeviceID, MCI_STATUS,
    Flags, Longint(@Info));
  if ErrorNumber <> 0 then
  begin
    raise ECDPlayerException.Create('Can''t get track location');
  end;
  Result := Info.dwReturn;
end;

function TCDPlayer.GetErrorMessage: string;
var
  S: array [0..255] of Char;
begin
  Result := '';
  if ErrorNumber <> 0 then
  begin
    mciGetErrorString(ErrorNumber, S, SizeOf(S));
    Result := StrPas(S);    
  end;
end;

function TCDPlayer.GetErrorNumber: Longint;
begin
  Result := ErrorNumber;
end;

function TCDPlayer.GetMode: Longint;
var
  Info: TMCI_Status_Parms;
  Flags: Longint;
begin
  Info.dwItem := MCI_STATUS_MODE;
  Flags := MCI_STATUS_ITEM or MCI_STATUS_MODE;
  ErrorNumber := mciSendCommand(DeviceID, MCI_STATUS,
    Flags, Longint(@Info));
  if ErrorNumber <> 0 then
  begin
    raise ECDPlayerException.Create('Can''t get mode');
  end;
  Result := Info.dwReturn;
end;

function TCDPlayer.GetTrackPosition(Number: Longint): Longint;
var
  Info: TMCI_Status_Parms;
  MSF: Longint;
begin
  FillChar(Info, SizeOf(TMCI_Status_Parms), #0);
  Info.dwTrack := Number;
  Info.dwItem := MCI_STATUS_POSITION;

  SetMSF;
  ErrorNumber := mciSendCommand(DeviceID, MCI_STATUS,
    MCI_STATUS_ITEM or MCI_TRACK, Longint(@Info));

  Result := Info.dwReturn;
end;

end.
