{*********************************************************}
{                                                         }
{  TDigitalDisplay component for Delphi 2.0 and later     }
{  Version 1.0                                            }
{                                                         }
{  Copyright (C) 1999 Jere Kpyaho.                       }
{  E-mail: jere@sci.fi                                    }
{  WWW: http://www.sci.fi/%7Ejere/                        }
{                                                         }
{*********************************************************}

unit Digital;

interface

uses
  Windows, Messages, Controls, Graphics,
  Classes, SysUtils;

type
  TDigitalDisplay = class(TGraphicControl)
  private
    FDigits: TBitmap;
    FDigitCount: Smallint;
    FValue: Longint;
    DigitWidth: Integer;
    procedure SetDigits(Value: TBitmap);
    procedure SetDigitCount(Value: Smallint);
    procedure SetValue(Value: Longint);
    procedure AdjustSize;
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop,
                    AWidth, AHeight: Integer); override;
  published
    property Digits: TBitmap read FDigits write SetDigits;
    property DigitCount: Smallint read FDigitCount
        write SetDigitCount default 1;
    property Value: Longint read FValue write SetValue
        default 0;
    property Enabled;
    property Visible;
    property Height default 16;
    property Width default 16;
  end;

procedure Register;


implementation

constructor TDigitalDisplay.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FDigits := TBitmap.Create;
  FDigitCount := 1;
  FValue := 0;
  DigitWidth := 16;
  Width := FDigitCount * DigitWidth;
  Height := 16;
end;

destructor TDigitalDisplay.Destroy;
begin
  FDigits.Free;

  inherited Destroy;
end;

procedure TDigitalDisplay.AdjustSize;
begin
  Width := FDigitCount * DigitWidth;
  Height := FDigits.Height;
end;

procedure TDigitalDisplay.SetBounds(ALeft, ATop,
                                AWidth, AHeight: Integer);
begin
  if csDesigning in ComponentState then
  begin
    AWidth := FDigitCount * DigitWidth;
    AHeight := FDigits.Height;
  end;
  inherited;
end;

procedure TDigitalDisplay.SetDigits(Value: TBitmap);
begin
  FDigits.Assign(Value);
  if Assigned(FDigits) then
  begin
    DigitWidth := FDigits.Width div 10;
    AdjustSize;
    Invalidate;
  end;
end;

procedure TDigitalDisplay.SetDigitCount(Value: Smallint);
begin
  if Value <> FDigitCount then
  begin
    FDigitCount := Value;
    AdjustSize;
    Invalidate;
  end;
end;

procedure TDigitalDisplay.SetValue(Value: Longint);
begin
  if Value <> FValue then
  begin
    FValue := Value;
    Invalidate;
  end;
end;

procedure TDigitalDisplay.Paint;
var
  ValueString: string;
  I: Integer;
  FromRect, ToRect: TRect;
  DigitValue: Integer;
begin
  inherited Paint;

  if Assigned(FDigits) then
  begin
    // Get the color of the top left pixel
    // and use it as the background color.
    Canvas.Brush.Color := FDigits.Canvas.Pixels[0, 0];
    Canvas.FillRect(Rect(0, 0, Width, Height));

    ValueString := Format(
      '%.' + IntToStr(FDigitCount) + 'd', [FValue]);

    for I := 1 to FDigitCount do
    begin
      DigitValue := Ord(ValueString[I]) - Ord('0');

      FromRect := Rect(DigitValue * DigitWidth, 0,
        0, Self.Height);
      FromRect.Right := FromRect.Left + DigitWidth;

      ToRect := Rect((I - 1) * DigitWidth, 0,
        0, Self.Height);
      ToRect.Right := ToRect.Left + DigitWidth;

      Canvas.CopyRect(ToRect, FDigits.Canvas, FromRect);
    end;
  end;
end;

procedure Register;
begin
  RegisterComponents('MikroBitti', [TDigitalDisplay]);
end;

end.

