/*
 * HUKKA.C -- laskee FAT-osiolla olevan hukkatilan
 * varausyksikn koon perusteella.
 * Kytt:    hukka [d:]      (d: = osion asematunnus)
 * Copyright (C) 1996 Jere Kpyaho
 */

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dos.h>
#include <dir.h>
#include <direct.h>
#ifdef __DJGPP__
#include <unistd.h>
#endif

struct drivestat_t
{
    char drive_spec[MAXDRIVE];
    unsigned long disk_size, disk_free, cluster_size;
    unsigned long total_bytes, total_alloc, slack_bytes;
    unsigned int file_count, dir_count;
    float free_percentage, wasted_percentage;
};

void spin_propeller( int *spin_pos );
void process_file( struct find_t *file_info, struct drivestat_t *drive_info );
void process_directory( char *path, struct drivestat_t *info );
void analyze( struct drivestat_t *info );
void report( struct drivestat_t *info );

const char progname[] = "hukka";

void spin_propeller( int *spin_pos )
{
    static const int PROP_LEN = 4;
    static const char prop_str[] = "|/-\\";

    if ( *spin_pos >= 0 && *spin_pos < PROP_LEN )
    {
        putch( prop_str[ *spin_pos ] );
        putch( '\b' );
        (*spin_pos)++;
        if ( *spin_pos == PROP_LEN - 1 )
            *spin_pos = 0;
    }
}

void process_file( struct find_t *file_info, struct drivestat_t *drive_info )
{
    (drive_info->file_count)++;
    drive_info->total_bytes += file_info->size;
    if ( file_info->size < drive_info->cluster_size )
        drive_info->slack_bytes += (drive_info->cluster_size - file_info->size);
    else
        drive_info->slack_bytes += (file_info->size % drive_info->cluster_size);
}

void process_directory( char *path, struct drivestat_t *drive_info )
{
    static int spin_pos = 0;
    struct find_t file_info;

    (drive_info->dir_count)++;
    chdir( path );

    if ( 0 == _dos_findfirst( "*.*", _A_HIDDEN | _A_SYSTEM | _A_RDONLY | _A_ARCH, &file_info ) )
    {
        process_file( &file_info, drive_info );
        spin_propeller( &spin_pos );
        while ( 0 == _dos_findnext( &file_info ) )
              process_file( &file_info, drive_info );
    }

    if ( 0 == _dos_findfirst( "*.*", _A_SUBDIR, &file_info ) )
    {
        if ( (file_info.attrib & _A_SUBDIR) && (file_info.name[0] != '.') )
            process_directory( file_info.name, drive_info );

        while ( 0 == _dos_findnext( &file_info ) )
            if ( (file_info.attrib & _A_SUBDIR) && (file_info.name[0] != '.' ) )
               process_directory( file_info.name, drive_info );
    }
    chdir( ".." );
}

void analyze( struct drivestat_t *info )
{
    struct diskfree_t df;

    if ( _dos_getdiskfree( 0, &df ) != 0 )
    {
        perror( progname );
        exit( EXIT_FAILURE );
    }

    /* varausyksikn koko = varausyksikn sektorimr * tavuja sektorilla */
    info->cluster_size = df.sectors_per_cluster * df.bytes_per_sector;
    info->disk_size = df.total_clusters * info->cluster_size;
    info->disk_free = df.avail_clusters * info->cluster_size;

    info->total_alloc = info->disk_size - info->disk_free;
    info->free_percentage = ((float) info->disk_free / (float) info->disk_size) * 100.0;

    info->file_count = info->dir_count = 0;
    info->total_bytes = info->slack_bytes = 0;

    process_directory( "\\", info );  /* aloita juurihakemistosta */
    putch( ' ' );

    info->wasted_percentage = ((float) info->slack_bytes / (float) info->disk_size) * 100.0;
}

void report( struct drivestat_t *info )
{
    printf( "\nAsema                          : %s", info->drive_spec );
    printf( "\nKoko tavuina                   : %10lu", info->disk_size );
    printf( "\nVapaa tila tavuina             : %10lu", info->disk_free );
    printf( "\nVapaan tilan osuus             : %8.1f %%", info->free_percentage );
    printf( "\nVarausyksikn koko tavuina     : %10lu", info->cluster_size );
    printf( "\nHakemistoja yhteens kpl       : %10u", info->dir_count );
    printf( "\nTiedostoja yhteens kpl        : %10u", info->file_count );
    printf( "\nTiedostojen yhteiskoko tavuina : %10lu", info->total_bytes );
    printf( "\nVarattu tila tavuina           : %10lu", info->total_alloc );
    printf( "\nHukkatilan mr tavuina       : %10lu", info->slack_bytes );
    printf( "\nHukkatilan osuus               : %8.1f %%", info->wasted_percentage );
    printf( "\n" );
}

int main( int argc, char *argv[] )
{
    unsigned old_drive, new_drive, verify_drive;
    char old_dir[MAXPATH];
    unsigned max_drives;
    struct drivestat_t info;

    printf( "\nHUKKA v1.0  Copyright (C) 1996 Jere Kpyaho\n" );
    printf( "Laskee levyn hukkatilan varausyksikn koon perusteella.\n" );

    _dos_getdrive( &old_drive );  /* palauttaa 1 = A:, 2 = B: jne. */

    if ( argc >= 2 )
    {
        strupr( argv[1] );
        if ( argv[1][0] >= 'A' && argv[1][0] <= 'Z' )
            info.drive_spec[0] = argv[1][0];
        else
        {
            fprintf( stderr, "Virheellinen asematunnus \"%s\"\n", argv[1] );
            exit( EXIT_FAILURE );
        }
    }
    else
        info.drive_spec[0] = old_drive + 'A' - 1;

    info.drive_spec[1] = ':';
    info.drive_spec[2] = '\0';

    new_drive = info.drive_spec[0] - 'A' + 1;
    _dos_setdrive( new_drive, &max_drives );
    _dos_getdrive( &verify_drive );
    if ( new_drive != verify_drive )
    {
        fprintf( stderr, "Ei voi siirty asemaan %s\n", info.drive_spec );
        exit( EXIT_FAILURE );
    }

    if ( NULL == getcwd( old_dir, sizeof(old_dir) ) )
    {
        perror( progname );
        exit( EXIT_FAILURE );
    }
    strupr( old_dir );

    if ( chdir( "\\" ) != 0 )
    {
        perror( progname );
        exit( EXIT_FAILURE );
    }

    printf( "Analyysi kynniss - odota hetkinen...  " );
    fflush( stdout );
    analyze( &info );
    report( &info );

    chdir( old_dir );
    _dos_setdrive( old_drive, &max_drives );
    return 0;
}
