// $Id: menu.h_v 1.1 1993/10/22 20:51:48 chris Exp chris $

// Chris Dodge -- Oct 93

// Class stuff for text mode menus.

#include <alloc.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include "point.h"
#include "menudefs.h"

void SetTextScreen();
void MenuShaddowBox(int, int, int, int, int, int);
void Printxy(int, int, char*);

int   MenuDel = MTRUE;       // Has the menu screen been deleted?

// --------------------------------------------------------------------
// Stuff for text string display ....
// --------------------------------------------------------------------

// Class to display a string on the screen (text mode)
class TextItem : public Location {
	char *msg;              // Message for output
	int  ColorFG;           // Foregroung colour
	int  ColorBG;           // Background colour

public:
	TextItem(int, int, int, int, char*);
	~TextItem();
	void HighLight(int, int);
	void UnHighLight();
	void SetMark(char*);
};

// Constructor - sets colours, allocs memory etc.
TextItem::TextItem(int msgX, int msgY, int ColFG, int ColBG,
									 char *text) : Location(msgX, msgY)
{
	ColorFG = ColFG;
	ColorBG = ColBG;
	if ((msg=(char *)calloc(strlen(text)+1,sizeof(char)))==NULL) {
		perror("Insufficient memory\n");
		exit(1);
	}
	strcpy(msg,text);
	return;
};

TextItem::~TextItem()
{
	free(msg);
	return;
}

// Highlight this particular menu item.
void TextItem::HighLight(int hlfg, int hlbg)
{
	struct text_info ti;

	// Find current text settings
	gettextinfo(&ti);

	// Highlight text...
	window(1,1,80,25);
	gotoxy(X, Y);
	textcolor(hlfg);
	textbackground(hlbg);
	cprintf(msg);

	// Put back into old colours
	textattr(ti.attribute);
	window(ti.winleft, ti.wintop, ti.winright, ti.winbottom);
	return;
}

// Un-highlight this menu item
void TextItem::UnHighLight()
{
	struct text_info ti;

	// Find current text settings
	gettextinfo(&ti);

	// Draw text normally...
	window(1,1,80,25);
	gotoxy(X, Y);
	textcolor(ColorFG);
	textbackground(ColorBG);
	cprintf(msg);

	// Put back into old colours
	textattr(ti.attribute);
	window(ti.winleft, ti.wintop, ti.winright, ti.winbottom);
	return;
}

// Place X in the toggle box [ ] if it exists. If not, then
// put it at the start of the line.
void TextItem::SetMark(char *mark)
{
	char *ptr;

	window(1,1,80,25);
	// Search for "[ ]"
	if ((ptr=strchr(msg, '['))&&(strchr(msg, ']')==ptr+2))
		gotoxy(X+(ptr-msg)+1, Y);
	else
		gotoxy(X, Y);
	cprintf(mark);
	strxfrm(&msg[(ptr-msg)+1], mark, strlen(mark));
	return;
}


// --------------------------------------------------------------------
// MenuList class - a linked list of TextItems ....
//                  There are two different types of menu item.
//                  type = 0 - normal.
//                  type = 1 - toggle.
// --------------------------------------------------------------------

// The node has 3 elements, 2 of which are links to point forwards and
// backwards in the list, and the third points to the TextItem.
class MNode {
public:
	MNode    *Next;          // Points to the next node in the list
	MNode    *Prev;          // Points to the previous node in the list
	TextItem *Item;          // Gets the item...
	int      (*Fn)();        // Function associated with this item
	int      Type;           // Type of menu item.
	int      TogOn;          // Current toggle state.
};

// The MenuList class....
class MenuList {
	int  size;               // Number of nodes in list
	int  FGCol;              // Menu text foreground colour
	int  BGCol;              // Menu text background colour
	int  HLFGCol;            // Menu highlight foreground colour
	int  HLBGCol;            // Menu highlight background colour
	char title[40];          // Menu title
	int  titleX, titleY;     // Position of title
	int  MenuX, MenuW;       // Menu x position and width
	int  MenuY, MenuH;       // Menu y position and height
	void     (*InfoFn)();    // Function to display info
	int      HaveInfoFn;     // Do we have an info function?
public:
	MNode *ThisNode;         // The current place in the list
	MenuList(int, int, int, int, int, int, int, int);
	~MenuList();
	void   AddMenuItem(int, char*, int (*func)(), int);
	void   Rewind(int);
	void   Forward(int);
	int    GetSize() {return size;};
	void   TakeInput();
	void   DrawMenu();
	void   SetTitle(char*);
	void   ToggleItem();
	void   ClearMenuArea(int, int, char*);
	void   SetTextScreen();
	void   AddInfo(void (*func)());
};

// Consructor, creates a node and sets up pointers to NULL
MenuList::MenuList(int fg, int bg, int hfg, int hbg,
									 int x, int w, int y, int h) {
	MNode *N;

	ThisNode = NULL;
	size = 0;
	FGCol = fg; BGCol = bg;
	HLFGCol = hfg; HLBGCol = hbg;
	MenuX = x; MenuW = w;
	MenuY = y; MenuH = h;
	HaveInfoFn = MFALSE;
	return;
}

MenuList::~MenuList()
{
	// Go through MenuList and remove all TextItems
	// declared with "new", and the list nodes.
	Forward(size);
	while (ThisNode->Prev!=NULL) {
		delete ThisNode->Item;    // Delete the TextItem at this node.
		Rewind(1);
		delete ThisNode->Next;    // Delete the node just rewound from.
	}
	delete ThisNode->Item;      // Delete the last one in the list.
	delete ThisNode;

	return;
}

// Info function.....
void MenuList::AddInfo(void (*fn)())
{
	InfoFn = fn;
	HaveInfoFn = MTRUE;
	return;
}

// Step back one TextItem in the list.
void MenuList::Rewind(int dist)
{
	if (dist==0) return;
	MNode *Current = ThisNode;         // Get the current pointer
	if (Current->Prev!=NULL)
		ThisNode = Current->Prev;        // Set it to the backwards pointer
	Rewind(--dist);
	return;
}

// Go forward one TextItem in list.
void MenuList::Forward(int dist)
{
	if (dist==0) return;
	MNode *Current = ThisNode;        // Get the current pointer
	if (Current->Next!=NULL)          // If not already at front, then
		ThisNode = Current->Next;       //    set current pointer to the next
	Forward(--dist);
	return;
}

// Add an item to the menu.
// y - horizontal menu posn.
// text - the text printed out for this menu item.
// fn - the function called when <return> pressed on this item.
// type - 0 = normal, 1 = toggle (puts a cross to indicate ON).
void MenuList::AddMenuItem(int y, char *text, int (*fn)(), int type)
{
	int x, mid;

	MNode *N = new MNode;                                  // New node
	mid = MenuX + (MenuW/2);
	x = mid-strlen(text)/2;
	TextItem *TI = new TextItem(x, y, FGCol, BGCol, text); // New TextItem
	MNode *Current = ThisNode;        // Get the current pointer

	N->Item = TI;                     // Setup values for the node.
	N->Next = NULL;
	N->Fn = fn;
	N->Type = type;
	N->TogOn = MFALSE;

	// First item in list
	if (size==0) {
		N->Prev = NULL;
	} else {
		// If not first item, go to front and put new item there
		Forward(size);
		Current->Next = N;
		N->Prev = Current;
	}
	size++;
	ThisNode = N;
	return;
}

// Take input from user, and act accordingly
void MenuList::TakeInput()
{
	char c;
	int  res=0;                        // Result from functions.

	Rewind(size);                      // Goto start
	MNode *Current = ThisNode;         // Get the current pointer
	Current->Item->HighLight(HLFGCol, HLBGCol);

	// Go round in a loop now taking input....
	while (res!=QUIT) {
		c=getch();
		switch (c) {
		// <RET> - do the function...
		case 13:
			if (res=Current->Fn()) break;
			if (Current->Type==MENU_TOGG) ToggleItem();
			else {
				SetTextScreen();
				DrawMenu();
			}
			break;
		// Up arrow - move up one in the menu
		case 72:
			Current->Item->UnHighLight();
			if (Current->Prev!=NULL) {
				ThisNode = Current->Prev;
				Current = ThisNode;
			} else {
				Forward(size);
				Current = ThisNode;
			}
			Current->Item->HighLight(HLFGCol, HLBGCol);
			break;
		// Down arrow - move down one
		case 80:
			Current->Item->UnHighLight();
			if (Current->Next!=NULL) {
				ThisNode = Current->Next;
				Current = ThisNode;
			} else {
				Rewind(size);
				Current = ThisNode;
			}
			Current->Item->HighLight(HLFGCol, HLBGCol);
			break;
		// Display info
		case 'i': case 'I':
			if (HaveInfoFn) {
				InfoFn();
				SetTextScreen();
				DrawMenu();
			}
			break;
		default:
			break;
		}
	}
	return;
};

// Draws the menu options....
void MenuList::DrawMenu()
{
	int i;

	// Draw title
	ClearMenuArea(titleX, titleY, title);

	// Draw the whole menu, and at end ThisNode points to the
	// current one still.
	for (i=0; i<size; i++) {
		if (ThisNode->Next==NULL)
			Rewind(size);
		else
			Forward(1);
		ThisNode->Item->UnHighLight();
	}
	// Draw the current one highlighted
	ThisNode->Item->HighLight(HLFGCol, HLBGCol);

	return;
}

// Set the menu title
void MenuList::SetTitle(char *iptitle)
{
	if (strlen(iptitle)>TSIZE) {
		strncpy(title, iptitle, TSIZE);
		title[TSIZE] = '\0';
	} else
		strcpy(title, iptitle);

	titleX = 1+(MenuW-strlen(iptitle))/2;
	titleY = 1;

	return;
};

// Toggle the current menu item output ie. put a cross in it.
void MenuList::ToggleItem()
{
	if (!ThisNode->TogOn) {
		ThisNode->Item->SetMark("X");
		ThisNode->TogOn = MTRUE;
	}
	else {
		ThisNode->Item->SetMark(" ");
		ThisNode->TogOn = MFALSE;
	}

	return;
}

// Clear the menu area on the text screen, when change menus etc.
void MenuList::ClearMenuArea(int x, int y, char* title)
{
	window(1,1,79,25);   // Set coords to full window
	textcolor(WHITE);
	textbackground(BLUE);
	MenuShaddowBox(MenuX,4,MenuX+MenuW,15,BLUE,BLACK);
	Printxy(x, y, title);                // Put up the menu title
	window(MenuX+1,5,MenuX+MenuW-1,14);  // Set the text window to be cleared
	clrscr();                            // Clear the window

	// Draw the help info
	window(51,5,69,14);  // Set the text window to be cleared
	gotoxy(4,2); putch(0x18); Printxy(7,2,"- Move Up");
	gotoxy(4,4); putch(0x19); Printxy(7,4,"- Move Down");
	gotoxy(4,6); putch(0x11); putch(0xbe); Printxy(7,6,"- Do It!");
	if (HaveInfoFn) Printxy(4,8,"i  - Info.");
	else Printxy(4,8,"           ");

	// Put defaults back to message window
	textbackground(LIGHTGRAY);
	textcolor(BLACK);
	window(5,20,76,23);

	return;
};

void MenuList::SetTextScreen()
{
	if (!MenuDel) return; // Don't refresh menu if not needed

	textmode(C80);

	// Set up the initial menu.
	window(1,1,80,26);
	textcolor(WHITE);
	textbackground(GREEN);
	clrscr();
	TextBox(1,1,80,25);
	char revstr[] = "$Revision: 1.0 $";
	char tempstr[12];
	int minr, majr;
	sscanf(revstr, "%s %d.%d", tempstr, &majr, &minr);
	char outstr[40];
	sprintf(outstr, " PC IR Remote Control - V %d.%d ", majr, minr);
	Printxy(25, 1, outstr);
	_setcursortype(_NOCURSOR);    // Turn cursor off

	// Add help window
	MenuShaddowBox(50,4,70,15,BLUE,BLACK);
	window(51,5,69,14);  // Set the text window to be cleared
	clrscr();            // Clear the window

	// Put defaults back to message window
	textbackground(LIGHTGRAY);
	textcolor(BLACK);
	window(5,20,76,23);

	// Set up message window
	MenuShaddowBox(4,18,77,23,LIGHTGRAY,BLACK);
	clrscr();
	textcolor(BLACK);
	TextBox(1,1,74,6);
	Printxy(29, 1, " Message Window ");
	textcolor(BLACK);
	window(5,19,76,22);
	MenuDel = MFALSE;

	return;
};

// -------------------------------------------------------------
// Misc. other routines used by menu stuff
// -------------------------------------------------------------

// Draw a double lined box in text mode
// First coords passed are the top left corner, second two
// are the bottom right.
void TextBox(int x1, int y1, int x2, int y2)
{
	int i;

	// Have a look at sent coords
	if (x1>x2) {
		i=x2; x2=x1; x1=i;   // Set x1 as the lowest X val
	}
	if (y1>y2) {
		i=y2; y2=y1; y1=i;   // Set y1 as the lowest Y val
	}
	// Put in corners
	gotoxy(x2, y2);	putch(188);
	gotoxy(x1, y1);	putch(201);
	gotoxy(x1, y2);	putch(200);
	gotoxy(x2, y1);	putch(187);
	// Draw horizontal lines
	for (i=x2-1; i>x1; i--) {
		gotoxy(i, y1); putch(205);
		gotoxy(i, y2); putch(205);
	}
	// Draw vertical lines
	for (i=y2-1; i>y1; i--) {
		gotoxy(x1, i); putch(186);
		gotoxy(x2, i); putch(186);
	}
	return;
}

// Draws a box in the specified colour, with a shadow
// a, b, c, d - box size/position (not including shaddow)
// Col, ShadCol - Colours of box and shaddow resp.
void MenuShaddowBox(int a, int b, int c, int d, int Col, int ShadCol)
{
	window(a+1,b+1,c+1,d+1);
	textbackground(ShadCol);
	clrscr();
	window(a,b,c,d);
	textbackground(Col);
	clrscr();

	return;
}


// Prints string at given position
void Printxy(int x, int y, char* str)
{
	gotoxy(x, y);
	cprintf(str);
	return;
}

// Function sent to Menu that is called when program exit required.
// Sets back defaults etc.
int QuitProgram()
{
	textcolor(WHITE);
	textbackground(BLACK);
	window(1,1,80,25);
	clrscr;
	_setcursortype(_NORMALCURSOR);    // Turn cursor back on
	return QUIT;
};

// Function sent to Menu that is called when exit required.
int QuitMenu()
{
	clrscr;
	return QUIT;
};
