/* pc_term.c */

/* This file is part of the MSDOS-DJGPP-port of the GNU standalone
   Info-reader

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <pc.h>
#include <conio.h>
#include "pc_term.h"
#include "terminal.h"
#include "termdep.h"

#define max(x,y) ((x)>(y) ? (x) : (y))
#define min(x,y) ((x)<(y) ? (x) : (y))
#define keywait  do {char dummy=0;\
                     fflush(stdout);\
                     while (!kbhit()) ;\
                     dummy=getkey();\
                 } while (0);

static struct text_info pc_info;    /* struct to hold the conio-status */
static struct text_info norm_info;  /* holds screen params outside Info */
void _set_screen_lines(int);
static unsigned char    norm_attr, inv_attr;

/* Put the screen in the video mode and colors which Info will use.  */
void pc_prep_terminal()
{
  textattr(norm_attr);

  /* Do not set screen height if we already have it, because
     doing so erases the screen.  */
  if (screenheight != ScreenRows())
    _set_screen_lines(screenheight);
}

void pc_unprep_terminal()
{
  /* Do not set screen height if we already have it, because
     doing so erases the screen.  */
  if (norm_info.screenheight != ScreenRows())
  {
    _set_screen_lines(norm_info.screenheight);
    textmode(LASTMODE);
  }
  textattr(norm_info.normattr);
  ScreenClear(); /* to leave behind a clean screen */
}

void pc_begin_inverse()
{
  textattr(inv_attr);
}

void pc_end_inverse()
{
  textattr(norm_attr);
}

void pc_up_line()
{
  int x, y;
  ScreenGetCursor(&y, &x);
  ScreenSetCursor(max(y-1, 0), x);
}

void pc_down_line()
{
  int x, y;
  ScreenGetCursor(&y, &x);
  ScreenSetCursor(min(screenheight-1, y+1), x);
}

void pc_clear_screen()
{
  ScreenClear();
}

void pc_clear_to_eol()
{
  clreol(); /* perhaps to be replaced by a loop */
}

void pc_get_screen_size()
{
  screenwidth  = ScreenCols();
  screenheight = ScreenRows();
}

void pc_goto_xy(x, y)
int x, y;
{
  ScreenSetCursor(y, x); /* yes, pc.h says ...(row, column) !! */
}

void pc_initialize_terminal(term_name)
    char *term_name;
{
  char *info_colors;

  /* Get current video information, to be restored later.  */
  if (norm_info.screenwidth == 0)
    {
      gppconio_init();
      gettextinfo(&norm_info);
    }

  /* Current screen dimensions and colors are the default.  */
  screenwidth  = norm_info.screenwidth;
  screenheight = norm_info.screenheight;
  norm_attr    = norm_info.normattr;
  inv_attr     = (((norm_info.normattr &    7) << 4) |
                  ((norm_info.normattr & 0x7f) >> 4));

  /* Environment variables "LINES" overrides the default.  */
  if (getenv("LINES") != NULL)
    screenheight = atoi(getenv("LINES"));

  /* Environment variable "INFO_LINES" overrides "LINES".  */
  if (getenv("INFO_LINES") != NULL)
      screenheight = atoi(getenv("INFO_LINES"));

  if (screenheight != norm_info.screenheight)
    {
      _set_screen_lines(screenheight);
      screenwidth  = ScreenCols();
      screenheight = ScreenRows();
    }

  /* Does the user want non-default colors?  */
  info_colors = getenv("INFO_COLORS");
  if (info_colors != (char *)0)
    {
      char *next;
      unsigned long coldesc = strtoul(info_colors, &next, 0);

      if (next == info_colors || coldesc > UCHAR_MAX)
        info_colors = (char *)0;
      else
        {
          char *endp;
          norm_attr = (unsigned char)coldesc;
          coldesc = strtoul(next + 1, &endp, 0);
          if (endp == next + 1 || coldesc > UCHAR_MAX)
            info_colors = (char *)0;
          else
            inv_attr = (unsigned char)coldesc;
        }
    }

  /* Try setting the colors user asked for.  */
  textattr(norm_attr);
  ScreenClear();

  /* We know how to produce a visible bell.  */
  terminal_has_visible_bell_p = 1;

  /* Set all the hooks to our PC-specific functions.  */
  terminal_begin_inverse_hook       = pc_begin_inverse;
  terminal_end_inverse_hook         = pc_end_inverse;
  terminal_prep_terminal_hook       = pc_prep_terminal;
  terminal_unprep_terminal_hook     = pc_unprep_terminal;
  terminal_up_line_hook             = pc_up_line;
  terminal_down_line_hook           = pc_down_line;
  terminal_clear_screen_hook        = pc_clear_screen;
  terminal_clear_to_eol_hook        = pc_clear_to_eol;
  terminal_get_screen_size_hook     = pc_get_screen_size;
  terminal_goto_xy_hook             = pc_goto_xy;
  terminal_new_terminal_hook        = pc_new_terminal;
  terminal_put_text_hook            = pc_put_text;
  terminal_ring_bell_hook           = pc_ring_bell;
  terminal_write_chars_hook         = pc_write_chars;
  terminal_scroll_terminal_hook     = pc_scroll_terminal;
}

void pc_new_terminal(term_name)
    char *term_name;
{
  pc_initialize_terminal(term_name);
}

void pc_put_text(string)
char *string;
{
  cputs(string);
}

void pc_ring_bell()
{
  if (terminal_has_visible_bell_p && terminal_use_visible_bell_p)
    ScreenVisualBell();
  else
    printf("%c",'\a');
}

void pc_write_chars(string, nchars)
    char *string;
    int nchars;
{
  cprintf("%.*s",nchars, string);
}

void pc_scroll_terminal(start, end, amount)
    int start, end, amount;
{
  movetext(pc_info.winleft, start, pc_info.winright, end,
           pc_info.winleft, start+amount);
}

int       tputs(char *a, int b, int (*c)())
{
  perror("tputs");
}

char*     tgoto(char*a, int b, int c)
{
  perror("tgoto");
}

int       tgetnum(char*a)
{
  perror("tgetnum");
}

int       tgetflag(char*a)
{
  perror("tgetflag");
}

char*     tgetstr(char *a, char **b)
{
  perror("tgetstr");
}

int       tgetent(char*a, char*b)
{
  perror("tgetent");
}

int       sigblock(int a)
{
  return 0;
}

