/* Copyright (C) 1996,1997 Robert Hhne, see COPYING.RH for details */
/* This file is part of RHIDE. */
#include <libtvuti.h>

#define Uses_ideFunctions
#define Uses_ideEnums
#include <libide.h>

#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>

#include <dos.h>

extern "C" time_t _file_time_stamp(unsigned int);

long time_of_file(const char * fname)
{
  long retval;
#ifdef USE_STAT
  struct stat s;
  if (stat(fname,&s) == 0) retval = s.st_atime;
  else retval = 0;
#else
  struct ftime ft;
  int handle;
  handle = open(fname,O_RDONLY);
  if (handle < 0) retval = 0;
  else
  {
    getftime(handle,&ft);
    retval = (long)_file_time_stamp(*((unsigned *)(&ft)));
    close(handle);
  }
#endif
  return retval;
}

FILE_TYPE get_file_type(const char * _ext)
{
  if (!_ext) return FILE_UNKNOWN;
  char *ext = strrchr(_ext,'.');
  if (!ext)
  {
    if (!*_ext) return FILE_UNKNOWN;
    return FILE_COFF;
  }
  if (!strcmp(ext,".c")) return FILE_C_SOURCE;
  if (!strcmp(ext,".i")) return FILE_C_SOURCE_I;
  if (!strcmp(ext,".cc")) return FILE_CC_SOURCE;
  if (!strcmp(ext,".cpp")) return FILE_CC_SOURCE;
  if (!strcmp(ext,".cxx")) return FILE_CC_SOURCE;
  if (!strcmp(ext,".ii")) return FILE_CC_SOURCE_II;
  if (!strcmp(ext,".C")) return FILE_CC_SOURCE; // note: capital C
  if (!strcmp(ext,".o")) return FILE_OBJECT;
  if (!strcmp(ext,".s")) return FILE_ASM_SOURCE;
  if (!strcmp(ext,".S")) return FILE_ASM_SOURCE;
  if (!strcmp(ext,".l")) return FILE_FLEX_SOURCE;
  if (!strcmp(ext,".y")) return FILE_BISON_SOURCE;
  if (!strcmp(ext,".a")) return FILE_LIBRARY;
  if (!strcmp(ext,".exe")) return FILE_EXE;
  if (!strcmp(ext,".h")) return FILE_HEADER;
  if (!strcmp(ext,".hpp")) return FILE_HEADER;
  if (!strcmp(ext,".m")) return FILE_OBJC_SOURCE;
  if (!stricmp(ext,".gpr")) return FILE_PROJECT; // ignore case
  if (!strcmp(ext,".p")) return FILE_PASCAL_SOURCE;
  if (!strcmp(ext,".pas")) return FILE_PASCAL_SOURCE;
  if (!strcmp(ext,".")) return FILE_COFF;
  if (!*ext) return FILE_COFF;
  return FILE_UNKNOWN;
}

COMPILE_ID how_to_compile(FILE_TYPE from,FILE_TYPE to)
{
  if (from == to) return COMPILE_NONE;
  switch (from)
  {
    case FILE_PROJECT:
      switch (to)
      {
        case FILE_LIBRARY:
          return COMPILE_PROJECT;
        default:
          return COMPILE_UNKNOWN;
      }
    case FILE_FLEX_SOURCE:
      switch (to)
      {
        case FILE_OBJECT:
          return COMPILE_FLEX;
        case FILE_C_SOURCE:
          return COMPILE_FLEX_TO_C;
        case FILE_CC_SOURCE:
          return COMPILE_FLEX_TO_CC;
        default:
          return COMPILE_UNKNOWN;
      }
    case FILE_BISON_SOURCE:
      switch (to)
      {
        case FILE_OBJECT:
          return COMPILE_BISON;
        case FILE_C_SOURCE:
          return COMPILE_BISON_TO_C;
        case FILE_CC_SOURCE:
          return COMPILE_BISON_TO_CC;
        default:
          return COMPILE_UNKNOWN;
      }
    case FILE_OBJC_SOURCE:
      switch (to)
      {
        case FILE_OBJECT:
          return COMPILE_OBJC;
        default:
          return COMPILE_UNKNOWN;
      }
    case FILE_CC_SOURCE:
    case FILE_CC_SOURCE_II:
      switch (to)
      {
	case FILE_OBJECT:
	  return COMPILE_CC;
        case FILE_ASM_SOURCE:
          return COMPILE_CC_LIKE;
	default:
	  return COMPILE_UNKNOWN;
      }
    case FILE_C_SOURCE:
    case FILE_C_SOURCE_I:
      switch (to)
      {
	case FILE_OBJECT:
	  return COMPILE_C;
        case FILE_ASM_SOURCE:
          return COMPILE_C_LIKE;
	default:
	  return COMPILE_UNKNOWN;
      }
    case FILE_ASM_SOURCE:
      switch (to)
      {
	case FILE_OBJECT:
	  return COMPILE_ASM;
	default:
	  return COMPILE_UNKNOWN;
      }
    case FILE_PASCAL_SOURCE:
      switch (to)
      {
	case FILE_OBJECT:
	  return COMPILE_PASCAL;
        case FILE_COFF:
        case FILE_EXE:
          return COMPILE_LINK_PASCAL_AUTOMAKE;
          break;
	default:
	  return COMPILE_UNKNOWN;
      }
    case FILE_UNKNOWN:
    case FILE_OBJECT:
      switch (to)
      {
	case FILE_COFF:
	case FILE_EXE:
	  return COMPILE_LINK;
	case FILE_LIBRARY:
	  return COMPILE_ARCHIVE;
	default:
	  return COMPILE_UNKNOWN;
      }
    default:
      return COMPILE_UNKNOWN;
  }
}

COMPILE_ID how_to_compile(const char * ext_from,const char * ext_to)
{
  FILE_TYPE from=get_file_type(ext_from),to=get_file_type(ext_to);
  return how_to_compile(from,to);
}
