/* Copyright (C) 1996,1997 Robert Hhne, see COPYING.RH for details */
/* This file is part of RHIDE. */
#define Uses_TProject
#define Uses_TOptions
#define Uses_TDependency
#define Uses_TDepCollection
#define Uses_TFlagCollection

#include <rhide.h>
#include <rhidehis.h>

#define Uses_ideFunctions
#include <libide.h>

#define Uses_TParamList
#define Uses_TDirList
#define Uses_tvutilFunctions
#include <libtvuti.h>

#include <string.h>
#include <stdlib.h>
#include <dir.h>
#include <stdio.h>

#if 1
#define A(n) __attribute__ (( regparm(n) ))
#else
#define A(n)
#endif

static char *default_variables[] = {
 "RHIDE_GCC",
 "gcc",

 "RHIDE_AS",
 "gcc",

 "RHIDE_GXX",
 "gcc",

 "RHIDE_GPC",
 "gpc",

 "RHIDE_AR",
 "ar",

 "RHIDE_LD",
 "gcc",

 "RHIDE_LD_PASCAL",
 "gpc",

 "RHIDE_ARFLAGS",
 "rcs",

 "RHIDE_INCLUDES",
 "$(SPECIAL_CFLAGS) $(addprefix -I,$(INCLUDE_DIRS))",

 "RHIDE_LIBDIRS",
 "$(addprefix -L,$(LIB_DIRS))",

 "RHIDE_LIBS",
 "$(addprefix -l,$(LIBS))",

 "RHIDE_LDFLAGS",
 "$(SPECIAL_LDFLAGS) $(addprefix -Xlinker ,$(LD_EXTRA_FLAGS))",

 "RHIDE_COMPILE_C",
 "$(RHIDE_GCC) $(RHIDE_INCLUDES) $(C_DEBUG_FLAGS) $(C_OPT_FLAGS)\
  $(C_WARN_FLAGS) $(C_C_LANG_FLAGS) $(C_EXTRA_FLAGS) $(LOCAL_OPT)\
  -c $(SOURCE_NAME) -o $(OUTFILE)",

 "RHIDE_COMPILE_C_FORCE",
 "$(RHIDE_GCC) $(RHIDE_INCLUDES) $(C_DEBUG_FLAGS) $(C_OPT_FLAGS)\
  $(C_WARN_FLAGS) $(C_C_LANG_FLAGS) $(C_EXTRA_FLAGS) -x c $(LOCAL_OPT)\
  -c $(SOURCE_NAME) -o $(OUTFILE)",

 "RHIDE_COMPILE_CC",
 "$(RHIDE_GXX) $(RHIDE_INCLUDES) $(C_DEBUG_FLAGS) $(C_OPT_FLAGS)\
  $(C_WARN_FLAGS) $(C_CXX_LANG_FLAGS) $(C_EXTRA_FLAGS) $(LOCAL_OPT)\
  -c $(SOURCE_NAME) -o $(OUTFILE)",

 "RHIDE_COMPILE_CC_FORCE",
 "$(RHIDE_GXX) $(RHIDE_INCLUDES) $(C_DEBUG_FLAGS) $(C_OPT_FLAGS)\
  $(C_WARN_FLAGS) $(C_CXX_LANG_FLAGS) $(C_EXTRA_FLAGS) -x c++ $(LOCAL_OPT)\
  -c $(SOURCE_NAME) -o $(OUTFILE)",

 "RHIDE_COMPILE_ASM",
 "$(RHIDE_AS) $(RHIDE_INCLUDES) $(C_DEBUG_FLAGS) $(C_OPT_FLAGS)\
  $(C_WARN_FLAGS) $(C_EXTRA_FLAGS) $(LOCAL_OPT)\
  -c $(SOURCE_NAME) -o $(OUTFILE)",

 "RHIDE_COMPILE_ASM_FORCE",
 "$(RHIDE_AS) $(RHIDE_INCLUDES) $(C_DEBUG_FLAGS) $(C_OPT_FLAGS)\
  $(C_WARN_FLAGS) $(C_EXTRA_FLAGS) -x assembler $(LOCAL_OPT)\
  -c $(SOURCE_NAME) -o $(OUTFILE)",

 "RHIDE_GPC_FLAGS",
 "$(RHIDE_INCLUDES) $(C_DEBUG_FLAGS) $(C_OPT_FLAGS) $(C_WARN_FLAGS) \
  $(C_P_LANG_FLAGS) $(C_EXTRA_FLAGS)",

 "RHIDE_COMPILE_PASCAL",
 "$(RHIDE_GPC) $(RHIDE_GPC_FLAGS) $(LOCAL_OPT)\
  -c $(SOURCE_NAME) -o $(OUTFILE)",

 "RHIDE_COMPILE_PASCAL_FORCE",
 "$(RHIDE_GPC) $(RHIDE_GPC_FLAGS) -x pascal $(LOCAL_OPT)\
  -c $(SOURCE_NAME) -o $(OUTFILE)",

 "RHIDE_COMPILE_LINK",
 "$(RHIDE_LD) $(RHIDE_LIBDIRS) $(C_EXTRA_FLAGS) $(RHIDE_LDFLAGS) -o $(OUTFILE)\
  $(OBJFILES) $(LIBRARIES) $(RHIDE_LIBS)",

 "RHIDE_COMPILE_LINK_PASCAL",
 "$(RHIDE_LD_PASCAL) $(RHIDE_LIBDIRS) $(C_EXTRA_FLAGS) $(RHIDE_LDFLAGS) \
  -o $(OUTFILE) $(OBJFILES) $(LIBRARIES) $(RHIDE_LIBS)",

 "RHIDE_COMPILE_LINK_PASCAL_AUTOMAKE",
 "$(RHIDE_LD_PASCAL) $(RHIDE_LIBDIRS) $(RHIDE_LDFLAGS) \
  -o $(OUTFILE) --automake=\"$(strip $(RHIDE_GPC_FLAGS))\" $(RHIDE_GPC_FLAGS) \
  $(SOURCE_NAME) $(LIBRARIES) $(RHIDE_LIBS)",

 "RHIDE_COMPILE_ARCHIVE",
 "$(RHIDE_AR) $(RHIDE_ARFLAGS) $(OUTFILE) $(OBJFILES)",

 "RHIDE_COMPILE.cc.s",
 "$(subst -c,-S,$(RHIDE_COMPILE_CC))",

 "RHIDE_COMPILE.cpp.s",
 "$(RHIDE_COMPILE.cc.s)",

 "RHIDE_COMPILE.cxx.s",
 "$(RHIDE_COMPILE.cc.s)",

 "RHIDE_COMPILE.C.s",
 "$(RHIDE_COMPILE.cc.s)",

 "RHIDE_COMPILE.ii.s",
 "$(RHIDE_COMPILE.cc.s)",

 "RHIDE_COMPILE.c.s",
 "$(subst -c,-S,$(RHIDE_COMPILE_C))",

 "RHIDE_COMPILE.i.s",
 "$(RHIDE_COMPILE.c.s)",

 "RHIDE_FSDB",
 "fsdb $(OUTFILE) $(addprefix -p ,$(SRC_DIRS)) $(PROG_ARGS)",

 "RHIDE_GDB",
 "gdb $(OUTFILE) $(addprefix -d ,$(SRC_DIRS))",

 "RHIDE_GREP",
 "grep -n $(prompt "__("arguments for GREP")",$(WUC))",

 "RHIDE_GPROF",
 "gprof $(OUTFILE)",

 0,
 0
};

static char **vars;
static int var_count = 0;

static
void add_variable(char *variable,char *contents)
{
  var_count++;
  vars = (char **)xrealloc(vars,var_count*2*sizeof(char *));
  string_dup(vars[var_count*2-2],variable);
  string_dup(vars[var_count*2-1],contents);
}

static
void insert_variable(char *variable,char *contents)
{
  int i;
  for (i=0;i<var_count;i++)
  {
    if (strcmp(vars[i*2],variable) == 0)
    {
      string_free(vars[2*i+1]);
      string_dup(vars[2*i+1],contents);
      return;
    }
  }
  add_variable(variable,contents);
}

static
void put_breakline(FILE *f,int start_len,int max_len,char *s)
{
  int len;
  while (1)
  {
    len = strlen(s);
    if (len+start_len <= max_len)
    {
      fprintf(f,"%s\n",s);
      return;
    }
    char *tmp;
    tmp = s + max_len-start_len;
    while (tmp > s && *tmp != ' ') tmp--;
    if (tmp == s)
    {
      fprintf(f,"%s\n",s);
      return;
    }
    *tmp = 0;
    fprintf(f,"%s\\\n\t",s);
    *tmp = ' ';
    start_len = 8; // tabsize
    s = tmp+1;
  }
}

static
const char * GetVariable(const char *variable)
{
  int i;
  for (i=0;i<var_count;i++)
  {
    if (strcmp(variable,vars[i*2]) == 0)
      return vars[i*2+1];
  }
  return getenv(variable);
}

extern char **environ;

static __attribute__ ((__constructor__))
void init_variables(void)
{
  char *variable,*contents;
  int i=0;
  while (default_variables[i])
  {
    variable = default_variables[i];
    contents = getenv(variable);
    if (!contents) contents = default_variables[i+1];
    add_variable(variable,contents);
    i += 2;
  }
  // Now check the env for any RHIDE_ variable
  for (i=0;environ[i];i++)
  {
    if (strncmp(environ[i],"RHIDE_",6) == 0)
    {
      contents = strchr(environ[i],'=');
      if (!contents) continue;
      contents++;
      char var[256];
      memcpy(var,environ[i],(int)(contents-environ[i])-1);
      var[(int)(contents-environ[i])-1] = 0;
      insert_variable(var,contents);
    }
  }
}

typedef A(1) char *(*string_function)(char *);

typedef struct
{
  char *name;
  int name_len;
  string_function function;
} string_function_rec;

static A(1)
char *string_function_strip(char *);
static A(1)
char *string_function_addsuffix(char *);
static A(1)
char *string_function_addprefix(char *);
static A(1)
char *string_function_notdir(char *);
static A(1)
char *string_function_dir(char *);
static A(1)
char *string_function_subst(char *);
static A(1)
char *string_function_prompt(char *);

static string_function_rec string_function_list[] =
{
#define SF(x) {#x,sizeof(#x)-1,string_function_##x}
  SF(strip),
  SF(addsuffix),
  SF(addprefix),
  SF(notdir),
  SF(dir),
  SF(subst),
  SF(prompt),
  {NULL,NULL}
#undef SF
};

static A(1)
char *find_close_brace(char *open_brace)
{
  int brace_count = 1;
  open_brace++;
  while (brace_count > 0 && *open_brace)
  {
    if (*open_brace == ')') brace_count--;
    else if (*open_brace == '(') brace_count++;
    open_brace++;
  }
  if (brace_count != 0) return NULL;
  return open_brace-1;
}

static A(1)
char *find_next_comma(char *arg)
{
  while (*arg)
  {
    if (*arg == ',') return arg;
    if (*arg == '(')
    {
      arg = find_close_brace(arg+1);
      if (!arg) return NULL;
    }
    arg++;
  }
  return NULL;
}

static
char * (*external_expand_token)(char *,char *(*)(char *));

static A(1)
char *expand_tokens(char *tokens);

static A(1)
char *expand_variable(char *token)
{
  char *end = token + strlen(token) - 1;
  const char *variable;
  char *retval;
  *end = 0;
  variable = GetVariable(token+2);
  *end = ')';
  if (!variable) return NULL;
  string_dup(end,variable);
  retval = expand_tokens(end);
  string_free(end);
  return retval;
}

static A(1)
char *check_for_string_function(char *token)
{
  int i=0;
  while (string_function_list[i].name)
  {
    if (strncmp(token+2,string_function_list[i].name,
                string_function_list[i].name_len) == 0)
    {
      char *start,*end,*expanded;
      start = token + 2 + string_function_list[i].name_len + 1;
      end = token + strlen(token)-1;
      *end = 0;
      expanded = expand_tokens(start);
      *end = ')';
      end = string_function_list[i].function(expanded);
      string_free(expanded);
      return end;
    }
    i++;
  }
  return NULL;
}

static A(1)
char *expand_token(char *token)
{
  char *retval = check_for_string_function(token);
  if (retval) return retval;
  retval = expand_variable(token);
  if (retval) return retval;
  if (external_expand_token)
    retval = external_expand_token(token,expand_tokens);
  if (retval) return retval;
  return strdup(token);
}

static A(1)
char *expand_tokens(char *tokens)
{
  char *start;
  char *end,c;
  char *_tokens=NULL;
  char *expanded;
  start = tokens;
  while (*start)
  {
    if (start[0] == '$' && start[1] == '(')
    {
      *start = 0;
      string_cat(_tokens,tokens);
      *start = '$';
      end = find_close_brace(start+1);
      if (!end)
      {
        string_cat(_tokens,start);
        return _tokens;
      }
      end++;
      c = *end;
      *end = 0;
      expanded = expand_token(start);
      string_cat(_tokens,expanded);
      string_free(expanded);
      *end = c;
      start = tokens = end;
    }
    else start++;
  }
  string_cat(_tokens,tokens);
  return _tokens;
}

char *expand_spec(char *spec,char * (*ext_func)(char *,char *(*)(char *)))
{
  char *retval;
  external_expand_token = ext_func;
  retval = expand_tokens(spec);
  external_expand_token = NULL;
  return retval;
}

char *build_command(char *program,char *args)
{
  char *spec;
  char *command;
  string_dup(spec,"$(strip ");
  string_cat(spec,program);
  string_cat(spec," ");
  string_cat(spec,args);
  string_cat(spec,")");
  command = expand_spec(spec,NULL);
  string_free(spec);
  return command;
}

static A(1)
char *string_function_prompt(char *arg)
{
  char *init_val;
  char *retval;
  init_val = find_next_comma(arg);
  if (!init_val)
    retval = expand_tokens(arg);
  else
  {
    *init_val = 0;
    retval = expand_tokens(arg);
    *init_val = ',';
    init_val = expand_tokens(init_val+1);
  }
  TParamList *params = new TParamList();
  if (init_val)
    params->insert(init_val);
  if (editParamList(params,_(retval),
      RHIDE_History_prompt) != cmOK)
  {
    destroy(params);
    string_free(retval);
    string_dup(retval,"");
    return retval;
  }
  string_free(retval);
  params->ToString(retval);
  destroy(params);
  return retval;
}

static A(1)
char *string_function_strip(char *arg)
{
  char *retval;
  char *tmp;
  char quote_char = '\'';
  int in_quote = 0;
  retval = expand_tokens(arg);
  tmp = retval;
  while (*tmp && *tmp == ' ') tmp++;
  strcpy(retval,tmp);
  if (!*retval) return retval;
  tmp =retval;
  while (*tmp)
  {
    if (in_quote)
    {
      if (*tmp == quote_char) in_quote = 0;
      tmp++;
      continue;
    }
    if (*tmp == '\'' || *tmp == '"')
    {
      in_quote = 1;
      quote_char = *tmp;
      tmp++;
      continue;
    }
    if (tmp[0] == ' ' && tmp[1] == ' ')
      strcpy(tmp,tmp+1);
    else tmp++;
  }
  tmp = retval + strlen(retval) - 1;
  while (*tmp == ' ') tmp--;
  tmp++;
  *tmp = 0;
  return retval;
}

static A(2)
char *string_function_add_pre_or_suffix(char *arg,int prefix)
{
  char *retval=NULL;
  char *tmp;
  char *arg1,*arg2;
  char *tok;
  tmp = find_next_comma(arg);
  if (!tmp) return NULL;
  *tmp = 0;
  arg1 = expand_tokens(arg);
  *tmp = ',';
  tmp++;
  arg2 = expand_tokens(tmp);
  for (tok = strtok(arg2," ");tok;tok = strtok(NULL," "))
  {
    if (retval) string_cat(retval," ");
    if (prefix)
    {
      string_cat(retval,arg1);
      string_cat(retval,tok);
    }
    else
    {
      string_cat(retval,tok);
      string_cat(retval,arg1);
    }
  }
  string_free(arg1);
  string_free(arg2);
  if (!retval) // arg2 may be consist of only ws's or empty
  {
    string_dup(retval," ");
  }
  return retval;
}

static A(1)
char *string_function_addprefix(char *arg)
{
  return string_function_add_pre_or_suffix(arg,1);
}

static A(1)
char *string_function_addsuffix(char *arg)
{
  return string_function_add_pre_or_suffix(arg,0);
}

static A(1)
char *string_function_notdir(char *arg)
{
  char *retval;
  BaseName(arg,retval);
  return retval;
}

static A(1)
char *string_function_dir(char *arg)
{
  char *dir,*name,*ext;
  split_fname(arg,dir,name,ext);
  string_free(name);
  string_free(ext);
  return dir;
}

static A(1)
char *string_function_subst(char *arg)
{
  char *arg1,*arg2,*arg3;
  char *retval=NULL;
  char *tmp,*temp;
  int l_arg1;
  tmp = find_next_comma(arg);
  if (!tmp) return NULL;
  arg3 = find_next_comma(tmp+1);
  if (!arg3) return NULL;
  *tmp = 0;
  arg1 = expand_tokens(arg);
  l_arg1 = strlen(arg1);
  *tmp = ',';
  tmp++;
  *arg3 = 0;
  arg2 = expand_tokens(tmp);
  *arg3 = ',';
  tmp = arg3+1;
  arg3 = expand_tokens(tmp);
  temp = arg3;
  while (l_arg1 && (tmp = strstr(temp,arg1)) != NULL)
  {
    char c = *tmp;
    *tmp = 0;
    string_cat(retval,temp);
    string_cat(retval,arg2);
    *tmp = c;
    temp = tmp + l_arg1;
  }
  if (*temp) string_cat(retval,temp);
  string_free(arg1);
  string_free(arg2);
  string_free(arg3);
  return retval;
}

static TDependency *_token_dep;

typedef struct
{
  char *name;
  int name_len;
  char *(*func)();
  char *(*make_func)();
} _rhide_tokens;

#define TF(x) static char *rhide_token_##x()
TF(INCLUDE_DIRS);
TF(LIB_DIRS);
TF(C_DEBUG_FLAGS);
TF(C_OPT_FLAGS);
TF(C_WARN_FLAGS);
TF(C_C_LANG_FLAGS);
TF(C_CXX_LANG_FLAGS);
TF(C_P_LANG_FLAGS);
TF(LIBS);
TF(LD_EXTRA_FLAGS);
TF(C_EXTRA_FLAGS);
TF(LOCAL_OPT);
TF(OBJFILES);
TF(LIBRARIES);
TF(SOURCE_NAME);
TF(OUTFILE);
TF(make_SOURCE_NAME);
TF(make_OUTFILE);
TF(make_LOCAL_OPT);
TF(SPECIAL_CFLAGS);
TF(SPECIAL_LDFLAGS);
TF(PROG_ARGS);
TF(SRC_DIRS);
TF(WUC);
TF(make_WUC);

static _rhide_tokens rhide_tokens[] =
{
#define SF(x,y) {"$("#x")",sizeof("$("#x")")-1,rhide_token_##x,y}
  SF(INCLUDE_DIRS,NULL),
  SF(LIB_DIRS,NULL),
  SF(C_DEBUG_FLAGS,NULL),
  SF(C_OPT_FLAGS,NULL),
  SF(C_WARN_FLAGS,NULL),
  SF(C_C_LANG_FLAGS,NULL),
  SF(C_CXX_LANG_FLAGS,NULL),
  SF(C_P_LANG_FLAGS,NULL),
  SF(LIBS,NULL),
  SF(LD_EXTRA_FLAGS,NULL),
  SF(C_EXTRA_FLAGS,NULL),
  SF(LOCAL_OPT,rhide_token_make_LOCAL_OPT),
  SF(OBJFILES,NULL),
  SF(LIBRARIES,NULL),
  SF(SOURCE_NAME,rhide_token_make_SOURCE_NAME),
  SF(OUTFILE,rhide_token_make_OUTFILE),
  SF(SPECIAL_CFLAGS,NULL),
  SF(SPECIAL_LDFLAGS,NULL),
  SF(PROG_ARGS,NULL),
  SF(SRC_DIRS,NULL),
  SF(WUC,rhide_token_make_WUC),
  {NULL,0,NULL,NULL}
#undef SF
};

TF(WUC)
{
  char *tmp;
  return (tmp = WUC()) ? tmp : strdup(" ");
}

TF(make_WUC)
{
  return NULL;
}

TF(make_LOCAL_OPT)
{
  return strdup("$(subst ___~~~___, ,$(subst $(notdir $<)___,,"
             "$(filter $(notdir $<)___%,$(LOCAL_OPTIONS))))\n");
}

TF(make_SOURCE_NAME)
{
  return strdup("$<");
}

TF(make_OUTFILE)
{
  return strdup("$@");
}

static A(1)
char *_dirs(TDirList *list)
{
  char *retval;
  list->ToString(retval," ");
  return retval;
}

TF(SRC_DIRS)
{
  return _dirs(Options.SrcDirs);
}

TF(INCLUDE_DIRS)
{
  return _dirs(Options.include_path);
}

TF(LIB_DIRS)
{
  return _dirs(Options.library_path);
}

static A(1)
char *_flags(TFlagCollection *flags)
{
  char *retval;
  flags->ToString(retval," ");
  return retval;
}

TF(C_DEBUG_FLAGS)
{
  return _flags(Options.debug_flags);
}

TF(C_OPT_FLAGS)
{
  return _flags(Options.opt_flags);
}

TF(C_WARN_FLAGS)
{
  return _flags(Options.warn_flags);
}

TF(C_C_LANG_FLAGS)
{
  return _flags(Options.c_flags);
}

TF(C_CXX_LANG_FLAGS)
{
  return _flags(Options.cxx_flags);
}

TF(C_P_LANG_FLAGS)
{
  return _flags(Options.pascal_flags);
}

static A(1)
char *_params(TParamList *params)
{
  char *retval;
  params->ToString(retval);
  return retval;
}

TF(PROG_ARGS)
{
  return _params(Options.ProgArgs);
}

TF(LD_EXTRA_FLAGS)
{
  return _params(Options.link_opt);
}

TF(SPECIAL_LDFLAGS)
{
  char *retval = NULL;
  if (NoStdLib)
  {
    string_dup(retval,"-nostdlib");
  }
  if (ForProfile)
  {
    if (retval) string_cat(retval," ");
    string_cat(retval,"-pg");
  }
  return retval;
}

TF(SPECIAL_CFLAGS)
{
  char *retval = NULL;
  if (NoStdInc)
  {
    if (retval) string_cat(retval," ");
    string_cat(retval,"-nostdinc");
  }
  return retval;
}

TF(C_EXTRA_FLAGS)
{
  return _params(Options.comp_opt);
}

TF(LOCAL_OPT)
{
  return _params(_token_dep->local_options);
}

TF(SOURCE_NAME)
{
  char *retval;
  FindFile(_token_dep->source_name,retval);
  AbsToRelPath(project_directory,retval);
  return retval;
}

TF(OUTFILE)
{
  char *retval;
  FindFile(_token_dep->dest_name,retval);
  AbsToRelPath(project_directory,retval);
  return retval;
}

TF(LIBRARIES)
{
  char *retval = NULL;
  char *tmp;
  int i,count=0;
  if (_token_dep->dependencies)
    count = _token_dep->dependencies->getCount();
  for (i=0;i<count;i++)
  {
    TDependency *dep = (TDependency *)_token_dep->dependencies->at(i);
    if (dep->dest_file_type != FILE_LIBRARY
        && dep->dest_file_type != FILE_PROJECT) continue;
    if (dep->exclude_from_link) continue;
    if (dep->dest_file_type == FILE_LIBRARY)
    {
      FindFile(dep->dest_name,tmp);
      AbsToRelPath(project_directory,tmp);
      if (retval) string_cat(retval," ");
      string_cat(retval,tmp);
      string_free(tmp);
    }
    else
    {
      string_dup(tmp,dep->dest_name);
      AbsToRelPath(project_directory,tmp);
      if (retval) string_cat(retval," ");
      string_cat(retval,tmp);
      string_free(tmp);
    }
  }
  return retval;
}

TF(OBJFILES)
{
  char *retval=NULL,*tmp;
  int i,count=0;
  if (_token_dep->dependencies)
    count = _token_dep->dependencies->getCount();
  for (i=0;i<count;i++)
  {
    TDependency *dep = (TDependency *)_token_dep->dependencies->at(i);
    if (dep->dest_file_type != FILE_OBJECT &&
        dep->dest_file_type != FILE_UNKNOWN) // unknown files also
      continue;
    if (dep->exclude_from_link) continue;
    FindFile(dep->dest_name,tmp);
    AbsToRelPath(project_directory,tmp);
    if (retval) string_cat(retval," ");
    string_cat(retval,tmp);
    string_free(tmp);
  }
  return retval;
}

TF(LIBS)
{
  char *retval;
  AddLibraries(retval);
  return retval;
}

static char *_handle_rhide_token(char *token,char *(*expand_tokens)(char *))
{
  int i=0;
  char *retval;
  while (rhide_tokens[i].name)
  {
    if (strncmp(token,rhide_tokens[i].name,rhide_tokens[i].name_len) == 0)
    {
      char *_token;
      _token = rhide_tokens[i].func();
      if (!_token) string_dup(_token,"");
      if (!expand_tokens) return _token;
      retval = expand_tokens(_token);
      string_free(_token);
      return retval;
    }
    i++;
  }
  return NULL;
}

static void WriteRule(FILE *f,char *from,char *to)
{
  TDependency *dep;
  dep = new TDependency();
  string_dup(dep->source_name,"dummy");
  string_cat(dep->source_name,from);
  string_dup(dep->dest_name,"dummy");
  string_cat(dep->dest_name,to);
  dep->source_file_type = get_file_type(from);
  dep->dest_file_type = get_file_type(to);
  dep->compile_id = how_to_compile(dep->source_file_type,
                                   dep->dest_file_type);
  Boolean is_user;
  char *spec = GetCompilerSpec(dep,is_user);
  destroy(dep);
  if (!spec) return;
  fprintf(f,"%c%s: %c%s\n\t%s\n",'%',to,'%',from,spec);
  string_free(spec);
}

static void WriteRules(FILE *f)
{
#define WR(from,to) WriteRule(f,"."#from,"."#to)
  WR(c,o);
  WR(i,o);
  WR(cc,o);
  WR(cpp,o);
  WR(cxx,o);
  WR(C,o);
  WR(ii,o);
  WR(s,o);
  WR(S,o);
  WR(c,s);
  WR(i,s);
  WR(cc,s);
  WR(cpp,s);
  WR(cxx,s);
  WR(C,s);
  WR(pas,o);
  WR(p,o);
  WR(pas,s);
  WR(p,s);
  WR(m,o);
  WR(m,s);
#undef WR
}

void WriteSpecData(FILE *f)
{
  int i;
  for (i=0;i<var_count;i++)
  {
    fprintf(f,"%s=",vars[i*2]);
    put_breakline(f,strlen(vars[i*2]),75,vars[i*2+1]);
  }
  i=0;
  _token_dep = project;
  while (rhide_tokens[i].name)
  {
    char *token;
    char *name;
    if (rhide_tokens[i].make_func) token = rhide_tokens[i].make_func();
    else token = rhide_tokens[i].func();
    string_dup(name,rhide_tokens[i].name+2);
    name[strlen(name)-1] = 0;
    fprintf(f,"%s=",name);
    put_breakline(f,strlen(name)+1,75,token?token:"");
    string_free(token);
    string_free(name);
    i++;
  }
  WriteRules(f);
}

static char *ExpandSpec(char *spec)
{
  return expand_spec(spec,_handle_rhide_token);
}

char *BuildCompiler(TDependency *dep,char *spec)
{
  _token_dep = dep;
  return ExpandSpec(spec);
}

/*
  search in the environment for a compile spec, for compiling
  files with specified suffixes taken from 'from' and 'to'.
  The name of the variable should

  RHIDE_COMPILE.extfrom.extto

  Example: define the variable
  RHIDE_COMPILE.cp.o
  to tell RHIDE, how to compile files with suffix '.cp' to files
  with suffix '.o'

  If the variable was found, the returned string is malloced, otherwise
  NULL is returned.
*/

static char *GetUserCompileSpec(const char *from,const char *to)
{
  char ext_from[256],ext_to[256];
  if (!from) strcpy(ext_from,".");
  else fnsplit(from,NULL,NULL,NULL,ext_from);
  if (!to) strcpy(ext_to,".");
  else fnsplit(to,NULL,NULL,NULL,ext_to);
  char *var;
  string_dup(var,"RHIDE_COMPILE");
  string_cat(var,ext_from);
  string_cat(var,ext_to);
  if (!GetVariable(var))
  {
    string_free(var);
    return NULL;
  }
  char *_var;
  string_dup(_var,"$(");
  string_cat(_var,var);
  string_cat(_var,")");
  string_free(var);
  return _var;
}

char *GetCompilerSpec(TDependency *dep,Boolean & is_user)
{
  is_user = False;
  char *user_spec = NULL;
  switch (dep->compiler_type)
  {
    case COMPILER_C:
      return strdup("$(RHIDE_COMPILE_C_FORCE)");
    case COMPILER_CC:
      return strdup("$(RHIDE_COMPILE_CC_FORCE)");
    case COMPILER_ASM:
      return strdup("$(RHIDE_COMPILE_ASM_FORCE)");
    case COMPILER_GPC:
      return strdup("$(RHIDE_COMPILE_PASCAL_FORCE)");
    case COMPILER_NONE:
      return NULL;
    case COMPILER_AUTO:
      user_spec = GetUserCompileSpec(dep->source_name,dep->dest_name);
      if (user_spec)
      {
        if (dep->compile_id == COMPILE_UNKNOWN)
          is_user = True;
        return user_spec;
      }
    case COMPILER_USER:
      switch (dep->compile_id)
      {
#       define C(x) case x: return strdup("$(RHIDE_"#x")");
        C(COMPILE_C);
        C(COMPILE_CC);
        C(COMPILE_PASCAL);
        C(COMPILE_ARCHIVE);
        C(COMPILE_ASM);
        C(COMPILE_OBJC);
        C(COMPILE_LINK_PASCAL_AUTOMAKE);
#       undef C
        default:
          break;
      }
      if (dep->compile_id == COMPILE_LINK)
      {
        int i,count = 0;
        if (dep->dependencies) count = dep->dependencies->getCount();
        for (i=0;i<count;i++)
        {
          if (((TDependency *)dep->dependencies->at(i))->source_file_type
              == FILE_PASCAL_SOURCE)
          {
            return strdup("$(RHIDE_COMPILE_LINK_PASCAL)");
          }
        }
        return strdup("$(RHIDE_COMPILE_LINK)");
      }
      if (dep->compile_id == COMPILE_USER)
      {
        is_user = True;
        return strdup(dep->compiler);
      }
      return GetUserCompileSpec(dep->source_name,dep->dest_name);
      break;
  }
  return NULL;
}


