/* dir.c
   Directory management.
*/

#define INCL_DOSERRORS
#define INCL_NOCOMMON
#define INCL_NOPMAPI
#define INCL_DOSFILEMGR
#include <os2.h>

#include <stdlib.h>

#include "fsd.h"
#include "MacFS/mountrec.h"
#include "MacFS/errors.h"
#include "MacFS/LibFuncs/resolve.h"
#include "MacFS/LibFuncs/catalog.h"
#include "names.h"

/******************************************************************************
**
** FS_CHDIR - Change current directory
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     CD_EXPLICIT      creating a new current directory
**     CD_VERIFY        verifying a current directory
**     CD_FREE          freeing an instance of a current directory
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pDir                           pointer to directory to change to
** unsigned short iCurDirEnd            offset to the end of the current directory in pDir
**
******************************************************************************/

#pragma argsused
short int FS_CHDIR(struct vpfsd *pvpfsd, unsigned short flag, struct cdfsi *pcdfsi, 
                   struct cdfsd *pcdfsd, char *pDir, 
                   unsigned short iCurDirEnd)
{
  char macname[CCHMAXPATH], name[CCHMAXPATH];
  CATALOG_RECORD *cr;
  LONGWORD pid;

  switch(flag) {
  case CD_EXPLICIT:
    /* Check that filename is a directory */
    if(os2_to_mac_name(pDir+2, macname))
      return ERROR_ACCESS_DENIED;
    cr=(CATALOG_RECORD *)malloc(sizeof(CATALOG_RECORD));
    if(macresolvepathname(pvpfsd->mr, macname, cr, &pid, name)) {
      free(cr);
      return ERROR_PATH_NOT_FOUND;
    }
    if(cr->cdrType != CDR_DIR) {
      free(cr);
      return ERROR_PATH_NOT_FOUND;
    }
    if(strlen(pDir)>3)
      pcdfsd->cr = cr;
    else
      free(cr);
    return NO_ERROR;

  case CD_VERIFY:
    /* Check that filename is a directory */
    if(os2_to_mac_name(pcdfsi->cdi_curdir+2, macname))
      return ERROR_PATH_NOT_FOUND;
    cr=(CATALOG_RECORD *)malloc(sizeof(CATALOG_RECORD));
    /* Release the old catalog record */
    if(pcdfsi->cdi_flags & CDI_ISVALID) {
      free(pcdfsd->cr);
      pcdfsd->cr=NULL;
    }
    if(macresolvepathname(pvpfsd->mr, macname, cr, &pid, name)) {
      free(cr);
      return ERROR_PATH_NOT_FOUND;
    }
    pcdfsd->cr = cr;
    return NO_ERROR;

  case CD_FREE:
    if(pcdfsd->cr)
      free(pcdfsd->cr);
    pcdfsd->cr=NULL;
    return NO_ERROR;
  default:
    return ERROR_NOT_SUPPORTED;
  }
}


/******************************************************************************
**
** FS_MKDIR - Make a new directory
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to directory name to create
** unsigned short iCurDirEnd            offset to the end of the current directory 
**                                      in pName
** char *pEABuf                         pointer to EAs to attach to new directory
** unsigned short flags                 0x40 = directory is non 8.3 filename
** unsigned long *oError                offset where error occurred in FEA list
**
******************************************************************************/

#pragma argsused
short int FS_MKDIR(struct vpfsd *pvpfsd, struct cdfsi *pcdfsi, struct cdfsd *pcdfsd,
                   char *pName, unsigned short iCurDirEnd, char *pEABuf, 
                   unsigned short flags, unsigned long *oError)
{
  char path[CCHMAXPATH], name[MAXFNLENGTH];
  CATALOG_RECORD cr;
  LONGWORD parid;
  int rc;

  if(os2_to_mac_name(pName+2, path))
    return ERROR_ACCESS_DENIED;
  /* This should use relative pathnames */
  rc = macresolvepathname(pvpfsd->mr, path, &cr, &parid, name);
  if(rc==0)
    return ERROR_ACCESS_DENIED; /* Dir already exists */
  else {
    CATALOG_RECORD parcr;
    LONGWORD parparid;
    char parname[MAXFNLENGTH];
    
    if(macresolvepnwithnonexistinglastelement(pvpfsd->mr, path, &parparid, 
					      parname, &parcr, name))
      return ERROR_ACCESS_DENIED;
    if(mcreatefolder(pvpfsd->mr, &parcr, name, &cr))
      return ERROR_ACCESS_DENIED;
    return NO_ERROR;
  }
}


/******************************************************************************
**
** FS_RMDIR - Delete directory
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to directory name to delete
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
**
******************************************************************************/

#pragma argsused
short int FS_RMDIR(struct vpfsd *pvpfsd, struct cdfsi *pcdfsi, struct cdfsd  *pcdfsd,
                   char *pName, unsigned short iCurDirEnd)
{
  char macname[CCHMAXPATH], name[MAXFNLENGTH], parname[MAXFNLENGTH];
  CATALOG_RECORD cr, parcr;
  LONGWORD parid, parparid;
  int rc;

  if(os2_to_mac_name(pName+2, macname))
    return ERROR_ACCESS_DENIED;
  if(macresolvepathname(pvpfsd->mr, macname, &cr, &parid, name))
    return ERROR_ACCESS_DENIED;
  if(cr.cdrType != CDR_DIR)
    return ERROR_ACCESS_DENIED;
  if(macresolvepnwithnonexistinglastelement(pvpfsd->mr, macname, &parparid,
					    parname, &parcr, name))
    return ERROR_ACCESS_DENIED;
  rc=mdeletefolder(pvpfsd->mr, &parcr, name, &cr);
  if(rc==MACDELFOLDERNOTEMPTY)
    return ERROR_DIR_NOT_EMPTY;
  else if(rc)
    return ERROR_WRITE_FAULT;
  return NO_ERROR;
}
