#include "../defs.h"
#include "../errors.h"
#include "../mountrec.h"
#include "../../diskfunc.h"
#include "readwrite.h"
#include "../Volume/volinfo.h"
#include "../Volume/volbmap.h"
#include "../btree/btree.h"
#include "partition.h"

/*
 * Call first to open a new device and allocate a state record for the 
 * device.
 *
 */

int initreadwrite(unsigned short hVPB, MOUNT_RECORD *mr,
		  unsigned short device_sectsize,
		  unsigned short write_protected)
{
  int rc;

  mr->sr.hVPB = hVPB;
  mr->sr.validflag = VALIDVAL;
  mr->sr.maxwritesbeforemrflush = MAXWRITESBEFOREMRFLUSH;
  mr->sr.currentnumwrites = 0;
  mr->sr.volinfo_dirty = mr->sr.volbitmap_dirty = 0;
  mr->sr.device_sectsize = device_sectsize;
  mr->sr.write_protected = write_protected;

  if ((rc=get_hfs_partition(mr))) {
    mr->sr.validflag=0;
    return rc;
  }
  return 0;
}


int closereadwrite(MOUNT_RECORD *mr)
{
  mr->sr.validflag=!VALIDVAL;

  return 0;
}

int readabsolutesector(MOUNT_RECORD *mr, SECTNUMTYPE sect, char *buf)
{
  struct staterec *what;

  what=&(mr->sr);

  if (what->validflag!=VALIDVAL)
    return INVALIDSTATEREC;

  if( read_blocks(sect, 1, what->sectsize, what->device_sectsize,
		  what->hVPB, buf) )
    return READSECTFAILED;

  return 0;
}

/*
 *
 * Returns the number of sectors read, or a negative number if an error
 * occurred.
 *
 */
int readsects(MOUNT_RECORD *mr, SECTNUMTYPE sect, int num, char *buf)
{
  struct staterec *what;

  what=&(mr->sr);

  sect+=what->sectoffsettopartition;

  if (what->validflag!=VALIDVAL)
    return INVALIDSTATEREC;

  if( read_blocks(sect, num, what->sectsize, what->device_sectsize,
		  what->hVPB, buf) )
    return READSECTFAILED;

  return num;
}

/*
 *
 * Returns the number of sectors written, or a negative number if an error
 * occurred.
 *
 */
int writesects(MOUNT_RECORD *mr, SECTNUMTYPE sect, int num, char *buf)
{
  struct staterec *what;
  int rc;

  what=&(mr->sr);

  sect+=what->sectoffsettopartition;

  if (what->validflag!=VALIDVAL)
    return INVALIDSTATEREC;

  if((rc=write_blocks(sect, num, what->sectsize, what->device_sectsize,
		      what->hVPB, buf)) != 0)
    return rc;

  if (what->currentnumwrites>=what->maxwritesbeforemrflush) {
    if ((rc=syncmountrecord(mr))) {
      return rc;
    } else {
      what->currentnumwrites=0;
    }
  } else {
    ++(what->currentnumwrites);
  }

  return num;
}


/*
 *
 * Returns the number of sectors written, or a negative number if an error
 * occurred.
 *
 */
int writesectsunbuf(MOUNT_RECORD *mr, SECTNUMTYPE sect, int num, char *buf)
{
  struct staterec *what;
  int rc;

  what=&(mr->sr);

  sect+=what->sectoffsettopartition;

  if (what->validflag!=VALIDVAL)
    return INVALIDSTATEREC;

  if((rc=write_blocks(sect, num, what->sectsize, what->device_sectsize,
		   what->hVPB, buf)) != 0)
    return rc;

  return num;
}

int syncmountrecord(MOUNT_RECORD *mr)
{
  int rc;

  if ((rc=BTreeSync(mr)))
    return rc;

  if ((rc=writevolbmap(mr)))
    return rc;
  
  if ((rc=writevolinfo(mr)))
    return rc;

  return 0;
}
