//
// $Header: D:/ext2-os2/RCS/fs_find.c,v 8.0 1996/05/31 00:22:28 Willm Exp Willm $
//

// Linux ext2 file system driver for OS/2 2.x and WARP - Allows OS/2 to
// access your Linux ext2fs partitions as normal drive letters.
// Copyright (C) 1995, 1996 Matthieu WILLM
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.



#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>                          // From the "Developer Connection Device Driver Kit" version 2.0

#include <fsd.h>
#include <fsh.h>

#include <linux/stat.h>
#include <os2/os2proto.h>
#include <os2/os2misc.h>

#include <os2/ifsdbg.h>
#include <os2/filefind.h>
#include <os2/cdfsd.h>
#include <os2/errors.h>

#include <os2/log.h>         /* Prototypes des fonctions de log.c                      */
#include <os2/volume.h>      /* Prototypes des fonctions de volume.c                   */
#include <os2/files.h>       /* Prototypes des fonctions de files.c                    */
#include <linux/fs.h>
#include <linux/fs_proto.h>

#define THISFILE FILE_FS_FIND_C

#define FIL_LEVEL7            7      // Level 7, return case preserved path name
#define FILE_NONFAT     0x0040       // File is non 8.3 compliant

extern char trace_FS_FINDFIRST;
extern char trace_FS_FINDCLOSE;
extern char trace_FS_FILEINFO;
extern char trace_FS_PATHINFO;

extern char Case_Retensive;

//
// date_dos2unix  and date_unix2dos are from the Linux FAT file system
//

/* Linear day numbers of the respective 1sts in non-leap years. */

static int day_n[] = { 0,31,59,90,120,151,181,212,243,273,304,334,0,0,0,0 };
                  /* JanFebMarApr May Jun Jul Aug Sep Oct Nov Dec */


// extern struct timezone sys_tz;
extern long timezone;

/* Convert a MS-DOS time/date pair to a UNIX date (seconds since 1 1 70). */

long date_dos2unix(unsigned short time,unsigned short date)
{
        long month,year,secs;

        month = ((date >> 5) & 15L)-1L;
        year = date >> 9;
        secs = (time & 31L)*2L+60*((time >> 5) & 63L)+(time >> 11)*3600L+86400L*
            ((date & 31L)-1L+day_n[month]+(year/4L)+year*365L-((year & 3L) == 0 &&
            month < 2L ? 1 : 0)+3653L);
                        /* days since 1.1.70 plus 80's leap day */
//        secs += sys_tz.tz_minuteswest*60;
        secs += timezone;
        return secs;
}


/* Convert linear UNIX date to a MS-DOS time/date pair. */

void date_unix2dos(long unix_date,unsigned short *time,
    unsigned short *date)
{
        long day,year,nl_day,month;
        unsigned long ud;
//        unix_date -= sys_tz.tz_minuteswest*60;
        unix_date -= timezone;
#if 0
        *time = (unsigned short)((unix_date % 60)/2+(((unix_date/60) % 60) << 5)+
            (((unix_date/3600) % 24) << 11));
#else
        ud = (unsigned long) unix_date;
        *time = (unsigned short)((ud % 60)/2+(((ud/60) % 60) << 5)+
            (((ud/3600) % 24) << 11));
#endif
        day = unix_date/86400-3652;
        year = day/365;
        if ((year+3)/4+365*year > day) year--;
        day -= (year+3)/4+365*year;
        if (day == 59 && !(year & 3)) {
                nl_day = day;
                month = 2;
        }
        else {
                nl_day = (year & 3) || day <= 59 ? day : day-1;
                for (month = 0; month < 12; month++)
                        if (day_n[month] > nl_day) break;
        }
        *date = (unsigned short)(nl_day-day_n[month-1]+1+(month << 5)+(year << 9));
}

/*
 * The following routines are meaningless for a mini FSD
 */
#ifndef MINIFSD

/* 1 - means OK, 0 means doesn't match */
int filter_with_attrs(struct inode * pino, unsigned short attrs, pchar name) {
    unsigned short mayhave  = attrs & 0xFF;
    unsigned short musthave = (attrs >> 8) & 0xFF;
    unsigned short mapped;

    mapped = Linux_To_DOS_Attrs(pino, name);


/*
          Actual   May have  Must have  Included   a&may&must ~a&~may&~must a&may&~must  ~a&may&~must
            1          1          1         y           1           0          0               0
            1          1          0         y           0           0          1               0
            1          0          1         n           0           0          0               0
            1          0          0         n           0           0          0               0
            0          1          1         n           0           0          0               0
            0          1          0         y           0           0          0               1
            0          0          1         n           0           0          0               0
            0          0          0         y           0           1          0               0

=> a & may | ~a & ~must

*/

    return !(~(( (mapped &  mayhave) | ( (~mapped) & (~musthave)))));
}

#define TYPEOP_FILEFIND 1
#define TYPEOP_FILEINFO 2

int ino_to_fileinfo(
                    struct inode * pino,
                    pchar databuf,
                    UINT32 maxlen,
                    PUINT32 plen,
                    unsigned short level,
                    unsigned short flags,
                    unsigned short attrs,
                    struct dirent *pDir,
                    INT32 position, int TypeOp) {
    pCommonFileInfo          pCommon;
    pFileName                pFile;
    PINT32                  pPosition;

    *plen = 0;

    /***************************************************************************/
    /*** First field : long position if FF_GETPOS - nothing if FF_NOPOS      ***/
    /***************************************************************************/
    if (TypeOp == TYPEOP_FILEFIND) {
        if (flags == FF_GETPOS) {
            if (sizeof(INT32) + *plen > maxlen) {
                fs_log("ino_to_fileinfo() - Buffer overflow for first field");
                return ERROR_BUFFER_OVERFLOW;
            }
            pPosition  = (PINT32)(databuf + *plen);
            *pPosition = position;
            *plen += sizeof(INT32);
        }
    }
    /***************************************************************************/

    /***************************************************************************/
    /*** Second field : common file information to all level/flags           ***/
    /***************************************************************************/
    if (sizeof(CommonFileInfo) + *plen > maxlen) {
        kernel_printf("ino_to_fileinfo() - Buffer overflow for second field - len = %lu - maxlen = %lu", (UINT32)sizeof(CommonFileInfo) + (UINT32)(*plen), (UINT32)maxlen);
        return ERROR_BUFFER_OVERFLOW;
    }
    pCommon = (pCommonFileInfo)(databuf + *plen);

    date_unix2dos(pino->i_ctime, &(pCommon->timeCreate), &(pCommon->dateCreate));
    date_unix2dos(pino->i_atime, &(pCommon->timeAccess), &(pCommon->dateAccess));
    date_unix2dos(pino->i_mtime, &(pCommon->timeWrite), &(pCommon->dateWrite));
    pCommon->cbEOF      = pino->i_size;
    pCommon->cbAlloc    = pino->i_blocks;
    pCommon->attr = FILE_NORMAL;

    if ((S_ISLNK(pino->i_mode))  ||
        (S_ISBLK(pino->i_mode))  ||
        (S_ISCHR(pino->i_mode))  ||
        (S_ISFIFO(pino->i_mode)) ||
        (S_ISSOCK(pino->i_mode))) {
        pCommon->attr |= FILE_SYSTEM;     /*** UNIXish special files are seen as SYSTEM files ***/
    } /* endif */

    if (S_ISDIR(pino->i_mode)) {
        pCommon->attr |= FILE_DIRECTORY;
    } /* endif */

    if ((!(pino->i_mode & S_IWUSR)) &&
        (!(pino->i_mode & S_IWGRP)) &&
        (!(pino->i_mode & S_IWOTH))) {
        pCommon->attr |= FILE_READONLY;
    }
    *plen += sizeof(CommonFileInfo);
    /***************************************************************************/

    /***************************************************************************/
    /*** Third field : nothing for level FIL_STANDARD                        ***/
    /***************************************************************************/
    /***************************************************************************/

    /***************************************************************************/
    /*** Third field : Size on disk of EA set for FIL_QUERYEASIZE            ***/
    /***************************************************************************/
    if (level == FIL_QUERYEASIZE) {
        if (sizeof(INT32) + *plen > maxlen) {
            fs_log("ino_to_fileinfo() - Buffer overflow for Third field - FIL_QUERYEASIZE");
            return ERROR_BUFFER_OVERFLOW;
        }
        *((PINT32)(databuf + *plen)) = 0;       /*** No EAS in ext2 ***/
        *plen += sizeof(INT32);
    }
    /***************************************************************************/

    /***************************************************************************/
    /*** Third field : FEAList for level QUERYEASFROMLIST                    ***/
    /***************************************************************************/
    if (level == FIL_QUERYEASFROMLIST) {
        if (sizeof(INT32) + *plen > maxlen) {
            fs_log("ino_to_fileinfo() - Buffer overflow for Third field - FIL_QUERYEASFROMLIST");
            return ERROR_BUFFER_OVERFLOW;
        }
        *((PINT32)(databuf + *plen)) = 4;       /*** No EAS in ext2 ***/ /* this is the length field of FEAList */
        *plen += sizeof(INT32);
    }
    /***************************************************************************/

    /***************************************************************************/
    /*** Fourth field : name                                                 ***/
    /***************************************************************************/
    if (TypeOp == TYPEOP_FILEFIND) {
        if (*plen + sizeof(FileName) + pDir->d_reclen > maxlen) {
            fs_log("ino_to_fileinfo() - Buffer overflow for fourth field");
            return ERROR_BUFFER_OVERFLOW;
        }
        pFile = (pFileName)(databuf + *plen);
        pFile->cbName = (unsigned char)pDir->d_reclen;   /* name length WITHOUT '\0' */
        strcpy(pFile->szName, pDir->d_name);             /* name WITH '\0'           */
        *plen += sizeof(FileName) + pDir->d_reclen;      /* sizeof(FileName) + strlen(Dir.d_name) */
    }
    /***************************************************************************/

    return NO_ERROR;
}

static int fileinfo_to_ino(char *databuf, struct inode *ino, int level, unsigned short datalen, struct sffsi *psffsi) {
    int             rc;
    pCommonFileInfo pCommon;
    unsigned short  time;
    unsigned short  date;

    rc = NO_ERROR;

    /*
     * Only FIL_STANDARD is significant. FIL_QUERYEASIZE is not because one can't SET EAS in an ext2
     * file system.
     */
    if (level == FIL_STANDARD) {
        if (datalen >= sizeof(pCommonFileInfo)) {
            pCommon = (pCommonFileInfo)databuf;


            /*
             * Creation date
             */
            date_unix2dos(ino->i_ctime, &time, &date);
            if (pCommon->timeCreate)
                time = pCommon->timeCreate;
            if (pCommon->dateCreate)
                date = pCommon->dateCreate;
            ino->i_ctime = date_dos2unix(time, date);
            if (psffsi) {
                psffsi->sfi_cdate = date;
                psffsi->sfi_ctime = time;
                if ((pCommon->timeCreate) || (pCommon->dateCreate)) {
                    /*
                     * Clears the stamp creation date bit if we've just modified the creation date.
                     */
                    psffsi->sfi_tstamp &= ~ST_SCREAT;   // disable kernel time stamp
                    psffsi->sfi_tstamp |=  ST_PCREAT;	// propagate new value to other sft

                }
            }

            /*
             * Last write date
             */
            date_unix2dos(ino->i_mtime, &time, &date);
            if (pCommon->timeWrite)
                time = pCommon->timeWrite;
            if (pCommon->dateWrite)
                date = pCommon->dateWrite;
            ino->i_mtime = date_dos2unix(time, date);
            if (psffsi) {
                psffsi->sfi_mdate = date;
                psffsi->sfi_mtime = time;
                if ((pCommon->timeWrite) || (pCommon->dateWrite)) {
                    /*
                     * Clears the stamp last write date bit if we've just modified the last write date.
                     */
                    psffsi->sfi_tstamp &= ~ST_SWRITE;   // disable kernel time stamp
                    psffsi->sfi_tstamp |=  ST_PWRITE;	// propagate new value to other sft

                }
            }

            /*
             * Last access date
             */
            date_unix2dos(ino->i_atime, &time, &date);
            if (pCommon->timeAccess)
                time = pCommon->timeAccess;
            if (pCommon->dateAccess)
                date = pCommon->dateAccess;
            ino->i_atime = date_dos2unix(time, date);
            if (psffsi) {
                psffsi->sfi_adate = date;
                psffsi->sfi_atime = time;
                if ((pCommon->timeAccess) || (pCommon->dateAccess)) {
                    /*
                     * Clears the stamp last access date bit if we've just modified the last access date.
                     */
                    psffsi->sfi_tstamp &= ~ST_SREAD;    // disable kernel time stamp
                    psffsi->sfi_tstamp |=  ST_PREAD;	// propagate new value to other sft

                }
            }

            /*
             * File size - neither the IFS document nor the Control Program Reference 
             *             are clear about if we must change the file size when DosSetPathInfo
             *             or DosSetFIleInfo is called. HPFS does NOT change it .... so we don't
             *             change it also.
             */

            /*
             * File attributes
             */
            DOS_To_Linux_Attrs(ino, pCommon->attr);

            /*
             * Marks the inode as dirty
             */
            ino->i_dirt = 1;

        } else {
            rc = ERROR_BUFFER_OVERFLOW;
        }
    }
    return rc;
}

int myfindnext(
             struct super_block    *p_volume,
             struct file           *p_file,
             unsigned short         attr,
             struct fsfsi   _FS_PTR pfsfsi,
             struct fsfsd   _FS_PTR pfsfsd,
             pchar                  pData,
             unsigned short         cbData,
             unsigned short _FS_PTR pcMatch,
             unsigned short         level,
             unsigned short         flags,
             UINT32                 index_dir,
             int                    is_findfirst,
             int                    caseRetensive
            ) {
    char                    *left;
    char                    *right;
    UINT32 cur_size;
    int    rc;
    pchar                    pNom;
    pchar                    pWork, pWorkUpper;
    pchar                    pName, pNameUpper;
    UINT32                   len;
    struct inode *                    ino;
    UINT16 nb_found;
    struct dirent Dir;

    pName               = ((p_hfind)pfsfsd)->pName;
    pNom                = pName + CCHMAXPATH;
    pWork               = pName + 2 * CCHMAXPATH;
    pNameUpper          = pName + 3 * CCHMAXPATH;
    pWorkUpper          = pName + 4 * CCHMAXPATH;

    nb_found  = 0;
    if (level == FIL_QUERYEASFROMLIST) {
        cur_size = sizeof(EAOP);
    } else {
        cur_size  = 0;
    }

    while (nb_found < *pcMatch) {
        /***********************************************************/
        /*** If we are at the end of the parent dir : stop       ***/
        /***********************************************************/
        if (VFS_readdir(p_file, &Dir) != NO_ERROR) {
            ((p_hfind)pfsfsd)->last = index_dir;
            *pcMatch = nb_found;

            if (nb_found > 0) {
#ifdef FS_TRACE
                kernel_printf("myfindnext()  - EOD - found %d entries", nb_found);
#endif
                return NO_ERROR;
            } else {

#if 1
                if (is_findfirst) {

                    /************************************************************************/
                    /*** Libration du buffer stockant les noms de recherche              ***/
                    /************************************************************************/
                    if ((rc = G_free(((p_hfind)pfsfsd)->pName)) != NO_ERROR) {
                        fs_err(FUNC_FS_FINDFIRST, FUNC_G_free, rc, THISFILE, __LINE__);
                        return rc;
                    } /* end if */
                    /************************************************************************/

                    if ((rc = vfs_close(p_file)) != NO_ERROR) {
                        fs_err(FUNC_FS_FINDFIRST, FUNC_CLOSE, rc, THISFILE, __LINE__);
                        return rc;
                    }
//                    ((p_hfind)pfsfsd)->FS_CLOSEd = SEARCH_ALREADY_CLOSED;
                }
#else
                    FS_FINDCLOSE(pfsfsi, pfsfsd);
                    ((p_hfind)pfsfsd)->FS_CLOSEd = SEARCH_ALREADY_CLOSED;
#endif
                return ERROR_NO_MORE_FILES;
            }
        } /* end if */
        /***********************************************************/

        /***********************************************************/
        /*** If we found an entry, see if it matches             ***/
        /***********************************************************/
        strcpy(pWork, pNom);
        strcat(pWork, "\\");
        strcat(pWork, Dir.d_name);
        if (caseRetensive) {
            FSH_UPPERCASE(pName, CCHMAXPATH, pNameUpper);
            left  = pNameUpper;
            FSH_UPPERCASE(pWork, CCHMAXPATH, pWorkUpper);
            right = pWorkUpper;
        } else {
            left  = pName;
            right = pWork;
        }
        if (FSH_WILDMATCH(left, right) == NO_ERROR) {
            if (p_file->f_inode->i_ino != Dir.d_ino) {
                if ((ino = iget(p_volume, Dir.d_ino)) == NULL) {
                    fs_err(FUNC_FS_FINDFIRST, FUNC_GET_VINODE, rc, THISFILE, __LINE__);
                    return ERROR_NO_MORE_FILES;
                } /* endif */
            } else {
                ino = p_file->f_inode;
            }
            if (filter_with_attrs(ino, attr | FILE_READONLY, Dir.d_name)) {
                if ((rc = ino_to_fileinfo(
                                          ino,
                                          pData + cur_size,
                                          cbData - cur_size,
                                          &len,
                                          level,
                                          flags,
                                          attr,
                                          &Dir,
                                          index_dir, TYPEOP_FILEFIND)) != NO_ERROR) {
                    *pcMatch                = nb_found;
                    ((p_hfind)pfsfsd)->last = index_dir;
                    fs_log("====> FS_FINDFIRST : erreur ino_to_fileinfo()");
                    if (p_file->f_inode->i_ino != Dir.d_ino) {
                        iput(ino);
                    }
                    return rc;
                }
                nb_found  ++;
                cur_size += len;
            }
            if (p_file->f_inode->i_ino != Dir.d_ino) {
                iput(ino);
            }
        } /* end if */
        /***********************************************************/

        index_dir ++;
    } /* end while */

    *pcMatch = nb_found;
    ((p_hfind)pfsfsd)->last = index_dir;
#ifdef FS_TRACE
    kernel_printf("myfindnext() - found %d entries", nb_found);
#endif

    return NO_ERROR;

}

/*************************************************************************************************/
/*** FS_FINDFIRST - IFS find first matching file(s) entry point (DosFindFirst API)             ***/
/*************************************************************************************************/
int     _FS_ENTRY FS_FINDFIRST(
                               struct cdfsi   _FS_PTR pcdfsi,
                               struct cdfsd   _FS_PTR pcdfsd,
                               char           _FS_PTR pName,
                               unsigned short         iCurDirEnd,
                               unsigned short         attr,
                               struct fsfsi   _FS_PTR pfsfsi,
                               struct fsfsd   _FS_PTR pfsfsd,
                               char           _FS_PTR pData,
                               unsigned short         cbData,
                               unsigned short _FS_PTR pcMatch,
                               unsigned short         level,
                               unsigned short         flags
                              )
{
    int                      rc, rc2;
    UINT32                   openmode = 0;
    pchar                    pNom;
    struct file        *     p_file;
    struct super_block *     p_volume;
//    id_t                     myid;
    char          lock[12];
    unsigned long lock_lin;
    int           caseRetensive;


    if (trace_FS_FINDFIRST) {
        kernel_printf("FS_FINDFIRST pre-invocation : %s , match = %u , level = %u, flag = %u)", pName, *pcMatch, level, flags);
    }

    /*
     * Checks that flags is valid.
     */
    if ((flags == FF_NOPOS) ||
        (flags == FF_GETPOS)) {

        /*
         * Checks that level is valid.
         */
        if ((level == FIL_STANDARD)    ||
            (level == FIL_QUERYEASIZE) ||
            (level == FIL_QUERYEASFROMLIST)) {

            /*
             * Checks that *pcMatch is not 0
             */
            if (*pcMatch) {
                /*
                 * Is it a case retensive search ?
                 */
                openmode = (caseRetensive = is_case_retensive() ? OPENMODE_DOSBOX : 0);

                    /*
                     * Gets the superblock from pfsfsi.
                     */
                    if ((p_volume = getvolume(pfsfsi->fsi_hVPB)) != 0) {

                        /*
                         * Saves the file name to search into pfsfsd
                         */
                        if ((((p_hfind)pfsfsd)->pName = G_malloc(5 * CCHMAXPATH)) != 0) {
                            strcpy(((p_hfind)pfsfsd)->pName, pName);
                            pNom = ((p_hfind)pfsfsd)->pName + CCHMAXPATH;

                            /*
                             * Adds MAY_HAVE_READONLY to the attributes to search for (cf doc : this bit is never passed to the IFS)
                             */
                            ((p_hfind)pfsfsd)->attr = attr  | FILE_READONLY;

                            /*
                             * Extracts the name of the parent directory.
                             */
                            ExtractPath(pName, pNom);

                            /*
                             * Tries to open the parent directory.
                             */
                            if ((p_file = _open_by_name(p_volume, pNom, openmode | OPENMODE_READONLY)) != 0) {
                                ((p_hfind)pfsfsd)->p_file = p_file;

                                /*
                                 * Locks the user buffer so that another thread can't free it from under us.
                                 */
                                if ((rc = LockUserBuffer(pData, cbData, lock, LOCK_WRITE, &lock_lin)) == NO_ERROR) {

                                   /*
                                    * Do the actual search
                                    */
                                    rc = myfindnext(
                                                    p_volume,
                                                    p_file,
                                                    attr | FILE_READONLY,
                                                    pfsfsi,
                                                    pfsfsd,
                                                    pData,
                                                    cbData,
                                                    pcMatch,
                                                    level,
                                                    flags,
                                                    0,
                                                    1,
                                                    caseRetensive
                                                   );
                                    /*
                                     * Unlocks the user buffer.
                                     */
                                    if ((rc2 = VMUnlock(lock_lin)) == NO_ERROR) {
                                        /* Nothing else to do */
                                    } else {
                                        kernel_printf("FS_READ : VMUnlock() returned %d", rc);
                                        rc = rc2;
                                    } /* VMUnlock failed */
                                } else {
                                    kernel_printf("FS_FINDFIRST - LockUserBuffer returned %d", rc);
                                } /* LockUserBuffer failed */
                            } else {
                                rc = ERROR_PATH_NOT_FOUND;
                            } /* Parent directory not found */
                        } else {
                            kernel_printf("FS_FINDFIRST - No more memory");
                            rc = ERROR_NOT_ENOUGH_MEMORY;
                        } /* G_malloc failed */
                    } else {
                        kernel_printf("FS_FINDFIRST - Couldn't get the superblock from pfsfsi");
                        rc = ERROR_INVALID_PARAMETER;
                    } /* p_volume invalid */
            } else {
                kernel_printf("FS_FINDFIRST - pcMatch = 0");
                rc =  ERROR_INVALID_PARAMETER;
            } /* pcMatch = 0 */
        } else {
            kernel_printf("FS_FINDFIRST - invalid level %d", rc);
            rc =  ERROR_INVALID_PARAMETER;
        }  /* Invalid level */
    } else {
         kernel_printf("FS_FINDFIRST() - invalid flag %d", flags);
         rc = ERROR_INVALID_PARAMETER;
    } /* Invalid flag */

    if (trace_FS_FINDFIRST) {
        kernel_printf("FS_FINDFIRST post-invocation - rc = %d - pcMatch = %d", rc, *pcMatch);
    }
    return rc;
}
/*************************************************************************************************/

/*************************************************************************************************/
/*** FS_FINDCLOSE - IFS close file search entry point (DosFindClose API)                       ***/
/*************************************************************************************************/
int     _FS_ENTRY FS_FINDCLOSE(
                               struct fsfsi _FS_PTR pfsfsi,
                               struct fsfsd _FS_PTR pfsfsd
                              )
{
    int                 rc;
    struct super_block *sb;

    if (trace_FS_FINDCLOSE) {
        kernel_printf("FS_FINDCLOSE"" pre-invocation : %s", ((p_hfind)pfsfsd)->pName);
    }

    /*
     * Gets the superblock from pfsfsi
     */
    if ((sb = getvolume(pfsfsi->fsi_hVPB)) != 0) {
        /*
         * Closes the search directory
         */
        if ((rc = vfs_close(((p_hfind)pfsfsd)->p_file)) == NO_ERROR) {
            /*
             * Frees the memory allocated for the search
             */
            if ((rc = G_free(((p_hfind)pfsfsd)->pName)) == NO_ERROR) {
                /*
                 * Zero out struct FS dependant file search area for safety purposes.
                 */
                memset(pfsfsd, 0, sizeof(struct fsfsd));
            } else {
                fs_err(FUNC_FS_FINDCLOSE, FUNC_G_free, rc, FILE_FS_FIND_C, __LINE__);
            }
        } else {
            kernel_printf("FS_FINDCLOSE"" - close returned %d", rc);
        }
    } else {
        kernel_printf("FS_FINDCLOSE"" - Unable to retrieveve superblock from pfsfsd");
        rc = ERROR_INVALID_PARAMETER;
    }

    if (trace_FS_FINDCLOSE) {
        kernel_printf("FS_FINDCLOSE"" post-invocation");
    }

    return rc;
}
/*************************************************************************************************/


/*************************************************************************************************/
/*** FS_FINDFROMNAME - IFS find next matching file(s) entry point (resume by name)             ***/
/*************************************************************************************************/
_FS_RET _FS_ENTRY FS_FINDFROMNAME(
                                  struct fsfsi   _FS_PTR pfsfsi,
                                  struct fsfsd   _FS_PTR pfsfsd,
                                  char           _FS_PTR pData,
                                  unsigned short         cbData,
                                  unsigned short _FS_PTR pcMatch,
                                  unsigned short         level,
                                  unsigned long          position,
                                  char           _FS_PTR pNameFrom,
                                  unsigned short         flags
                                 )
{
    int                 rc;
    struct super_block *p_volume;

#ifdef FS_TRACE
    kernel_printf("FS_FINDFROMNAME( %s , match = %u , level = %u, flag = %u)", ((p_hfind)pfsfsd)->pName, *pcMatch, level, flags);
#endif

    //
    // Verify we have write access to the user buffer
    //
    if ((rc = FSH_PROBEBUF(PB_OPWRITE, pData, cbData)) != NO_ERROR) {
        fs_err(FUNC_FS_FINDFROMNAME, FUNC_FSH_PROBEBUF, rc, THISFILE, __LINE__);
        return rc;
    } /* end if */

    if ((flags != FF_NOPOS) &&
        (flags != FF_GETPOS)) {
        if ((rc = kernel_printf("FS_FINDFROMNAME() - invalid flags  : %u", flags)) != NO_ERROR) {
            return rc;
        } /* end if */
        return ERROR_INVALID_PARAMETER;
    }

    if ((level != FIL_STANDARD)    &&
        (level != FIL_QUERYEASIZE) &&
        (level != FIL_QUERYEASFROMLIST)) {
        if ((rc = kernel_printf("FS_FINDFROMNAME() - invalid level  : %u", level)) != NO_ERROR) {
            return rc;
        } /* end if */
        return ERROR_INVALID_PARAMETER;
    }

    //
    // Gets the superblock from pfsfsi
    //
     p_volume = getvolume(pfsfsi->fsi_hVPB);

    //
    // Verifies the user asked something !
    //
    if (*pcMatch == 0) {
        fs_log("FS_FINDFROMNAME - *pcMatch = 0");
        return ERROR_INVALID_PARAMETER;
    } /* end if */

    //
    // Do the actual search
    //
    return myfindnext(
                      p_volume,
                      ((p_hfind)pfsfsd)->p_file,
                      ((p_hfind)pfsfsd)->attr,
                      pfsfsi,
                      pfsfsd,
                      pData,
                      cbData,
                      pcMatch,
                      level,
                      flags,
                      ((p_hfind)pfsfsd)->last,
                      0,
                      is_case_retensive()
                     );


}


/*************************************************************************************************/
/*** FS_FINDNEXT - IFS find next matching file(s) entry point (resume by index)                ***/
/*************************************************************************************************/
int     _FS_ENTRY  FS_FINDNEXT(
                               struct fsfsi   _FS_PTR pfsfsi,
                               struct fsfsd   _FS_PTR pfsfsd,
                               char           _FS_PTR pData,
                               unsigned short         cbData,
                               unsigned short _FS_PTR pcMatch,
                               unsigned short         level,
                               unsigned short         flags
                              )
{
    int                 rc;
    struct super_block *p_volume;

#ifdef FS_TRACE
    kernel_printf("FS_FINDNEXT( %s , match = %u , level = %u, flag = %u)", ((p_hfind)pfsfsd)->pName, *pcMatch, level, flags);
#endif

    //
    // Verifies we have write access to the user buffer
    //
    if ((rc = FSH_PROBEBUF(PB_OPWRITE, pData, cbData)) != NO_ERROR) {
        fs_err(FUNC_FS_FINDNEXT, FUNC_FSH_PROBEBUF, rc, THISFILE, __LINE__);
        return rc;
    } /* end if */

    if ((flags != FF_NOPOS) &&
        (flags != FF_GETPOS)) {
        if ((rc = fs_log("FS_FINDNEXT() - invalid flag")) != NO_ERROR) {
            return rc;
        } /* end if */
        return ERROR_INVALID_PARAMETER;
    }

    if ((level != FIL_STANDARD)    &&
        (level != FIL_QUERYEASIZE) &&
        (level != FIL_QUERYEASFROMLIST)) {
        if ((rc = fs_log("FS_FINDNEXT() - invalid level")) != NO_ERROR) {
            return rc;
        } /* end if */
        return ERROR_INVALID_PARAMETER;
    }


    //
    // Gets the superblock from pfsfsi
    //
     p_volume = getvolume(pfsfsi->fsi_hVPB);

    //
    // Verifies the user asked something !
    //
    if (*pcMatch == 0) {
        fs_log("FS_FINDNEXT - *pcMatch = 0");
        return ERROR_INVALID_PARAMETER;
    } /* end if */

    //
    // Do the actual search
    //
    return myfindnext(
                      p_volume,
                      ((p_hfind)pfsfsd)->p_file,
                      ((p_hfind)pfsfsd)->attr,
                      pfsfsi,
                      pfsfsd,
                      pData,
                      cbData,
                      pcMatch,
                      level,
                      flags,
                      ((p_hfind)pfsfsd)->last,
                      0,
                      is_case_retensive()
                     );


}
/*************************************************************************************************/



_FS_RET _FS_ENTRY FS_FILEINFO(
                              unsigned short         flag,
                              struct sffsi   _FS_PTR psffsi,
                              struct sffsd   _FS_PTR psffsd,
                              unsigned short         level,
                              char           _FS_PTR pData,
                              unsigned short         cbData,
                              unsigned short         IOflag
                             )
{
    int     rc, rc2;
    UINT32  len;
    struct file        *p_file;
    struct super_block *p_volume;
    PEAOP   peaop;
    char          lock[12];
    unsigned long lock_lin;

    if ((level != FIL_STANDARD)         &&
        (level != FIL_QUERYEASIZE)      &&
        (level != FIL_QUERYEASFROMLIST) &&
        (level != FIL_QUERYALLEAS)      && // Undocumented level - similar to level 3 but for full EA set
        (level != FIL_LEVEL7)) {
        if ((rc = kernel_printf("FS_FILEINFO() - invalid level %u", level)) != NO_ERROR) {
            return rc;
        } /* end if */
        return ERROR_INVALID_PARAMETER;
    }

    //
    // Get the superblock from psffsi
    //
    p_volume = getvolume(psffsi->sfi_hVPB);

    //
    // Gets the file structure from psffsd
    //
    p_file = ((_sffsd _FS_PTR)psffsd)->p_file;

    switch(flag) {
        case FI_SET :
            if (trace_FS_FILEINFO) {
                kernel_printf("FS_FILEINFO - FI_SET");
            }

            rc = NO_ERROR;
            if (Read_Write) {
                if ((rc = LockUserBuffer(pData, cbData, lock, LOCK_READ, &lock_lin)) == NO_ERROR) {
                    if ((rc = fileinfo_to_ino(pData, p_file->f_inode, level, cbData, psffsi)) == NO_ERROR) {
                        /*
                         * Nothing else to do
                         */
                    } else {
                        kernel_printf("fileinfo_to_ino returned %d", rc);
                    }
                    if ((rc2 = VMUnlock(lock_lin)) == NO_ERROR) {
                        /* Nothing else to do */
                    } else {
                        kernel_printf("FS_FILEINFO : VMUnlock() returned %d", rc2);
                        rc = rc2;
                    } /* VMUnlock failed */
                } else {
                    kernel_printf("FS_FILEINFO : LockUserBuffer() returned %d", rc);
                }
            } else {
                rc = ERROR_WRITE_PROTECT;
            }
            break;

        case FI_RETRIEVE :
            if (trace_FS_FILEINFO) {
                kernel_printf("FS_FILEINFO - FI_RETRIEVE - lvl = %u - inode %lu )", level, p_file->f_inode->i_ino);
            }

            //
            // Verify we have write access to the user buffer
            //
            if ((rc = FSH_PROBEBUF(PB_OPWRITE, pData, cbData)) != NO_ERROR) {
                fs_err(FUNC_FS_FILEINFO, FUNC_FSH_PROBEBUF, rc, THISFILE, __LINE__);
                return rc;
            } /* end if */

            if ((level == FIL_QUERYEASFROMLIST) ||
                (level == FIL_QUERYALLEAS)) {
                peaop = (PEAOP)pData;
                if ((rc = FSH_PROBEBUF(PB_OPWRITE, (pchar)peaop, 4)) != NO_ERROR) {
                    fs_err(FUNC_FS_FILEINFO, FUNC_FSH_PROBEBUF, rc, THISFILE, __LINE__);
                    return rc;
                } /* end if */
                peaop->fpFEAList->cbList = 4;
            } else {
            if ((rc = ino_to_fileinfo(
                                      p_file->f_inode,
                                      pData,
                                      cbData,
                                      &len,
                                      level,
                                      0,
                                      0,
                                      0,
                                      0, TYPEOP_FILEINFO)) != NO_ERROR) {
                fs_log("FS_FILEINFO() - ino_to_fileinfo()");
                return rc;
            }
            }
            break;

        default :
            fs_log("FS_FILEINFO() - Unknown flag");
            rc = ERROR_INVALID_PARAMETER;
            break;
    }


    return rc;

}




_FS_RET _FS_ENTRY FS_PATHINFO(
                              unsigned short         flag,
                              struct cdfsi   _FS_PTR pcdfsi,
                              struct cdfsd   _FS_PTR pcdfsd,
                              char           _FS_PTR pName,
                              unsigned short         iCurDirEnd,
                              unsigned short         level,
                              char           _FS_PTR pData,
                              unsigned short         cbData
                             )
{
    int      rc, rc2, rc3;
    struct super_block *p_volume;
    struct file        *p_file;
    UINT32   len, openmode = 0;
    PEAOP    peaop;
    char          lock[12];
    unsigned long lock_lin;

    //
    // Verify the level is valid
    //
    if ((level != FIL_STANDARD)         &&
        (level != FIL_QUERYEASIZE)      &&
        (level != FIL_QUERYEASFROMLIST) &&
        (level != FIL_QUERYALLEAS)      &&  // Undocumented level - similar to level 3 but for full EA set
        (level != FIL_LEVEL7)) {
        kernel_printf("FS_PATHINFO() - invalid level %u", level);
        return ERROR_INVALID_PARAMETER;
    }

    openmode = (is_case_retensive() ? OPENMODE_DOSBOX : 0);

    //
    // Retrieves the superblock from psffsi
    //
    p_volume = getvolume(pcdfsi->cdi_hVPB);

    switch(flag) {
        case PI_RETRIEVE :
            if (trace_FS_PATHINFO) {
                kernel_printf("FS_PATHINFO - PI_RETRIEVE - lvl = %u - %s", level, pName);
            }

            //
            // Verify we have write access to the user buffer
            //
            if ((rc = FSH_PROBEBUF(PB_OPWRITE, pData, cbData)) != NO_ERROR) {
                fs_err(FUNC_FS_PATHINFO, FUNC_FSH_PROBEBUF, rc, THISFILE, __LINE__);
                return rc;
            } /* end if */

            switch(level) {
                case FIL_LEVEL7 :
                    if (strlen(pName) + 1 > cbData) {
                        return ERROR_BUFFER_OVERFLOW;
                    }
                    memcpy(pData, pName,strlen(pName) + 1);
                    return NO_ERROR;

                default :
                    if ((p_file = _open_by_name(p_volume, pName, openmode | OPENMODE_READONLY)) == 0) {
#ifdef FS_TRACE
                        fs_err(FUNC_FS_PATHINFO, FUNC_OPEN_BY_NAME, ERROR_PATH_NOT_FOUND, THISFILE, __LINE__);
#endif
                        return ERROR_FILE_NOT_FOUND;
                    }

            if ((level == FIL_QUERYEASFROMLIST) ||
                (level == FIL_QUERYALLEAS)) {
                peaop                    = (PEAOP)pData;
                if ((rc = FSH_PROBEBUF(PB_OPWRITE, (pchar)peaop, 4)) != NO_ERROR) {
                    fs_err(FUNC_FS_PATHINFO, FUNC_FSH_PROBEBUF, rc, THISFILE, __LINE__);
                    return rc;
                } /* end if */
                peaop->fpFEAList->cbList = 4;       // NO EAS for ext2
            } else {
                    if ((rc = ino_to_fileinfo(
                                              p_file->f_inode,
                                              pData,
                                              cbData,
                                              &len,
                                              level,
                                              0,
                                              0,
                                              0,
                                              0, TYPEOP_FILEINFO)) != NO_ERROR) {
                        fs_log("FS_PATHINFO() - ino_to_fileinfo()");
                        if ((rc3 = vfs_close(p_file)) != NO_ERROR) {
                            fs_err(FUNC_FS_PATHINFO, FUNC_CLOSE, rc3, THISFILE, __LINE__);
                        }
                        return rc;
                    }
                    }
                    if ((rc = vfs_close(p_file)) != NO_ERROR) {
                        fs_err(FUNC_FS_PATHINFO, FUNC_CLOSE, rc, THISFILE, __LINE__);
                        return rc;
                    }
                    return NO_ERROR;
            }
            break;

        case PI_SET :
        case PI_SET + 0x10 :
            if (trace_FS_PATHINFO) {
                kernel_printf("FS_PATHINFO - PI_SET - %s", pName);
            }
            rc = NO_ERROR;
            if (Read_Write) {
                if ((p_file = _open_by_name(p_volume, pName, openmode | OPENMODE_READONLY)) != 0) {
                    if ((rc = LockUserBuffer(pData, cbData, lock, LOCK_READ, &lock_lin)) == NO_ERROR) {
                        if ((rc = fileinfo_to_ino(pData, p_file->f_inode, level, cbData, 0)) == NO_ERROR) {
                            /*
                             * Nothing else to do
                             */
                        } else {
                            kernel_printf("fileinfo_to_ino returned %d", rc);
                        }
                        if ((rc2 = VMUnlock(lock_lin)) == NO_ERROR) {
                            /* Nothing else to do */
                        } else {
                            kernel_printf("FS_PATHINFO : VMUnlock() returned %d", rc2);
                            rc = rc2;
                        } /* VMUnlock failed */
                    } else {
                        kernel_printf("FS_PATHINFO : LockUserBuffer() returned %d", rc);
                    }

                    if ((rc2 = vfs_close(p_file)) == NO_ERROR) {
                        /*
                         * Nothing else to do.
                         */
                    } else {
                        fs_err(FUNC_FS_PATHINFO, FUNC_CLOSE, rc, THISFILE, __LINE__);
                        rc = rc2;
                    }
                } else {
#ifdef FS_TRACE
                    fs_err(FUNC_FS_PATHINFO, FUNC_OPEN_BY_NAME, ERROR_PATH_NOT_FOUND, THISFILE, __LINE__);
#endif
                    rc = ERROR_FILE_NOT_FOUND;
                }
            } else {
                rc = ERROR_WRITE_PROTECT;
            }
            break;


        default :
            kernel_printf("FS_PATHINFO( %s ) - unknown flag %d", pName, flag);
            rc = ERROR_INVALID_PARAMETER;
            break;
    } /* end switch */



    return rc;

}

#endif /* #ifndef MINIFSD */
