unit Version; { Encapsulates the version information resource. }

interface

uses
  Windows, SysUtils;

type
  { ID numbers for the different version information strings. These are
    the variable file information strings, only some of which are required. }
  TVariableFileInfoID = (VF_VF_COMMENTS, VF_VF_COMPANY, VF_VF_FILEDESCRIPTION,
    VF_VF_FILEVERSION, VF_VF_INTERNALNAME, VF_VF_LEGALCOPYRIGHT,
    VF_VF_LEGALTRADEMARKS, VF_VF_ORIGINALFILENAME, VF_VF_PRIVATEBUILD,
    VF_VF_PRODUCTNAME, VF_VF_PRODUCTVERSION, VF_VF_SPECIALBUILD);

  { The version information class. }
  TVersionInfo = class(TObject)
  private
    Data: Pointer; { pointer to a block of data containing version information }
    UnknownString: string; { the string that is used for missing fields }
    BaseString: string;  { Base of version query }    
    Prepared: Boolean; { True if the fields have been initialized }
    procedure InitVariableFields; { reads and saves the information }
  public
    constructor Create(UnknownStringID: Integer); virtual;
    destructor Destroy; override;
    function GetVarString(StringID: TVariableFileInfoID): string; { get a variable info field }
    function GetFixedString(StringID: Integer): string;
    function GetFixedFileInfo: TVSFixedFileInfo; { return the fixed file information }
  end;

implementation

const
  NUMVARIABLEFIELDS = 12;

type
  { Variable file information field record }
  TVSVariableFileInfo = record
    Name:  string[32];
    Value: string[64];
  end;

var
  Fixed: TVSFixedFileInfo; { the fixed file information }
  Variable: array [0..NUMVARIABLEFIELDS - 1] of TVSVariableFileInfo; { the variable file information }

{ Initialize the variable field names, and the field values to the "unknown" string. }
procedure TVersionInfo.InitVariableFields;
var
  FieldNumber: Integer;
begin
  Variable[0].Name := 'Comments';
  Variable[1].Name := 'CompanyName';
  Variable[2].Name := 'FileDescription';
  Variable[3].Name := 'FileVersion';
  Variable[4].Name := 'InternalName';
  Variable[5].Name := 'LegalCopyright';
  Variable[6].Name := 'LegalTrademarks';
  Variable[7].Name := 'OriginalFilename';
  Variable[8].Name := 'PrivateBuild';
  Variable[9].Name := 'ProductName';
  Variable[10].Name := 'ProductVersion';
  Variable[11].Name := 'SpecialBuild';

  for FieldNumber := 0 to NUMVARIABLEFIELDS - 1 do
    Variable[FieldNumber].Value := UnknownString;
end;

{ Construct the version information object. Read and save the fields. }
constructor TVersionInfo.Create(UnknownStringID: Integer);
var
  Filename: array [0..259] of Char;
  FixedPointer: Pointer;
  Size, Handle: DWORD;
  OK: Boolean;
  InfoSize: Integer;
  TranslationInfoPointer: Pointer;
  LangID, CharSetID: UINT;
  StringName, VerName: string[64];
  FieldNumber: Integer;
  LangIDStr, CharSetIDStr: string[8];
begin
  inherited Create;

  UnknownString := LoadStr(UnknownStringID);

  InitVariableFields;
  Prepared := False;

  Data := nil;
  FillChar(Fixed, SizeOf(Fixed), 0);

  { Get the filename of our executable. }
  Size := GetModuleFileName(HInstance, Filename, SizeOf(Filename));
  if Size = 0 then
    Exit;

  { Find out the size of the version information block and
    allocate memory for it: }
  Size := GetFileVersionInfoSize(Filename, Handle);
  GetMem(Data, Size);
  if Data = nil then
    Exit;

  { Copy the version information to our data block: }
  OK := GetFileVersionInfo(Filename, Handle, Size, Data);
  if not OK then
    Exit;

  { Get a pointer to the fixed file information: }
  OK := VerQueryValue(Data, '\', FixedPointer, InfoSize);
  if (not OK) or (InfoSize = 0) then
    Exit;

  { Copy the fixed file information to our own record: }
  Move(FixedPointer, Fixed, SizeOf(TVSFixedFileInfo));

  { Get the current translation (language and character set ID: }
  VerQueryValue(Data, '\VarFileInfo\Translation', TranslationInfoPointer, InfoSize);
  LangID := LoWord(DWORD(TranslationInfoPointer^));
  CharSetID := HiWord(DWORD(TranslationInfoPointer^));

  { Convert language and character set IDs to hexadecimal strings: }
  LangIDStr    := IntToHex(LangID, 4);
  CharSetIDStr := IntToHex(CharSetID, 4);

  { Construct a version query string for the relevant translation block: }
  FmtStr(BaseString, '\StringFileInfo\%s%s\', [LangIDStr, CharSetIDStr]);

  { Iterate through the variable file information fields and save them: }
  for FieldNumber := 0 to NUMVARIABLEFIELDS - 1 do
  begin
    VerName := StringName + Variable[FieldNumber].Name;
    Variable[FieldNumber].Value := GetVarString(TVariableFileInfoID(FieldNumber));
  end;
  Prepared := True; { indicate that the variable fields are now prepared }
end;

{ Destroy the version information object }
destructor TVersionInfo.Destroy;
begin
  if Data <> nil then
    FreeMem(Data);

  inherited Destroy;
end;

{ Get the contents of a variable file information field based on the
  field ID number }
function TVersionInfo.GetVarString(StringID: TVariableFileInfoID): string;
var
  StringLength: Integer;
  StringData: Pointer;
  VerName: string;
begin
  if Prepared then { no need to read them from the version information resource }
  begin
    Result := Variable[Integer(StringID)].Value;
    Exit;
  end;

  Result := UnknownString; { prepare for the worst }
  if Data = nil then { no data, can't continue }
    Exit;

  { Construct a version information query for the translation/field }
  VerName := BaseString + Variable[Integer(StringID)].Name;

  { Get the value of the variable file information field: }
  if not VerQueryValue(Data, PChar(VerName), StringData, StringLength) then
    Exit;

  { The query was OK but the string length was zero. Maybe we should
    return an empty string instead of the "unknown" string? }
  if StringLength = 0 then
    Exit;

  Result := PChar(StringData);
end;

{ Format a fixed file information field as a string }
function TVersionInfo.GetFixedString(StringID: Integer): string;
begin
  Result := 'Not implemented.';
end;

function TVersionInfo.GetFixedFileInfo: TVSFixedFileInfo;
begin
  Result := Fixed;
end;

end.

