program Taquin32;
{
Purpose:       Implements the game of Jeu de Taquin (a.k.a. 15-puzzle).
Version:       1.1
System:        Microsoft Windows 95
Compiler:      Borland Delphi 2.01
Programmer:    Jere Kpyaho (jere@sci.fi)
Last modified: 1997-05-01
}

{$DESCRIPTION 'Jeu de Taquin Copyright (C) 1997 Jere Kpyaho' }

{
NOTE: This program contains no hard-coded display strings. All the text that
is rendered to the screen that could possibly be localized into another
language is in the string table. The strings are loaded on demand using the
LoadStr call from the Delphi run-time library.
}

uses
  Windows,
  Messages,
  MMSystem,
  CommDlg,
  CommCtrl,
  SysUtils,
  Classes, { this is a VCL unit, but Delphi has a smart linker --> no bulk }
  Utility in 'Utility.pas',
  Data in 'Data.pas',
  Settings in 'Settings.pas',
  About in 'About.pas',
  WPlace in 'Wplace.pas';

{$I Resource.inc}           { read in the resource ID numbers }
{$R Resource.res}           { link the binary resource file }
{$I Setup.inc}              { pull in BASEKEY for registry operations }

const
  IDC_STATUSBAR = 1;  { status bar control ID }

var
  WindowClassName, WindowTitle: string;
  HelpFileName: string;
  MainWindow, StatusBarWindow: HWND;
  Accelerators: HACCEL;
  GameData: TGameData;
  GameSettings: TSettings;
  TileFont: TLogFont;        { logical font description for tile numbers }
  LastStatusMessageID: UINT; { string ID of last status message }
  DefaultStatusBarText: string;
  MoveCount: Integer;        { count of user moves in current game }

{********************************************************************}

{ Select a new font for the tile numbers using the Windows common
  font selection dialog. }
procedure SelectTileFont(var OK: Boolean);
var
  CF: TChooseFont; { record to drive the font selection dialog }
  IsFontBold, IsFontItalic: Boolean;
begin
  { Initialize the ChooseFont data record and fill in the fields: }
  FillChar(CF, SizeOf(TChooseFont), #0);
  with CF do
  begin
    lStructSize := SizeOf(TChooseFont);
    hwndOwner := MainWindow;
    lpLogFont := @TileFont;
    Flags := CF_SCREENFONTS or       { screen fonts }
             CF_ANSIONLY    or       { ANSI characters only (no symbols) }
             CF_TTONLY      or       { TrueType fonts only }
             CF_FORCEFONTEXIST or    { only allow selection of existing fonts }
             CF_NOVECTORFONTS or     { no vector fonts allowed }
             CF_INITTOLOGFONTSTRUCT; { initialize the LOGFONT structure }
  end;

  { Invoke the font selection dialog: }
  OK := ChooseFont(CF);

  { Adjust our settings based on the selected font: }
  IsFontBold := (TileFont.lfWeight = FW_BOLD);
  { NOTE: some TrueType fonts do not set the font height to exactly FW_BOLD }
  IsFontItalic := (TileFont.lfItalic <> 0);

  GameSettings.SetFont(
    TileFont.lfFaceName,        { name of typeface }
    CF.iPointSize div 10,       { font size, in tenths of a point }
    IsFontBold, IsFontItalic);  { bold/italic flags }
end;

{ Play a sound for various game events }
procedure PlayGameSound(Sound: TGameSound);
var
  SoundName: string;
begin
  if not GameSettings.IsSoundEnabled then { be silent }
    Exit;

  { Get a sound file name for the event: }
  SoundName := GameSettings.GetSoundName(Sound);

  { Use the PlaySound function from the Win32 MM API to play the sound file: }
  PlaySound(PChar(SoundName), 0, SND_ASYNC or SND_NODEFAULT or SND_FILENAME);
end;

{ Initialize the status bar at the bottom of the main window }
function InitStatusBar(ParentWindow: HWND): HWND;
var
  StatusBarWindow: HWND;
  StatusBarRect: TRect;
  StatusBarWidth: Integer;
  Panes: array [0..1] of Integer;
begin
  { Load the default text to use in the status bar: }
  DefaultStatusBarText := LoadStr(IDS_STAT_READY);

  { Create the status bar window and return its handle: }
  StatusBarWindow := CreateStatusWindow(
    WS_CHILD or WS_VISIBLE or WS_CLIPSIBLINGS or CCS_BOTTOM,
    PChar(DefaultStatusBarText),
    ParentWindow,
    IDC_STATUSBAR); { *NOT* the number of parts, but the ID }

  if StatusBarWindow <> 0 then
  begin { initialize the parts of the status bar }
    GetWindowRect(StatusBarWindow, StatusBarRect);
    StatusBarWidth := StatusBarRect.Right - StatusBarRect.Left;
    Panes[0] := (3 * StatusBarWidth) div 4; { three quarters of the width }
    Panes[1] := -1;
    SendMessage(StatusBarWindow, SB_SETPARTS, 2, LParam(@Panes));
  end;
  Result := StatusBarWindow;
end;

{ Set the text of the status bar to the string with the number MessageID }
procedure SetStatusMessage(MessageID: UINT);
var
  StatusMessage: string;
begin
  StatusMessage := LoadStr(MessageID);
  SendMessage(StatusBarWindow, SB_SETTEXT, 0, LParam(PChar(StatusMessage)));
  LastStatusMessageID := MessageID; { save the last message ID }
end;


procedure UpdateMoveCountDisplay;
begin
  SendMessage(StatusBarWindow, SB_SETTEXT, 1,
    LParam(PChar(#9 + #9 + IntToStr(MoveCount))));
  { The two #9 or TAB characters indicate that the status bar text }
  { should be right-justified. }
end;

{********************************************************************}

{ Handle the WM_CREATE window message for the main window. }
procedure Taquin_OnCreate(Window: HWND; var Msg: TWMCreate);
var
  DC: HDC; { device context handle for getting font information }
begin
  { Load the name of our help file from the string table: }
  HelpFileName := LoadStr(IDS_HELPFILENAME);

  { Try to create the status bar window: }
  StatusBarWindow := InitStatusBar(Window);
  if StatusBarWindow = 0 then
  begin
    Msg.Result := Longint(False);
    Exit
  end;
  if not GameSettings.StatusBar then
    ShowWindow(StatusBarWindow, SW_HIDE);

  { Set default tile font: }
  FillChar(TileFont, SizeOf(TLogFont), 0); { initialize the record }
  DC := GetDC(Window); { get a device context handle for the window }
  TileFont.lfHeight := -MulDiv(GameSettings.GetFontSize,
    GetDeviceCaps(DC, LOGPIXELSY), 72);
  StrPCopy(TileFont.lfFaceName, GameSettings.GetFontName);
  if GameSettings.IsFontBold then
    TileFont.lfWeight := FW_BOLD
  else
    TileFont.lfWeight := FW_NORMAL;
  if GameSettings.IsFontItalic then
    TileFont.lfItalic := 1
  else
    TileFont.lfItalic := 0;
  ReleaseDC(Window, DC);

  { Set the initial status bar message: }
  SetStatusMessage(IDS_STAT_PRESSTOPLAY);

  Msg.Result := Longint(True); { window creation was successful }
end;

{ Handle the WM_SIZE window message for the main window. }
{ NOTE: the size of our main window is fixed, but we need to handle }
{ this message to get the initial size of the status bar right. }
procedure Taquin_OnSize(Window: HWND; var Msg: TWMSize);
var
  StatusBarRect: TRect;
  StatusBarWidth, StatusBarHeight: Word;
  StatusBarX, StatusBarY: Word;
begin
  { Get the dimensions of the status bar window: }
  GetWindowRect(StatusBarWindow, StatusBarRect);
  StatusBarHeight := StatusBarRect.Bottom - StatusBarRect.Top;

  { Calculate new position and size for the status bar }
  { to keep it aligned to the bottom of the client area. }
  StatusBarX := 0;
  StatusBarY := Msg.Height - StatusBarHeight;
  StatusBarWidth := Msg.Width;

  { Resize the status bar window to match the main window: }
  MoveWindow(StatusBarWindow,
    StatusBarX, StatusBarY,
    StatusBarWidth, StatusBarHeight, True);
end;

{ Handle the WM_MENUSELECT message to show the menu help strings
  in the status bar. }
procedure Taquin_OnMenuSelect(Window: HWND; var Msg: TWMMenuSelect);
var
  OriginalMessage: TMessage;
  StringNumberBase: UINT;
begin
  OriginalMessage := TMessage(Msg); { cast back to get wParam and lParam }
  StringNumberBase := 0;
  MenuHelp(WM_MENUSELECT, { always }
    OriginalMessage.WParam,
    OriginalMessage.LParam,
    0, { main menu handle not needed }
    HInstance,
    StatusBarWindow,
    @StringNumberBase);
end;

{ Handle the WM_INITMENU message, which is sent to the window when
  a menu is opened. This handler sets the checkmark alongside the
  Sound menu command depending on the user interface settings. }
procedure Taquin_OnInitMenu(Window: HWND; var Msg: TWMInitMenu);
var
  State: Integer;
begin
  if GameSettings.IsSoundEnabled then
    State := mf_Checked
  else
    State := mf_Unchecked;
  CheckMenuItem(Msg.Menu, idm_Options_Sound, State);
end;

{ Handle the WM_COMMAND message for menu commands. }
procedure Taquin_OnCommand(Window: HWND; var Msg: TWMCommand);
var
  ReturnCode: Integer;
  OldCursor: HCURSOR;
  OK: Boolean;
  SoundEnabled: Boolean;
  StatusBarRect, WindowRect: TRect;
  StatusBarHeight, WindowHeight, WindowWidth: Integer;
  StatusBarVisible: Boolean;
begin
  case Msg.ItemID of { handle each menu item ID we know }
    IDM_GAME_NEW:
    begin
      { If there is a game in progress, ask for confirmation before scrambling: }
      if GameData.State = gsPlaying then { updated to use property }
      begin
        ShowMessage(HInstance, Window,
          IDS_MSG_RESTARTGAME, IDS_WINDOWTITLE,
          MB_ICONQUESTION or MB_YESNO, ReturnCode);
        if ReturnCode = IDNO then { don't mess up the current game }
          Exit;
      end;
      MoveCount := 0; { reset the move count }
      UpdateMoveCountDisplay;
      GameData.Initialize;
      GameData.State := gsScrambling; { updated to use property }
      SetStatusMessage(IDS_STAT_SCRAMBLING);

      { Set the hourglass cursor, scramble the game, and restore the cursor. }
      OldCursor := SetCursor(LoadCursor(0, IDC_WAIT));
      ShowCursor(True);
      GameData.Scramble(Window);
      ShowCursor(False);
      SetCursor(OldCursor);

      SetStatusMessage(IDS_STAT_READY);
      GameData.State := gsPlaying;
    end;

    IDM_GAME_EXIT:
      SendMessage(Window, WM_CLOSE, 0, 0);

    IDM_OPTIONS_SOUND:
    begin
      SoundEnabled := GameSettings.IsSoundEnabled;
      if SoundEnabled then
      begin
        GameSettings.DisableSound;
        CheckMenuItem(GetMenu(Window), IDM_OPTIONS_SOUND, MF_UNCHECKED);
      end
      else
      begin
        GameSettings.EnableSound;
        CheckMenuItem(GetMenu(Window), IDM_OPTIONS_SOUND, MF_CHECKED);
      end;
    end;

    IDM_OPTIONS_FONT:
    begin
      SelectTileFont(OK);
      if OK then
        InvalidateRect(Window, nil, True);
    end;

    IDM_OPTIONS_STATUSBAR:
    begin
      GetWindowRect(StatusBarWindow, StatusBarRect);
      StatusBarHeight := StatusBarRect.Bottom - StatusBarRect.Top;
      GetWindowRect(Window, WindowRect);
      WindowWidth := WindowRect.Right - WindowRect.Left;
      WindowHeight := WindowRect.Bottom - WindowRect.Top;

      StatusBarVisible := IsWindowVisible(StatusBarWindow);
      if StatusBarVisible then
      begin
        ShowWindow(StatusBarWindow, SW_HIDE);
        CheckMenuItem(GetMenu(Window), IDM_OPTIONS_STATUSBAR, MF_UNCHECKED);
        GameSettings.StatusBar := False;
      end
      else { status bar not visible }
      begin
        ShowWindow(StatusBarWindow, SW_SHOW);
        CheckMenuItem(GetMenu(Window), IDM_OPTIONS_STATUSBAR, MF_CHECKED);
        GameSettings.StatusBar := True;
      end;
      AdjustForStatusBar(Window, StatusBarWindow, not StatusBarVisible);
    end;

    IDM_HELP_CONTENTS:
      WinHelp(Window, PChar(HelpFileName), HELP_FINDER, 0);

    IDM_HELP_ABOUT:
      DialogBox(HInstance, 'AboutDlg', Window, @About_DlgProc);
  end; { case Msg.ItemID }
end;

{ Handle the WM_LBUTTONDOWN message to respond to mouse clicks. }
procedure Taquin_OnLButtonDown(Window: HWND; var Msg: TWMLButtonDown);
var
  MouseColumn, MouseRow: Integer;
  BlankColumn, BlankRow: Integer;
  Column, Row: Integer;
  ReturnCode: Integer;
  PaintRect: TRect;
  SolvedMessage: string;
begin
  { If there is no game in progress, we don't need to do anything }
  if GameData.State = gsIdling then
    Exit;

  { If the last attempt to move was blocked, restore the 'Ready' message. }
  if LastStatusMessageID = IDS_STAT_BLOCKED then
    SetStatusMessage(IDS_STAT_READY);

  { Calculate the row and column of the mouse click. The mouse coordinates
    are zero-based, so we adjust the row and column to be one-based. }
  MouseRow := Msg.YPos div TileHeight;
  Inc(MouseRow);

  MouseColumn := Msg.XPos div TileWidth;
  Inc(MouseColumn);

  BlankRow    := GameData.GetBlankRow;
  BlankColumn := GameData.GetBlankColumn;

  { Check that the mouse row and column are valid, and that the clicked cell
    is not the blank cell. }
  if (not (MouseRow in [1..NUMTILES])) or (not (MouseColumn in [1..NUMTILES])) or { out of bounds }
     ((MouseRow <> BlankRow) and (MouseColumn <> BlankColumn)) or { not adjacent to the blank cell }
     ((MouseRow = BlankRow) and (MouseColumn = BlankColumn)) then { user clicked the blank cell }
  begin
    if GameData.State <> gsScrambling then { only respond while actually playing }
    begin
      SetStatusMessage(IDS_STAT_BLOCKED);  { show the "blocked" message }
      PlayGameSound(gsBlocked);            { play the "blocked" sound }
    end;
    Exit;
  end;

  { What follows is a lengthy series of checks to find out if the click is
    supposed to move just one or several tiles. }
  if MouseRow = BlankRow then
  begin
    if MouseColumn < BlankColumn then { clicked to the left of the blank cell }
    begin
      Column := BlankColumn;
      while Column > MouseColumn do
      begin
        GameData.SetTileAt(BlankRow, Column, GameData.GetTileAt(BlankRow, Column - 1));
        Dec(Column);
      end;
    end
    else { clicked to the right of the blank cell }
    begin
      Column := BlankColumn;
      while Column < MouseColumn do
      begin
        GameData.SetTileAt(BlankRow, Column, GameData.GetTileAt(BlankRow, Column + 1));
        Inc(Column);
      end;
    end;
  end
  else { the blank cell is not on the clicked row }
  begin
    if MouseRow < BlankRow then { clicked above the blank cell }
    begin
      Row := BlankRow;
      while Row > MouseRow do
      begin
        GameData.SetTileAt(Row, BlankColumn, GameData.GetTileAt(Row - 1, BlankColumn));
        Dec(Row);
      end;
    end
    else { clicked below the blank cell }
    begin
      Row := BlankRow;
      while Row < MouseRow do
      begin
        GameData.SetTileAt(Row, BlankColumn, GameData.GetTileAt(Row + 1, BlankColumn));
        Inc(Row);
      end;
    end;
  end;

  { The blank tile is shifted to the place of the original mouse click. }
  GameData.SetTileAt(MouseRow, MouseColumn, EMPTY);

  { For optimized repainting of the window,
    calculate the invalid rectangle based on the position
    of the blank cell and the clicked cell. }
  Dec(MouseRow);
  Dec(MouseColumn);
  Dec(BlankRow);
  Dec(BlankColumn);

  { Now the (x,y) coordinates are zero-based once again. }

  { Compute the rectangle that encloses the cells whose positions
    did move as a result of the mouse click. }
  PaintRect := Rect(TileWidth * Min(MouseColumn, BlankColumn),
                    TileHeight * Min(MouseRow, BlankRow),
                    TileWidth * (Max(MouseColumn, BlankColumn) + 1),
                    TileHeight * (Max(MouseRow, BlankRow) + 1));

  { Invalidate only the computed rectangle: }
  InvalidateRect(Window, @PaintRect, True);

  if GameData.State = gsScrambling then
    Exit;

  Inc(MoveCount);        { update move count }
  UpdateMoveCountDisplay;
  PlayGameSound(gsMove); { play the "move" sound }

  { Already tested idling and scrambling, must be playing. }
  if GameData.Solved then
  begin
    { Update window to reflect the true ending position on the board. }
    InvalidateRect(Window, nil, True);
    UpdateWindow(Window);

    PlayGameSound(gsSolved); { play the "solved" fanfare }

    { Show a message indicating that the puzzle has been solved: }
    FmtStr(SolvedMessage, LoadStr(IDS_MSG_SOLVED), [MoveCount]);
    MessageBox(Window, PChar(SolvedMessage), PChar(WindowTitle), MB_ICONEXCLAMATION or MB_OK);

    GameData.State := gsIdling;

    { Ask the player if s/he wants to have another go: }
    ShowMessage(HInstance, Window, IDS_MSG_ANOTHER, IDS_WINDOWTITLE,
      MB_ICONQUESTION or MB_YESNO, ReturnCode);
    if ReturnCode = IDYES then
      SendMessage(Window, WM_COMMAND, IDM_GAME_NEW, 0) { send a menu command message }
    else
      SendMessage(Window, WM_CLOSE, 0, 0); { close the window }
  end;
end;

{ Handle the WM_KEYDOWN message by sending simulated left mouse button clicks. }
procedure Taquin_OnKeyDown(Window: HWND; var Msg: TWMKeyDown);
var
  MouseRow, MouseColumn: Integer;
begin
  MouseRow := GameData.GetBlankRow - 1;
  MouseColumn := GameData.GetBlankColumn - 1;

  case Msg.CharCode of
    VK_LEFT:  Inc(MouseColumn);
    VK_RIGHT: Dec(MouseColumn);
    VK_UP:    Inc(MouseRow);
    VK_DOWN:  Dec(MouseRow);
  else
    Exit
  end;

  { Send the updated mouse position as a WM_LBUTTONDOWN message. }
  SendMessage(Window, WM_LBUTTONDOWN, 0,
    MakeLParam(MouseColumn * TileWidth, MouseRow * TileHeight));
end;

{ Handle the WM_PAINT message. The tiles are drawn as a result of this message. } 
procedure Taquin_OnPaint(Window: HWND; var Msg: TWMPaint);
var
  PaintDC: HDC;
  PaintStruct: TPaintStruct;
  Row, Column: Integer;
  PaintRect, DestRect: TRect;
begin
 { Get a device context for painting on the window. The PaintStruct record
   is updated and contains the invalid rectangle. }
  PaintDC := BeginPaint(Window, PaintStruct);

  { For each tile, check if it needs to be redrawn. }
  for Row := 1 to NUMTILES do
  begin
    for Column := 1 to NUMTILES do
    begin
      { Compute the rectangle that encloses the tile at (Row, Column) and
        save it in the PaintRect record. }
      PaintRect := Bounds(TileWidth * (Column - 1), TileHeight * (Row - 1),
                          TileWidth, TileHeight);

      { Compute the intersection of the invalid rectangle and the
        current tile's rectangle. If they intersect, redraw the tile. }
      if IntersectRect(DestRect, PaintRect, PaintStruct.rcPaint) then
        GameData.PaintTileAt(Row, Column, PaintDC, TileFont);
    end; { for Column }
  end; { for Row }

  { End the painting operation, releasing the device context: }
  EndPaint(Window, PaintStruct);
end;

{ Handle the WM_CLOSE message. }
procedure Taquin_OnClose(Window: HWND; var Msg: TWMClose);
var
  ReturnCode: Integer;
begin
  if GameData.State <> gsPlaying then
    DestroyWindow(Window)
  else { there is a game in progress }
  begin
    { Ask for confirmation: }
    ShowMessage(HInstance, Window,
      IDS_MSG_REALLYQUIT, IDS_WINDOWTITLE,
      MB_ICONQUESTION or MB_YESNO, ReturnCode);
    if ReturnCode = IDYES then
      DestroyWindow(Window);
  end;
end;

{ Handle the WM_DESTROY message. }
procedure Taquin_OnDestroy(Window: HWND; var Msg: TWMDestroy);
var
  WinPlace: TWinPlace; { our window placement class }
  OK: Boolean;
begin
  try
    WinPlace := TWinPlace.Create;
    WinPlace.Save(BaseKey, Window);
  finally
    WinPlace.Free;
  end;

  WinHelp(Window, PChar(HelpFileName), HELP_QUIT, 0); { cancel WinHelp }
  PostQuitMessage(0);
end;

procedure Taquin_OnEndSession(Window: HWND; var Msg: TWMEndSession);
var
  OK: Boolean;
  WinPlace: TWinPlace;
begin
  if Msg.EndSession then
  begin
    try
      WinPlace := TWinPlace.Create;
      WinPlace.Save(BaseKey, Window);
    finally
      WinPlace.Free;
    end;
  end;
end;

{********************************************************************}

{ The window procedure for the main window. Use's Delphi's polymorphic
  message types to pass the correct message records to the message
  handler procedures. }
function Taquin_WndProc(Window: HWND; Msg, WParam, LParam: Longint): Longint;
  stdcall; export;
var
  WindowMessage: TMessage;
begin
  WindowMessage.Msg := Msg;
  WindowMessage.WParam := WParam;
  WindowMessage.LParam := LParam;
  WindowMessage.Result := 0;

  case Msg of
    WM_CREATE:
      Taquin_OnCreate(Window, TWMCreate(WindowMessage));
    WM_SIZE:
      Taquin_OnSize(Window, TWMSize(WindowMessage));
    WM_MENUSELECT:
      Taquin_OnMenuSelect(Window, TWMMenuSelect(WindowMessage));
    WM_INITMENU:
      Taquin_OnInitMenu(Window, TWMInitMenu(WindowMessage));
    WM_COMMAND:
      Taquin_OnCommand(Window, TWMCommand(WindowMessage));
    WM_LBUTTONDOWN:
      Taquin_OnLButtonDown(Window, TWMLButtonDown(WindowMessage));
    WM_KEYDOWN:
      Taquin_OnKeyDown(Window, TWMKeyDown(WindowMessage));
    WM_PAINT:
      Taquin_OnPaint(Window, TWMPaint(WindowMessage));
    WM_CLOSE:
      Taquin_OnClose(Window, TWMClose(WindowMessage));
    WM_DESTROY:
      Taquin_OnDestroy(Window, TWMDestroy(WindowMessage));
    WM_ENDSESSION:
      Taquin_OnEndSession(Window, TWMEndSession(WindowMessage));
  else
    Result := DefWindowProc(Window, Msg, WParam, LParam);
    Exit
  end;

  Result := WindowMessage.Result;
end;

{********************************************************************}

{ Fill in and register the window class. }
procedure InitApplication(var OK: Boolean);
var
  WindowClass: TWndClassEx;
begin
  WindowClass.cbSize        := SizeOf(TWndClassEx);
  WindowClass.style         := 0;
  WindowClass.lpfnWndProc   := @Taquin_WndProc;
  WindowClass.cbClsExtra    := 0;
  WindowClass.cbWndExtra    := 0;
  WindowClass.hInstance     := HInstance;
  WindowClass.hIcon         := LoadIcon(HInstance, 'AppIcon');
  WindowClass.hCursor       := LoadCursor(HInstance, 'HandCursor');
  WindowClass.hbrBackground := HBrush(COLOR_WINDOW + 1);
  WindowClass.lpszMenuName  := 'MainMenu';
  WindowClass.lpszClassName := PChar(WindowClassName);
  { We use LoadImage here instead of LoadIcon because the latter can
    only load standard size icons, and we need a small icon here.
    At least that's what the Win32 SDK documentation says. I think
    I even tried it once. }
  WindowClass.hIconSm       := LoadImage(HInstance, 'AppIcon', IMAGE_ICON,
                                         GetSystemMetrics(SM_CXSMICON),
                                         GetSystemMetrics(SM_CYSMICON),
                                         LR_DEFAULTCOLOR );
  OK := (RegisterClassEx(WindowClass) <> 0);
end;

{ Initialize game data and create the program window. }
procedure InitInstance(var OK: Boolean);
var
  WindowWidth, WindowHeight: Integer;
  WindowRect, StatusRect: TRect;
  WinPlace: TWinPlace;
begin
  { Game data class' constructor initializes tiles and
    sets the game state to "idling" }
  GameData := TGameData.Create;

  { Create the game settings object and load settings from the registry: }
  GameSettings := TSettings.Create;
  GameSettings.ReadFromRegistry;

  { Load the keyboard accelerators: }
  Accelerators := LoadAccelerators(HInstance, 'AccTable');
  OK := Accelerators <> 0;
  if not OK then
    Exit;

  { Calculate the window dimensions: }
  WindowWidth  := NUMTILES * TileWidth  + 2 * GetSystemMetrics(SM_CXFIXEDFRAME);
  WindowHeight := NUMTILES * TileHeight + 2 * GetSystemMetrics(SM_CYFIXEDFRAME) +
                                              GetSystemMetrics(SM_CYCAPTION) +
                                              GetSystemMetrics(SM_CYMENU);

  { Load the window caption from the string table: }
  WindowTitle := LoadStr(IDS_WINDOWTITLE);

  { Create the main window }
  MainWindow := CreateWindow(PChar(WindowClassName), PChar(WindowTitle),
    WS_OVERLAPPED or WS_SYSMENU or WS_MINIMIZEBOX,
    CW_USEDEFAULT, CW_USEDEFAULT,
    WindowWidth, WindowHeight,
    0, 0, HInstance, nil);
  OK := MainWindow <> 0;
  if not OK then
    Exit;

  try
    WinPlace := TWinPlace.Create;
    WinPlace.Restore(BaseKey, MainWindow, OK);
  finally
    WinPlace.Free;
  end;

  if not OK then { restore failed, so this is the first run }
  begin
    { Center the window on the screen: }
    CenterWindow(MainWindow, GetDesktopWindow, OK);
    ShowWindow(MainWindow, SW_SHOWDEFAULT);
    if GameSettings.StatusBar then
      AdjustForStatusBar(MainWindow, StatusBarWindow, True);
  end;
  UpdateWindow(MainWindow);

  OK := True;
end;

{ Actions necessary before terminating the program }
procedure TermInstance;
begin
  GameData.Free;
  GameSettings.WriteToRegistry;
  GameSettings.Free;
end;

{********************************************************************}

var
  FirstWindow: HWND;
  OK: Boolean;
  ReturnCode: Integer;
  WindowMessage: TMsg;

begin
  Randomize;  { initialize the random number generator }

  { Load the name of our window class from the string table: }
  WindowClassName := LoadStr(IDS_WINDOWCLASSNAME);

  { If there is a window based on our class, activate it and quit: }
  FirstWindow := FindWindow(PChar(WindowClassName), nil);
  if FirstWindow <> 0 then
  begin
    if IsIconic(FirstWindow) then
      ShowWindow(FirstWindow, SW_RESTORE);
    SetForegroundWindow(FirstWindow);
    Halt(0);
  end;

  InitApplication(OK);
  if not OK then
  begin
    ShowMessage(HInstance, 0, IDS_MSG_CLASSREGFAILED, IDS_MSG_ERROR,
                MB_ICONSTOP or MB_OK, ReturnCode);
    Halt(0);
  end;

  InitInstance(OK);
  if not OK then
  begin
    ShowMessage(HInstance, 0, IDS_MSG_WINCREATEFAILED, IDS_MSG_ERROR,
                MB_ICONSTOP or MB_OK, ReturnCode);
    Halt(0);
  end;

  InitCommonControls(); { initialize the Win32 common control library }

  { The message loop starts here: }
  while GetMessage(WindowMessage, 0, 0, 0) do
  begin
    if TranslateAccelerator(MainWindow, Accelerators, WindowMessage) = 0 then
    begin
      TranslateMessage(WindowMessage);
      DispatchMessage(WindowMessage);
    end;
  end;

  TermInstance; { clean up before leaving }

  Halt(WindowMessage.wParam);
end.

