unit Settings;
{ The Jeu de Taquin settings that are saved in the Windows 95 registry.
  Unit version: 1.1 }

interface

uses
  Windows, Data;

type
  TGameSound = (gsMove, gsBlocked, gsSolved);

  TSettings = class(TObject)
  private
    Changed: Boolean;
    SoundEnabled: Boolean;
    FontName: string;
    FontSize: Integer;
    FontBold, FontItalic: Boolean;
    MoveSoundName: string;
    BlockedSoundName: string;
    SolvedSoundName: string;
    FStatusBar: Boolean;
  public
    constructor Create; virtual;
    procedure   EnableSound;
    procedure   DisableSound;
    function    IsSoundEnabled: Boolean;
    procedure   SetFont(NewFontName: string; NewFontSize: Integer; IsFontBold, IsFontItalic: Boolean);
    function    GetFontSize: Integer;
    function    IsFontBold: Boolean;
    function    IsFontItalic: Boolean;
    function    GetFontName: string;
    procedure   SetSoundName(Sound: TGameSound; NewSoundName: string);
    function    GetSoundName(Sound: TGameSound): string;
    property    StatusBar: Boolean read FStatusBar write FStatusBar;
    procedure   ReadFromRegistry;
    procedure   WriteToRegistry;
  end;

implementation

uses
  SysUtils,
  Registry;

{$I Resource.inc}
{$I Setup.inc} { Load our base registry key (BASEKEY) }

constructor TSettings.Create;
begin
  inherited Create;

  Changed := False;
  SoundEnabled := True;
  FontName := 'Arial';
  FontSize := 18;
  FontBold := True;
  FontItalic := False;

  StatusBar := True;

  { Load default names for the sounds: }
  MoveSoundName    := LoadStr(IDS_DEFAULTMOVESOUND);
  BlockedSoundName := LoadStr(IDS_DEFAULTBLOCKEDSOUND);
  SolvedSoundName  := LoadStr(IDS_DEFAULTSOLVEDSOUND);
end;

procedure TSettings.EnableSound;
begin
  SoundEnabled := True;
  Changed := True;
end;

procedure TSettings.DisableSound;
begin
  SoundEnabled := False;
  Changed := True;
end;

function TSettings.IsSoundEnabled: Boolean;
begin
  Result := SoundEnabled;
end;

procedure TSettings.SetFont(NewFontName: string; NewFontSize: Integer; IsFontBold, IsFontItalic: Boolean);
begin
  FontName := NewFontName;

  if NewFontSize < 8 then
    NewFontSize := 8;
  if NewFontSize > 48 then
    NewFontSize := 48;
  FontSize := NewFontSize;

  FontBold := IsFontBold;
  FontItalic := IsFontItalic;

  Changed := True;
end;

function TSettings.GetFontSize: Integer;
begin
  Result := FontSize;
end;

function TSettings.IsFontBold: Boolean;
begin
  Result := FontBold;
end;

function TSettings.IsFontItalic: Boolean;
begin
  Result := FontItalic;
end;

function TSettings.GetFontName: string;
begin
  Result := FontName;
end;

procedure TSettings.ReadFromRegistry;
var
  R: TRegistry;
begin
  R := TRegistry.Create;
  try
    { Check if the HKLM key exists. }
    R.RootKey := HKEY_LOCAL_MACHINE;
    if not R.KeyExists(BASEKEY) then
      Exit; { use defaults set by constructor }

    R.RootKey := HKEY_CURRENT_USER;
    R.OpenKey(BASEKEY + '\Options', True);
    try
      if R.ValueExists('Sound') then
        SoundEnabled := R.ReadBool('Sound');
      if R.ValueExists('FontName') then
        FontName := R.ReadString('FontName');
      if R.ValueExists('FontSize') then
        FontSize := R.ReadInteger('FontSize');
      if R.ValueExists('FontBold') then
        FontBold := R.ReadBool('FontBold');
      if R.ValueExists('FontItalic') then
        FontItalic := R.ReadBool('FontItalic');
      if R.ValueExists('StatusBar') then
        StatusBar := R.ReadBool('StatusBar');
    finally
      R.CloseKey;
    end;

    R.OpenKey(BASEKEY + '\Sounds', True);
    try
      if R.ValueExists('MoveSound') then
        MoveSoundName := R.ReadString('MoveSound');
      if R.ValueExists('BlockedSound') then
        BlockedSoundName := R.ReadString('BlockedSound');
      if R.ValueExists('SolvedSound') then
        SolvedSoundName := R.ReadString('SolvedSound');
    finally
      R.CloseKey;
    end;

  finally
    R.Free;
  end;

  Changed := True;
end;

procedure TSettings.WriteToRegistry;
var
  R: TRegistry;
begin
  if not Changed then
    Exit;

  R := TRegistry.Create;
  try
    R.RootKey := HKEY_LOCAL_MACHINE;
    R.CreateKey(BASEKEY);

    R.RootKey := HKEY_CURRENT_USER;
    R.OpenKey(BASEKEY + '\Options', True);
    try
      R.WriteBool   ('Sound',      SoundEnabled);
      R.WriteString ('FontName',   FontName);
      R.WriteInteger('FontSize',   FontSize);
      R.WriteBool   ('FontBold',   FontBold);
      R.WriteBool   ('FontItalic', FontItalic);
      R.WriteBool   ('StatusBar',  StatusBar);
    finally
      R.CloseKey;
    end;

    R.OpenKey(BASEKEY + '\Sounds', True);
    try
      R.WriteString('MoveSound',    MoveSoundName);
      R.WriteString('BlockedSound', BlockedSoundName);
      R.WriteString('SolvedSound',  SolvedSoundName);
    finally
      R.CloseKey;
    end;

  finally
    R.Free;
  end;

  Changed := False; { we actually close immediately after this anyway... }
end;

procedure TSettings.SetSoundName(Sound: TGameSound; NewSoundName: string);
begin
  case Sound of
    gsMove:    MoveSoundName    := NewSoundName;
    gsBlocked: BlockedSoundName := NewSoundName;
    gsSolved:  SolvedSoundName  := NewSoundName;
  end;
end;

function TSettings.GetSoundName(Sound: TGameSound): string;
var
  SoundName: string;
begin
  case Sound of
    gsMove:    SoundName := MoveSoundName;
    gsBlocked: SoundName := BlockedSoundName;
    gsSolved:  SoundName := SolvedSoundName;
  end;
  Result := SoundName;
end;

end.

