unit MainForm;

{
Revision history:

1997 Jan 14  V0.0.0  First version
1997 Jan 16  V0.0.2  Allow timestamp instead of checksum
                     Use inherited file properties dialog box
1997 Jan 19  V1.0.0  Add icon display to file properties dialog box
                     Make timestamp the default method for speed and
                     integrity of file access date
1997 Jan 27  V1.0.2  Remove code timing the components of the compare phase
                     Store directory name separately from file name to save space
1997 Jan 29  V1.0.4  Clear directory list at end of compare phase
1997 Feb 03  V1.0.6  Make drives listbox owner-draw
                     Make floppy, if chosen, a "master" directory
1997 Feb 12  V1.0.8  Update FileProperties form, decode date/time 0 as "unknown"
1997 Apr 02  V1.1.0  Make file list box hint the filename (for long paths!)
                     Save and optionally restore duplicate file list
                     By default, ignore files in Win 95 SYSBCKUP directory
                     By default, ignore zero-length files
                     Replace ListBox with ListView (both Drives and Results)
                     Correct: missing FindClose in do_checksum routine
                     Correct: remove deleted file from the duplicates list
1997 Apr 07  V1.1.2  Use ShellAPI function to move file to recycle bin
1997 May 13  V1.1.4  Use my own TFileList component
                     Don't show properties/delete box for non-existant files
                     Put source files in sub-folder
                     Force checksum routine to return 31-bit value
1997 May 23  V1.2.0  Move to Delphi 3.0
                     Don't leave singletons in the duplicates list
                     Correct property display for sequential compressed files
                     Don't allow ColumnClick on the FileListView - set False
1997 Oct 08  V1.2.2  Move to Delphi 3.01
                     Handle large font displays better.
                     Use TreeScanner with FindHiddenXX options
                     Don't build against run-time VCL30.DPL
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, TreeScan, ComCtrls, ExtCtrls, DiskList, FileList;

type
  TChecksum = Longint;   // for rapid checksum on the first 512 bytes of a file

type
  TfrmMain = class(TForm)
    StatusBar1: TStatusBar;
    GroupBox1: TGroupBox;
    edtFileMask: TEdit;
    Label2: TLabel;
    btnSearch: TButton;
    GroupBox2: TGroupBox;
    edtRootDir: TEdit;
    Label7: TLabel;
    Timer1: TTimer;
    btnExit: TButton;
    GroupBox3: TGroupBox;
    chkCheckFileTimestamp: TCheckBox;
    chkFloppyIsMaster: TCheckBox;
    chkSkipSysbckup: TCheckBox;
    chkSkipEmptyFiles: TCheckBox;
    TreeScanner1: TTreeScanner;
    DiskListView1: TDiskListView;
    FileListView1: TFileListView;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnCloseClick(Sender: TObject);
    procedure TreeScanner1FileFound(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure TreeScanner1DirectoryFound(Sender: TObject);
    procedure btnSearchClick(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
    procedure btnExitClick(Sender: TObject);
    procedure TreeScanner1DirectoryDone(Sender: TObject);
    procedure FileListView1DblClick(Sender: TObject);
    procedure FileListView1Click(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure DiskListView1Change(Sender: TObject; Item: TListItem;
      Change: TItemChange);
  private
    { Private declarations }
    file_list: TStringList;       // list of files, with pointer to directory
    dir_list: TStringList;        // list of directories
    master_list: TStringList;     // simple list of ASCII sizes
    current_dir: integer;         // index of current directory in its list
    comparing: boolean;           // true when a compare in progress
    continue_compare: boolean;    // user sets this false to stop a scan
    duplicate_bytes: integer;     // total number of bytes (excluding "master")
    duplicate_files: integer;     // total number of duplicate files
    saved_duplicates_filename: string;
    scan_state: (all_files, master_scan, filtered_scan);
    skip_sysbckup_folder: boolean;   // true if we ignore Win 95 backup files
    skip_this_folder: boolean;       // true if the current folder is to be skipped
    process_empty_files: boolean;    // true if not ingoring zero-length files
    check_file_timestamp: boolean;   // true for proper checksum comparison
    progress_bar: TProgressBar;
    min_width, min_height: integer;
    procedure find_files_to_compare;
    procedure compare_files;
    procedure update_compare_display;
    function do_checksum (size: integer;  filename: String): TChecksum;
    function files_match (size: integer;  name1, name2: String): boolean;
    function parse_list_view (lvw: TListView): string;
  protected
    procedure GetMinMaxInfo (var info: TWMGetMinMaxInfo);  message WM_GETMINMAXINFO;
  public
    { Public declarations }
  end;

var
  frmMain: TfrmMain;


implementation

uses ShellAPI, FilProps;

{$R *.DFM}

const
  saved_filename = 'FindDupl.lis';        // where the duplicate list is saved
  duplicate_group_prefix = 'Duplicate files .....';


procedure TfrmMain.FormCreate(Sender: TObject);
begin
  // limit the minimum form size
  min_width := Width - 10;
  min_height := Height - 50;

  // Create the list to hold the file sizes and names found from the
  // directory scan.  This must be a sorted list to keep same sized
  // files together - a fundamental requirement.  The "object" field
  // contains pointers to the DIR_LIST
  file_list := TStringList.Create;
  with file_list do
    begin
    Sorted := True;
    Duplicates := dupAccept;
    end;

  // list of directories
  dir_list := TStringList.Create;
  dir_list.Sorted := False;

  // master list - contains just the master sizes
  master_list := TStringList.Create;
  with master_list do
    begin
    Sorted := True;
    Duplicates := dupAccept;
    end;

  progress_bar := TProgressBar.Create (Self);
  with progress_bar do
    begin
    Parent := StatusBar1;
    Left := 0;
    Top := 2;
    Width := StatusBar1.Panels [0].Width;
    Height := StatusBar1.Height - 2;
    Visible := False;
    end;

  saved_duplicates_filename :=
                   ExtractFileDir (Application.ExeName) + '\' + saved_filename;

  comparing := False;
end;


procedure TfrmMain.FormDestroy(Sender: TObject);
var
  i: integer;
  sl: TStringList;
begin
  // save the contents of the current search - if not blank
  // convert the ListView into a string list
  if FileListView1.Items.Count > 0 then     // do the save
    begin
    sl := TStringList.Create;
    with FileListView1.Items do
      for i := 0 to Count - 1 do       // scan down the list view
        with Item [i] do               // the current item (row)
          if SubItems.Count = 0        // are there any sub-items?
          then sl.Add ('"' + Caption + '"')    // no - just save the caption
          else sl.Add ('"' + Caption + '",' + SubItems.CommaText);  // yes - save the lot
    // just in case the "save" fails (e.g. read-only directory or file)
    try
      sl.SaveToFile (saved_duplicates_filename);
    except
    end;
    sl.Free;
    end;

  // return the items we allocated in the Create method
  progress_bar.Free;
  master_list.Free;
  file_list.Free;
  dir_list.Free;
end;


procedure TfrmMain.GetMinMaxInfo (var info: TWMGetMinMaxInfo);
begin
  with info.MinMaxInfo.ptMinTrackSize do
    begin
    x := min_width;
    y := min_height;
    end;
end;


procedure TfrmMain.FormResize(Sender: TObject);
begin
  // Handle all the resizing stuff.
  // We could do this with panels rather than all this computation.....
  // position the search controls group box
  GroupBox3.Left := frmMain.ClientWidth - 8 - GroupBox3.Width;

  // size the what to search for group box
  GroupBox1.Width := GroupBox3.Left - 24;

  // position the buttons in this group box
  btnExit.Left := GroupBox1.Width - 8 - btnExit.Width;
  btnSearch.Left := btnExit.Left - 8 - btnSearch.Width;

  // position the root directory and root file edit boxes, and their captions
  edtRootDir.Left := GroupBox1.Width - 8 - edtRootDir.Width;
  edtFileMask.Left := edtRootDir.Left;
  Label7.Left := edtRootDir.Left - Label7.Width;
  Label2.Left := edtFileMask.Left - Label2.Width;

  // expand the disk list view to fill the remaining available width
  DiskListView1.Width := Label2.Left - 24;

  // fill the lower half of the form with the list of duplicates found
  GroupBox2.Width := frmMain.ClientWidth - 16;
  GroupBox2.Height := frmMain.ClientHeight - GroupBox1.Height - StatusBar1.Height - 32;

  FileListView1.Width := GroupBox2.Width - 24;
  FileListView1.Height := GroupBox2.Height - 40;
end;


procedure TfrmMain.Timer1Timer(Sender: TObject);
var
  i: integer;
  j: integer;
  s: TSearchRec;
  sl_items: TStringList;
  sl_sub_items: TStringList;
  new_item: TListItem;
  in_group: boolean;
  item_caption: string;
begin
  // this routine contains all the initialisation code that is handled in the
  // context of the running program.  First we check for the results of a
  // previous run, then fill in the Drives list view.
  Timer1.Enabled := False;

  // Was there a previous run?
  if FindFirst (saved_duplicates_filename, faAnyFile, s) = 0 then
    begin
  // Does the user want to load the data from the previous run?
    if Application.MessageBox (PChar (
          'A list of duplicate files was found from a'#13 +
          FormatDateTime ('"previous run on" yyyy-mmm-dd "at" hh:nn:ss',
                          FileDateToDateTime (s.Time)) + #13#13 +
          'Reloading that list will save time.' + #13#13 +
          'Do you want to reload that list?'),
          'Find Duplicate Files - previous list found',
          MB_YESNO or MB_ICONQUESTION) = IDYES then
      begin
      sl_items := TStringList.Create;    // list containing an image of the file
      StatusBar1.Panels [0].Text := 'Reading previous list...';  // tell the user
      StatusBar1.Update;
      sl_items.LoadFromFile (saved_duplicates_filename);   // load the list
      FileListView1.Visible := False;                           // hide the view
      in_group := False;
      sl_sub_items := TStringList.Create;   // temp list for each line
      for i := 0 to sl_items.Count - 1 do   // scan down the saved list view
        begin
        new_item := FileListView1.Items.Add;     // add the next item in the list view
        // assume each string in the file contains comma text for each item(list)
        sl_sub_items.CommaText := sl_items.Strings [i];
        // assume we'll at least have a caption, even though it could be blank
        item_caption := Trim (sl_sub_items.Strings [0]);
        // reset IN_GROUP if we've just read a caption
        if Length (item_caption) = 0 then in_group := False;
        new_item.Caption := item_caption;
        // scan across the remaining items in the list, adding as sub-items
        for j := 1 to sl_sub_items.Count - 1 do
          new_item.SubItems.Add (Trim (sl_sub_items [j]));
        // in a list of duplicates? - then bump the file an byte count
        // just in case the string isn't a valid number, use try/except block
        if in_group then
          try
            Inc (duplicate_files);
            Inc (duplicate_bytes, StrToInt (sl_sub_items [1]));
          except
          end;

        // only now do we see if we might be in a group of duplicates
        in_group := sl_sub_items.Count > 1;
        end;

      // done reading the file, clean up
      sl_items.Free;
      sl_sub_items.Free;
      StatusBar1.Panels [0].Text := 'Duplicates read from file .....';
      FileListView1.Visible := True;
      update_compare_display;
      end;
    end;
  FindClose (s);

  // now find out what drives are available, add an icon and the display name to
  // the Drives list view.  If a drive is local, pre-select it for the user.
  DiskListView1.ScanDrives;

end;


procedure TfrmMain.DiskListView1Change(Sender: TObject; Item: TListItem;
  Change: TItemChange);
// this procedure updates the main form's caption to reflect the drives selected
var
  s, t: string;
  i: integer;
begin
  s := DiskListView1.SelectedDrives;
  t := 'Find Duplicate Files on ';
  for i := 1 to Length (s) do t := t + s [i] + ': ';
  Caption := t;
end;


procedure TfrmMain.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
// just note that we are required to close
begin
  TreeScanner1.Continue := False;
  continue_compare := False;
end;


procedure TfrmMain.btnCloseClick(Sender: TObject);
begin
  Close;
end;


procedure TfrmMain.btnSearchClick(Sender: TObject);
begin
  if TreeScanner1.SearchInProgress
  then                                 // must be in directory scan phase
    begin
    TreeScanner1.Continue := False;    // stop the tree scanner
    continue_compare := False;         // and stop looping over drives
    end
  else                                 // in comparison phase
    if comparing
    then
      continue_compare := False        // stop the comparison
    else
      begin
      btnSearch.Caption := 'Stop';     // prepare for a stop
      continue_compare := True;
      find_files_to_compare;           // build list of files sorted by size
      compare_files;                   // and compare the contents of those files
      end;
end;


procedure TfrmMain.find_files_to_compare;
// Find all the files in the tree that match the specification on all the
// selected drives.  The files are sorted by size.

var
  i: integer;
  root_dir: string;
  drives: string;
  this_drive: string;
begin
  with StatusBar1 do
    begin
    Panels [1].Text := 'Bytes:';
    Panels [2].Text := 'Files:';
    end;

  FileListView1.Hint := '';
  // clear out any existing matches
  FileListView1.Items.Clear;
  file_list.Clear;
  dir_list.Clear;
  master_list.Clear;
  root_dir := edtRootDir.Text;
  if root_dir = '' then root_dir := '\';
  if edtFileMask.Text = '' then edtFileMask.Text := '*.*';
  TreeScanner1.FileMask := edtFileMask.Text;
  skip_sysbckup_folder := chkSkipSysbckup.Checked;
  process_empty_files := not chkSkipEmptyFiles.Checked;

  drives := DiskListView1.SelectedDrives;       // get string of all selected drives
  for i := 1 to Length (drives) do          // do one drive letter at a time
    begin
    this_drive := drives [i] + ':';         // build a valid drive string
    scan_state := all_files;                // assume normal scan to start with
    // if we've just built the master list, all subsequent scans are filtered
    if scan_state = master_scan then scan_state := filtered_scan;
    // if we're just starting and this is a floppy, get the master list
    if (scan_state = all_files) and
       (chkFloppyIsMaster.Checked) and
       (this_drive = 'A:') then scan_state := master_scan;
    with StatusBar1.Panels [0] do           // report scan state to user
      case scan_state of
            all_files: Text := 'Scanning directories ....';
          master_scan: Text := 'Scanning master directories ....';
        filtered_scan: Text := 'Scanning filtered directories ....';
      end;
    // form the root scan path by drive + directory
    TreeScanner1.InitialDirectory := this_drive + root_dir;
    TreeScanner1.ScanTree;                  // scan selected drive
    if not continue_compare then Break;     // check if we can continue
    end;

  // clear out any messages from scanning
  StatusBar1.Panels [0].Text := '';
  StatusBar1.Panels [3].Text := '';
end;


procedure TfrmMain.TreeScanner1DirectoryFound(Sender: TObject);
// this procedure is called back by the tree scanner for each directory
var
  dir_name: string;
begin
  dir_name := Trim (TreeScanner1.FileFound);  // get the directory name
  dir_list.Add (dir_name);                    // and add it to the list of dirs
  current_dir := dir_list.Count - 1;          // taking note of the index
  // determine if we should skip this folder (because its' Win 95 System Backup)
  skip_this_folder := (skip_sysbckup_folder) and
                      (Pos ('\SYSBCKUP\', UpperCase (dir_name)) <> 0);
  if skip_this_folder
    then StatusBar1.Panels [3].Text := 'Skipping ' + dir_name
    else StatusBar1.Panels [3].Text := dir_name;
  file_list.BeginUpdate;                      // only really affects visible comps
end;


procedure TfrmMain.TreeScanner1DirectoryDone(Sender: TObject);
begin
  file_list.EndUpdate;
end;


procedure TfrmMain.TreeScanner1FileFound(Sender: TObject);
// this procedure is called back by the tree scanner for each file
var
  file_name: String;
  file_size: String;
  f: TSearchRec;
  dummy: integer;
begin
  if skip_this_folder then Exit;           // ignore skipped folders
  file_name := TreeScanner1.FileFound;     // save the file name
  // if we can find the file, add an entry to the file list with the size as a
  // numeric prefix (to allow sorting) and with the directory as an index into
  // the directory list (store the index as an integer, type-cast to a pointer)
  if FindFirst (file_name, faAnyFile, f) = 0 then
    begin
    if (process_empty_files) or (f.size <> 0) then
      begin
      file_size := Format ('%9d ', [f.size]);
      if scan_state = master_scan then master_list.Add (file_size);
      // should we add this file?  Yes if doing all files, yes if scanning the
      // master directory, and yes if filtered and we get a match
      if (scan_state = all_files) or
         (scan_state = master_scan) or
         ((scan_state = filtered_scan) and (master_list.Find (file_size, dummy))) then
        begin
        file_size := file_size + ExtractFileName (file_name);    // add the name string
        file_list.AddObject (file_size, Pointer (current_dir));  // integer type-cast as pointer
        end;
      end;
    end;
  FindClose (f);
end;


procedure TfrmMain.update_compare_display;
// formatted update of duplicate bytes and files found
begin
  StatusBar1.Panels[1].Text := Format ('Bytes: %.0n', [duplicate_bytes + 0.0]);
  StatusBar1.Panels[2].Text := Format ('Files: %.0n', [duplicate_files + 0.0]);
end;


function TfrmMain.files_match (size: integer;  name1, name2: String): boolean;
// primary file comparison routine, returns True if files match
// zero length files automatically match!
// updates number of files and bytes that matched
// uses memory-mapped file I/O
// could be improved by matching 32-bit quantities at a time, rather
// than the bytes used in this version
type
  PByte = ^byte;
var
  same: boolean;                   // true if files match
  handle1, handle2: THandle;       // stuff for memory-mapped file I/O
  mapping1, mapping2: THandle;
  base1, base2: Pointer;
  p1, p2: PByte;                   // pointers within each file
  i: integer;                      // loop over bytes in the file
begin
  if size = 0 then                 // special case - zero length files match!
    begin
    Inc (duplicate_files);
    Result := True;
    Exit;
    end;

  Result := False;
  handle1 := CreateFile (PChar(name1), GENERIC_READ, FILE_SHARE_READ, nil,
                         OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
  if handle1 <> INVALID_HANDLE_VALUE then
    begin
    handle2 := CreateFile (PChar(name2), GENERIC_READ, FILE_SHARE_READ, nil,
                           OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
    if handle2 <> INVALID_HANDLE_VALUE then
      begin
      mapping1 := CreateFileMapping (handle1, nil, PAGE_READONLY, 0, 0, nil);
      if mapping1 <> 0 then
        begin
        mapping2 := CreateFileMapping (handle2, nil, PAGE_READONLY, 0, 0, nil);
        if mapping2 <> 0 then
          begin
          base1 := MapViewOfFile (mapping1, FILE_MAP_READ, 0, 0, 0);
          if base1 <> nil then
            begin
            base2 := MapViewOfFile (mapping2, FILE_MAP_READ, 0, 0, 0);
            if base2 <> nil then
              begin
              same := True;
              p1 := PByte (base1);         // point to start of first file
              p2 := PByte (base2);         // and second file
              i := 0;                      // initialise loop counter
              while same and (i < size) do
                begin
                same := p1^ = p2^;         // compare a single byte
                Inc (i);                   // bump the count
                Inc (p1);                  // and the two read pointers
                Inc (p2)
                end;
              Result := same;              // record the outcome
              if same then
                begin
                Inc (duplicate_bytes, size);
                Inc (duplicate_files);
                end;
              UnmapViewOfFile (base2);     // undo all the mapped I/O stuff
              end;
            UnmapViewOfFile (base1);
            end;
          CloseHandle (mapping2);
          end;
        CloseHandle (mapping1);
        end;
      CloseHandle (handle2);
      end;
    CloseHandle (handle1);
    end;
end;


function TfrmMain.do_checksum (size: integer;  filename: String): TChecksum;
// checks either date/time or the first 512 bytes of the file
// uses memory-mapped I/O
// zero byte files return a zero checksum
const
  bytes_per_sector = 512;
  max_dwords_in_checksum = bytes_per_sector div SizeOf (TChecksum);
var
  checksum: TChecksum;
  handle: THandle;
  mapping: THandle;
  base: Pointer;
  dwords_to_checksum: integer;
  p: ^DWORD;
  dw: DWORD;
  search: TSearchRec;
begin
  Result := 0;
  if size = 0 then Exit;    // special case

  checksum := 0;

  if check_file_timestamp   // see what method is to be used
  then
    begin
    if FindFirst (filename, faAnyFile, search) = 0 then checksum := search.Time;
    FindClose (search);
    end
  else
    begin
    handle := CreateFile (PChar(filename), GENERIC_READ, FILE_SHARE_READ, nil,
                          OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
    if handle <> INVALID_HANDLE_VALUE then
      begin
      mapping := CreateFileMapping (handle, nil, PAGE_READONLY, 0, 0, nil);
      if mapping <> 0 then
        begin
        base := MapViewOfFile (mapping, FILE_MAP_READ, 0, 0, 0);
        if base <> nil then
          begin
          p := base;
          dwords_to_checksum := size div SizeOf (DWORD);
          if dwords_to_checksum > max_dwords_in_checksum then
            dwords_to_checksum := max_dwords_in_checksum;
          for dw := 1 to dwords_to_checksum do
            begin
            checksum := checksum + p^;
            Inc (p);
            end;
          UnmapViewOfFile (base);
          end;
        CloseHandle (mapping);
        end;
      CloseHandle (handle);
      end;
    end;
  // There's a chance that the total has hit -2147483648, which is not a valid
  // integer when you use StrToInt.  To avoid this, kill the top bit and return
  // a 31-bit result.  It doesn't matter that the value isn't a true sum,
  // it _will_ be the same for each summation which is what matters.
  Result := checksum and $7FFFFFFF;
end;


procedure TfrmMain.compare_files;
// This is the main procedure to compare all files found from the directory
// scan.  This is sorted by size, so we now look for blocks of files that
// have the same size.  For each of these blocks of multiples, create a list
// with the file initial checksum and the file name and pass that to an
// inner procedure.
//
// Measurements show that the critical routine here (in terms of execution time)
// is the Checksum function, hence its default replacement by a simple
// date and time check.
var
  items_done: integer;             // count for progress bar display

  procedure do_multiples (size: integer;  list: TStringList);
  // handle a list of files of the same size
  var
    master_filename: string;
    old_filename: string;
    checksum: TChecksum;
    old_checksum: TChecksum;
    in_checksum_group: boolean;
    in_match_group: boolean;
    i: integer;
    s: String;
    space: integer;
    same_checksum: boolean;
    exact_match: boolean;
    new_item: TListItem;
    filesize: integer;
  begin
    old_filename := '';                  // set to impossible initial values
    old_checksum := -1;                  // unlikely, if not impossible!
    in_checksum_group := False;          // initially, not in checksum group
    in_match_group := False;             // nor in match group

    // scan down the list of files which are the same size
    for i := 0 to list.Count - 1 do
      begin
      s := Trim (list.Strings [i]);       // separate out the size part of the string
      space := Pos (' ', s);
      checksum := StrToInt (Copy (s, 1, space-1));   // get the checksum part
      s := Trim (Copy (s, space, 999));              // and the file name part
      same_checksum := checksum = old_checksum;      // is this the same as before?

      if same_checksum and (not in_checksum_group)
        then master_filename := old_filename;

      old_checksum := checksum;
      old_filename := s;

      if same_checksum
        then exact_match := files_match (size, master_filename, s)
        else exact_match := False;

      in_checksum_group := same_checksum;

      if (not in_match_group) and exact_match then
        with FileListView1 do
          begin
          if Items.Count <> 0 then Items.Add;  // add a blank
          new_item := Items.Add;               // add a title line for the group
          new_item.Caption := duplicate_group_prefix;
          FileListView1.AddFile (master_filename, filesize);
          end;

      in_match_group := exact_match;

      if in_checksum_group and exact_match then
        FileListView1.AddFile (LowerCase (s), filesize);  // and note this duplicate
      end;

    update_compare_display;                          // the bytes and files
    progress_bar.Position := items_done;             // show the progress
  end;

var
  current_size: integer;
  old_size: integer;
  old_filename: String;
  current_filename: String;
  item: integer;
  s: String;
  in_multiple: boolean;
  same_size: boolean;
  multiples_list: TStringList;
  chk: TChecksum;
  space: integer;
begin
  // scan the list of files finding all the files of duplicate size
  comparing := True;
  duplicate_bytes := 0;     // initially, no duplicate files
  duplicate_files := 0;     // or bytes duplicated
  old_size := -1;
  old_filename := '';
  in_multiple := False;

  // each group of similarly sized files will be kept in this sorted list
  multiples_list := TStringList.Create;
  with multiples_list do
    begin
    Sorted := True;
    Duplicates := dupAccept;
    end;

  progress_bar.Max := file_list.Count - 1;  // the maximum number of files to check
  progress_bar.Visible := True;             // show the progress bar
  items_done := 0;                          // with nothing done, as yet

  check_file_timestamp := chkCheckFileTimestamp.Checked;  // note the checksum method

  FileListView1.Items.Clear;
  FileListView1.Hint := 'Double-click a file to review';

  // Use a WHILE loop to scan the list rather than a DO to allow for the
  // comparison to be interrupted.  Start with the last item which has the
  // largest file size, so that we record the worst-case duplicates first.
  item := file_list.Count - 1;
  while (item >= 0) and continue_compare do
    begin
    Application.ProcessMessages;             // look for any button press
    s := Trim (file_list.Strings [item]);    // get the current size and filename
    space := Pos (' ', s);                   // split into components
    // retrieve the directory name from the stored list, the index into the
    // list is given by the integer, stored as a pointer in the Object field
    current_filename := dir_list.Strings [Integer (file_list.Objects [item])] +
                        Trim (Copy (s, space, 999));  // add the file name
    Delete (s, space, 999);                  // retain just the size string
    current_size := StrToInt (s);            // convert the file size to integer

    same_size := current_size = old_size;    // is size the same as before ?
    if same_size and (not in_multiple) then  // start of a new set of files
      begin
      multiples_list.Clear;                  // clear out any existing stuff
      chk := do_checksum (old_size, old_filename);   // get the checksum and add to list
      multiples_list.Add (Format ('%12d ', [chk]) + ' ' + old_filename);
      // update the user on the progress of the comparison
      StatusBar1.Panels [3].Text := 'Comparing ' +
          Format ('%.0n', [current_size + 0.0]) + ' byte files ...';
      StatusBar1.Update;
      end;

    in_multiple := same_size;
    if in_multiple
    then
      // if in a list, checksum this file and add it to the list
      begin
      chk := do_checksum (current_size, current_filename);
      multiples_list.Add (Format ('%12d ', [chk]) + ' ' + current_filename);
      end
    else
      // if not in a list, process any existing list and reset it by clearing out
      begin
      if multiples_list.Count <> 0 then
        begin
        do_multiples (old_size, multiples_list);
        multiples_list.Clear;
        end;
      end;

    old_size := current_size;
    old_filename := current_filename;
    Inc (items_done);              // update the progress
    file_list.Delete (item);       // give the memory back for this item
    Dec (item);
    end;

  // we might get here and still have a list of multiples pending
  // so process it, but there's no point in clearing out the list
  if multiples_list.Count <> 0 then
    do_multiples (old_size, multiples_list);

  // try and make the status text reflect the most recent user command
  if TreeScanner1.Continue
    then StatusBar1.Panels [3].Text := 'Scan complete'
    else StatusBar1.Panels [3].Text := 'Scan interrupted';

  if not continue_compare
    then StatusBar1.Panels [3].Text := 'Comparison interrupted';

  if duplicate_files <> 0
    then StatusBar1.Panels [0].Text := 'Duplicates found....';

  // clear up after this procedure
  progress_bar.Visible := False;       // hide the progress bar display
  progress_bar.Position := 0;          // reset for next time
  multiples_list.Free;                 // give back the memory
  file_list.Clear;
  dir_list.Clear;
  comparing := False;
  btnSearch.Caption := 'Start Search';
end;


function TfrmMain.parse_list_view (lvw: TListView): string;
// function to return filename from selected TListView item caption
// ignores those items starting "Duplicate group..."
var
  filename: string;
begin
  Result := '';
  with lvw do
    begin
    if Selected = nil then Exit;            // nothing selected - skip
    filename := Trim (Selected.Caption);    // get the caption string
    if Length (filename) = 0 then Exit;     // blank - skip
    if Pos (duplicate_group_prefix, filename) <> 0 then Exit;
    Result := filename;                     // return the file name
    end;
end;


procedure TfrmMain.FileListView1DblClick(Sender: TObject);
// handle a double-click, get the filename and display properties to the user
var
  filename: string;
  first, middle, last: TListItem;
  num_files: integer;
begin
  filename := parse_list_view (FileListView1);
  if filename = '' then Exit;
  first := FileListView1.GetNextItem (FileListView1.Selected, sdAbove, [isNone]);

  if FileExists (filename)
  then
    begin
    frmFileProperties1.filename := filename;
    if frmFileProperties1.ShowModal = idYes then FileListView1.Selected.Delete;
    end
  else
    FileListView1.Selected.Delete;

  // we may changed the list, so check for the number in a group
  // scan up the list until we find the "start of group" message
  while Pos (duplicate_group_prefix, first.Caption) = 0 do
    first := FileListView1.GetNextItem (first, sdAbove, [isNone]);
  last := first;
  middle := nil;
  num_files := 0;

  // scan down the list until a blank is found or we fall off the list end
  // count the number of files remaining
  repeat
    last := FileListView1.GetNextItem (last, sdBelow, [isNone]);
    if last <> nil then              // not off the list end
      if last.Caption <> '' then     // still in group of files
        begin
        Inc (num_files);
        middle := last;              // save pointer to real file name entry
        end;
  until (last = nil) or (last.Caption = '');

  // only one file left? - then we delete the header, the file and any trailer
  if num_files = 1 then
    begin
    if last <> nil then FileListView1.Items.Delete (last.index);
    FileListView1.Items.Delete (middle.index);
    FileListView1.Items.Delete (first.index);
    end;

  FileListView1.Hint := 'Double-click a file to review';
end;


procedure TfrmMain.FileListView1Click(Sender: TObject);
// handle a single-click, make the hint echo the file name or reset to default
var
  filename: string;
begin
  filename := parse_list_view (FileListView1);
  with FileListView1 do
    if filename = ''
    then Hint := 'Double-click a file to review'
    else Hint := filename;
end;


procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Close;
end;


end.

