unit Main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  MPlayer, StdCtrls, ComCtrls, ExtCtrls, Menus, RxGrdCpt, LCD_Lab, TransBtn,
  Buttons, Hemibtn, Rbutton, RXCtrls, Placemnt;

const
  S_UnknownArtist = '(tuntematon esittj)';
  S_UnknownTitle = '(tuntematon levy)';
  S_NoDisc = 'Asemassa ei ole levy';
  S_InsertDisc = 'Syt ni-CD asemaan';
  S_NoTrack = '';

type
  TTimeMode = (tmNone, tmDiscTotal, tmTrackElapsed, tmTrackRemaining,
               tmDiscElapsed, tmDiscRemaining);

const
  TimeModes: array [TTimeMode] of string =
    ('No disc', 'Disc length', 'Track elapsed', 'Track remaining',
     'Disc elapsed', 'Disc remaining');

type
  TFrmMain = class(TForm)
    StatusBar: TStatusBar;
    RxGradientCaption1: TRxGradientCaption;
    Player: TMediaPlayer;
    MainMenu: TMainMenu;
    MnuDisc: TMenuItem;
    MnuDiscEdit: TMenuItem;
    N1: TMenuItem;
    MnuDiscExit: TMenuItem;
    MnuHelp: TMenuItem;
    MnuHelpAbout: TMenuItem;
    TmrStatus: TTimer;
    Panel1: TPanel;
    LblArtist: TLCDLabel;
    LblTitle: TLCDLabel;
    LcdTrackInfo: TLCDLabel;
    LblTrackName: TLCDLabel;
    BtnPlay: TRoundButton;
    BtnStop: TRoundButton;
    BtnPause: TRoundButton;
    BtnEject: TRoundButton;
    BtnPrevious: TRoundButton;
    BtnNext: TRoundButton;
    BtnTimeMode: TRoundButton;
    LblTimeMode: TLabel;
    FormPlacement1: TFormPlacement;
    procedure FormCreate(Sender: TObject);
    procedure TmrStatusTimer(Sender: TObject);
    procedure BtnEjectClick(Sender: TObject);
    procedure BtnPreviousClick(Sender: TObject);
    procedure BtnNextClick(Sender: TObject);
    procedure BtnStopClick(Sender: TObject);
    procedure BtnPlayClick(Sender: TObject);
    procedure BtnPauseClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure MnuDiscExitClick(Sender: TObject);
    procedure MnuDiscEditClick(Sender: TObject);
    procedure MnuHelpAboutClick(Sender: TObject);
    procedure BtnBackClick(Sender: TObject);
    procedure BtnForwardClick(Sender: TObject);
    procedure BtnTimeModeClick(Sender: TObject);
  private
    CurrentTrack: Longint;
    Paused: Boolean;
    HaveDisc: Boolean;
    Changed: Boolean;
    TimeMode: TTimeMode;
    procedure UpdateStatus;
    procedure WriteDiscInfo;
  public
    { Public declarations }
  end;

var
  FrmMain: TFrmMain;

implementation

uses
  MMSystem, Disc, IniFiles, DiscEdit, About;

{$R *.DFM}

var
  WinDir: array [0..256] of Char;
  IniFileName: string;
  DiscInfo: TDiscInfo; (* see the Disc unit *)

procedure TFrmMain.FormCreate(Sender: TObject);
begin
  GetWindowsDirectory(@WinDir, SizeOf(WinDir));
  IniFileName := WinDir + '\cdplayer.ini';

  FillChar(DiscInfo, SizeOf(DiscInfo), 0);

  HaveDisc := False;
  Paused := False;
  Changed := False;
  TimeMode := tmNone;

  Player.TimeFormat := tfTMSF;  { set the time format }
  Player.Wait := False;

  CurrentTrack := 1;
  StatusBar.Panels[0].Text := 'Pysytetty';
end;

procedure TFrmMain.TmrStatusTimer(Sender: TObject);
var
  Ini: TIniFile;
  TrackNumber: Longint;
begin
  if Player.Mode = mpNotReady then
  begin
    Exit;
  end;

  (* A CD was inserted since we last were here. *)
  if (not HaveDisc) and MediaPresent(Player.DeviceID) then
  begin
    StatusBar.Panels[0].Text := 'Haetaan levyn tietoja...';
    HaveDisc := True;
    Paused := False;
    CurrentTrack := 1;

    DiscInfo.DiscID := MakeDiscID(Player);

    (* Must read from INI file *)
    Ini := TIniFile.Create(IniFileName);
    try
      DiscInfo.Artist := Ini.ReadString(DiscInfo.DiscID, 'artist', S_UnknownArtist);
      DiscInfo.Title := Ini.ReadString(DiscInfo.DiscID, 'title', S_UnknownTitle);

      Player.TimeFormat := tfTMSF;

      (* CDPLAYER.INI stores track names with zero based offsets. *)
      for TrackNumber := 0 to Player.Tracks - 1 do
      begin
        DiscInfo.TrackNames[TrackNumber + 1] :=
          Ini.ReadString(DiscInfo.DiscID,
                         IntToStr(TrackNumber),
                         'Track ' + IntToStr(TrackNumber + 1));
      end;

      TimeMode := tmDiscTotal;

      MnuDiscEdit.Enabled := True;
    finally
      Ini.Free;
    end;

    LblArtist.Caption := DiscInfo.Artist;
    LblTitle.Caption := DiscInfo.Title;
    UpdateStatus;
    StatusBar.Panels[0].Text := 'Pysytetty';
  end
 (* CD was removed since we last were here *)
  else if HaveDisc and (not MediaPresent(Player.DeviceID)) then
  begin
    HaveDisc := False;
    Paused := False;
    FillChar(DiscInfo, SizeOf(DiscInfo), 0);
    MnuDiscEdit.Enabled := False;
    Exit;
  end
  (* no disc in drive *)
  else if (not HaveDisc) and
          (not MediaPresent(Player.DeviceID)) then
  begin
    FillChar(DiscInfo, SizeOf(DiscInfo), 0);
    LblArtist.Caption := S_NoDisc;
    LblTitle.Caption := S_InsertDisc;
    UpdateStatus;
    Paused := False;
    StatusBar.Panels[0].Text := S_NoDisc;
    MnuDiscEdit.Enabled := False;
    Exit;
  end;

  if (Player.Mode = mpPlaying) or Paused then
  begin
    UpdateStatus;
    if Player.Mode = mpPlaying then
      StatusBar.Panels[0].Text := 'Toistaa'
    else
      StatusBar.Panels[0].Text := 'Tauko toistossa';
  end
  else if Player.Mode = mpStopped then
  begin
    Player.TimeFormat := tfMSF;
    UpdateStatus;
    StatusBar.Panels[0].Text := 'Pysytetty';
  end
  else (* maybe skip pressed? *)
  begin
    Player.TimeFormat := tfMSF;
    UpdateStatus;
  end;
end;

(* Update the display status *)
procedure TFrmMain.UpdateStatus;
var
  Track, Min, Sec: Longint;
  DisplayTime: Longint;
  TimeTemplate: string;
  TrackName: string;
  TrackLen: Longint;
begin
  DisplayTime := Player.Position;

  case TimeMode of
    tmNone:
    begin
      Min := 0;
      Sec := 0;
      Track := 0;
    end;

    tmDiscTotal:
    begin
      Player.TimeFormat := tfMSF;
      DisplayTime := Player.Length;
      Min := MCI_MSF_MINUTE(DisplayTime);
      Sec := MCI_MSF_SECOND(DisplayTime);
      Track := Player.Tracks;
    end;

    tmTrackElapsed:
    begin
      Player.TimeFormat := tfTMSF;
      DisplayTime := Player.Position;
      Min := MCI_TMSF_MINUTE(DisplayTime);
      Sec := MCI_TMSF_SECOND(DisplayTime);
      Track := GetCurrentTrack(Player.DeviceID);
      CurrentTrack := Track;
    end;

    tmTrackRemaining:
    begin
      Player.TimeFormat := tfMilliseconds;
      (* the player's position should be further than the track starting point *)
      Track := GetCurrentTrack(Player.DeviceID);
      DisplayTime := Player.TrackLength[Track] - (Player.Position - Player.TrackPosition[Track]);
      MillisecToMinSec(DisplayTime, Min, Sec);
      CurrentTrack := Track;
    end;

    tmDiscElapsed:
    begin
      Player.TimeFormat := tfMSF;
      DisplayTime := Player.Position;
      Min := MCI_MSF_MINUTE(DisplayTime);
      Sec := MCI_MSF_SECOND(DisplayTime);
      Track := GetCurrentTrack(Player.DeviceID);
      CurrentTrack := Track;
    end;

    tmDiscRemaining:
    begin
      Player.TimeFormat := tfMilliseconds;
      DisplayTime := Player.Length - Player.Position;
      MillisecToMinSec(DisplayTime, Min, Sec);
      Track := GetCurrentTrack(Player.DeviceID);
      CurrentTrack := Track;
    end;
  end;

  TimeTemplate := '%.2d  %.2d:%.2d';

  if (TimeMode in [tmTrackRemaining, tmDiscRemaining]) and
     (Player.Mode <> mpStopped) then
  begin
    TimeTemplate := '%.2d -%.2d:%.2d';
  end;

  LcdTrackInfo.Caption := Format(TimeTemplate, [Track, Min, Sec]);

  if Player.Mode = mpStopped then
    TrackName := S_NoTrack
  else
    TrackName := DiscInfo.TrackNames[CurrentTrack];

  (* comparing strings should be faster than updating the display *)
  if LblTrackName.Caption <> TrackName then
    LblTrackName.Caption := TrackName;
end;

procedure TFrmMain.BtnPlayClick(Sender: TObject);
begin
  if Paused then
  begin
    (* set no starting position when resuming paused play *)
    Player.Play;
    (* time display mode stays the same *)
  end
  else
  begin
    Player.TimeFormat := tfTMSF;
    Player.Position := MCI_MAKE_TMSF(CurrentTrack, 0, 0, 0);
    Player.Play;
    TimeMode := tmTrackElapsed;
  end;
end;

procedure TFrmMain.BtnPauseClick(Sender: TObject);
begin
  if Player.Mode <> mpPlaying then Exit;
  Player.Pause;
  Paused := True;
end;

procedure TFrmMain.BtnStopClick(Sender: TObject);
begin
  if Paused then
  begin
    Player.Play;
    Paused := False;
  end;
  Player.Stop;
  CurrentTrack := 1;        (* reset to the first track *)
  TimeMode := tmDiscTotal;  (* display total disc time *)
end;

procedure TFrmMain.BtnEjectClick(Sender: TObject);
begin
  if not HaveDisc then
  begin
    LoadDisc(Player.DeviceID);
    TimeMode := tmDiscTotal;
  end
  else
  begin
    Player.Eject;
  end;
  TimeMode := tmNone;
end;

procedure TFrmMain.BtnPreviousClick(Sender: TObject);
begin
  if CurrentTrack > 1 then
  begin
    Dec(CurrentTrack);
    if Player.Mode = mpPlaying then
    begin
      Player.TimeFormat := tfTMSF;
      Player.Previous;
    end;
  end;
end;

procedure TFrmMain.BtnNextClick(Sender: TObject);
begin
  if CurrentTrack < Player.Tracks then
  begin
    Inc(CurrentTrack);
    if Player.Mode = mpPlaying then
    begin
      Player.TimeFormat := tfTMSF;
      Player.Next;
      if Player.Error <> 0 then
      begin
        StatusBar.Panels[0].Text := Player.ErrorMessage;
      end;
    end;
  end;
end;

procedure TFrmMain.FormDestroy(Sender: TObject);
begin
  if Changed then
  begin
    WriteDiscInfo;
  end;
end;

procedure TFrmMain.MnuDiscExitClick(Sender: TObject);
begin
  Close;
end;

procedure TFrmMain.MnuDiscEditClick(Sender: TObject);
var
  I: Longint;
begin
  FrmDiscEditor.TxtArtist.Text := LblArtist.Caption;
  FrmDiscEditor.TxtTitle.Text := LblTitle.Caption;

  FrmDiscEditor.LstTracks.Items.Clear;

  for I := 1 to Player.Tracks do
  begin
    FrmDiscEditor.LstTracks.Items.Add(
      DiscInfo.TrackNames[I]
    );
  end;
  
  if FrmDiscEditor.ShowModal = mrOK then
  begin
    Changed := FrmDiscEditor.HasChanged;

    (* Zap the track name list and fill it again with the
       information from the form. *)
    FillChar(DiscInfo.TrackNames, SizeOf(DiscInfo.TrackNames), 0);
    for I := 0 to FrmDiscEditor.LstTracks.Items.Count - 1 do
    begin
      DiscInfo.TrackNames[I + 1] :=
        FrmDiscEditor.LstTracks.Items[I];
    end;

    LblArtist.Caption := FrmDiscEditor.TxtArtist.Text;
    LblTitle.Caption := FrmDiscEditor.TxtTitle.Text;

    WriteDiscInfo;
  end;
end;

procedure TFrmMain.WriteDiscInfo;
var
  Ini: TIniFile;
  I: Longint;
  Order: string;
begin
  if not Changed then Exit;

  (* First erase the old section for this disc *)
  Ini := TIniFile.Create(IniFileName);
  try
    Ini.EraseSection(DiscInfo.DiscID);

    with Ini do
    begin
      WriteInteger(DiscInfo.DiscID, 'EntryType', 1);
      WriteString(DiscInfo.DiscID, 'artist', LblArtist.Caption);
      WriteString(DiscInfo.DiscID, 'title', LblTitle.Caption);
      WriteInteger(DiscInfo.DiscID, 'numtracks', Player.Tracks);

      Order := '';
      for I := 1 to Player.Tracks do
      begin
        Order := Order + IntToStr(I - 1) + ' ';
        WriteString(DiscInfo.DiscID, IntToStr(I - 1),
          DiscInfo.TrackNames[I]);
      end;
      WriteString(DiscInfo.DiscID, 'order', Trim(Order));
      WriteInteger(DiscInfo.DiscID, 'numplay', Player.Tracks);
    end;
  finally
    Ini.Free;
  end;

  Changed := False;
end;

procedure TFrmMain.MnuHelpAboutClick(Sender: TObject);
begin
  FrmAbout.ShowModal;
end;

procedure TFrmMain.BtnBackClick(Sender: TObject);
begin
  Player.Back;
end;

procedure TFrmMain.BtnForwardClick(Sender: TObject);
begin
  Player.Step;
end;

procedure TFrmMain.BtnTimeModeClick(Sender: TObject);
begin
  if Player.Mode = mpStopped then
  begin
    Exit;
  end;

  if TimeMode = tmDiscRemaining then
  begin
    TimeMode := tmTrackElapsed;
  end
  else
  begin
    TimeMode := TTimeMode(Succ(Ord(TimeMode)));
  end;

  StatusBar.Panels[1].Text := TimeModes[TimeMode];
end;

end.
