unit Version; { Encapsulates the version information resource. }

interface

uses
  Windows, SysUtils, Classes;

type
  TVariableFieldNumber =
    (vfComments, vfCompany, vfFileDescription,
     vfFileVersion, vfInternalName, vfLegalCopyright,
     vfLegalTrademarks, vfOriginalFilename, vfPrivateBuild,
     vfProductName, vfProductVersion, vfSpecialBuild);

  { The version information class. }
  TVersionInfo = class(TComponent)
  private
    // The following are the variable fields of the version information.
    FComments: string;
    FCompany: string;
    FFileDescription: string;
    FFileVersion: string;
    FInternalName: string;
    FLegalCopyright: string;
    FLegalTrademarks: string;
    FOriginalFilename: string;
    FPrivateBuild: string;
    FProductName: string;
    FProductVersion: string;
    FSpecialBuild: string;
    // End of variable fields.

    FUnknownString: string;
    Data: Pointer;       { pointer to version information data }
    BaseString: string;  { base of version query string }

  protected
    procedure ReadVersionInfo;
    procedure SetVariableFields;
    function GetVariableString(FieldNumber: TVariableFieldNumber): string;

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure Loaded; override;

    // The variable version information fields as read-only properties:
    property Comments: string read FComments stored False;
    property Company: string read FCompany stored False;
    property FileDescription: string read FFileDescription stored False;
    property FileVersion: string read FFileVersion stored False;
    property InternalName: string read FInternalName stored False;
    property LegalCopyright: string read FLegalCopyright stored False;
    property LegalTrademarks: string read FLegalTrademarks stored False;
    property OriginalFilename: string read FOriginalFilename stored False;
    property PrivateBuild: string read FPrivateBuild stored False;
    property ProductName: string read FProductName stored False;
    property ProductVersion: string read FProductVersion stored False;
    property SpecialBuild: string read FSpecialBuild stored False;
    // End of variable field properties.

  published
    property UnknownString: string read FUnknownString write FUnknownString;
  end;

procedure Register;

///////////////////////////////////////////////////////////////////////////

implementation

resourcestring
  SUnknown = '(unknown)';

const
  VariableFieldCount = 12;

  VariableFieldNames : array [0..VariableFieldCount-1] of string =
    ('Comments', 'CompanyName', 'FileDescription', 'FileVersion',
     'InternalName', 'LegalCopyright', 'LegalTrademarks', 'OriginalFilename',
     'PrivateBuild', 'ProductName', 'ProductVersion', 'SpecialBuild');

{ Construct the version information object. }
constructor TVersionInfo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FUnknownString := SUnknown;

  Data := nil;
end;

{ Destroy the version information object }
destructor TVersionInfo.Destroy;
begin
  if Assigned(Data) then
    FreeMem(Data);

  inherited Destroy;
end;


procedure TVersionInfo.Loaded;
begin
  inherited Loaded;

  if not (csDesigning in ComponentState) then
  begin
    ReadVersionInfo;    // read the version information from the executable
    SetVariableFields;  // initialize the fields based on the data that was read
  end;
end;


procedure TVersionInfo.SetVariableFields;
begin
  FComments        := GetVariableString(vfComments);
  FCompany         := GetVariableString(vfCompany);
  FFileDescription := GetVariableString(vfFileDescription);
  FFileVersion     := GetVariableString(vfFileVersion);
  FInternalName    := GetVariableString(vfInternalName);
  FLegalCopyright  := GetVariableString(vfLegalCopyright);
  FLegalTrademarks := GetVariableString(vfLegalTrademarks);
  FOriginalFilename:= GetVariableString(vfOriginalFilename);
  FPrivateBuild    := GetVariableString(vfPrivateBuild);
  FProductName     := GetVariableString(vfProductName);
  FProductVersion  := GetVariableString(vfProductVersion);
  FSpecialBuild    := GetVariableString(vfSpecialBuild);
end;


{ Get the contents of a variable file information field based on the
  field ID number }
function TVersionInfo.GetVariableString(FieldNumber: TVariableFieldNumber): string;
var
  StringLength: UINT;
  StringData: Pointer;
  VerName: string;
begin
  if not Assigned(Data) then { no data, can't continue }
    Exit;

  { Construct a version information query for the translation/field }
  VerName := BaseString + VariableFieldNames[Ord(FieldNumber)];

  { Get the value of the variable file information field: }
  if not VerQueryValue(Data, PChar(VerName), StringData, StringLength) then
    Exit;

  { If the query was OK and the string length was non-zero, return
    the string data. Otherwise, return the default "unknown" string. }
  if StringLength <> 0 then
    Result := PChar(StringData)
  else
    Result := FUnknownString;
end;


procedure TVersionInfo.ReadVersionInfo;
var
  Filename: array [0..259] of Char;  { the name of our executable }
  Handle: DWORD;      { dummy handle to receive zero from GetFileVersionInfo(Size) }
  InfoSize: DWORD;    { length of version information from GetFileVersionInfoSize }
  ValueLength: UINT;  { length of version field from VerQueryValue }
  TranslationInfoPointer: Pointer;
  LangID, CharSetID: UINT;
  LangIDStr, CharSetIDStr: string[8];
  OK: Boolean;
begin
  { Get the filename of our executable. }
  GetModuleFileName(HInstance, Filename, SizeOf(Filename));

  { Find out the size of the version information block and
    allocate memory for it: }
  InfoSize := GetFileVersionInfoSize(Filename, Handle);
  GetMem(Data, InfoSize);
  if Data = nil then
    Exit;

  { Copy the version information to our data block: }
  OK := GetFileVersionInfo(Filename, Handle, InfoSize, Data);
  if not OK then
    Exit;

  { Get the current translation (language and character set ID: }
  VerQueryValue(Data, '\VarFileInfo\Translation', TranslationInfoPointer, ValueLength);
  LangID := LoWord(DWORD(TranslationInfoPointer^));
  CharSetID := HiWord(DWORD(TranslationInfoPointer^));

  { Convert language and character set IDs to hexadecimal strings: }
  LangIDStr    := IntToHex(LangID, 4);
  CharSetIDStr := IntToHex(CharSetID, 4);

  { Construct a version query string for the relevant translation block: }
  FmtStr(BaseString, '\StringFileInfo\%s%s\', [LangIDStr, CharSetIDStr]);

end;

procedure Register;
begin
  RegisterComponents('Win95', [TVersionInfo]);
end;

end.

