unit ctrltrap;
// Copyright  1999 by Ziff-Davis, Inc.
// Written by Neil J. Rubenking

interface

USES Windows, SysUtils, Classes;

function  StartWatching : Integer;
procedure StopWatching;
function  GetObsfucator : DWORD;
function  GetVxdVersion : Integer;
procedure AddAProcess(P : THandle);
function  GetAList(idx : Integer) : TStringList;

const
  // constants for StartWatching
  IsWatching : Boolean = False;
  sw_OK       = 0;
  sw_NoOpen   = -1;
  sw_NoVer    = -2;
  sw_Already  = -3;
  // constants for GetAList
  L_FilOpe = 0;
  L_FilCre = 1;
  L_FilDel = 2;
  L_DirCre = 3;
  L_DirDel = 4;
  L_KeyCre = 5;
  L_KeyDel = 6;
  L_ValCre = 7;
  L_ValCha = 8;
  L_ValDel = 9;
  L_ProcID = 10;


//*********************
    implementation
//*********************
USES forms, In4share;

CONST
  VXD_GET_VER        = 1;
  VXD_REGISTER_CB    = 2; //register a callback and logging
                          // conditions
  VXD_RELEASE_REC    = 3; //release the trap buffer received from VxD
  VXD_UPDATE         = 4; //update logging conditions
  VXD_RELEASE_EXT    = 5; //release extended buffer used by VxD
  VXD_GET_OBFUSCATOR = 6; //get obfuscator from VxD
  DISABLE_LOGGING    = 1;
  APP_EXITING        = 16;
  IFSMGR_FUNCTIONS   = $40; //Installable File System Mgr functions
  REG_FUNCTIONS      = $1;  //Registry functions

  ACCESS_MODE_MASK   = 7;
  ACCESS_WRITEONLY   = 1;
  ACCESS_READWRITE   = 2;

  REG_CREATE_KEY     = 0;
  REG_DELETE_KEY     = 1;
  REG_CREATE_VALUE   = 2;
  REG_CHANGE_VALUE   = 3;
  REG_DELETE_VALUE   = 4;

  IFSFN_OPEN     = 36; //function codes for file system operations
  IFSFN_READ     = 0 ;
  IFSFN_CLOSE    = 11;
  IFSFN_DELETE   = 31;
  IFSFN_DIR      = 32; //directory manipulation
                       //(tr_ir_flags indicate function)
  IFSFN_RENAME   = 37;

  LOG_ALL         = $0ffffffff;
  NO_FLAGS_SPECIFIED = $ff;
  NO_MASK_SPECIFIED = 0;
  CREATE_DIR = 0;
  DELETE_DIR = 1;

  OPENS_FOR_WRITE = $8000; //only report opens that can potentially
                           //change file
  REPORT_ALL_CREATES = $4000; //report back all RegCreateKey calls,
                              // even if it's just opening the key

  MAX_PROCESSES = 10;

type
  PTrap_criteria = ^trap_criteria;
  trap_criteria = packed RECORD
    tc_category    : DWORD; //indicator of functional area
                            // (e.g. file system, registry, etc.)
    tc_func_num    : DWORD; //function app wants trapped (-1 = all)
    tc_flags       : BYTE;  //if function uses flags, can also
                            // specify here (zero if none)
    tc_flags_mask  : BYTE;  //if function uses flags, can also mask
                            // off before compare (zero if none)
    tc_drive       : DWORD; //drive (-1 = all drives)
    tc_proc_handle : ARRAY[0..MAX_PROCESSES-1] OF DWORD;
                            //process ID  (-1 if all processes)
    tc_callback    : DWORD; //address to call back
    tc_misc        : WORD;  //misc flags
  end;

//trap record used for logging installable file system manager
// functions

  Ptrap_record = ^trap_record;
  trap_record = packed RECORD
    tr_category        : DWORD; //indicator of functional area
                                // (e.g. file system, registry, etc.)
    tr_function        : DWORD; //can be IFSFN_OPEN, IFSFN_DELETE,
                                // or IFSFN_DIR
    tr_drive           : DWORD;
    tr_ir_flags        : Byte {unsigned char};
    tr_ir_options      : WORD;
    tr_res_type        : DWORD;
    tr_handle          : DWORD;
    tr_file1           : ARRAY [0..MAX_PATH-1] OF char;
    tr_file2           : ARRAY [0..MAX_PATH-1] OF char;
    tr_program         : ARRAY [0..8] OF char;
    tr_proc_handle     : DWORD;
    tr_error           : WORD;
    tr_drv_context     : DWORD;
    tr_drv_status      : WORD;
    tr_drv_miscflag    : WORD;
//The following fields are valid if an OPEN operation has been
// logged. These indicate: whether the file existed prior to the
// open (i.e., it is not being created), and the data/time and
// size of the file if it existed prior to the open.
    tr_create_flag     : BYTE;  //TRUE if file is being created
    tr_low_size        : DWORD;
    tr_high_size       : DWORD;
    tr_low_datetime    : DWORD;
    tr_high_datetime   : DWORD;
  end;

  //trap record used for logging registry functions
  Preg_trap_record = ^reg_trap_record;
  reg_trap_record = packed RECORD
    rtr_category            : DWORD; //indicator of functional area
                              // (e.g. file system, registry, etc.)
    rtr_function            : DWORD;
    rtr_program             : ARRAY [0..8] OF Char;
    rtr_proc_handle         : DWORD;
    rtr_lp_key_name         : DWORD;
    rtr_lp_subkey_name      : DWORD;
    rtr_lp_value_name       : DWORD;
    rtr_value_type          : DWORD;
    rtr_lp_value_data       : DWORD;
    rtr_len_value_data      : DWORD;
    rtr_lp_prev_value_data  : DWORD;
    rtr_len_prev_value_data : DWORD; //zero if no previous data
//These elements handle storage for values that exceed what can be
//stored in the typical logging records. Most likely, the value
//is something unusual such as a large binary field. In such
//cases the VxD must allocate additional space (which must be
//released once the trap record has been retrieved by the client).
//If the rtr_extra_rec contains a nonzero value, client must use
//the CntrlTrap VXD_RELEASE_EXT IOCTL to release these extended
//trapping record(s).
    rtr_extra_rec1          : DWORD;
    rtr_extra_rec2          : DWORD;
    rtr_extra_rec3          : DWORD;
  end;

  upcall = record
    trap_rec_num : DWORD;
    trap_dat_ptr : DWORD;
    trap_dat_len : WORD;
  end;
VAR
  VXD_NAME    : String;
  trap_update : upcall;
  the_version : WORD;
  vxd_handle  : THandle;
  critsec     : TRTLCRITICALSECTION;

CONST
  NumConds = 11;
  IfsConds = 6;
  chs : ARRAY[0..NumConds-1] OF DWORD = (0,0,0,0,0,0,0,0,0,0,0);
  tcs : ARRAY[0..NumConds-1] OF trap_criteria =
   ((tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_DIR; tc_flags : CREATE_DIR;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive : LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : 0),
    (tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_DIR; tc_flags : DELETE_DIR;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive : LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : 0),
    (tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_DELETE; tc_flags : NO_FLAGS_SPECIFIED;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive : LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : 0),
    (tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_OPEN; tc_flags : ACCESS_WRITEONLY;
      tc_flags_mask : ACCESS_MODE_MASK; tc_drive : LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : OPENS_FOR_WRITE),
    (tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_RENAME; tc_flags : NO_FLAGS_SPECIFIED;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive :  LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : 0),
    (tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_OPEN; tc_flags : ACCESS_READWRITE;
      tc_flags_mask : ACCESS_MODE_MASK; tc_drive : LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : OPENS_FOR_WRITE),
    (tc_category : REG_FUNCTIONS;
      tc_func_num : REG_CREATE_KEY; tc_flags : NO_FLAGS_SPECIFIED;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive : LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : 0),
    (tc_category : REG_FUNCTIONS;
      tc_func_num : REG_DELETE_KEY; tc_flags : NO_FLAGS_SPECIFIED;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive :  LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : 0),
    (tc_category : REG_FUNCTIONS;
      tc_func_num : REG_DELETE_VALUE; tc_flags : NO_FLAGS_SPECIFIED;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive :  LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : 0),
    (tc_category : REG_FUNCTIONS;
      tc_func_num : REG_CHANGE_VALUE; tc_flags : NO_FLAGS_SPECIFIED;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive :  LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : 0),
    (tc_category : REG_FUNCTIONS;
      tc_func_num : REG_CREATE_VALUE; tc_flags : NO_FLAGS_SPECIFIED;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive :  LOG_ALL;
      tc_proc_handle : (LOG_ALL,0,0,0,0,0,0,0,0,0); tc_callback : 0;
      tc_misc : 0));

VAR
  FilOpeList : TStringList;
  FilCreList : TStringList;
  FilDelList : TStringList;
  DirCreList : TStringList;
  DirDelList : TStringList;
  KeyCreList : TStringList;
  KeyDelList : TStringList;
  ValCreList : TStringList;
  ValChaList : TStringList;
  ValDelList : TStringList;
  ProcIdList : TStringList;
  dummy      : DWORD;

procedure AddAProcess(P : THandle);
VAR Idx : Integer;
begin
  Idx := ProcIdList.IndexOfObject(Pointer(P));
  IF Idx = -1 THEN
    ProcIdList.AddObject(ExeNameFromPID(P), Pointer(P));
end;

procedure release_rec(VAR rec_address : DWORD);
begin
  DeviceIoControl(vxd_handle, VXD_RELEASE_REC,
    @rec_address, 4, nil, 0, dummy, nil);
end;

procedure release_extrec(VAR rec_address : DWORD);
begin
  if rec_address <> 0 THEN
    DeviceIoControl(vxd_handle, VXD_RELEASE_EXT,
      @rec_address, 4, nil, 0, dummy, nil);
end;

procedure record_data(trap_rec_ptr : Ptrap_record);
VAR
  rt_ptr : PReg_trap_record;
  Idx    : Integer;
  sKey, sSub, sAll, sVal, sDat, sPrv : String;
  PP    : Pointer;

  function StringForValue(P : Pointer; Len, rType : Integer)
    : String;
  VAR
    D : DWord;
    N : Integer;
    B : PByteArray;
  begin
    CASE rType OF
      REG_DWORD : begin
        D := PDWORD(P)^;
        Result := IntToStr(D);
      end;
      REG_DWORD_BIG_ENDIAN : begin
        D := PDWORD(P)^;
        D := MakeLong(Swap(HiWord(D)), Swap(LoWord(D)));
        Result := IntToStr(D);
      end;
      REG_SZ,
      REG_MULTI_SZ,
      REG_EXPAND_SZ : begin
        Result := StrPas(PChar(P));
        FOR D := 1 TO Length(Result) DO
          IF Result[D] < #32 THEN
            Result[D] := '.';
      end;
      ELSE // treat as binary
        begin
          Result := '';
          N := 0;
          B := PByteArray(P);
          WHILE N+15 < Len DO
            begin
              Result := Format('%s%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,'+
                '%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x ',
                [Result, B[N], B[N+1], B[N+2], B[N+3], B[N+4],
                B[N+5], B[N+6], B[N+7], B[N+8], B[N+9], B[N+10],
                B[N+11], B[N+12], B[N+13], B[N+14], B[N+15]]);
              N := N + 16;
            end;
          WHILE N < Len DO
            begin
              Result := Format('%s%.2x', [Result, B[N]]);
              Inc(N);
            end;
        end;
    END;
  end;

begin
  EnterCriticalSection(critsec);
  try
    rt_ptr := PReg_trap_record(trap_rec_ptr);
    IF trap_rec_ptr^.tr_category = IFSMGR_FUNCTIONS THEN
      WITH trap_rec_ptr^ DO
        begin
          IF tr_error > 0 THEN Exit;
          AddAProcess(tr_proc_handle);
          PP := Pointer(tr_proc_handle);
          CASE tr_function OF
            IFSFN_OPEN   : IF tr_create_flag = 0 THEN
              begin
                // If not already on Cre or Ope list, add to Ope
                IF (FilOpeList.IndexOf(tr_file1) < 0) AND
                   (FilCreList.IndexOf(tr_file1) < 0) THEN
                  FilOpeList.AddObject(tr_file1, FilOpeObject.Create(
                    tr_high_size, tr_low_size, tr_high_datetime,
                    tr_low_datetime, tr_proc_handle));
              end
            ELSE
              begin
                // If not already on Cre list, add to Cre
                IF (FilCreList.IndexOf(tr_file1) < 0) THEN
                  FilCreList.AddObject(tr_file1, PP);
              end;
            IFSFN_RENAME : begin
              // Treat as delete tr_file1/create tr_file2
              // If not already on Del list, add to Del
              IF (FilDelList.IndexOf(tr_file1) < 0) THEN
                FilDelList.AddObject(tr_file1, PP);
              // If on Cre list, remove from Cre
              idx := FilCreList.IndexOf(tr_file1);
              IF idx >= 0 THEN FilCreList.Delete(idx);
              // If on Ope list, remove from Ope
              idx := FilOpeList.IndexOf(tr_file1);
              IF idx >= 0 THEN FilOpeList.Delete(idx);
              // If new-name not on create list, add it
              IF (FilCreList.IndexOf(tr_file2) < 0) THEN
                FilCreList.AddObject(tr_file2, PP);
            end;
            IFSFN_READ   : Exit;
            IFSFN_CLOSE  : Exit;
            IFSFN_DELETE : begin
              // If not already on Del list, add to Del
              // If not on Del, Cre, or Ope list, add to Del
              IF (FilDelList.IndexOf(tr_file1) < 0) AND
                 (FilOpeList.IndexOf(tr_file1) < 0) AND
                 (FilCreList.IndexOf(tr_file1) < 0) THEN
                FilDelList.AddObject(tr_file1, PP);
              // If on Cre list, remove from Cre
              idx := FilCreList.IndexOf(tr_file1);
              IF idx >= 0 THEN FilCreList.Delete(idx);
              // If on Ope list, remove from Ope
              idx := FilOpeList.IndexOf(tr_file1);
              IF idx >= 0 THEN FilOpeList.Delete(idx);
            end;
            IFSFN_DIR    : IF tr_ir_flags = CREATE_DIR THEN
              begin
                // if not on DirCre list, add to DirCre
                IF (DirCreList.IndexOf(tr_file1) < 0) THEN
                  DirCreList.AddObject(tr_file1, PP);
              end
            ELSE
              begin
                // if not on DirDel list, add to DirDel
                IF (DirDelList.IndexOf(tr_file1) < 0) THEN
                  DirDelList.AddObject(tr_file1, PP);
                // if on DirCre list, remove from DirCre
                idx := DirCreList.IndexOf(tr_file1);
                IF idx >= 0 THEN DirCreList.Delete(idx);
              end;
          END;
        end
    ELSE IF trap_rec_ptr^.tr_category = REG_FUNCTIONS THEN
      WITH rt_ptr^ DO
        begin
          AddAProcess(rtr_proc_handle);
          IF rtr_lp_key_name = 0 THEN
            Exit; // it's a "fake" report.
          IF (rtr_lp_subkey_name = 0) AND
            ((rtr_function = REG_CREATE_KEY) OR
             (rtr_function = REG_DELETE_KEY)) THEN
              Exit; // it's a "fake" report
          IF rtr_lp_key_name = 0 THEN sKey := ''
          ELSE sKey := StrPas(PChar(rtr_lp_key_name));
          IF rtr_lp_subkey_name = 0 THEN sSub := ''
          ELSE sSub := StrPas(PChar(rtr_lp_subkey_name));
          IF rtr_lp_value_name = 0 THEN sVal := ''
          ELSE sVal := StrPas(PChar(rtr_lp_value_name));
(*          IF (rtr_function = REG_CHANGE_VALUE) OR
            (rtr_function = REG_CREATE_VALUE) THEN*)//!!0.3
          IF rtr_lp_value_data <> 0 THEN //!!0.3  
            sDat := StringForValue(Pointer(rtr_lp_value_data),
              rtr_len_value_data, rtr_value_type)
          ELSE sDat := '';
          IF rtr_lp_prev_value_data <> 0 THEN
            sPrv := StringForValue(Pointer(rtr_lp_prev_value_data),
              rtr_len_prev_value_data, rtr_value_type)
          ELSE sPrv := '';
          IF (rtr_function = REG_CHANGE_VALUE) OR
            (rtr_function = REG_CREATE_VALUE) THEN
            begin
              IF (rtr_lp_prev_value_data = 0) AND
                (rtr_lp_value_data = 0) THEN
                begin
                  // both null - it's a no-operation
                  Exit;
                end;
              IF (rtr_lp_prev_value_data <> 0) AND
                (rtr_lp_value_data <> 0) AND
                (rtr_len_prev_value_data = rtr_len_value_data) AND
                CompareMem(Pointer(rtr_lp_value_data), Pointer(
                  rtr_lp_prev_value_data), rtr_len_value_data) THEN
                begin
                  // identical - it's a no-operation
                  Exit;
                end;
              CASE rtr_value_type OF
                REG_SZ,
                REG_MULTI_SZ,
                REG_EXPAND_SZ : IF sDat = sPrv THEN
                  begin
                    // Strings identical - it's a no-operation
                    Exit;
                  end;
              END;
            end;
          sAll := Format('%s\%s', [sKey, sSub]);
          CASE rtr_function OF
            REG_CREATE_KEY   : KeyCreList.AddObject(sAll,
              RegObject.Create(rtr_value_type, rtr_proc_handle, sVal,
                sDat, sPrv));
            REG_DELETE_KEY   : KeyDelList.AddObject(sAll,
              RegObject.Create(rtr_value_type, rtr_proc_handle, sVal,
                sDat, sPrv));
            REG_CREATE_VALUE : ValCreList.AddObject(sAll,
              RegObject.Create(rtr_value_type, rtr_proc_handle, sVal,
                sDat, sPrv));
            REG_CHANGE_VALUE : ValChaList.AddObject(sAll,
              RegObject.Create(rtr_value_type, rtr_proc_handle, sVal,
                sDat, sPrv));
            REG_DELETE_VALUE : ValDelList.AddObject(sAll,
              RegObject.Create(rtr_value_type, rtr_proc_handle, sVal,
                sDat, sPrv));
          END;
        end;
  finally
    LeaveCriticalSection(critsec);  
  end;
end;


procedure our_callback(d : DWORD); stdcall;
VAR trap_rec_ptr : Ptrap_record;
begin
  trap_rec_ptr := PTrap_record(d);
  record_data(trap_rec_ptr);
  if trap_rec_ptr^.tr_category = REG_FUNCTIONS THEN
    WITH PReg_Trap_record(trap_rec_ptr)^ DO
      begin
        release_extrec(rtr_extra_rec1);
        release_extrec(rtr_extra_rec2);
        release_extrec(rtr_extra_rec3);
      end;
  release_rec(d);
end;


//**************************
//  API-style threads - begin

VAR
  stop_flag : THandle;
  thread_handle : THandle;
  thread_id : DWORD;

function do_getcalls : DWORD; export; stdcall; // thread funktion
VAR N : Integer;
begin
  FOR N := 0 TO NumConds-1 DO
    DeviceIoControl(vxd_handle, VXD_REGISTER_CB, @tcs[N],
      sizeof(trap_criteria), @chs[N], sizeof(DWORD), dummy, nil);
  while WaitForSingleObjectEx(stop_flag, INFINITE, TRUE) =
    WAIT_IO_COMPLETION do;
  result := 0;
end;

function GetObsfucator : DWORD;
VAR PID : DWORD;
begin
  result := 0;
  vxd_handle := INVALID_HANDLE_VALUE;
  vxd_handle := CreateFile(PChar(VXD_NAME), 0, 0, nil,
    0, FILE_FLAG_DELETE_ON_CLOSE, 0);
  IF vxd_handle <> INVALID_HANDLE_VALUE THEN
    begin
      PID := GetCurrentProcessID;
      IF NOT DeviceIoControl(vxd_handle, VXD_GET_OBFUSCATOR,
        @PID, 4, @result, 4, dummy, nil) THEN result := 0;
      CloseHandle(vxd_Handle);
    end
  else
    MessageBox(0, PChar('Failed to open ' + VXD_NAME + ': ' +
      SysErrorMessage(GetLastError)), 'DEBUG', MB_OK);

end;

function GetVxdVersion : Integer;
begin
  result := -1;
  vxd_handle := INVALID_HANDLE_VALUE;
  vxd_handle := CreateFile(PChar(VXD_NAME), 0, 0, nil,
    0, FILE_FLAG_DELETE_ON_CLOSE, 0);
  IF vxd_handle <> INVALID_HANDLE_VALUE THEN
    begin
      FillChar(the_version, SizeOf(the_version), 0);
      IF DeviceIoControl(vxd_handle, VXD_GET_VER,
        nil, 0, @the_version, 2, dummy, nil) THEN
        Result := the_version;
      CloseHandle(vxd_Handle);
    end;
end;

function StartWatching : Integer;
VAR N : Integer;
begin
  FilOpeList.Clear; FilCreList.Clear; FilDelList.Clear;
  DirCreList.Clear; DirDelList.Clear; KeyCreList.Clear;
  KeyDelList.Clear; ValCreList.Clear; ValChaList.Clear;
  ValDelList.Clear; ProcIdList.Clear;
  vxd_handle := INVALID_HANDLE_VALUE;
  vxd_handle := CreateFile(PChar(VXD_NAME), 0, 0, nil, 0,
    FILE_FLAG_DELETE_ON_CLOSE, 0);
  IF vxd_handle = INVALID_HANDLE_VALUE THEN
    begin
      result := sw_NoOpen;
      MessageBox(0, PChar('Failed to open ' + VXD_NAME + ': ' +
        SysErrorMessage(GetLastError)), 'InCtrl4', MB_OK OR
        MB_ICONSTOP);
    end
  else
    begin
      FillChar(the_version, SizeOf(the_version), 0);
      IF NOT DeviceIoControl(vxd_handle, VXD_GET_VER, nil, 0,
        @the_version, 2, dummy, nil) THEN result := sw_NoVer
      else
        begin
          FOR N := 0 TO NumConds-1 DO
            tcs[N].tc_callback := DWORD(@our_callback);
          InitializeCriticalSection(critsec);
          stop_flag := CreateEvent(nil, false, false, nil);
          ResetEvent(stop_flag);
          thread_handle := CreateThread(nil, 4096, @do_getcalls,
            nil, 0, thread_id);
          result := sw_OK;
          IsWatching := True;
        end;
    end;
end;

procedure StopWatching;
VAR N : Integer;
begin
  IF NOT IsWatching THEN Exit;
  IsWatching := False;
  FOR N := 0 TO NumConds-1 DO
    begin
      trap_update.trap_rec_num := chs[N];
      trap_update.trap_dat_ptr := Integer(@tcs[N]);
      trap_update.trap_dat_len := sizeof(trap_criteria);
      tcs[N].tc_misc := (APP_EXITING OR DISABLE_LOGGING);
      DeviceIoControl(vxd_handle,VXD_UPDATE, @trap_update,
        sizeof(upcall), nil, 0, dummy, nil );
    end;
  CloseHandle(vxd_handle);
  SetEvent(stop_flag);
  CloseHandle(thread_handle);
  DeleteCriticalSection(critsec);
end;

function GetAList(idx : Integer) : TStringList;
VAR
  TempList : TStringList;
  N        : Integer;
  ID       : Pointer;
begin
  CASE idx OF
    L_FilOpe : TempList := FilOpeList;
    L_FilCre : TempList := FilCreList;
    L_FilDel : TempList := FilDelList;
    L_DirCre : TempList := DirCreList;
    L_DirDel : TempList := DirDelList;
    L_KeyCre : TempList := KeyCreList;
    L_KeyDel : TempList := KeyDelList;
    L_ValCre : TempList := ValCreList;
    L_ValCha : TempList := ValChaList;
    L_ValDel : TempList := ValDelList;
    L_ProcID : begin
      Result := ProcIDList;
      exit;
    end;
    ELSE
      begin
        result := nil;
        exit;
      end;
  END;
  // Separate the items by ProcId, while keeping
  // the order otherwise the same.
  Result := TStringList.Create;
  WHILE TempList.Count > 0 DO
    begin
      ID := TempList.Objects[0];
      FOR N := 0 TO TempList.Count-1 DO
        IF TempList.Objects[N] = ID THEN
          Result.AddObject(TempList[N], ID);
      FOR N := TempList.Count-1 DOWNTO 0 DO
        IF TempList.Objects[N] = ID THEN
          TempList.Delete(N);
    end;
end;

VAR Buffer : ARRAY[0..MAX_PATH] OF Char;
initialization
  FilOpeList := TStringList.Create;
  FilCreList := TStringList.Create;
  FilDelList := TStringList.Create;
  DirCreList := TStringList.Create;
  DirDelList := TStringList.Create;
  KeyCreList := TStringList.Create;
  KeyDelList := TStringList.Create;
  ValCreList := TStringList.Create;
  ValChaList := TStringList.Create;
  ValDelList := TStringList.Create;
  ProcIdList := TStringList.Create;
  vxd_name := ExtractFileDir(Application.ExeName);
  GetShortPathName(PChar(vxd_name), Buffer, MAX_PATH);
  vxd_name := StrPas(Buffer);
  IF vxd_name[length(vxd_name)] = '\' THEN
    vxd_name := vxd_name + 'CTRLTRAP.VXD'
  ELSE vxd_name := vxd_name + '\CTRLTRAP.VXD';
  vxd_name := '\\.\'+vxd_name;
end.
