unit in32Regu;
(* NOTE: Weird programs that store *key names* with embedded CR/LFs
   will cause trouble. If more than one person reports the problem
   consider slowing thing down by adding a check for embedded CR/LFs.
   (current report from John Lortschner 72662,1742 *)


interface

uses
  Windows, Messages, SysUtils, Classes, Forms, in32allU, (*!*) dialogs;

procedure RememberRegistry(const OutPath : String; Ext : String4;
  Ignores : TStrings; UpStatus : UpdateStatusFunc);

procedure CompareRegistry(const OldN, NewN : TFilename;
  vAddKey, vDelKey, vChaVal, vAddVal, vDelVal : TStrings;
  UpStatus : UpdateStatusFunc);

implementation
uses registry;
CONST
  MaxBinaryBytes = 16;
  MaxStoreString = 256;

procedure RememberRegistry(const OutPath : String; Ext : String4;
  Ignores : TStrings; UpStatus : UpdateStatusFunc);
VAR
  R     : TRegistry;
  RootK : String;
  OutF  : TextFile;

  function StringForKey(const S: String; Level : Char) : String;
  {Create a string to represent a registry value. Strings are
   unchanged, numbers are converted to string, and short binary
   or other values ("short" means less than MaxBinaryBytes) are
   converted to a series of hex digits separated by commas. Longer
   "other" values are XORed together into a "short" series of
   binary values and represented the same; adding 16 to data
   type signals a "long" value.}
  VAR
   dSize, dType,
   {L,} P1    : Integer;
   Buffer   : PChar;
   Bytes    : PByteArray;
   PLong    : PDWord;

    function StringOfBytes : String;
    VAR L, MaxShow : Integer;
    begin
      IF dSize >= 32768 THEN
        Result := 'Binary data over 32K - changes not tracked'
      ELSE
        BEGIN
          MaxShow := dSize;
          IF dSize > MaxBinaryBytes THEN {XOR together too-long data}
            BEGIN
              FOR L := MaxBinaryBytes TO dSize-1 DO
                Bytes^[L MOD MaxBinaryBytes] :=
                  Bytes^[L MOD MaxBinaryBytes] XOR Bytes^[L];
              Inc(dType,16);
              MaxShow := MaxBinaryBytes;
            END;
          Result := IntToHex(Bytes^[0], 2);
          FOR L := 1 TO MaxShow-1 DO
            Result := Result + ',' + IntToHex(Bytes^[L], 2);
        END;
   end;
  BEGIN
    dType := 0; dSize := 0;
    Result := '';
    {get required buffer size and allocate buffer}
    RegQueryValueEx(R.CurrentKey, PChar(S), Nil, @dType, NIL, @dSize);
    GetMem(Bytes, dSize);
    Buffer := PChar(Bytes);
    PLong := PDWord(Bytes);
    try (*!*) try
      {get value data}
      IF RegQueryValueEx(R.CurrentKey, PChar(S), Nil, NIL,
           @Bytes[0], @dSize) <> ERROR_SUCCESS THEN
         Exit;
      CASE dType OF
        REG_DWORD             : BEGIN   {double-word value}
          Result := Format('%d', [PLong^]);
        END;
        REG_DWORD_BIG_ENDIAN  : BEGIN   {double-word value LSB first}
          Result := Format('%d', [MakeLong(Swap(HiWord(PLong^)),
            Swap(LoWord(PLong^)))]);
        END;
        REG_SZ, REG_EXPAND_SZ : BEGIN   {string value; remove CRLF}
          IF dSize=0 THEN Result := ''
          ELSE IF dSize >= MaxStoreString THEN
            Result := StringOfBytes
          ELSE
            BEGIN
              {Help for GetQueryValueEx *says* that dSize includes the
               terminating NUL character; experience says it sometimes
               does, sometimes does not}
              IF Buffer[dSize-1] = #0 THEN
                SetString(Result, Buffer, dSize-1)
              ELSE SetString(Result, Buffer, dSize);
              REPEAT
                P1 := Pos(#13, Result);
                IF P1 <> 0 THEN Result[P1] := ' ';
              UNTIL P1=0;
              REPEAT
                P1 := Pos(#10, Result);
                IF P1 <> 0 THEN Result[P1] := ' ';
              UNTIL P1=0;
            END;
        END;
        ELSE IF dSize > 0 THEN
          Result := StringOfBytes;
        END;
    finally
      FreeMem(Bytes);
      Result := Format('%.1s%.1s%s',
        [Level, Char(dType+2+Ord('0')), Result]);
    end;
    except (*!*)
      ON Exception DO
        BEGIN
          WriteLn(OutF, 'ERROR GETTING DATA: ', R.CurrentPath, ' (',S,')');
          Raise;
        END;
    end;
  END;

  procedure RememberKey(const Pth : String; Level : Char);
  VAR
    I, J : Integer;
    TL   : TStringList; {local TStringList}
    S    : String;
  BEGIN
    {Let Windows process messages; Exit if app terminated}
    IF ProcMsgTerminated THEN Exit;
    {don't store our own keys}
    IF Pos(IC32Key, RootK+Pth) > 0 THEN
      Exit;
    {don't store these keys; they change too much}
    S := RootK+Uppercase(Pth);
    FOR J := 0 TO Ignores.Count-1 DO
      IF S = Ignores[J] THEN
        Exit;
    IF NOT R.OpenKey(Pth, False) THEN Exit;
    TL := TStringList.Create;
    try (*!*) try
      IF Assigned(UpStatus) AND (Level <= '3') THEN
        UpStatus('Recording Registry keys:', RootK+Pth);
      TL.Sorted := True;
      {write out values}
      R.GetValueNames(TL);
      FOR I := 0 TO TL.Count-1 DO
        BEGIN
          IF TL.Strings[I]='' THEN
            WriteLn(OutF,  Format('%.1s1@', [Level]))
          ELSE WriteLn(OutF,  Format('%.1s1%s', [Level,TL.Strings[I]]));
          WriteLn(OutF, StringForKey(TL.Strings[I], Succ(Level)));
        END;
      {write out subkeys}
      R.GetKeyNames(TL);
      FOR I := 0 TO TL.Count-1 DO
        BEGIN
          IF TL.Strings[I] = '' THEN Continue;
          WriteLn(OutF, Format('%.1s0%s', [Level, TL.Strings[I]]));
          RememberKey(Pth + '\' + TL.Strings[I], Succ(Level));
        END;
    finally
      TL.Free;
    END;
    except (*!*)
      ON Exception DO
        BEGIN
          WriteLn(OutF, 'ERROR GETTING KEYS/VALUES: ', Pth);
          Raise;
        END;
    end;
  END;

begin
  AssignFile(OutF, OutPath + Ext);
  Rewrite(OutF);
  R := TRegistry.Create;
  WITH R DO
    try (*!*) try
      R.RootKey := HKEY_USERS;
      RootK := 'HKEY_USERS';
      WriteLn(OutF, '00HKEY_USERS');
      RememberKey('', '1');
      R.RootKey := HKEY_LOCAL_MACHINE;
      RootK := 'HKEY_LOCAL_MACHINE';
      WriteLn(OutF, '00HKEY_LOCAL_MACHINE');
      RememberKey('', '1');
    finally
      Free;
      CloseFile(OutF);
    end;
    except (*!*)
      ON Exception DO
        BEGIN
          ShowMessage('An error occurred. email '+OutPath+Ext+' to Neil');
          Raise;
        END;
    end;
end;

procedure CompareRegistry(const OldN, NewN : TFilename;
  vAddKey, vDelKey, vChaVal, vAddVal, vDelVal : TStrings;
  UpStatus : UpdateStatusFunc);
{Take two registry-layout files in the format created by the
 RememberRegistry function and compare them, reporting keys
 added or deleted and values added, deleted, or changed}
VAR
  OldFile, NewFile : TextFile;
  OldPath, NewPath : TStringList;
  OldData, NewData : String;
  OldType, NewType,
  OldLevl, NewLevl : Char;
  OldDone, NewDone : Boolean;

  FUNCTION StringFromList(TS : TStringList) : String;
  VAR N : Integer;
  BEGIN
    IF TS.Count > 0 THEN
      BEGIN
        Result := TS[0];
        FOR N := 1 TO TS.Count-1 DO
          Result := Result + '\' + TS[N];
      END
    ELSE Result := '';
  END;

  FUNCTION KeyFromList(TS : TStringList) : String;
  VAR N : Integer;
  BEGIN
    IF TS.Count > 1 THEN
      BEGIN
        Result := TS[0];
        FOR N := 1 TO TS.Count-2 DO
          Result := Result + '\' + TS[N];
      END
    ELSE Result := '';
  END;

  PROCEDURE ReadNext(VAR T : TextFile; VAR Pth: TStringList;
    VAR Data : String; VAR Level, dType : Char);
  {Read next item from specified file, adding to Pth list;
   report key value (if any) in Data parameter, subkey
   level in Level parameter, data type (if any) in dType
   parameter. If at end of file, clear the Pth list.}
  VAR
    TempStr, NewName : String;
    NewLevel         : Char;
  BEGIN
    IF EoF(T) THEN
      BEGIN
        Pth.Clear;
        Level := '/';
        Exit;
      END;
    ReadLn(T, TempStr);
    NewLevel := TempStr[1];
    IF NewLevel = '0' THEN
      BEGIN
        Pth.Clear;
        Pth.Add(Copy(TempStr, 3, length(TempStr)-2));
        Level := '0';
        Exit;
      END;
    dType   := TempStr[2];
    Data    := '';
    NewName := Copy(TempStr, 3, Length(TempStr)-2);
    IF NewLevel <= Level THEN Pth.Delete(Pth.Count-1);
    WHILE (NewLevel < Level) AND (Pth.Count > 0) DO
      BEGIN
        Pth.Delete(Pth.Count-1);
        Dec(Level);
      END;
    Pth.Add(NewName);
    Level := NewLevel;
    IF dType = '1' THEN {it's a value name}
      BEGIN
        ReadLn(T, TempStr);
        IF (TempStr[1] <> Succ(NewLevel)) OR (TempStr[2] <= '1') THEN
          Raise Exception.Create('Stored Registry data is corrupted');
        dType := TempStr[2];
        Data  := Copy(TempStr, 3, Length(TempStr)-2);
       END;
  END;

  FUNCTION ComparePath(SL1, SL2 : TStringList) : Integer;
  {Empty list indicates EoF file, and thus is always LAST.
   If non-zero list length differs, shorter is first.
   Otherwise, compare path items}
  VAR N : Integer;
  BEGIN
    IF (SL1.Count=0) AND (SL2.Count=0) THEN
      Result := 0
    ELSE IF SL1.Count=0 THEN
      Result := 1
    ELSE IF SL2.Count=0 THEN
      Result := -1
    ELSE IF SL1.Count < SL2.Count THEN
      Result := 1
    ELSE IF SL1.Count > SL2.Count THEN
      Result := -1
    ELSE
      BEGIN
        N := 0;
        Result := 0;
        WHILE (N < SL1.Count) AND (Result = 0) DO
          BEGIN
           {Use AnsiCompareText to be same as sorted list}
            Result := AnsiCompareText(SL1[N], SL2[N]);
            Inc(N);
          END;
        IF Result <> 0 THEN
          Result := Result DIV Abs(Result);
      END;
  END;

BEGIN
  vAddKey.Clear;
  vDelKey.Clear;
  vChaVal.Clear;
  vAddVal.Clear;
  vDelVal.Clear;
  AssignFile(OldFile, OldN);
  AssignFile(NewFile, NewN);
  OldPath := TStringList.Create;
  NewPath := TStringList.Create;
  try
    FileMode := 0; {read-only}
    Reset(OldFile);
    try
      Reset(NewFile);
        OldLevl := '0';
        NewLevl := '0';
        OldType := '0';
        NewType := '0';
        OldData := '';
        NewData := '';
        OldDone := False;
        NewDone := False;
        REPEAT
          IF EoF(OldFile) THEN OldDone := True;
          IF EoF(NewFile) THEN NewDone := True;
          IF ProcMsgTerminated THEN Exit;
          IF Assigned(UpStatus) AND (OldLevl <= '2') THEN
            UpStatus('Comparing Registry keys:', StringFromList(OldPath));
          CASE ComparePath(OldPath, NewPath) OF
            -1 : BEGIN {old is less; was deleted}
              IF OldType <= '1' THEN
                vDelKey.Add(StringFromList(OldPath))
              ELSE
                vDelVal.Add(Format('%s="%s"', [StringFromList(OldPath),
                   OldData]));
              ReadNext(OldFile, OldPath, OldData, OldLevl, OldType);
            END;
            0  : BEGIN {same; check for change}
              IF OldData <> NewData THEN
                BEGIN
                  vChaVal.Add(KeyFromList(OldPath));
                  IF Ord(OldType)-Ord('2') >= 16 THEN
                    CASE Ord(OldType)-Ord('2')-16 OF
                      REG_SZ, REG_EXPAND_SZ :
                        vChaVal.Add(Format('Value "%s": string data changed',
                          [OldPath[OldPath.Count-1]]));
                      ELSE
                        vChaVal.Add(Format('Value "%s": binary data changed',
                          [OldPath[OldPath.Count-1]]));
                    END
                  ELSE
                    vChaVal.Add(Format('Value "%s": from "%s" to "%s"',
                      [OldPath[OldPath.Count-1], OldData, NewData]));
                END;
              ReadNext(OldFile, OldPath, OldData, OldLevl, OldType);
              ReadNext(NewFile, NewPath, NewData, NewLevl, NewType);
            END;
            1  : BEGIN {new is less; was added}
              IF NewType <= '1' THEN
                vAddKey.Add(StringFromList(NewPath))
              ELSE
                vAddVal.Add(Format('%s="%s"', [StringFromList(NewPath),
                  NewData]));
              ReadNext(NewFile, NewPath, NewData, NewLevl, NewType);
            END;
          END;
        UNTIL OldDone AND NewDone;
      finally
        CloseFile(NewFile);
      end;
  finally
    CloseFile(OldFile);
    FileMode := 2; {read-write}
    OldPath.Free;
    NewPath.Free;
  end;
END;

end.
