unit in32dsku;

interface
uses SysUtils, Windows, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, in32AllU;

procedure RememberDisk(const fName : TFilename;
  DirsWatched : TStrings; UpStatus : UpdateStatusFunc);
procedure CompareDisk(const OldN, NewN : TFilename;
  vAdd, vDel, vCha : TStrings; UpStatus : UpdateStatusFunc);
procedure FindNewFiles(L : TStrings; StartTime : TFileTime;
  DirsWatched : TStringList; UpStatus : UpdateStatusFunc);

implementation

function SkinnyName(const S : TFilename) : TFilename;
CONST MaxWid = 30;
VAR
  NameOnly : String;
  Path     : TFileName;
  Len      : Integer;
BEGIN
  IF Length(S) < MaxWid THEN Result := S
  ELSE
    BEGIN
      NameOnly := ExtractFilename(S);
      Len  := MaxWid - 3 - Length(NameOnly);
      Path := ExtractFileDir(S);
      WHILE (Length(Path) > Len) AND (Length(Path) > 3) DO
        Path := ExtractFileDir(Path);
      IF Length(Path) = 3 THEN
        Result := Copy(Path, 1, 2) + '..\' + NameOnly
      ELSE Result := Path + '\..\' + NameOnly;
      IF Length(Result) > MaxWid THEN
        Result := Copy(Result, 1, MaxWid-3)+'...'
    END;
END;

(* FIG 4 BEGIN *)
procedure RememberDisk(const fName : tFilename;
  DirsWatched : TStrings; UpStatus : UpdateStatusFunc);
{NOTE: This function frees the DirsWatched list on exiting}
VAR
  OutF     : TextFile;
  StartDir : TFilename;
  N        : Integer;

  function AddDir(const Dir : TFileName; const line : String): String;
  {First char of line indicates level; last char indicates whether
   file or directory. Strip both and append to Dir}
  BEGIN
    Result := Dir + '\' + Copy(line, 2, Length(line)-2);
  END;

  procedure AddOneFile(TL : TStringList; VAR vTS : TSearchRec;
    vLevel : Char);
  VAR
    sTimeSize : ARRAY[0..3] OF Char;
    N         : Integer;
  begin
    {Combine date/time and file size using XOR}
    LongInt(sTimeSize) := vTS.Time XOR vTS.Size;
    {Treat date/time/size as a string; fix bad control characters}
    FOR N := 0 TO 3 DO
      CASE sTimeSize[N] OF
        #0, #8, #9, #10, #13, #26 : Inc(sTimeSize[N], $20);
      END;
    {Store the filename and its date/time/size in the list}
    TL.AddObject(Format('%.1s%s%.1x',[vLevel, vTS.Name,
      (vTS.Attr AND faDirectory) SHR 4]), Pointer(sTimeSize));
  end;

  procedure FindFiles(const Dir : TFileName; Level : Char);
  VAR
    TS     : TSearchRec;
    I      : Integer;
    TL     : TStringList; {local TStringList}
    TempS  : String;
    DirC   : Char;
    barray : ARRAY[0..4] OF Char;
    P      : Pointer;
  BEGIN
    IF Assigned(UpStatus) AND (Level <= '3') THEN
      UpStatus('Recording directory:', SkinnyName(Dir));
    {Let Windows process messages; Exit if app terminated}
    IF ProcMsgTerminated THEN Exit;
    TL := TStringList.Create;
    try
      TL.Sorted := True;
      {add file strings to sorted list box}
      I := FindFirst(Dir+'\*.*', faReadOnly OR faDirectory OR faHidden
        OR faSysFile, TS);
      WHILE I = 0 DO
        BEGIN
          IF TS.Name[1] <> '.' THEN AddOneFile(TL, TS, Level);
          I := FindNext(TS);
        END;
      FindClose(TS.FindHandle);
      {Write out strings in sorted order}
      FOR I := 0 TO TL.Count-1 DO
        BEGIN
          TempS := TL.Strings[I];
          {Strip the last char, '0' for file, '1' for dir}
          DirC  := TempS[Length(TempS)];
          SetLength(TempS, Length(TempS)-1);
          {Insert the date/time/size just after the level char}
          P := TL.Objects[I];
          Move(P, barray, 4);
          barray[4] := #0;
          System.Insert(StrPas(barray), TempS, 2);
          WriteLn(OutF, TempS);
          {If it's a directory, make a recursive call to FindFiles}
          IF DirC = '1' THEN
            FindFiles(AddDir(Dir, TL.Strings[I]), Succ(Level));
        END;
    finally
      TL.Free;
    END;
  END;

BEGIN
  AssignFile(OutF, fName);
  Rewrite(OutF);
  try
    FOR N := 0 TO DirsWatched.Count-1 DO
      BEGIN
        StartDir := DirsWatched[N];
        IF ProcMsgTerminated THEN Exit;
        WriteLn(OutF, '0',StartDir);
        {Chop final backslash if present}
        IF Length(StartDir) = 3 THEN SetLength(StartDir, 2);
        FindFiles(StartDir, '1');
      END;
  finally
    CloseFile(OutF);
    DirsWatched.Free;
  end;
END;
(* FIG 4 END *)

procedure CompareDisk(const OldN, NewN : TFilename;
  vAdd, vDel, vCha : TStrings; UpStatus : UpdateStatusFunc);
{Take two disk-layout files in the format created by the
 RememberDisk method and compare them, reporting files and
 directories added or deleted and files changed}
VAR
  OldFile, NewFile : TextFile;
  OldPath, NewPath : String;
  OldData, NewData : String4;
  OldLevl, NewLevl : Char;
  OldDone, NewDone : Boolean;

  PROCEDURE ReadNext(VAR T : TextFile; VAR Pth: String;
    VAR DateSize : String4; VAR Level : Char);
  {Read next item from specified file, converting to full
   path in Pth parameter; report date/time/size data in
   DateSize parameter and subdirectory level in Level
   parameter. If at end of file, set level before '0'
   and clear the string}
  VAR
    TempStr  : String;
    NewLevel : Char;
    NewName  : String;
  BEGIN
    IF EoF(T) THEN
      BEGIN
        Pth := '';
        {Set level to char before '0'; this handles
         case where ENTIRE DIRECTORY has been demolished}
        Level := '/';
        Exit;
      END;
    ReadLn(T, TempStr);
    NewLevel := TempStr[1];
    IF NewLevel = '0' THEN
      BEGIN
        Pth := Copy(TempStr, 2, length(TempStr)-1);
        DateSize := '';
        Level := '0';
        Exit;
      END;
    DateSize := Copy(TempStr, 2, 4);
    NewName := Copy(TempStr, 6, Length(TempStr)-5);
    IF NewLevel <= Level THEN Pth := ExtractFilePath(Pth);
    WHILE NewLevel < Level DO
      BEGIN
        Pth := ExtractFilePath(Copy(Pth,1,Length(Pth)-1));
        Dec(Level);
      END;
    IF Pth[length(Pth)] <> '\' THEN Pth := Pth + '\';
    Pth   := Pth + NewName;
    Level := NewLevel;
  END;

  FUNCTION ComparePath(L1, L2 : Char; const P1, P2 : String) : Integer;
  {First compare depth of subdirectory. If same
   level, then compare paths}
  BEGIN
    IF (L1='\') AND (L2='\') THEN
      Result := 0
    ELSE IF L1='\' THEN
      Result := 1
    ELSE IF L2='\' THEN
      Result := -1
    ELSE IF L1 < L2 THEN
      Result := 1
    ELSE IF L1 > L2 THEN
      Result := -1
    ELSE
      BEGIN
        Result := AnsiCompareText(P1, P2);
        {Return precisely -1, 0, or 1}
        IF Result <> 0 THEN
          Result := Result DIV Abs(Result);
      END;
  END;

BEGIN
  vAdd.Clear;
  vDel.Clear;
  vCha.Clear;
  AssignFile(OldFile, OldN);
  AssignFile(NewFile, NewN);
  try
    FileMode := 0; {read-only}
    Reset(OldFile);
    try
      Reset(NewFile);
      OldLevl := '0';
      NewLevl := '0';
      OldDone := False;
      NewDone := False;
      OldData := '    ';
      NewData := '    ';
      OldPath := '';
      NewPath := '';
      REPEAT
        IF Assigned(UpStatus) AND (OldLevl <= '2') THEN
          UpStatus('Comparing directory:', SkinnyName(OldPath));
        IF ProcMsgTerminated THEN Exit;
        IF EoF(OldFile) THEN OldDone := True;
        IF EoF(NewFile) THEN NewDone := True;
        CASE ComparePath(OldLevl, NewLevl, OldPath, NewPath) OF
          -1 : BEGIN {old is less; was deleted}
            vDel.Add(OldPath);
            ReadNext(OldFile, OldPath, OldData, OldLevl);
          END;
          0  : BEGIN
            IF OldData <> NewData THEN
              vCha.Add(OldPath);
            ReadNext(OldFile, OldPath, OldData, OldLevl);
            ReadNext(NewFile, NewPath, NewData, NewLevl);
          END;
          1  : BEGIN {new is less; was added}
            vAdd.Add(NewPath);
            ReadNext(NewFile, NewPath, NewData, NewLevl);
          END;
        END;
      UNTIL OldDone AND NewDone;
    finally
      CloseFile(NewFile);
    end;
  finally
    CloseFile(OldFile);
    FileMode := 0; {read-write}
  end;
END;

procedure FindNewFiles(L : TStrings; StartTime : TFileTime;
  DirsWatched : TStringList; UpStatus : UpdateStatusFunc);
VAR N : Integer;

  function AddDir(const Dir : TFileName; const line : String): String;
  BEGIN
    Result := Dir + '\' + Copy(line, 2, Length(line)-2);
  END;

  function Newer(FD : TWin32FindData) : Boolean;
  CONST NoTime : TFileTime = (dwLowDateTime: 0; dwHighDateTime: 0);
  BEGIN
    Result :=
      ((CompareFileTime(FD.ftCreationTime, NoTime) <> 0) AND
       (CompareFileTime(FD.ftCreationTime, StartTime) > 0)) OR
      ((CompareFileTime(FD.ftLastWriteTime, NoTime) <> 0) AND
       (CompareFileTime(FD.ftLastWriteTime, StartTime) > 0));
  END;

  procedure FindNewFilesR(L : TStrings; const Dir : TFilename);
  VAR
    TS   : TSearchRec;
    Rslt : Integer;
  BEGIN
    IF Assigned(UpStatus) THEN
      UpStatus('Time-checking directory:', SkinnyName(Dir));
    Rslt := FindFirst(Dir+'\*.*', faDirectory OR faReadOnly OR faHidden
      OR faSysfile, TS);
    WHILE Rslt = 0 DO
      BEGIN
        IF TS.Attr AND faDirectory > 0 THEN
          BEGIN
            IF TS.Name[1] <> '.' THEN
              BEGIN
                IF Newer(TS.FindData) THEN
                  L.Add(Dir+'\'+TS.FindData.cFilename);
                FindNewFilesR(L, Dir + '\' + TS.FindData.cFilename);
              END;
          END
        ELSE IF Newer(TS.FindData) THEN
          L.Add(Dir+'\'+TS.FindData.cFilename);
        Rslt := FindNext(TS);
      END;
    FindClose(TS.FindHandle);
  END;
BEGIN
  try
    FOR N := 0 TO DirsWatched.Count-1 DO
      BEGIN
        IF Length(DirsWatched[N]) = 3 THEN
          DirsWatched[N] := Copy(DirsWatched[N], 1, 2);
        FindNewFilesR(L, DirsWatched[N]);
        IF ProcMsgTerminated THEN Exit;
      END;
  finally
    DirsWatched.Free;
  end;
END;

end.
