unit ctrltrap;

interface

USES Windows, SysUtils;

function StartWatching : Integer;
function StopWatching : String;

const
  sw_OK = 0;
  sw_NoOpen = -1;
  sw_NoVer = -2;
  sw_Already = -3;

CONST
  VXD_NAME  = '\\.\ctrltrap.VXD';
  VXD_GET_VER       = 1;
  VXD_REGISTER_CB   = 2; //register a callback and logging conditions
  VXD_RELEASE_REC   = 3; //release the trap buffer received from VxD
  VXD_RELEASE_EXT   = 5; //release extended buffer used by VxD
  VXD_UPDATE        = 4; //update logging conditions

  DISABLE_LOGGING    = 1;
  APP_EXITING        = 16;
  IFSMGR_FUNCTIONS   = $40;    //Installable File System Manager functions
  REG_FUNCTIONS      = $1;     //Registry functions

  ACCESS_MODE_MASK   = 7;
  ACCESS_WRITEONLY   = 1;
  ACCESS_READWRITE   = 2;

  REG_CREATE_KEY     = 0;
  REG_DELETE_KEY     = 1;
  REG_CREATE_VALUE   = 2;
  REG_CHANGE_VALUE   = 3;
  REG_DELETE_VALUE   = 4;

  IFSFN_OPEN     = 36; //function codes for file system operations
  IFSFN_READ     = 0 ;
  IFSFN_CLOSE    = 11;
  IFSFN_DELETE   = 31;
  IFSFN_DIR      = 32; //directory manipulation (tr_ir_flags indicate function)

  LOG_ALL         = $0ffffffff;
  REP_BUF_SIZE    = (60 * 1024);
//  MAX_PATH = 260
  NO_FLAGS_SPECIFIED = $ff;
  NO_MASK_SPECIFIED = 0;
  CREATE_DIR = 0;
  DELETE_DIR = 1;

  OPENS_FOR_WRITE = $8000; //only report opens that can potentially change file
  REPORT_ALL_CREATES = $4000; //report back all RegCreateKey calls --even if it's just opening the key

type
  PTrap_criteria = ^trap_criteria;
  trap_criteria = packed RECORD
    tc_category    : DWORD; //indicator of functional area (e.g. file system, registry, etc.)
    tc_func_num    : DWORD; //function app wants trapped (-1 = all)
    tc_flags       : BYTE;  //if function uses flags, can also specify here (zero if none)
    tc_flags_mask  : BYTE;  //if function uses flags, can also mask off before compare (zero if none)
    tc_drive       : DWORD; //drive (-1 = all drives)
    tc_proc_handle : DWORD; //process ID  (-1 if all processes)
    tc_callback    : DWORD; //address to call back
    tc_misc        : WORD;  //misc flags
  end;

//trap record used for logging installable file system manager functions

  Ptrap_record = ^trap_record;
  trap_record = packed RECORD
    tr_category        : DWORD; //indicator of functional area (e.g. file system, registry, etc.)
    tr_function        : DWORD; //can be IFSFN_OPEN, IFSFN_DELETE, or IFSFN_DIR
    tr_drive           : DWORD;
    tr_ir_flags        : {unsigned} char;
    tr_ir_options      : WORD;
    tr_res_type        : DWORD;
    tr_handle          : DWORD;
    tr_file1           : ARRAY [0..MAX_PATH-1] OF char;
    tr_file2           : ARRAY [0..MAX_PATH-1] OF char;
    tr_program         : ARRAY [0..7] OF char;
    tr_proc_handle     : DWORD;
    tr_error           : WORD;
    tr_drv_context     : DWORD;
    tr_drv_status      : WORD;
    tr_drv_miscflag    : WORD;
//The following fields are valid if an OPEN operation has been logged.
//These indicate: whether the file existed prior to the open (i.e., it is
//not being created), and the data/time and size of the file if it
//existed prior to the open.
    tr_create_flag     : BYTE;  //TRUE if file is being created
    tr_low_size        : DWORD;
    tr_high_size       : DWORD;
    tr_low_datetime    : DWORD;
    tr_high_datetime   : DWORD;
  end;

  //trap record used for logging registry functions
  Preg_trap_record = ^reg_trap_record;
  reg_trap_record = packed RECORD
    rtr_category            : DWORD; //indicator of functional area (e.g. file system, registry, etc.)
    rtr_function            : DWORD;
    rtr_program             : ARRAY [0..8] OF Char;
    rtr_proc_handle         : DWORD;
    rtr_lp_key_name         : DWORD;
    rtr_lp_subkey_name      : DWORD;
    rtr_lp_value_name       : DWORD;
    rtr_value_type          : DWORD;
    rtr_lp_value_data       : DWORD;
    rtr_len_value_data      : DWORD;
    rtr_lp_prev_value_data  : DWORD;
    rtr_len_prev_value_data : DWORD; //zero if no previous data
//These elements handle storage for values that exceed what can be
//stored in the typical logging records. Most likely, the value
//is something unusual such as a large binary field. In such
//cases the VxD must allocate additional space (which must be
//released once the trap record has been retrieved by the client).
//If the rtr_extra_rec contains a nonzero value, client must use
//the CntrlTrap VXD_RELEASE_EXT IOCTL to release these extended
//trapping record(s).
//!!!not currently implemented
    rtr_extra_rec1          : DWORD;
    rtr_extra_rec2          : DWORD;
    rtr_extra_rec3          : DWORD;
  end;

//*********************
    implementation
//*********************
uses Classes;

type
  upcall = record
    trap_rec_num : DWORD;
    trap_dat_ptr : DWORD;
    trap_dat_len : WORD;
  end;
VAR
  trap_update : upcall;
  general_io_buf : ARRAY [0..512] OF Char; //holds data to/from Vxd


  the_bytes_returned : DWORD;
  the_version : WORD;

  rep_buf_ptr : String;
  start_rep_buf : String;
  vxd_handle : THandle;
  critsec : TRTLCRITICALSECTION;


CONST
  NumConds = 10;
  chs : ARRAY[0..NumConds-1] OF DWORD = (0,0,0,0,0,0,0,0,0,0);
  tcs : ARRAY[0..NumConds-1] OF trap_criteria =
   ((tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_DIR; tc_flags : CREATE_DIR;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive : LOG_ALL;
      tc_proc_handle : LOG_ALL; tc_callback : 0; tc_misc : 0),
    (tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_DIR; tc_flags : DELETE_DIR;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive : LOG_ALL;
      tc_proc_handle : LOG_ALL; tc_callback : 0; tc_misc : 0),
    (tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_DELETE; tc_flags : NO_FLAGS_SPECIFIED;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive : LOG_ALL;
      tc_proc_handle : LOG_ALL; tc_callback : 0; tc_misc : 0),
    (tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_OPEN; tc_flags : ACCESS_WRITEONLY;
      tc_flags_mask : ACCESS_MODE_MASK; tc_drive : LOG_ALL;
      tc_proc_handle : LOG_ALL; tc_callback : 0; tc_misc : OPENS_FOR_WRITE),
    (tc_category : REG_FUNCTIONS;
      tc_func_num : REG_CREATE_KEY; tc_flags : NO_FLAGS_SPECIFIED;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive : LOG_ALL;
      tc_proc_handle : LOG_ALL; tc_callback : 0; tc_misc : 0),
    (tc_category : IFSMGR_FUNCTIONS;
      tc_func_num : IFSFN_OPEN; tc_flags : ACCESS_READWRITE;
      tc_flags_mask : ACCESS_MODE_MASK; tc_drive : LOG_ALL;
      tc_proc_handle : LOG_ALL; tc_callback : 0; tc_misc : OPENS_FOR_WRITE),
    (tc_category : REG_FUNCTIONS;
      tc_func_num : REG_DELETE_KEY ; tc_flags : NO_FLAGS_SPECIFIED ;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive :  LOG_ALL;
      tc_proc_handle : LOG_ALL; tc_callback : 0; tc_misc : 0),
    (tc_category : REG_FUNCTIONS;
      tc_func_num : REG_DELETE_VALUE ; tc_flags : NO_FLAGS_SPECIFIED ;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive :  LOG_ALL;
      tc_proc_handle : LOG_ALL; tc_callback : 0; tc_misc : 0),
    (tc_category : REG_FUNCTIONS;
      tc_func_num : REG_CHANGE_VALUE  ; tc_flags : NO_FLAGS_SPECIFIED ;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive :  LOG_ALL;
      tc_proc_handle : LOG_ALL; tc_callback : 0; tc_misc : 0),
    (tc_category : REG_FUNCTIONS;
      tc_func_num : REG_CREATE_VALUE  ; tc_flags : NO_FLAGS_SPECIFIED ;
      tc_flags_mask : NO_MASK_SPECIFIED; tc_drive :  LOG_ALL;
      tc_proc_handle : LOG_ALL; tc_callback : 0; tc_misc : 0));


function DeviceIoControlB(hDevice: THandle; dwIoControlCode: DWORD; lpInBuffer: Pointer;
  nInBufferSize: DWORD; lpOutBuffer: Pointer; nOutBufferSize: DWORD;
  lpBytesReturned: PDWORD; lpOverlapped: POverlapped): BOOL; stdcall;
  external 'kernel32.dll' name 'DeviceIoControl';

procedure release_rec(VAR rec_address : DWORD);
begin
  DeviceIoControlB(vxd_handle, VXD_RELEASE_REC,
    @rec_address, 4, nil, 0, nil, nil);
end;

procedure format_log_record(trap_rec_ptr : Ptrap_record);
VAR
  S : String;
  rt_ptr : PReg_trap_record;
begin
  EnterCriticalSection(critsec);
  S := '';
  rt_ptr := PReg_trap_record(trap_rec_ptr);
  IF trap_rec_ptr^.tr_category = IFSMGR_FUNCTIONS THEN
    WITH trap_rec_ptr^ DO
      begin
        S := Format('File: %s'#13#10, [tr_file1]);
        S := Format('%s ir_flags: %02x  io_options: %04x',
          [S, Byte(tr_ir_flags), tr_ir_options]);
        S := Format('%s reenter: %08x  func: %04x'#13#10,
          [S, tr_res_type, tr_function]);
        S := Format('%s drive: %04x  status: %04x  File accessed by: %s'#13#10,
          [S, tr_drive, tr_error, tr_program]);
        CASE tr_function OF
          IFSFN_OPEN   : S := S + 'function: OPEN';
          IFSFN_READ   : S := S + 'function: READ';
          IFSFN_CLOSE  : S := S + 'function: CLOSE';
          IFSFN_DELETE : S := S + 'function: DELETE';
          IFSFN_DIR    : S := S + 'function: DIR';
        end;
        S := Format('%s Handle: %04x'#13#10, [S, tr_handle]);
        IF (tr_function = IFSFN_OPEN) AND (tr_create_flag = 0) THEN
          S := Format('%s file existed, former size was: %4u'#13#10,
            [S, tr_low_size]);
      end
  ELSE IF trap_rec_ptr^.tr_category = REG_FUNCTIONS THEN
    WITH rt_ptr^ DO
      begin
        S := 'Registry functions: ';
        CASE rtr_function OF
          REG_CREATE_KEY   : S := S + 'Create key';
          REG_DELETE_KEY   : S := S + 'Delete key';
          REG_CREATE_VALUE : S := S + 'Create value';
          REG_CHANGE_VALUE : S := S + 'Change value';
          REG_DELETE_VALUE : S := S + 'Delete value';
          ELSE S := S + 'Unknown function';
        END;
        S := Format(#13#10'%s Key = %s'#13#10, [S, PChar(rtr_lp_key_name)]);
        IF rtr_lp_subkey_name <> 0 THEN
          S := Format('%s SubKey = %s'#13#10, [S, PChar(rtr_lp_subkey_name)]);
        IF rtr_lp_value_name <> 0 THEN
          S := Format('%s Value name = %s'#13#10, [S, PChar(rtr_lp_value_name)]);
        IF (rtr_function = REG_CHANGE_VALUE) OR (rtr_function = REG_CREATE_VALUE) THEN
          begin
            // TODO: handle non-string types
            case rtr_value_type OF
              REG_SZ : S := Format('%s Value = %s'#13#10, [S, PChar(rtr_lp_value_data)]);
              ELSE S := S + 'Value = (not this type yet)'#13#10;
            end;
          end;
        IF rtr_lp_prev_value_data <> 0 THEN
          begin
            // TODO: handle non-string types
            case rtr_value_type OF
              REG_SZ : S := Format('%s Previous Value = %s'#13#10,
                [S, PChar(rtr_lp_prev_value_data)]);
              ELSE S := S + 'Previous Value = (not this type yet)'#13#10;
            end;
          end;
        S := Format('%s By program %s'#13#10, [S, rtr_program]);
      end;
  rep_buf_ptr := rep_buf_ptr + S + #13#10;
  LeaveCriticalSection(critsec);
end;

procedure our_callback(d : DWORD); stdcall;
VAR trap_rec_ptr : Ptrap_record;
begin
  trap_rec_ptr := PTrap_record(d);
  format_log_record(trap_rec_ptr);
  release_rec(d);
end;


//**************************
//  API-style threads - begin

VAR
  stop_flag : THandle;
  thread_handle : THandle;
  thread_id : DWORD;

function do_getcalls : DWORD; export; stdcall; // thread funktion
VAR N : Integer;
begin
  FOR N := 0 TO NumConds-1 DO
    DeviceIoControlB(vxd_handle, VXD_REGISTER_CB, @tcs[N],
      sizeof(trap_criteria), @chs[N], sizeof(DWORD), nil, nil);
  while WaitForSingleObjectEx(stop_flag, INFINITE, TRUE) = WAIT_IO_COMPLETION do;
  result := 0;
end;

function StartWatching : Integer;
VAR N : Integer;
begin
  rep_buf_ptr := '';
  vxd_handle := INVALID_HANDLE_VALUE;
  vxd_handle := CreateFile( VXD_NAME, 0, 0, nil,
    0, FILE_FLAG_DELETE_ON_CLOSE, 0);
  IF vxd_handle = INVALID_HANDLE_VALUE THEN result := sw_NoOpen
  else
    begin
      FillChar(the_version, SizeOf(the_version), 0);
      IF NOT DeviceIoControlB(vxd_handle, VXD_GET_VER,
        nil, 0, @the_version, 2, nil, nil) THEN result := sw_NoVer
      else
        begin
          FOR N := 0 TO NumConds-1 DO
            tcs[N].tc_callback := DWORD(@our_callback);
          InitializeCriticalSection(critsec);
          stop_flag := CreateEvent(nil, false, false, nil);
          ResetEvent(stop_flag);
          thread_handle := CreateThread(nil, 4096, @do_getcalls, nil, 0, thread_id);
          result := sw_OK;
        end;
    end;
end;

function StopWatching : String;
VAR N : Integer;
begin
  FOR N := 0 TO NumConds-1 DO
    begin
      trap_update.trap_rec_num := chs[N];
      trap_update.trap_dat_ptr := Integer(@tcs[N]);
      trap_update.trap_dat_len := sizeof(trap_criteria);
      tcs[N].tc_misc := (APP_EXITING OR DISABLE_LOGGING);
      IF NOT DeviceIoControlB(vxd_handle,VXD_UPDATE, @trap_update,
        sizeof(upcall), nil, 0, nil, nil ) THEN
        rep_buf_ptr := Format('%s%d (%x) %s'#13#10, [rep_buf_ptr, N,
          GetLastError, SysErrorMessage(GetLastError)]);
    end;
  CloseHandle(vxd_handle);
  SetEvent(stop_flag);
  CloseHandle(thread_handle);
  DeleteCriticalSection(critsec);
  result := rep_buf_ptr;
end;

end.
