//tabs=4
//
//		==========
//		LOGCYCLE.C
//		==========
//
// Utility to signal Windows httpd to cycle its logfile(s). See comments
// at SigHTTPD() below. I also produced a DLL that does this for use with
// Visual Basic, etc.
//
// Author:	Robert B. Denny
//			<rdenny@netcom.com>
//			05-Jun-94
//
// History:
//
// Who               When        What
// ----------------- ---------   ---------------------------------------
// Robert B. Denny   05-Jun-94   Created - First release
//
//----------------------------------------------------------------------

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "getopt.h"

#define VERSION "V1.0/06-Jun-94"

extern HINSTANCE _hInstance;		// BC4 has this in library

HWND hTheWindow;				// Used by WindCatcher() during enumeration

static BOOL SigHTTPD(int mask);
static void usage(void);
BOOL CALLBACK WindCatcher(HWND hWnd, LPARAM lparam);

// ====
// MAIN
// ====
main(int argc, char *argv[])
{
	int c;
	WORD mask = 0;

	while((c = getopts(argc, argv, "aeh")) != -1)
	{
		switch(tolower(c))
		{
		  case 'a':							// Cycle access log
			mask |= 1;
			break;

		  case 'e':
			mask |= 2;
			break;

		  case '?':
		  case 'h':
			usage();
		}
	}

	if(mask == 0)							// Default to cycle access log only
		mask = 1;

	if(!SigHTTPD(mask))
	{
		printf("It doesn't appear that the Web server 'httpd' is running.\n");
		usage();
	}

	return(0);
}

//--------------------------------------------------------------------------
//
// SigHTTPD() - Signal Windows httpd to cycle log(s).
//
// Enumerates windows looking for one whose name starts with 'httpd'. Once
// found, it posts a (WM_USER + 13) message to that window. The wParam of
// the message tells which logfiles to cycle (mask):
//
//	wParam = 1		Cycle access log
//	wParam = 2		Cycle Error log
//	wParam = 3		Cycle both logs
//
// Returns non-zero (-1, true) if successful, else 0 (false)
//--------------------------------------------------------------------------
static BOOL SigHTTPD(int mask)
{
	FARPROC lpfnEnumWinCB;			// Thunk for enum callback

	hTheWindow = NULL;				// Assume failure

	lpfnEnumWinCB = MakeProcInstance((FARPROC)WindCatcher, _hInstance);
	EnumWindows(lpfnEnumWinCB, NULL);
	FreeProcInstance(lpfnEnumWinCB);

	if(hTheWindow != NULL)			// If we found an httpd running
	{
		//
		// NOTE: This is hard-coded for httpd V1.2b8.
		//
		return(PostMessage(hTheWindow, (WM_USER + 13), (WPARAM)mask, 0));
	}
	return(0);
}


//------------------------------------------------------------------------
//
// WindCatcher() - Callback function for EnumTaskWindows(). Passes
//                 back a copy of the next window handle.
//
//------------------------------------------------------------------------
#define BUF_LEN 64
#pragma argsused
BOOL CALLBACK WindCatcher (HWND hWnd, LPARAM lparam)
{
	char buf[BUF_LEN+1];
	static const char *tgtName = "httpd";
	register int i;

	GetWindowText(hWnd, (LPSTR)buf, BUF_LEN);	// Get name
	for(i=0; i < 5; i++)						// Avoid huge C library
		if(buf[i] != tgtName[i])
			break;
	if(i == 5)
	{											// Found it
		hTheWindow = hWnd;
		return(FALSE);
	}
	return(TRUE);								// Keep going
}

//------------------------------------------------------------------------
//
// Usage() - Display usage information then exit.
//
//------------------------------------------------------------------------
static void usage(void)
{
	printf("\nlogcycle %s\nusage:\n", VERSION);
	printf("    logcycle [-a] [-e] [-h]\n\n");
	printf("-a  Cycle the access log\n");
	printf("-e  Cycle the error log\n");
	printf("-h  Show this information (also '-?')\n\n");
	printf("If no options given, cycles only the access log.\n");
	printf("NOTE: The server must be running for this to work.\n\n");
	printf("Active log(s) renamed as \".001\", and previous old log(s) renamed\n");
	printf("from \".001\" to \".002\" and so forth up to 30 cycles. The\n");
	printf("oldest is deleted.\n");

	exit(-1);
}


