{**************************************************************************}
{                                                                          }
{    Calmira shell for Microsoft Windows(TM) 3.1                          }
{    Source Release 2.1                                                    }
{    Copyright (C) 1997-1998 Li-Hsin Huang                                 }
{                                                                          }
{    This program is free software; you can redistribute it and/or modify  }
{    it under the terms of the GNU General Public License as published by  }
{    the Free Software Foundation; either version 2 of the License, or     }
{    (at your option) any later version.                                   }
{                                                                          }
{    This program is distributed in the hope that it will be useful,       }
{    but WITHOUT ANY WARRANTY; without even the implied warranty of        }
{    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         }
{    GNU General Public License for more details.                          }
{                                                                          }
{    You should have received a copy of the GNU General Public License     }
{    along with this program; if not, write to the Free Software           }
{    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             }
{                                                                          }
{**************************************************************************}

unit Task;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Buttons, ExtCtrls, Stylsped, Menus, CalMsgs, Hooks, StdCtrls, Profile,
  Referenc;

type
  TWindowType = (wtGeneral, wtIconWindow, wtExplorer);

  TTaskButton = class(TStyleSpeed)
  private
    FWindow : HWnd;
    FTask   : THandle;
    FWindowType : TWindowType;
    FWinControl : TWinControl;
    procedure SetWindow(value : HWND);
  public
    constructor Create(AOwner : TComponent); override;
    procedure RefreshCaption;
    procedure AssignGlyph;
    function MinimizeCaption(s : string): string;
    property Window : HWND read FWindow write SetWindow;
    property Task : THandle read FTask;
    property WindowType : TWindowType read FWindowType;
    property WinControl : TWinControl read FWinControl write FWinControl;
  end;


  TButtonList = class(TList)
  private
    function GetButtons(i: Integer): TTaskButton;
  public
    property Buttons[i: Integer]: TTaskButton read GetButtons;
  end;

  TApplet = class(TGraphicControl)
  private
    FPressed : Boolean;
    procedure SetPressed(value: boolean);
  protected
    FGlyph : TBitmap;
    procedure Paint; override;
    property Pressed : Boolean read FPressed write SetPressed;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  end;

  TTrayProgram = class(TApplet)
  private
    FModuleFile : TFilename;
    FCommand : string;
    procedure HideAppIcon;
  public
    procedure SetProgram(const command: string);
    procedure Click; override;
    procedure CheckModule;
  end;

  TTrayAlias = class(TApplet)
  private
    FRef : TReference;
  public
    constructor Create(AOwner : TComponent; filename: TFilename);
    destructor Destroy; override;
    procedure Click; override;
  end;


  TTaskbar = class(TForm)
    TaskMenu: TPopupMenu;
    Restore: TMenuItem;
    Minimize: TMenuItem;
    Maximize: TMenuItem;
    CloseItem: TMenuItem;
    StartButton: TStyleSpeed;
    SysMenu: TPopupMenu;
    Timer: TTimer;
    Clock: TPanel;
    Stay: TMenuItem;
    HideItem: TMenuItem;
    HintTimer: TTimer;
    Spy: TMenuItem;
    N2: TMenuItem;
    TaskbarProperties: TMenuItem;
    StartProperties: TMenuItem;
    N1: TMenuItem;
    Terminate: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure FormPaint(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure RestoreClick(Sender: TObject);
    procedure MinimizeClick(Sender: TObject);
    procedure MaximizeClick(Sender: TObject);
    procedure CloseItemClick(Sender: TObject);
    procedure TaskMenuPopup(Sender: TObject);
    procedure TerminateClick(Sender: TObject);
    procedure StartButtonMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure QuitClick(Sender: TObject);
    procedure SysMenuPopup(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure TimerTimer(Sender: TObject);
    procedure ClockMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ClockMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ClockMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure StayClick(Sender: TObject);
    procedure HideItemClick(Sender: TObject);
    procedure HintTimerTimer(Sender: TObject);
    procedure SpyClick(Sender: TObject);
    procedure StartPropertiesClick(Sender: TObject);
    procedure TaskbarPropertiesClick(Sender: TObject);
    procedure ClockDblClick(Sender: TObject);
    procedure StartButtonClick(Sender: TObject);
    procedure FormDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure StartButtonMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
  private
    { Private declarations }
    Excludes      : TStringList;
    HintWindow    : THintWindow;
    HintControl   : TControl;
    Pressed       : Integer;
    InTaskClick   : Boolean;
    HiddenList    : TList;
    procedure TaskClick(Sender : TObject);
    procedure TaskButtonMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure WMMouseActivate(var Msg : TWMMouseActivate); message WM_MOUSEACTIVATE;
    procedure WMEnable(var Msg : TWMEnable); message WM_ENABLE;
    procedure WMDropFiles(var Msg : TWMDropFiles); message WM_DROPFILES;
    procedure WMSysCommand(var Msg : TWMSysCommand); message WM_SYSCOMMAND;
    procedure ShellWndCreate(var Msg : TMessage); message WM_SHELLWNDCREATE;
    procedure ShellWndDestroy(var Msg : TMessage); message WM_SHELLWNDDESTROY;
    procedure WMMouseHook(var Msg : TMessage); message WM_MOUSEHOOK;
    procedure WMHideQuery(var Msg : TMessage); message WM_HIDEQUERY;
    procedure WMWinActivate(var Msg : TMessage); message WM_WINACTIVE;
    procedure WMAddButton(var Msg : TMessage); message WM_ADDBUTTON;
    {procedure CMShowingChanged(var Message: TMessage); message CM_SHOWINGCHANGED;}
    function TaskToButton(task: THandle): Integer;
    function WndToButton(Wnd : HWnd): Integer;
    function ShouldExclude(Wnd : HWND): Boolean;
    procedure ShowMinimized(Wnd : HWND);
    procedure SetMouseMonitor;
    procedure TaskButtonDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure TaskButtonDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure UpdateStartButtonState;
  protected
    procedure CreateParams(var Params : TCreateParams); override;
  public
    { Public declarations }
    ButtonList    : TButtonList;
    BarShowing    : Boolean;
    procedure ShowBar;
    procedure HideBar;
    procedure Press(Wnd: HWND);
    procedure RefreshCaptions;
    procedure RefreshButtons;
    procedure ArrangeButtons;
    procedure UpdateButtons;
    procedure UpdateApplets;
    procedure AddButton(Wnd : HWND);
    procedure DeleteButton(Wnd : HWND);
    procedure Configure;
    procedure ActivateHint(p: TPoint);
    procedure CancelHint;
    procedure SetClock(const s : string);
    procedure StartKeyPopup;
    procedure MinimizeAll;
  end;

var
  Taskbar: TTaskbar;

implementation

uses ShellAPI, ToolHelp, Strings, Settings, Files, Start, Desk, Compsys,
  MiscUtil, IconWin, Tree, Resource, MultiGrd, FileFind, Environs, Streamer;

{$R *.DFM}

var
  YLimit : Integer;
  UseMouseHook  : Boolean;
  ConciseDT     : string[127];
  FullDT        : string[127];
  ExplorerBmp   : TBitmap;
  FolderBmp     : TBitmap;


procedure RaiseWindow(Wnd: HWnd);
var p: TPoint;
begin
  { Shifts a minimized window up a little }
  p := GetMinPosition(Wnd);
  if (p.y > YLimit - MinAppHeight) and (p.y < Screen.Height) then begin
    p.y := YLimit - MinAppHeight;
    MoveDesktopIcon(Wnd, p);
  end;
end;


function TButtonList.GetButtons(i: Integer): TTaskButton;
begin
  Result := TTaskButton(Items[i]);
end;


procedure GetModuleAndClass(Wnd: HWND; var f, c: OpenString);
begin
  { Fills two strings with the module and class names of a window }
  f[0] := Chr(GetModuleFilename(GetWindowWord(Wnd, GWW_HINSTANCE), @f[1], High(f)-1));
  c[0] := Chr(GetClassName(Wnd, @c[1], High(c)-1));
end;



function IsTaskWindow(Wnd: HWND): Boolean;
var
  Style: Longint;
begin
  { Returns true if the window qualifies as a "task" }

  Style  := GetWindowLong(Wnd, GWL_STYLE);
  Result := (GetWindowWord(Wnd, GWW_HWNDPARENT) = 0) and
             Bool(GetWindowTextLength(Wnd)) and
             ((Style and WS_MINIMIZEBOX <> 0) or
              (Style and WS_MAXIMIZEBOX <> 0) or
              (Style and WS_THICKFRAME <> 0) or
              (Style and WS_SYSMENU <> 0));
end;


function IsVisibleTaskWindow(Wnd: HWND): Boolean;
begin
  Result := IsTaskWindow(Wnd) and IsWindowVisible(Wnd);
end;


function IsHiddenTaskWindow(Wnd: HWND): Boolean;
begin
  Result := IsTaskWindow(Wnd) and not IsWindowVisible(Wnd);
end;


function EnumWinProc(Wnd: HWnd; Taskbar: TTaskbar): Bool; export;
begin
  { Adds all visible task windows to the bar }
  if IsVisibleTaskWindow(Wnd) {and (GetWindowTask(Wnd) <> GetCurrentTask)} then begin
    Taskbar.Perform(WM_SHELLWNDCREATE, Wnd, 0);
    if IsIconic(Wnd) then Taskbar.Perform(WM_HIDEQUERY, Wnd, 0);
  end;
  Result := True;
end;


{ TTaskButton }

constructor TTaskButton.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  Style := sbWin95;
  Margin := 2;
  Spacing := 1;
  GroupIndex := 1;
  AllowAllUp := True;
end;


procedure TTaskButton.SetWindow(value : HWND);
begin
  FWindow := value;
  FTask := GetWindowTask(FWindow);
  FWinControl := FindControl(FWindow);

  if FWinControl is TIconWindow then FWindowType := wtIconWindow
  else if FWinControl is TExplorer then FWindowType := wtExplorer
  else FWindowType := wtGeneral;

  AssignGlyph;
  RefreshCaption;
end;

procedure ChooseBitmap(Dest, Source: TBitmap; Res: PChar);
begin
  if Source.Empty then Dest.Handle := LoadBitmap(HInstance, Res)
  else Dest.Assign(Source);
end;


procedure TTaskButton.AssignGlyph;
var
  m, c : string[127];
  h : HIcon;
begin
  if (IconWindowTask or ExplorerTask) and (FWindowType <> wtGeneral) then
    case FWindowType of
      wtIconWindow : ChooseBitmap(Glyph, FolderBmp, 'FOLDERBMP');
      wtExplorer   : ChooseBitmap(Glyph, ExplorerBmp, 'EXPLORERBMP');
    end

  else begin
    { Ask Calmira to provide an icon }
    Application.ProcessMessages;
    h := ProvideLastIcon(GetWindowWord(Window, GWW_HINSTANCE));

    if h > 1 then begin
      ShrinkIcon(h, Glyph);
      DestroyIcon(h);
    end;
  end;

  if Glyph.Empty then begin
    GetModuleAndClass(Window, m, c);
    h := ExtractIcon(HInstance, StringAsPChar(m), 0);
    if h > 0 then begin
      ShrinkIcon(h, Glyph);
      DestroyIcon(h);
    end;
  end;
end;


function TTaskButton.MinimizeCaption(s : string): string;

var i, j   : Integer;    { counters }
    target : Integer;    { maximum width of text that can fit }
    dw     : Integer;    { width of three dots }
    tw     : Integer;    { current text width }
    app, doc : string[79];
begin
  { Given a string and a button width, truncate it so that it fits
    comfortably on the button.  First check if it fits.  If it doesn't,
    keep chopping the end off until it does and append three dots to it.

    To avoid calling Canvas.TextWidth too many times, the string
    is cut in half if the width is over twice the desired width

    Bizzare bug: change Taskbar.Canvas to just Canvas and something very
    strange happens...because MinimizeCaption is called before
    the button is added to the form? }

  if DocNameFirst then begin
    i := Pos(' - ', s);
    if i > 0 then begin
      app := Copy(s, 1, i-1);
      doc := Copy(s, i+3, 255);
      if DocNameLower then doc := Lowercase(doc);
      s := Format('%s - %s', [doc, app]);
    end;
  end;

  tw := Taskbar.Canvas.TextWidth(s);
  dw := Taskbar.Canvas.TextWidth('...');
  target := Width - 6;
  if not Glyph.Empty then Dec(target, 16);

  if (tw > target) then begin
    Dec(target, dw);

    if target < dw then begin
      Result := '';
      exit;
    end;

    repeat
      if (tw > target * 2) and (s[0] > #1)  then Dec(s[0], ord(s[0]) div 2)
      else Dec(s[0]);
      tw := Taskbar.Canvas.TextWidth(s);
    until ((tw <= Target) or (Length(s) = 1));
    if Length(s) <= 1 then s := ''
    else AppendStr(s, '...');
  end;

  Result := s;
end;


procedure TTaskButton.RefreshCaption;
var
  s: string[127];
begin
  s[0] := Chr(GetWindowText(Window, @s[1], 126));

  if (FWindowType = wtIconWindow) then begin
    Hint := TIconWindow(WinControl).Dir.Fullname;
    if not FullFolderPath and (Length(s) > 3) and (s[2] = ':') and (s[3] = '\') then
      s := ExtractFilename(s);
  end
  else Hint := s;

  Caption := MinimizeCaption(s);
end;


{ routine for finding a window belonging to a module -- the module handle,
  not instance handle, is given so GetWindowWord can't be used }

var FoundWindow : HWND;

function WinModuleProc(Wnd: HWnd; Filename: PChar): Bool; export;
var
  buf : array[0..127] of char;
begin
  if IsTaskWindow(Wnd) then begin
    GetModuleFilename(GetWindowWord(Wnd, GWW_HINSTANCE), buf, 127);
    if StrComp(Filename, buf) = 0 then begin
      FoundWindow := Wnd;
      Result := False;
      Exit;
    end;
  end;
  FoundWindow := 0;
  Result := True;
end;


{ TApplet }

constructor TApplet.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FGlyph := TBitmap.Create;
  SetBounds(0, 0, 20, 20);
  Align := alLeft;
end;

destructor TApplet.Destroy;
begin
  FGlyph.Free;
  inherited Destroy;
end;

procedure TApplet.Paint;
var R: TRect;
begin
  R := ClientRect;
  InflateRect(R, -1, -1);
  if FPressed then Frame3D(Canvas, R, clBtnShadow, clBtnHighlight, 1);
  Canvas.Draw((Width - FGlyph.Width) div 2, (Height - FGlyph.Height) div 2, FGlyph);
end;

procedure TApplet.SetPressed(value: Boolean);
begin
  if FPressed <> value then begin
    FPressed := value;
    Invalidate;
  end;
end;


{ TTrayProgram }


procedure TTrayProgram.SetProgram(const command: string);
var
  h : HIcon;
  p : Integer;
begin
  FCommand := command;
  FModuleFile := Uppercase(command);
  p := Pos(' ', FModuleFile);
  if p > 1 then FModuleFile[0] := Chr(p-1);

  h := ExtractIcon(HInstance, StringAsPChar(FModuleFile), 0);
  if h > 0 then
    try
      ShrinkIcon(h, FGlyph);
    finally
      DestroyIcon(h);
    end;

  HideAppIcon;
end;


procedure TTrayProgram.HideAppIcon;
begin
  EnumWindows(@WinModuleProc, Longint(@FModuleFile[1]));
  if FoundWindow > 0 then MoveDesktopIcon(FoundWindow, Point(0, Screen.Height));
end;


procedure TTrayProgram.Click;
begin
  if GetModuleHandle(@FModuleFile[1]) > 0 then begin
    { Re-activate the utility }
    EnumWindows(@WinModuleProc, Longint(@FModuleFile[1]));
    if FoundWindow > 0 then
      if IsIconic(FoundWindow) then ShowWindow(FoundWindow, SW_RESTORE)
      else BringWindowToTop(FoundWindow)
  end
  else begin
    { run a new instance }
    WinExec(StringAsPChar(FCommand), SW_SHOW);
    HideAppIcon;
    Pressed := True;
  end;
end;

procedure TTrayProgram.CheckModule;
begin
  Pressed := GetModuleUsage(GetModuleHandle(@FModuleFile[1])) > 0;
end;


function LoadBitmapExtern(filename: TFilename): TBitmap;
begin
  Result := TBitmap.Create;
  if FileExists(filename) then Result.LoadFromFile(filename);
end;


constructor TTrayAlias.Create(AOwner : TComponent; filename : TFilename);
var
  s: TStreamer;
  Icon : TIcon;
begin
  inherited Create(AOwner);

  s := TStreamer.Create(filename, fmOpenRead);
  s.ReadString;
  FRef := TReference.Create;
  FRef.LoadFromStream(s);
  s.Free;

  Icon := TIcon.Create;
  FRef.AssignIcon(Icon);
  ShrinkIcon(Icon.Handle, FGlyph);
  Icon.Free;
  Hint := FRef.Caption;
end;

destructor TTrayAlias.Destroy;
begin
  FRef.Free;
  inherited Destroy;
end;

procedure TTrayAlias.Click;
begin
  FRef.Open;
end;

{ Main taskbar }


procedure TTaskbar.FormCreate(Sender: TObject);
var
  i: Integer;
  Wnd : HWND;
  buf : TFilename;
begin
  Pressed := -1;
  SetCallBackWnd(Handle);

  HintWindow := THintWindow.Create(Application);
  HintWindow.Visible := False;

  if Screen.PixelsPerInch > 96 then
    StartButton.Width := StartButton.Width + 6;

  Desktop.SetCursor(crHourGlass);
  try
    ExplorerBmp := LoadBitmapExtern(ApplicationPath + 'TASKEXP.BMP');
    FolderBmp := LoadBitmapExtern(ApplicationPath + 'TASKFOLD.BMP');

    Setbounds(0, Screen.Height -1, Screen.Width, Height);
    ButtonList := TButtonList.Create;
    HiddenList := TList.Create;

    Configure;

    StartButton.OnDragOver := Computer.FormDragOver;
    StartButton.OnDragDrop := Computer.FormDragDrop;

    YLimit := Screen.Height - ClientHeight;
    SetYLimit(YLimit);

    StartTaskMonitor;
    if UseMouseHook then StartMouseMonitor;
    SetWndHook;

    if DisableTaskbar then Exit;

    if Stay.Checked then ShowBar else HideBar;

    EnumWindows(@EnumWinProc, Longint(self));
  finally
    Desktop.ReleaseCursor;
    DragAcceptFiles(Handle, True);
  end;
end;


procedure TTaskbar.WMMouseHook(var Msg : TMessage);
begin
  { Called by the DLL when the cursor leaves the taskbar }
  if not Stay.Checked and (GetCapture = 0) and BarShowing then HideBar
  else if ButtonHints and HintWindow.Visible then CancelHint;
end;


procedure TTaskbar.FormMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
begin
  if not BarShowing then ShowBar;
  CancelHint;
end;


procedure TTaskbar.HideBar;
var i : Integer;
begin
  { Suspends the taskbar until it is re-activated by the mouse }
  CancelHint;
  Timer.Enabled := False;
  BarShowing := False;
  DisableMouseMonitor;
  Top := Screen.Height - 1;
  if Animate then for i := 0 to ControlCount-1 do Controls[i].Hide;
end;


function TTaskbar.TaskToButton(task: THandle): Integer;
begin
  { Returns the button index for a given task handle, -1 if the
    task is not shown on the bar }

  with ButtonList do
    for Result := 0 to Count-1 do
      if task = Buttons[Result].Task then Exit;
  Result := -1;
end;


function TTaskbar.WndToButton(Wnd : HWnd): Integer;
begin
  { Returns the button index for a given window handle, -1 if the
  task is not shown on the bar }

  with ButtonList do
    for Result := 0 to Count-1 do
      if Wnd = Buttons[Result].Window then Exit;
  Result := -1;
end;


procedure TTaskbar.Press(Wnd: HWND);
var
  i: Integer;
begin
  { Called when a window receives a WM_ACTIVATE message.  If there is
    a button for that window or the task it belongs to, then that
    button is pressed }
  if IsIconic(Wnd) then Exit;

  i := WndToButton(Wnd);
  if i = -1 then i := TaskToButton(GetWindowTask(Wnd));

  with ButtonList do
    if i > -1 then
      Buttons[i].Down := True
    else if (Pressed > -1) and (Pressed < Count) then
      Buttons[Pressed].Down := False;

  Pressed := i;
end;


procedure TTaskbar.UpdateButtons;
begin
  RefreshButtons;
  ArrangeButtons;
  Press(GetActiveWindow);
end;


procedure TTaskbar.ShowBar;
var
  i : Integer;
  Wnd : HWND;
begin
  if DisableTaskbar then Exit;
  Timer.Enabled := True;
  SetClock(FormatDateTime(ConciseDT, Now));
  UpdateButtons;

  { Move the form up 5 pixels at a time and then show the buttons }

  if Animate then begin
    i := Screen.Height - 1;
    while i >= Screen.Height - ClientHeight + 2 do begin
      Top := i;
      Dec(i, 2);
    end;
    Top := Screen.Height - ClientHeight;
  end;

  if not StartButton.Visible then
    for i := 0 to ControlCount-1 do Controls[i].Show;

  Top := Screen.Height - ClientHeight;
  BarShowing := True;
  SetMouseMonitor;
end;


procedure TTaskbar.FormPaint(Sender: TObject);
begin
  with Canvas do begin
    if BarShowing then begin
      { Paint the 3D effect around the edges }
      Pen.Color := clBtnHighLight;
      MoveTo(0, ClientHeight-1);
      LineTo(0, 1);
      LineTo(ClientWidth-1, 1);
      Pen.Color := clBtnShadow;
      LineTo(ClientWidth-1, ClientHeight-1);
    end;

    { Draw a black line across the top }
    Pen.Color := clBlack;
    MoveTo(0, 0);
    LineTo(ClientWidth, 0);
  end;
end;


procedure TTaskbar.ArrangeButtons;
var i, t, h, w, x, avail: Integer;
begin
  { w is the width of a button plus the gap to its right}

  avail := ClientWidth - StartButton.Width - Clock.Width - 8;

  case ButtonList.Count of
    0: Exit;
    1..2: w := avail div 3;
  else
    w := avail div ButtonList.Count;
  end;
  if w > 256 then w := 256;

  { x is initialised to the left side of the first button }

  x := StartButton.Left + StartButton.Width + 3;
  t := StartButton.Top;
  h := StartButton.Height;

  with ButtonList do
    for i := 0 to Count-1 do begin
      Buttons[i].SetBounds(x, t, w - 3, h);
      Inc(x, w);
    end;

  RefreshCaptions;
end;



procedure TTaskbar.RefreshCaptions;
var
  i: Integer;
begin
  with ButtonList do
    for i := 0 to Count-1 do Buttons[i].RefreshCaption;
end;


procedure TTaskbar.RefreshButtons;
var
  i, j: Integer;
  Wnd : HWND;
  FoundDupe : Boolean;
begin
  { remove any windows that no longer exist or have disappeared }

  i := 0;
  with ButtonList do
  for i := Count-1 downto 0 do begin
    Wnd := Buttons[i].Window;

    FoundDupe := False;
    j := i-1;
    while (j >= 0) and not FoundDupe do begin
      FoundDupe := Buttons[j].Window = Wnd;
      Dec(j);
    end;

    if FoundDupe or ((Buttons[i].WindowType = wtGeneral) and (not IsWindow(Wnd) or
      not IsWindowVisible(Wnd) or (GetWindowTextLength(Wnd) = 0))) then begin
      Buttons[i].Free;
      Delete(i);
    end;
  end;
end;


procedure TTaskbar.AddButton(Wnd : HWND);
var
  button : TTaskButton;
begin
  button := TTaskButton.Create(self);

  with button do begin
    Left := -64;
    Window := Wnd;
    OnClick := TaskClick;
    OnMouseDown := TaskButtonMouseDown;
    OnMouseMove := ClockMouseMove;
    OnDragOver := TaskButtonDragOver;
    OnDragDrop := TaskButtonDragDrop;
  end;

  if not IsWindow(Wnd) or (WndToButton(Wnd) <> -1) then begin
    button.Free;
    Exit;
  end;

  InsertControl(button);
  button.Down := True;
  ButtonList.Add(button);
  if BarShowing then ArrangeButtons;
end;


procedure TTaskbar.DeleteButton(Wnd : HWND);
var i: Integer;
begin
  { When Wnd is destroyed, look for a button with the matching window
    and remove it, then rearrange the other buttons }

  with ButtonList do
  for i := 0 to Count-1 do
    if Buttons[i].Window = Wnd then begin
      Buttons[i].Free;
      Delete(i);
      ArrangeButtons;
      Exit;
    end;
end;


procedure TTaskbar.TaskClick(Sender : TObject);
var
  wnd : HWND;
  i : Integer;
begin
  { This is the event handler for normal task buttons.

    Disabled child windows are skipped in case they cover up the
    active window (e.g. if an icon window covers up a modal dialog,
    there is no way to end the modal state).

    The SendMessage trick is required to access full screen DOS boxes
    because of a bug (solution provided by Microsoft) }

  Wnd := (Sender as TTaskButton).Window;

  if not IsWindowEnabled(Wnd) and (
    TTaskButton(Sender).WindowType <> wtGeneral) then begin
    MessageBeep(0);
    Exit;
  end;

  InTaskClick := True;
  SendMessage(Wnd, WM_ACTIVATE, WA_ACTIVE, MakeLong(Wnd, Word(True)));
  InTaskClick := False;

  if IsIconic(Wnd) then ShowWindow(Wnd, SW_RESTORE)
  else BringWindowToTop(Wnd);
end;



function TTaskbar.ShouldExclude(Wnd : HWND): Boolean;
var
  fname, cname: string[127];
begin
  { Returns True if Wnd should be excluded from the bar }

  GetModuleAndClass(Wnd, fname, cname);
  fname := ExtractFilename(fname);

  Result := (Excludes.IndexOf(fname) > -1) or
            (Excludes.IndexOf(Format('%s %s', [fname, cname])) > -1);
end;


procedure TTaskbar.ShellWndCreate(var Msg : TMessage);
begin
  { Called by the shell hook when a top-level window is created }

  with msg do
    if not ShouldExclude(wParam) then
      if IsHiddenTaskWindow(wParam) then
        HiddenList.Add(Pointer(wParam))
      else if IsVisibleTaskWindow(wParam) then begin
        AddButton(wParam);
        if IsIconic(wParam) then Perform(WM_HIDEQUERY, wParam, 0);
      end;
end;


procedure TTaskbar.ShellWndDestroy(var Msg : TMessage);
var i: Integer;
begin
  { Called by the shell hook when a top-level window is created }
  i := HiddenList.IndexOf(Pointer(msg.wParam));
  if i > -1 then HiddenList.Delete(i)
  else DeleteButton(msg.wParam);
end;


procedure TTaskbar.FormDestroy(Sender: TObject);
var i: Integer;
begin
  StopMouseMonitor;
  StopTaskMonitor;
  UnhookWndHook;

  { Apps which have had their icon moved off the screen must be restored
    properly.  If Calmira is active, then its ArrangeIcons function is
    called, but the icons must be moved above Screen.Height so that
    Calmira knows that they are not supposed to be hidden }

  for i := 0 to ButtonList.Count-1 do
    MoveDesktopIcon(ButtonList.Buttons[i].Window,
    Point(0, Screen.Height-1));
  Desktop.ArrangeIcons;

  Excludes.Free;
  HiddenList.Free;
  ButtonList.Free;
  FolderBmp.Free;
  ExplorerBmp.Free;
end;


procedure TTaskbar.FormMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
const
  MouseButtons : array[Boolean] of Word = (VK_LBUTTON, VK_RBUTTON);
var
  control : TControl;
  i : Integer;
begin
  { "Terminate" mode distinguished by the cursor being crTerminate }
  
  if Cursor = crTerminate then begin

    if Button = mbLeft then begin
      control := ControlAtPos(Point(X, Y), True);
      if (control is TTaskButton) and (TTaskButton(control).Task <> GetCurrentTask) then
        TerminateApp(TTaskButton(control).Task, NO_UAE_BOX);
    end;

    for i := 0 to ControlCount-1 do Controls[i].Enabled := True;
    Cursor := crDefault;
  end
  else if (Button = mbRight) and
    (GetAsyncKeyState(MouseButtons[Bool(GetSystemMetrics(SM_SWAPBUTTON))]) < 0) then
    Computer.Perform(WM_DESKACTIVATE, 0, 0);
end;


procedure TTaskbar.TaskButtonMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var p: TPoint;
begin
  { To remember which button the right mouse button was pressed over,
    tha Tag is used rather than using the PopupComponent property --
    just in case the button gets deleted before the menu click occurs }
  
  if Button = mbLeft then exit;
  TaskMenu.Tag := (Sender as TTaskButton).Window;
  DisableMouseMonitor;
  GetCursorPos(p);
  TaskMenu.Popup(p.X, p.Y);
  SetMouseMonitor;
end;

procedure TTaskbar.RestoreClick(Sender: TObject);
begin
  ShowWindow(TaskMenu.Tag, SW_RESTORE);
end;

procedure TTaskbar.MinimizeClick(Sender: TObject);
begin
  CloseWindow(TaskMenu.Tag);
end;

procedure TTaskbar.MaximizeClick(Sender: TObject);
begin
  ShowWindow(TaskMenu.Tag, SW_SHOWMAXIMIZED);
end;

procedure TTaskbar.CloseItemClick(Sender: TObject);
begin
  PostMessage(TaskMenu.Tag, WM_CLOSE, 0, 0);
end;


procedure TTaskbar.TaskMenuPopup(Sender: TObject);
var
  Wnd : HWND;
  Zoomed, Iconic, E: Boolean;
  Style : Longint;
begin
  with TaskMenu do begin
    Wnd := Tag;
    Zoomed := IsZoomed(Wnd);
    Iconic := IsIconic(Wnd);
    Style := GetWindowLong(Wnd, GWL_STYLE);

    E := IsWindowEnabled(Wnd);
    Restore.Enabled := E and (Zoomed or Iconic);
    Minimize.Enabled := E and not Iconic and (Style and WS_MINIMIZEBOX <> 0);
    Maximize.Enabled := E and not Zoomed and (Style and WS_MAXIMIZEBOX <> 0);
    CloseItem.Enabled := E;
  end;
end;



procedure TTaskbar.TerminateClick(Sender: TObject);
var i: Integer;
begin
  { Start terminate mode by disabling buttons and setting crTerminate cursor }

  StartButton.Enabled := False;
  with ButtonList do
  for i := 0 to Count-1 do begin
    Buttons[i].Down := False;
    Buttons[i].Enabled := False;
  end;
  Cursor := crTerminate;
  Pressed := -1;
end;




procedure TTaskbar.QuitClick(Sender: TObject);
begin
  Close;
end;


procedure TTaskbar.SysMenuPopup(Sender: TObject);
begin
  Terminate.Enabled := ButtonList.Count > 0;
end;


procedure TTaskbar.FormResize(Sender: TObject);
begin
  Clock.Left := ClientWidth - 3 - Clock.Width;
end;


procedure TTaskbar.UpdateApplets;
var i: Integer;
begin
  with Clock do
    for i := 0 to ControlCount-1 do
      if Controls[i] is TTrayProgram then
        TTrayProgram(Controls[i]).CheckModule;
end;

procedure TTaskbar.TimerTimer(Sender: TObject);
const
  MouseButtons : array[Boolean] of Word = (VK_LBUTTON, VK_RBUTTON);
begin
  if GetAsyncKeyState(MouseButtons[Bool(
    GetSystemMetrics(SM_SWAPBUTTON))]) >= 0 then
   SetClock(FormatDateTime(ConciseDT, Now));

  if BarShowing then begin
    UpdateButtons;
    UpdateApplets;
  end;
end;



procedure TTaskbar.ClockMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  SetClock(IntToStr(GetFreeSpace(0) div 1024) + ' KB');
end;


procedure TTaskbar.ClockMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  SetClock(FormatDateTime(ConciseDT, Now));
end;

procedure TTaskbar.ClockMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
begin
  if (HintControl = Sender) or ((Sender is TTaskButton) and not ButtonHints) or
      ((Sender = Clock) and not (PopupRes or PopupDate)) then Exit;

  HintControl := Sender as TControl;

  if Hintwindow.Visible then
    ActivateHint(HintControl.ClientToScreen(Point(X, Y)))
  else
    HintTimer.Enabled := True;
end;


procedure TTaskbar.ShowMinimized(Wnd : HWND);
begin
  if not IsIconic(Wnd) and
    (GetWindowLong(Wnd, GWL_STYLE) and WS_MINIMIZEBOX <> 0) then begin
    Perform(WM_HIDEQUERY, Wnd, 0);
    ShowWindow(Wnd, SW_SHOWMINIMIZED);
  end;
end;


procedure TTaskbar.WMSysCommand(var Msg : TWMSysCommand);
begin
  if Msg.CmdType = SC_SCREENSAVE then HideBar
  else if Msg.CmdType = SC_CLOSE then Exit;
  inherited;
end;


procedure TTaskbar.WMDropFiles(var Msg : TWMDropFiles);
var
  p: TPoint;
  control : TControl;
  i : Integer;
  Wnd : HWND;
begin
  inherited;
  { Find the target window and check that it accepts files before
  forwarding the message on }

  DragQueryPoint(Msg.Drop, p);
  control := ControlAtPos(p, False);
  if control <> nil then with ButtonList do begin
    i := IndexOf(control);
    if (i > -1) and (Buttons[i].WindowType = wtGeneral) then begin
      Wnd := Buttons[i].Window;
      if GetWindowLong(Wnd, GWL_EXSTYLE) and WS_EX_ACCEPTFILES <> 0 then begin
        PostMessage(Wnd, WM_DROPFILES, Msg.Drop, Msg.Unused);
        Exit;
      end;
    end;
  end;
  { release files after an error }
  DragFinish(Msg.Drop);
  MessageBeep(0);
end;


type TProtectedControl = class(TControl);

procedure TTaskbar.Configure;

procedure AddApplet(applet: TGraphicControl);
begin
  with Clock do begin
    if Alignment <> taRightJustify then Alignment := taRightJustify;
    Left := Left - 20;
    Width := Width + 20;
  end;
  applet.Left := Clock.ControlCount * 20;
  applet.Parent := Clock;
  TProtectedControl(applet).OnMouseMove := ClockMouseMove;
end;


var
  i : Integer;
  TrayApps : TStringList;
  s : string;
  tp : TTrayProgram;
begin
  { reads settings and adjusts controls to reflect the changes }

  Excludes.Free;
  Excludes := TStringList.Create;

  with ini do begin
    ReadStrings('Exclude', Excludes);
    Timer.Interval     := ReadInteger('Taskbar', 'Refresh', 5) * 1000;
    HintTimer.Interval := ReadInteger('Taskbar', 'HintDelay', 800);
    UseMouseHook       := ReadBool('Taskbar', 'UseMouseHook', True);
    Stay.Checked       := StayVisible;

    if Clock24 then
      ConciseDT := ReadString('Taskbar', '24HourFormat', 'h:mm')
    else
      ConciseDT := ReadString('Taskbar', '12HourFormat', 'h:mm AM/PM');

    FullDT := ReadString('Taskbar', 'FullDateTime', 'dddd, mmmm d, yyyy');
    Color := StringToColor(ReadString('Colors', 'Taskbar', 'clSilver'));

    with StartButton do begin
      Caption := ReadString('Start button', 'Caption', 'Start');
      Left := ReadInteger('Start button', 'Left', Left);
      Width := ReadInteger('Start button', 'Width', Width);
      s := ApplicationPath + 'startbtn.bmp';
      if FileExists(s) then Glyph.LoadFromFile(s);
    end;

    ReadFont('Taskbar', Font);
    ReadFont('Start button', StartButton.Font);
  end;

  SetMaxEnabled(Stay.Checked and ShrinkMax);


  { Clear Calmira buttons if they have been turned off, and also
    adjust button states }

  with ButtonList do
    for i := Count-1 downto 0 do with Buttons[i] do
      if (IconWindowTask and (WindowType = wtIconWindow)) or
         (ExplorerTask and (WindowType = wtExplorer)) then begin
        Free;
        ButtonList.Delete(i)
      end else begin
        GroupIndex := 1;
        Down := False;
      end;

  { Clear the Applet Tray }

  with Clock do begin
    i := ControlCount * 20;
    Left := Left + i;
    Width := Width - i;
    while ControlCount > 0 do Controls[0].Free;
  end;
  Clock.Alignment := taCenter;

  TrayApps := TStringList.Create;
  ini.ReadSectionValues('Applet Tray', TrayApps);

  { Load Applet Tray programs }

  for i := 0 to TrayApps.Count-1 do begin
    s := TrayApps[i];
    tp := TTrayProgram.Create(self);
    tp.setProgram(GetStrValue(s));
    tp.Hint := GetStrKey(s);
    AddApplet(tp);
    Excludes.Add(ExtractFilename(GetStrValue(s)));
  end;

  TrayApps.Clear;
  FindFiles(ApplicationPath + 'tray\*' + AliasExtension,
    faAnyFile and not faDirectory, TrayApps);

  for i := 0 to TrayApps.Count-1 do
    AddApplet(TTrayAlias.Create(self, ApplicationPath + 'tray\' + TrayApps[i]));

  TrayApps.Free;

  TimerTimer(self);
end;


procedure TTaskbar.StayClick(Sender: TObject);
begin
  Stay.Checked := not Stay.Checked;
  SetMaxEnabled(Stay.Checked and ShrinkMax);
  SetMouseMonitor;
end;


procedure TTaskbar.HideItemClick(Sender: TObject);
begin
  HideBar;
end;

procedure TTaskbar.SetMouseMonitor;
begin
  if HintWindow.Visible or ((Top < (Screen.Height - 1)) and not Stay.Checked) then
    EnableMouseMonitor
  else
    DisableMouseMonitor;
end;


procedure TTaskbar.CancelHint;
begin
  with HintWindow do begin
    Visible := False;
    if HandleAllocated then ShowWindow(Handle, SW_HIDE);
  end;
  HintControl := nil;
  SetMouseMonitor;
end;


procedure TTaskbar.ActivateHint(P: TPoint);
var
  HintStr: string;
  fname, cname: string[127];
  r : TRect;

procedure AddField(const s: string);
begin
  if HintStr > '' then AppendStr(HintStr, '  ');
  AppendStr(Hintstr, s);
end;

begin
  if HintControl = nil then Exit;
  if HintWindow.HandleAllocated then ShowWindow(HintWindow.Handle, SW_HIDE);

    if HintControl = Clock then begin
      HintStr := '';
      if PopupDate then AddField(FormatDateTime(FullDT, Now));
      if PopupRes then AddField(
        Format('sys %d%%  gdi %d%%  user %d%%',
        [GetFreeSystemResources(GFSR_SYSTEMRESOURCES),
         GetFreeSystemResources(GFSR_GDIRESOURCES),
         GetFreeSystemResources(GFSR_USERRESOURCES)]));
    end
    else if HintControl is TTaskButton then begin
      HintStr := HintControl.Hint;
      if Spy.Checked then begin
        GetModuleAndClass(TTaskButton(HintControl).Window, fname, cname);
        AppendStr(HintStr, Format('   %s(%s)', [ExtractFilename(fname), cname]));
      end;
    end
    else HintStr := HintControl.Hint;

  r.Left := HintControl.ClientToScreen(Point(0, 0)).X;
  r.Bottom := Top - 2;

  with HintWindow do begin
    r.Right := r.Left + Canvas.TextWidth(HintStr) + 6;
    r.Top := r.Bottom - Abs(Canvas.Font.Height) - 4;
    ActivateHint(r, HintStr);
    Visible := True;
  end;
  EnableMouseMonitor;
end;


procedure TTaskbar.HintTimerTimer(Sender: TObject);
var
  P: TPoint;
  Control: TControl;
begin
  GetCursorPos(P);
  Control := FindDragTarget(P, True);
  if Control = HintControl then ActivateHint(P);
  HintTimer.Enabled := False;
end;


procedure TTaskbar.SpyClick(Sender: TObject);
begin
  with Spy do Checked := not Checked;
end;


procedure TTaskbar.WMHideQuery(var Msg : TMessage);
var
  i: Integer;
begin
  if HideMinApps then begin
    i := WndToButton(Msg.wParam);
    if i > -1 then begin
      MoveDesktopIcon(Msg.wParam, Point(0, Screen.Height));
      Exit;
    end;
  end;

  if ArrangeMin then RaiseWindow(Msg.wParam);
end;


procedure TTaskbar.WMWinActivate(var Msg : TMessage);
var i: Integer;
begin
  if not InTaskClick then begin
    i := HiddenList.IndexOf(Pointer(Msg.wParam));
    if (i > -1) and IsVisibleTaskWindow(Msg.wParam) then begin
      if not ShouldExclude(msg.wParam) then
        PostMessage(Handle, WM_ADDBUTTON, Word(HiddenList[i]), 0);
      HiddenList.Delete(i);
    end
    else Press(Msg.WParam);
  end;
end;


procedure TTaskbar.WMMouseActivate(var Msg : TWMMouseActivate);
begin
  Msg.Result := MA_NOACTIVATE;
end;


procedure TTaskbar.WMAddButton(var Msg : TMessage);
begin
  AddButton(Msg.wParam);
  Press(Msg.wParam);
end;


procedure TTaskbar.StartPropertiesClick(Sender: TObject);
begin
  Computer.ConfigStartMenu.Click;
end;


procedure TTaskbar.TaskbarPropertiesClick(Sender: TObject);
begin
  Computer.ConfigTaskbar.Click;
end;


procedure TTaskbar.SetClock(const s : string);
begin
  with Clock do
    if ControlCount > 0 then Caption := s + '  ' else Caption := s;
end;


procedure TTaskbar.CreateParams(var Params : TCreateParams);
begin
  inherited CreateParams(Params);
  Params.WndParent := GetDesktopWindow;
end;

procedure TTaskbar.WMEnable(var Msg : TWMEnable);
begin
  inherited;
  SetWindowLong(Handle, GWL_STYLE,
    GetWindowLong(Handle, GWL_STYLE) and not WS_DISABLED);
end;



procedure TTaskbar.ClockDblClick(Sender: TObject);
var buf: array[0..255] of Char;
begin
  WinExec(StrPCopy(buf,
    EnvironSubst(ini.ReadString('Taskbar', 'AdjustClock', 'control Date/Time'))), SW_SHOW);
end;


procedure TTaskbar.StartKeyPopup;
var temp: Boolean;
begin
  if Top > Screen.Height-3 then ShowBar;
  StartButton.Down := True;
  temp := StartMouseUp;
  StartMouseUp := True;
  StartButton.Click;
  StartMouseUp := temp;
end;


procedure TTaskbar.MinimizeAll;
var i: Integer;
begin
  with ButtonList do
    for i := 0 to Count-1 do
      ShowMinimized(Buttons[i].Window);
end;


procedure TTaskbar.UpdateStartButtonState;
var p: TPoint;
begin
  GetCursorPos(p);
  with StartButton do
    if not (PtInRect(ClientRect, ScreenToClient(p)) {and MousePressed}) then
      Down := False;
end;

procedure TTaskbar.StartButtonClick(Sender: TObject);
var
  p: TPoint;
  Msg: TMessage;
  MousePressed : Boolean;
begin
  if StartMouseUp and StartButton.Down then begin
    DisableMouseMonitor;
    StartMenu.Popup(0, Top - StartMenu.Height, True);
    UpdateStartButtonState;
    GetCursorPos(p);
    if not (Stay.Checked or PtInRect(BoundsRect, p)) then HideBar;
    SetMouseMonitor;
  end;
end;



procedure TTaskbar.StartButtonMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var p: TPoint;
begin
  if Button = mbRight then begin
    DisableMouseMonitor;
    GetCursorPos(p);
    SysMenu.Popup(p.X, p.Y);
    SetMouseMonitor;
  end
  else if not StartMouseUp then begin
    { Restore start button state by simulating a mouse click }
    DisableMouseMonitor;
    StartButton.Down := True;
    Update;
    StartMenu.Popup(0, Top - StartMenu.Height, True);
    PostMessage(Handle, WM_LBUTTONUP, 0,
      MakeLong(StartButton.Left + X, StartButton.Top + Y));
    PostMessage(Handle, WM_LBUTTONUP, 0,
      MakeLong(StartButton.Left + X, StartButton.Top + Y));
  end;
end;

procedure TTaskbar.TaskButtonDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
var
  IconWindow: TIconWindow;
begin
  with Sender as TTaskButton do
    if WindowType = wtIconWindow then begin
      IconWindow := WinControl as TIconWindow;
      IconWindow.FormDragOver(IconWindow, Source, X, Y, State, Accept);
    end
    else if (WindowType = wtGeneral) and ((Source = FindList) or
      ((Source is TMultiGrid) and (Source <> Computer.Grid))) then

      Accept := GetWindowLong(Window, GWL_EXSTYLE) and WS_EX_ACCEPTFILES <> 0;
end;

procedure TTaskbar.TaskButtonDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
  IconWindow: TIconWindow;
begin
  with Sender as TTaskButton do
    if WindowType = wtIconWindow then begin
      IconWindow := WinControl as TIconWindow;
      IconWindow.FormDragDrop(IconWindow, Source, X, Y);
    end
    else if WindowType = wtGeneral then
      if (Source is TMultiGrid) and (Source <> Computer.Grid) then
        (TMultiGrid(Source).Owner as TIconWindow).DropServer.DropFiles(Window, Point(1,1))
      else if Source = FindList then
        FindForm.DropServer.DropFiles(Window, Point(1,1));
end;

procedure TTaskbar.FormDragOver(Sender, Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  Accept := False;
  if not BarShowing then ShowBar;
end;



procedure TTaskbar.StartButtonMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var p: TPoint;
begin
  if StartMouseUp then UpdateStartButtonState
  else begin
    GetCursorPos(p);
    SetMouseMonitor;
    if not (Stay.Checked or PtInRect(BoundsRect, p)) then HideBar;;
  end;
end;

end.
