{ copyright  Grahame Grieve, Kim Kirkpatrick

  refer to xyGraph.pas, xyGraph.hlp for more information }

{ definitions - see xyGraph.pas for info }

{$DEFINE STATISTICS }

{--------------- implementation section heading labels ---------------
    #1. Administration
    #2. Data engine
    #2. Data property servers
    #3. Other property servers
    #4. Statistical analysis
 ----------------------------------------------------------------------}

unit xyData;

interface

uses
  sysutils, classes;

type
{$IFNDEF WIN32}
  ShortString = string [255];
{$ENDIF}

{$IFDEF STATISTICS}
  et_regressions = ( rg_None,
                     {1. forms of regression: }
                     rg_Linear, rg_passingBablok, rg_quadratic,
                     {2. Curve Fitting: }
                     rg_Spline, rg_dwls,
                     {3. Other: }
                     rg_RunningAverage);
    {note that a function exports these names for use at run time}

  et_interceptResult = (ir_None, ir_found, ir_invalid);
         { ir_invalid if there is a continuous range, or no regression set }

{linked lists - sorting types}
  psPoint = ^TsPoint;
  TsPoint = record
            x: Real;
            next: psPoint;
            end;
  pSortP  = ^TSortP;
  TSortP  = record
            Point: psPoint;
            next: pSortP;
            end;

{$ENDIF}{STATISTICS}

  { each point is stored in a linked list as an x,y coordinate with a longint
    index value. This puts a theoretical limit on the number of points in a
    series to 2147483647 points, but since this is >40000 Mb of data, it's
    probably not a problem }

  pgPoint = ^tgPoint;
  tgPoint = record
            i: Longint;
            xv, yv, rv: Double;
            next: pgPoint;
            end;

  tDoublePoint = record
            x,y:double;
            end;

  {$IFDEF STATISTICS}

  TDataStats = record
    {admin:}
     current:boolean;
     comment:string;
     error:boolean;
    {basic statistics}
     count:integer;
     mean, median, mode, Total, SD, Skew, Kurtosis,
     lowquintile, highquintile, lowquartile, highquartile:double;
    end;

  TCompStats = record
    {admin:}
     current:boolean;
     comment:string;
     error:boolean;
    {correlation}
     Rsquared, PearsonR,  {pearson correlation}
     PRMin, PRMax, RSqMin, RSqMax, {conf intervals of r}
     PValue,
    {regressions}
     { y = FRegSlope2 * x^2 + FRegSlope * x + FRegIntercept }
     {1. Available in all forms of regression: }
     RegSlope, regintercept,
     {2. available during quadratic regression: }
     RegSlope2,
     {3. available for Passing/Bablok regression }
     SlopeSD, IntSD: Double;
    end;
{$ENDIF}

{ messages sent to requestpaint & objects to identify the reason }
 TDSChangeType = (dsUnLoad, dsDataChange, dsZeroChange, dsClearUpdates,
                  {$IFDEF STATISTICS} dsRegressionChange, {$ENDIF}
                  dsChangeLook);

{ allowance for other objects to register an interest in the data of the series.
  xyGraph actually defines a descendent and uses requestpaint, but only one
  descendent can exist. This allows other analytical tools to use the same
  TDataSeries }

 TDataChangeEvent=procedure (Sender:TObject; TheMessage:TDSChangeType) of object;

 pDataChangeHook=^TDataChangeHook;
 TDataChangeHook=record
      FEvent:TDataChangeEvent;
      FObject:TObject;
      next:pDataChangeHook;
   end;

 TDataSeries = class
   private
     FOnTooManyPoints: TNotifyEvent;
     FDataChangeHooks:PDataChangeHook;
     FPointWarningLimit: longint;
     procedure SetAutoZero(v: Boolean);
     procedure setZeroOffset(v: Double);
     procedure SetAllowDuplicates(v: Boolean);
    {$IFDEF STATISTICS}
     procedure SetRegType(v: et_regressions);
     procedure SetRegControl1(v: LongInt);
     procedure setRegcontrol2(v: Double);
     procedure DoLinearRegression;
     procedure DoQuadraticRegression;
     procedure DoPassingBablok;
     procedure DoRunningAverage;
     procedure DoSplineRegression;
     function  GetDWLS(x, xDiff: Double): Double;
     procedure doDWLS;
     procedure DoCalcStats(p: pSPoint; n: integer; var stats: TDataStats);
     procedure CalcStats(var stats: TDataStats; isxaxis:boolean);
     procedure correlate;
     function  GetXStats: TDatastats;
     function  GetYStats: TDatastats;
     function  GetCompStats: TCompstats;
    {$ENDIF}
     function  getpointval(x: Double): Double;
     procedure setpointval(x: Double; y: Double);
     function  getPointndx(i: longint): Double;
     procedure setPointndx(i: longint; y: Double);
     function  getAllPoints(x: Double): pGPoint;
     function  getPointCount(x: Double): Word;
     procedure DoDataChangeHooks(themessage:TDSChangeType);
     procedure clearDataChangeHooks;
     function  getDataChangeHook(o:TObject):TDataChangeEvent;
     procedure setDataChangeHook(o:TObject; v:TDataChangeEvent);

    protected
      FIndexCount, FPointCount: Longint;
      FData, FDataLast, FCurrent: pGPoint;
      xvMin, xvMax, yvMin, yvMax, FZeroOffset, FTolerance: Double;
      Changed, FAutoZero, FAllowDuplicates: Boolean;
     {$IFDEF STATISTICS}
      FXStats,FYStats: TDataStats;
      FCompStats: TCompStats;
      FRegType: et_regressions;
      FRegControl1: LongInt;
      FRegControl2: Double;
      FRegrData: pgPoint;
      NeedRegreCalc: Boolean;
     {$ENDIF}
      FSeriesName:Shortstring;
      procedure RequestPaint(TheMessage:TDSChangeType); virtual;
      procedure WarnTooManyPoints; virtual; { call event to notify. overridden
                          to call a different event in xyGraph descendent }
      procedure ExtractZero;
      procedure GetMinMax;
    public
      constructor Create;
      destructor Destroy; override;
      procedure Clear;
      property SeriesName:shortstring read FSeriesName write FSeriesName;

      property  Tolerance: Double read FTolerance write FTolerance;
      property  ValueAt[x: Double]: Double read getpointval write setpointval; default;
        { Adds a point, returns last point. while this is a very convenient
            interface, not overly powerful re mutiple points, error control, etc.}

      property  AllPoints[x: Double]: pGPoint read getAllPoints;
      property  PointByIndex[i: longint]: Double read getPointndx write setPointndx;
      property  PointCount[x:Double]:Word read GetPointCount;
      property  AllowDuplicates: Boolean read FAllowDuplicates write SetAllowDuplicates;
      function  Add(x, y: Double): longint;
      procedure AddPointSeries(x:Double; y:array of Double);
      procedure AddPoints(p:array of TDoublepoint);
      function  GetValue(curr: longint; var index: longint;
                           var x, y, r: Double): Boolean;
      function  GetPoint(var index: longint;
                           var x, y, r: Double): Boolean;

      function  DeleteValue(x: Double): Boolean;
      function  DeletePoint(index: longint): Boolean;
      function  EliminateDuplicates: Boolean;

     {simple sequential access to data:}
      function  GetFirstPoint(var x, y, r: Double; var index: longint): Boolean;
      function  GetNextPoint(var x, y, r: Double; var index: longint): Boolean;
          {both return false if no more points}
      procedure UpdateCurrentPoint(y: Double);
      function  DeleteCurrentPoint(var x, y, r: Double; var index: longint): Boolean;
          {deletes current point, and performs GetNextPoint}

     {$IFDEF STATISTICS}
      procedure DoRegression;
      procedure getintercepts(y: double; var result: et_interceptResult;
                              var count: word; var intercepts: PsPoint);
      procedure disposeintercepts(var intercepts:psPoint);
     {$ENDIF}

      property Data: Pgpoint read FData;
      property AutoZero: Boolean read FAutoZero write SetAutoZero;
      property InternalZero: Double read FZeroOffset write setZeroOffset;
     {$IFDEF STATISTICS}
      property XStats: TDataStats read GetXStats;
      property YStats: TDataStats read GetYStats;
      property CompStats: TCompStats read GetCompStats;
      property RegType: et_regressions read FRegType write SetRegType;
      property RegControl1: LongInt read FRegControl1 write SetRegControl1;
      property RegControl2: Double read FRegControl2 write setRegcontrol2;
      property RegressionData: pgPoint read FRegrData;
     {$ENDIF}

     { protection against unlimited memory usage: warn when
       number of points exceed limit }
     property OnTooManyPoints: TNotifyEvent read FOnTooManyPoints write FOnTooManyPoints;
     property PointWarningLimit: longint read FPointWarningLimit write FPointWarningLimit;
     property PointNumber:longint read FPointCount;

     property XMinVal:double read xvmin;
     property XMaxVal:double read xvmax;
     property YMinVal:double read yvmin;
     property YMaxVal:double read yvmax;
     property Hooks [o:tobject] :TDataChangeEvent
                 read getDataChangeHook write setDataChangeHook;
    end {class TdataSeries};


implementation

const
   tiny=1.0e-20;  { used to avoid divide by zero errors }

{--------------------------------------------------------------------------
    #1. Administration
 --------------------------------------------------------------------------}

constructor TDataSeries.create;
begin
  FData := nil;
  FDataLast := nil;
  FCurrent := nil;
  Changed := false;
  xvMin := 0;
  xvMax := 0;
  yvMin := 0;
  yvMax := 0;
  FZeroOffset := 0;
  FTolerance := 0;
  FAutoZero := false;
  FIndexCount := 0;
  FAllowDuplicates := false;
  FDataChangeHooks := nil;
  {$IFDEF STATISTICS}
  FRegType := rg_none;
  FRegControl1 := 100;
  FRegControl2 := 0.05;
  FRegrData := nil;
  NeedRegreCalc := false;
  FXStats.Current := false;
  FYStats.Current := false;
  FCompStats.Current := false;
  {$ENDIF}
  FPointWarningLimit := 30000;
  FSeriesName := 'UnNamed Series';
end;

procedure TDataSeries.RequestPaint;
{ this procedure exists as a notification point for state changes.
  any registered objects will be notified. This procedure is defined
  as virtual so that the xyGraph descendant can override it and decide
  a/ if registered objects are notified of data changes; and
  b/ who gets to paint first - the graph or the other objects

  Note!! The descendant procedure must call this one (i.e. inherited
  RequestPaint at some stage)

  This procedure is called when:
  - the object is being destroyed.
     This is intended to save you work at run-time, because all the
     registered objects are automatically notified, and should unhook themselves
     It may not matter if they only respond to the event.
     BUT!! there's a gotcha. :(  If your application is terminating,
     and any of your objects try to paint themselves, you'll get a GPF because
     their window has gone. So do "if not application.terminating then ..."
     before responding to this one
  - data changes
  - zero changes
  - regression settings change
  - appearance changes}
begin
  DoDataChangeHooks(TheMessage);
end;

procedure EmptySeriesData(var p: pgPoint);
var p2, p1: pgPoint;
begin
  p2 := p;
  while p2 <> nil do
  begin
    p1 := p2;
    p2 := p2^.next;
    Dispose(p1);
  end;
  p := nil;
end;

destructor TDataSeries.destroy;
begin
  clearDataChangeHooks;
  if FData <> nil then EmptySeriesData(FData);
  {$IFDEF STATISTICS}
  if FRegrData <> nil then EmptySeriesData(FRegrData);
  {$ENDIF}
  inherited destroy;
end;


procedure TDataSeries.Clear;
begin
  if FData <> nil then EmptySeriesData(FData);
 {$IFDEF STATISTICS}
  if FRegrData <> nil then EmptySeriesData(FRegrData);
 {$ENDIF}
  FDataLast := nil;
  FIndexCount := 0;
  FPointCount := 0;
  Changed := True;
  RequestPaint(dsDataChange);
end;

procedure TDataSeries.DoDataChangeHooks(themessage:TDSChangeType);
var p:pDataChangeHook;
begin
 p := FDataChangeHooks;
 while p <> nil do
   begin
   if assigned(p^.Fevent) then p^.FEvent(self, themessage);
   p := p^.next;
   end;
end;

procedure TDataSeries.clearDataChangeHooks;
var p,p1:pDataChangeHook;
begin
 p := FDataChangeHooks;
 while p <> nil do
   begin
   p1 := p;
   p := p^.next;
   if assigned(p1^.Fevent) then p1^.FEvent(self, dsUnLoad);
   dispose(p1);
   end;
end;

function TDataSeries.getDataChangeHook(o:TObject):TDataChangeEvent;
var p,p1:pDataChangeHook;
begin
 p := FDataChangeHooks;
 while (p <> nil) and (p^.Fobject <> o) do p := p^.next;
 if p <> nil then result := p^.FEvent else
   if FDataChangeHooks = nil then
     begin
     new(FDataChangeHooks);
     FDataChangeHooks^.Fevent := nil;
     FDataChangeHooks^.Fobject := o;
     FDataChangeHooks^.next := nil;
     result := FDataChangeHooks^.FEvent;
     end
   else
     begin
     p := FDataChangeHooks;
     while p <> nil do
       begin
       p1 := p;
       p := p^.next;
       end;
     new(p);
     p^.Fevent := nil;
     p^.Fobject := o;
     p^.next := nil;
     p1^.next := p;
     result := p^.FEvent;
     end;
end;

procedure TDataSeries.setDataChangeHook(o:TObject; v:TDataChangeEvent);
var p,p1:pDataChangeHook;
begin
 p := FDataChangeHooks;
 while (p <> nil) and (p^.Fobject <> o) do p := p^.next;
 if p <> nil then p^.Fevent := v else
   if FDataChangeHooks = nil then
     begin
     new(FDataChangeHooks);
     FDataChangeHooks^.Fevent := v;
     FDataChangeHooks^.Fobject := o;
     FDataChangeHooks^.next := nil;
     end
   else
     begin
     p := FDataChangeHooks;
     while p <> nil do
       begin
       p1 := p;
       p := p^.next;
       end;
     new(p);
     p^.Fevent := v;
     p^.Fobject := o;
     p^.next := nil;
     p1^.next := p;
     end;
end;

{--------------------------------------------------------------------------
    #2. Data engine
 --------------------------------------------------------------------------}

function TDataSeries.Add(x, y: Double): longint;
var pp, pp1, pp2: pgPoint;
begin
  Result := 0;
  if FData = nil then
  begin
    New(pp);
    Inc(FIndexCount);
    Inc(FPointCount);
    pp^.i := FIndexCount;
    pp^.next := nil;
    pp^.xv := x;
    pp^.yv := y;
    pp^.rv := 0;
    FData := pp;
    FDataLast := pp;
  end
  else if FDataLast^.xv <= x then
  begin
    if FAllowDuplicates or (FDataLast^.xv < x - FTolerance) then
    begin
      New(pp);
      Inc(FIndexCount);
      Inc(FPointCount);
      pp^.i := FIndexCount;
      pp^.next := nil;
      pp^.xv := x;
      pp^.yv := y;
      pp^.rv := 0;
      FDataLast^.next := pp;
      FDataLast := pp;
    end
    else
    begin
      FDataLast^.xv := x;
      FDataLast^.yv := y;
    end;
  end
  else
  begin
    pp1 := nil;
    pp := FData;
    while (pp <> nil) and (pp^.xv < x - FTolerance) do
    begin
      pp1 := pp;
      pp := pp^.next;
    end;
    if FAllowDuplicates or not (abs(pp^.xv - x) <= FTolerance) then
    {This represents potential changing of the data. That's a problem for the user
    to consider.  FTolerance defaults to 0}
    begin
      New(pp2);
      Inc(FIndexCount);
      Inc(FPointCount);
      pp2^.i := FIndexCount;
      pp2^.xv := x;
      pp2^.yv := y;
      pp2^.rv := 0;
      pp2^.next := pp;
      if pp1 = nil then FData := pp2 else pp1^.next := pp2;
    end
    else
    begin
      pp^.xv := x;
      pp^.yv := y;
    end;
  end;

  if FData^.next = nil then
  begin
    xvMin := x;
    xvMax := x;
    if FAutoZero then FZeroOffset := y;
    yvMin := y - FZeroOffset;
    yvMax := y - FZeroOffset;
    Changed := false;
  end
  else if FAutoZero and (y < FZeroOffset) then
  begin
    FZeroOffset := y;
    Changed := True;
  end
  else if not Changed then
  begin
    if y - FZeroOffset < yvMin then yvMin := y - FZeroOffset;
    if y - FZeroOffset > yvMax then yvMax := y - FZeroOffset;
    if x < xvMin then xvMin := x;
    if x > xvMax then xvMax := x;
  end;

 {$IFDEF STATISTICS}
  NeedRegreCalc := True;
  FXStats.Current := false;
  FYStats.Current := false;
  FCompStats.Current := false;
 {$ENDIF}
  if (FPointWarningLimit <> 0) and (FPointCount > FPointWarningLimit) then
      WarnTooManyPoints;
  RequestPaint(dsDataChange);
end {Add};

procedure TDataSeries.WarnTooManyPoints;
begin
  if assigned(FOnTooManyPoints) then FOnTooManyPoints(self);
end;

procedure TDataSeries.setPointndx(i:longint; y:double);
var p:pgpoint;
begin
  p := FData;
  while (p <> nil) and (p^.i <> i) do p := p^.next;
  if p <> nil then p^.yv := y;
  if FAutoZero and (y < FZeroOffset) then
  begin
    FZeroOffset := y;
    Changed := True;
  end
  else if not Changed then
  begin
    if y - FZeroOffset < yvMin then yvMin := y - FZeroOffset;
    if y - FZeroOffset > yvMax then yvMax := y - FZeroOffset;
  end;
  RequestPaint(dsDataChange);
end;

function TDataSeries.getAllPoints(x:double):pgpoint;
var p,bp,lp,sp:pgpoint;
    stopvalue:double;
begin
  result := nil;
  p := FData;
  stopvalue := x - Ftolerance;
  while (p <> nil) and (p^.xv < stopvalue) do p := p^.next;
  if p = nil then exit;
  if p^.xv <= x + FTolerance then
  begin
    new(bp);
    bp^ := p^;
    bp^.next := nil;
    lp := bp;
    p := p^.next;
  end else bp := nil;
  while p^.xv <= x + Ftolerance do
  begin
    new(sp);
    sp^ := p^;
    sp^.next := nil;
    lp^.next := sp;
    lp := sp;
    p := p^.next;
  end;
  result := bp;
end;

function TDataSeries.getpointcount(x:double):word;
var p:pgpoint;
    stopvalue:double;
begin
  result := 0;
  p := FData;
  stopvalue := x + tolerance;
  while (p <> nil) and (p^.xv < Stopvalue) do
  begin
    if abs(p^.xv - x)<= FTolerance then inc(result);
    p := p^.next;
  end;
end;

function TDataSeries.GetValue(curr: longint; var index: longint;
                       var x, y, r: Double): Boolean;
var pp: pgPoint;
begin
  pp := FData;
  while (pp <> nil) and (pp^.xv < x) do pp := pp^.next;
  while (pp <> nil) and (curr > 1) do
  begin
    pp := pp^.next;
    Dec(curr);
  end;
  Result := (pp <> nil) and (abs(pp^.xv-x) <= FTolerance);
  if Result then
  begin
    y := pp^.yv;
    index := pp^.i;
    r := pp^.rv;
  end;
end;

function TDataSeries.GetPoint(var index: longint;
                       var x, y, r: Double): Boolean;
var pp: pgPoint;
begin
  pp := FData;
  while (pp <> nil) and (pp^.i <> index) do
    pp := pp^.next;
  Result := (pp <> nil);
  if Result then
  begin
    x := pp^.xv;
    y := pp^.yv;
    r := pp^.rv;
    if pp^.next = nil then
      index := 0
    else
      index := pp^.next^.i;
  end;
end;

procedure TDataSeries.UpdateCurrentPoint(y: Double);
begin
  if (FCurrent <> nil) and (y <> FCurrent^.yv) then
  begin
    if FAutoZero and (y < FZeroOffset) then
      begin
      FZeroOffset := y;
      changed := true;
      end
    else if not changed then
      if (y < yvMin) then yvMin := y
       else if (y > yvMax) then yvMax := y;
    FCurrent^.yv := y;
   {$IFDEF STATISTICS}
    needregrecalc := true;
    FXStats.Current := false;
    FYStats.Current := false;
    FCompStats.Current := false;
   {$ENDIF}
    RequestPaint(dsDataChange);
  end;
end;

function  TDataSeries.DeleteCurrentPoint(var x, y, r: Double; var index: longint): Boolean;
var t: pGPoint;
begin
  if FCurrent <> nil then
  begin
    t := FCurrent;
    FCurrent := FCurrent^.next;
    DeletePoint(t^.i);    {not very efficient, but should work}
  end;
  Result := (FCurrent <> nil);   { return next point values }
  if Result then
  begin
    x := FCurrent^.xv;
    y := FCurrent^.yv;
    r := FCurrent^.rv;
    index := FCurrent^.i;
  end;
end;


function  TDataSeries.DeletePoint(index: longint): Boolean;
var pp, pp1: pgPoint;
begin
  pp := FData;
  pp1 := nil;
  while (pp <> nil) and (pp^.i <> index) do
  begin
    pp1 := pp;
    pp := pp^.next;
  end;
  Result := (pp <> nil);
  if Result then
  begin
    if pp1 = nil then FData := pp^.next else pp1^.next := pp^.next;
    if pp^.next = nil then FDataLast := pp1;
    dec(FPointCount);
    {Let's only recalc MaxMin if necessary:}
    if ((pp^.xv = xvMin) or (pp^.xv = xvMax)) or
       ((pp^.yv = yvMin) or (pp^.yv = yvMax)) or
       ((pp^.yv <= FZeroOffset) and FAutoZero) then
      Changed := True;
    {$IFDEF STATISTICS}
    needregrecalc := true;
    FXStats.Current := false;
    FYStats.Current := false;
    FCompStats.Current := false;
    {$ENDIF}
    Dispose(pp);
    RequestPaint(dsDataChange);
  end;
end;

function  TDataSeries.DeleteValue(x: Double): Boolean;
var i: longint;
    y, dummy: Double;
begin
  Result := False;
  while GetValue(0, i, x, y, dummy) do
    begin
      Result := True;
      DeletePoint(i);
    end;
end;

function TDataSeries.EliminateDuplicates: Boolean;
var pp, pp1: pgPoint;
begin
  Result := False;
  pp := FData;
  if pp <> nil then
    while pp^.next <> nil do
    begin
      pp1 := pp^.next;
      if abs(pp1^.xv-pp^.xv) <= FTolerance then
      begin
        pp^.next := pp1^.next;
        Dispose(pp1);
        dec(FPointCount);
        Changed := True;
       {$IFDEF STATISTICS}
        NeedregreCalc := true;
       {$ENDIF}
      end else pp := pp^.next;
    end;
  if changed then RequestPaint(dsDataChange);
end;

procedure TDataSeries.ExtractZero;
var t: Double;
  pt: pgPoint;
begin
 if FAutoZero then
  begin
    pt := FData;
    t := pt^.yv;
    while pt <> nil do
    begin
      if pt^.yv < t then t := pt^.yv;
      pt := pt^.next;
    end;
    FZeroOffset := t;
  end;
end;

procedure TDataSeries.GetMinMax;
var
  pt: pgPoint;
begin
   if FData <> nil then
    begin
      pt := FData;
      yvMin := pt^.yv;
      yvMax := pt^.yv;
      xvMin := pt^.xv;
      xvMax := pt^.xv;
      while pt <> nil do
      begin
        if yvMin > pt^.yv then yvMin := pt^.yv;
        if yvMax < pt^.yv then yvMax := pt^.yv;
        if xvMin > pt^.xv then xvMin := pt^.xv;
        if xvMax < pt^.xv then xvMax := pt^.xv;
        pt := pt^.next;
      end;
      ExtractZero;
      yvMin := yvMin - FZeroOffset;
      yvMax := yvMax - FZeroOffset;
      Changed := False;
    end;
end;

{--------------------------------------------------------------------------
    #3. Data property servers
 --------------------------------------------------------------------------}

function TDataSeries.getpointval(x:double):double;
var d1:longint;
    r:double;
begin
  if not getvalue(0,d1,x,result,r) then result := 0;
{ this returns the last point added. (if allowduplicates = false, this
  is the only point). If there is any problems, returns 0}
end;

procedure TDataSeries.setpointval(x:double; y:double);
begin
  add(x,y);
end;

function TDataSeries.getPointndx(i:longint):double;
var x, y, r: Double;
begin
  Getpoint(i,x,y,r);
  result := y;
end;

procedure TDataSeries.AddPointSeries(x:double; y:array of double);
var i: word;
begin
  for i := 0 to High(y) do add(x,y[i]);
end;

procedure TDataSeries.AddPoints(p:array of TDoublepoint);
var i: word;
begin
  for i := 0 to High(p) do add(p[i].x,p[i].y);
end;

function TDataSeries.GetFirstPoint(var x, y, r: Double; var index: longint): Boolean;
begin
  FCurrent := FData;
  Result := (FCurrent <> nil);
  if Result then
  begin
    x := FCurrent^.xv;
    y := FCurrent^.yv;
    r := FCurrent^.rv;
    index := FCurrent^.i;
  end;
end;

function TDataSeries.GetNextPoint(var x, y, r: Double; var index: longint): Boolean;
begin
  if FCurrent <> nil then
    FCurrent := FCurrent^.next;
  Result := (FCurrent <> nil);
  if Result then
  begin
    x := FCurrent^.xv;
    y := FCurrent^.yv;
    r := FCurrent^.rv;
    index := FCurrent^.i;
  end;
end;

{--------------------------------------------------------------------------
    #4. Other property servers
 --------------------------------------------------------------------------}

procedure TDataSeries.SetAutoZero(v:boolean);
begin
  if v <> FAutoZero then
  begin
    FAutoZero := v;
    changed := true;
    RequestPaint(dsZeroChange);
  end;
end;

procedure TDataSeries.setZeroOffset(v:Double);
begin
  if FAutoZero or (v <> FZeroOffset) then
  begin
    FZeroOffset := v;
    FAutoZero := false;
    changed := true;
    RequestPaint(dsZeroChange);
  end;
end;

procedure TDataSeries.SetAllowduplicates;
begin
 if v <> FAllowDuplicates then
 begin
  FAllowDuplicates := v;
  if not FAllowDuplicates then eliminateduplicates;
 end;
end;

{$IFDEF STATISTICS}
procedure TDataSeries.SetRegType(v:et_regressions);
begin
  if v <> FRegType then
  begin
    FRegType := v;
    NeedRegrecalc := true;
    RequestPaint(dsRegressionChange);
  end;
end;

procedure TDataSeries.SetRegControl1(v:longint);
begin
  if v <> FRegControl1 then
  begin
    FRegControl1 := v;
    NeedRegrecalc := true;
    RequestPaint(dsRegressionChange);
  end;
end;

procedure TDataSeries.setRegcontrol2(v:double);
begin
  if v <> FRegcontrol2 then
  begin
    FRegcontrol2 := v;
    NeedRegrecalc := true;
    RequestPaint(dsRegressionChange);
  end;
end;
{$ENDIF}{STATISTICS}

{--------------------------------------------------------------------------
    #5. Statistical analysis
 --------------------------------------------------------------------------}
{$IFDEF STATISTICS}

procedure FillData(p: pgPoint; m2, m, c: Double);
begin
  while p <> nil do
  begin
    p^.rv := (p^.xv * p^.xv * m2) + (p^.xv * m) + c;
    p := p^.next;
  end;
end;

{ Sorting of secondary lists. This is used for Passing-Bablok regression and
  for stats retrieval. Merge sort }
procedure Empty(p: psPoint);
var p1: psPoint;
begin
  while p <> nil do
  begin
    p1 := p;
    p := p^.next;
    Dispose(p1);
  end;
end;

procedure Sort(var orig: psPoint);

  function merge(p1, p2: psPoint): psPoint;
  var p, pc: psPoint;
  begin
    p := nil;
    while (p1 <> nil) or (p2 <> nil) do
    begin
      if p1 = nil then begin
        if p = nil then begin
          p := p2; pc := p2;
        end else begin pc^.next := p2; pc := pc^.next; end;
        p2 := p2^.next;
      end else
        if p2 = nil then begin
          if p = nil then begin
            p := p1; pc := p1;
          end else begin pc^.next := p1; pc := pc^.next; end;
          p1 := p1^.next;
      end else
        if p1^.x < p2^.x then begin
          if p = nil then begin
            p := p1; pc := p1;
          end else begin pc^.next := p1; pc := pc^.next; end;
          p1 := p1^.next;
      end else
      begin
        if p = nil then begin
          p := p2; pc := p2;
        end else begin pc^.next := p2; pc := pc^.next; end;
        p2 := p2^.next;
    end; end;
    Result := p;
  end {Merge};

var p, p1, p2, pt: psPoint;
    k2, kb, kl: pSortP;
begin {Sort}
 { first pass: a list of nodes 1 long. merge them once as they go into
  a secondary structure, a list of sorted lists }
  p := orig;
  kb := nil;
  if (p = nil) or (p^.next = nil) then Exit;
  while p <> nil do
  begin
    p1 := p;
    if p^.next <> nil then p2 := p^.next else p2 := nil;
    if p2 = nil then p := nil else p := p2^.next;
    p1^.next := nil;
    if p2 <> nil then p2^.next := nil;
    pt := merge(p1, p2);
    if kb = nil
    then begin New(kb); kb^.next := nil; kl := kb; k2 := kb; end
    else begin New(k2); k2^.next := nil; kl^.next := k2; kl := k2; end;
    k2^.Point := pt;
  end;
  { next pass: pull 2 lists off, merge them, and add them at the end }
  { when there is only 1 list left, it has been sorted }
  while kb <> kl do
  begin
    p1 := kb^.Point;
    p2 := kb^.next^.Point;
    p := merge(p1, p2);
    k2 := kb;
    kb := kb^.next;
    Dispose(k2);
    k2 := kb;
    kb := kb^.next;
    Dispose(k2);
    if kb = nil
      then begin New(kb); kb^.next := nil; kl := kb; k2 := kb; end
      else begin New(k2); k2^.next := nil; kl^.next := k2; kl := k2; end;
    k2^.Point := p;
  end;
  orig := kb^.Point;
  Dispose(kb);
end {Sort};


procedure TDataSeries.DoCalcStats;
var mostVal,totalsq,last,v,svar,v1:double;
    mostCount,currCount,c,i1,i2,i3,i4,i5:longint;
    pt:pspoint;
    i1m,i2m,i3m,i4m,i5m:boolean;
begin
 if n < 2 then
   begin
   stats.comment := 'No Statistics are available unless more than one point exists';
   stats.error := true;
   exit;
   end;
 {no stats unless more than 2 points}
 with stats do
  begin
  comment := '';
  stats.error := false;
  count := n;
  pt := p;
  total := 0;
  while pt <> nil do
    begin
    total := total + pt^.x;
    pt := pt^.next;
    end;
  mean := total / count;

  i1 := n div 5;
  i1m := (n mod 5) <> 0;
  i2 := n div 4;
  i2m := (n mod 4) <> 0;
  i3 := n div 2;
  i3m := (n mod 2) <> 0;
  i4 := n - n div 4;
  i4m := (n mod 4) <> 0;
  i5 := n - n div 5;
  i5m := (n mod 5) <> 0;

  mostcount := 0;
  totalsq := 0;
  svar := 0;
  skew := 0;
  kurtosis := 0;
  c := 0;
  last := p^.x;
  CurrCount := 0;
  while p <> nil do
  begin
    inc(C);
    v := p^.x;
    if c = i1 then if i1m and (p^.next <> nil)
          then stats.lowquintile := (v + p^.x)/2
          else stats.lowquintile := v;
    if c = i5 then if i5m and (p^.next <> nil)
          then stats.highquintile := (v + p^.x)/2
          else stats.highquintile := v;
    if c = i2 then if i2m and (p^.next <> nil)
          then stats.lowquartile := (v + p^.x)/2
          else stats.lowquartile := v;
    if c = i4 then if i4m and (p^.next <> nil)
          then stats.highquartile := (v + p^.x)/2
          else stats.highquartile := v;
    if c = i3 then if i3m and (p^.next <> nil)
          then stats.median := (v + p^.x)/2
          else stats.median := v;
    totalsq := totalsq + v * v;
    if not (abs(v - last) <= FTolerance) then
    begin
      last := v;
      currCount := 0;
    end
    else
    begin
      inc(CurrCount);
      if CurrCount > MostCount then
      begin
        MostCount := CurrCount;
        MostVal := v;
      end;
    end;
    v := v - mean;
    v1 := v * v;
    svar := svar + v1;
    v1 := v1 * v;
    skew := skew + v1;
    v1 := v1 * v;
    kurtosis := kurtosis + v1;
    p := p^.next;
  end;
  mode := MostVal;
  SD := (totalsq - (Total*total)/n) / (n-1);
  if svar = 0 THEN
    begin
    comment := 'No skew/kurtosis are available because there is no deviance in the data';
    skew := 0;
    Kurtosis := 0;
    end
  else
    begin
    skew := skew/(n*SD*SD*SD);
    kurtosis := kurtosis/(n*sqr(svar))-3.0;
    end;
  end;
end;

{------------------------- regression routines -----------------------------}
{1. Passing Bablok Regression.
  Suited for analytical comparisons - no assumption that x is without error
  reference Bablok W. and Passing, H.: "Application of Statistical Procedures
  in Analytical instrument testing", J. Auto. Chem. v7, 1985, pp74-79. }
procedure TDataSeries.DoPassingBablok;
var j: LongInt;
  datat, pp: psPoint;
  p1, p2: pgPoint;
  rstats:TDataStats;
begin
  if (FData = nil) or (FData^.next = nil) or (XStats.SD = 0) then
  begin
    FCompStats.RegSlope2 := 0;
    if XStats.sd = 0 then FCompStats.Regslope := 0 else FCompStats.RegSlope := 1;
    FCompStats.regintercept := 0;
    FillData(FData, FCompStats.RegSlope2, FCompStats.RegSlope, FCompStats.regintercept);
  end
  else
  begin
    j := 0;
    datat := nil;
    p1 := FData;
    while p1 <> nil do
    begin
      p2 := p1^.next;
      while p2 <> nil do
      begin
        if not (abs(p2^.xv - p1^.xv) <= FTolerance) then
    {check is really here to eliminate divide by zero errors. so FTolerance
     is perhaps superfluous. However conceptually points considered equal
     by FTolerance shouldn't really contribute to the regression}
        begin
          Inc(j);
          New(pp); pp^.next := datat; datat := pp;
          pp^.x := (p1^.yv - p2^.yv) / (p1^.xv - p2^.xv);
        end;
        p2 := p2^.next;
      end;
      p1 := p1^.next;
    end;
    Sort(datat);

    DoCalcStats(datat, j, rstats);
    FCompStats.RegSlope2 := 0;
    FCompStats.RegSlope := rstats.median;
    FCompStats.SlopeSD := rstats.SD;
    Empty(datat);

    j := 0;
    datat := nil;
    p1 := FData;
    while p1 <> nil do
    begin
      New(pp);
      Inc(j);
      pp^.next := datat;
      datat := pp;
      pp^.x := p1^.yv - (FCompStats.RegSlope * p1^.xv);
      p1 := p1^.next;
    end;
    Sort(datat);
    DoCalcStats(datat, j, rstats);
    FCompStats.regintercept := rstats.median;
    FCompStats.IntSD := rstats.SD;  {!! this gives a falsely low sense
         of the sd on the intercept, as it does not include contributory
         uncertainty from the slope sd. yet to be fixed }
    Empty(datat);
  end;
  FillData(FData, FCompStats.RegSlope2, FCompStats.RegSlope, FCompStats.regintercept);
end;

{2. Linear Regression.
  reference, Devore, JL.: "Probability and Statistics for the engineering
  and Scientists", Duxbury Press, Belmont, Ca. 1990, pp460 - 461}
procedure TDataSeries.DoLinearRegression;
var sumX, sumY, sumXY, sumX2: Double;
  pp: pgPoint;
  j: Integer;
begin
  if (FData = nil) or (FData^.next = nil) or (XStats.sd = 0) then
  begin
    FCompStats.RegSlope2 := 0;
    if XStats.SD = 0 then FCompStats.RegSlope := 0 else FCompStats.RegSlope := 1;
    FCompStats.regintercept := 0;
    FillData(FData, FCompStats.RegSlope2, FCompStats.RegSlope, FCompStats.regintercept);
  end
  else
  begin
    j := 0;
    sumX := 0;
    sumY := 0;
    sumX2 := 0;
    sumXY := 0;
    pp := FData;
    while pp <> nil do
    begin
      Inc(j);
      sumX := sumX + pp^.xv;
      sumY := sumY + pp^.yv;
      sumX2 := sumX2 + (pp^.xv * pp^.xv);
      sumXY := sumXY + (pp^.xv * pp^.yv);
      pp := pp^.next;
    end;
    FCompStats.RegSlope2 := 0;
    FCompStats.RegSlope := ((j * sumXY) - (sumX * sumY)) / ((j * sumX2) - (sumX * sumX));
    FCompStats.regintercept := (sumY / j) - FCompStats.RegSlope * (sumX / j);
  end;
  FillData(FData,  FCompStats.RegSlope2, FCompStats.RegSlope, FCompStats.regintercept);
end;

{2. Quadratic Regression.
  reference, Devore, JL.: "Probability and Statistics for the engineering
  and Scientists", Duxbury Press, Belmont, Ca. 1990, pp516 - 517}
procedure TDataSeries.DoQuadraticRegression;
var sumX, sumY, sumXY, sumX2, Sumx2y, Sumx3, Sumx4: Double;
    s1y,s22,s2y,s12,s11,xmean,ymean, x2mean: extended;
  pp: pgPoint;
  j: Integer;
begin
  if (FData = nil) or (FData^.next = nil) or (FData^.next^.next = nil)
              or (XStats.SD = 0)  then
      { more checks for more degrees of freedom! }
  begin
    FCompStats.RegSlope2 := 0;
    if XStats.sd = 0 then FCompStats.Regslope := 0 else
    FCompStats.regintercept := 0;
  end
  else
  begin
    j := 0;
    sumX := 0;
    sumY := 0;
    sumX2 := 0;
    sumXY := 0;
    sumx2y := 0;
    sumx3 := 0;
    sumx4 := 0;
    pp := FData;
    while pp <> nil do
    begin
      Inc(j);
      sumX := sumX + pp^.xv;
      sumY := sumY + pp^.yv;
      sumX2 := sumX2 + (pp^.xv * pp^.xv);
      sumXY := sumXY + (pp^.xv * pp^.yv);
      sumx2y := sumx2y + (pp^.xv * pp^.xv * pp^.yv);
      sumx3 := sumx3 +  + (pp^.xv * pp^.xv * pp^.xv);
      sumx4 := sumx4 + (pp^.xv * pp^.xv * pp^.xv * pp^.xv);
      pp := pp^.next;
    end;
    xmean := sumx/j;
    ymean := sumy/j;
    x2mean := sumx2/j;
    s1y := sumxy - j * xmean * ymean;
    s2y := sumx2y - j * xmean * xmean * ymean;
    s11 := sumx2 - j * xmean * xmean;
    s12 := sumx3 - j * xmean * x2mean;
    s22 := sumx4 - j * x2mean * x2mean;
    FCompStats.RegSlope2 := (s2y*s11 - s1y*s12) / (s11*s22 - s12*s12);
    FCompStats.RegSlope := (s1y*s22 - s2y*s12) / (s11*s22 - s12*s12);
    FCompStats.regintercept := (sumY / j) - FCompStats.RegSlope * (sumX / j) -
                             (FCompStats.RegSlope2 * (sumX2 / j));
  end;
  FillData(FData,  FCompStats.RegSlope2, FCompStats.RegSlope, FCompStats.regintercept);
end;

{ Running Average - applications in QA procedures. No reference }
procedure TDataSeries.DoRunningAverage;
var pp: pgPoint;
  avg: Double;
begin
  if FData <> nil then
  begin
    pp := FData;
    avg := pp^.yv;
    pp^.rv := pp^.yv;
    pp := pp^.next;
    while pp <> nil do
    begin
      avg := FRegControl2 * (pp^.yv - FZeroOffset) + (1 - FRegControl2) * avg;
      pp^.rv := avg;
      pp := pp^.next;
    end;
  end;
end;

{ Distance Weighted Least Squares line fitting
 reference - McLain DH.: "Drawing Contours from arbitrary data points." Comp. J.,
 v17, 1973? pp 318 - 324.
 His discussion there is for 3d fitting - I adapted this to tp4.0 code along
 time ago, and some day'll I'll port my 3d code to a delphi graph component
 The weighting is discussed there. The least squares bit came off the top of
 my head, and I missed something, because it doesn't work quite as it should.
 But with a quick hack, it works fine - see the comment in the main routine }

function TDataSeries.GetDWLS(x, xDiff: Double): Double;
var sumX, sumY, sumXY, sumX2, j, Weight, vx, cx, cy: Double;
  pp: pgPoint;

  function w(x: Double): Double;
  (* for DWLS,  w(d^2) = 1/(d^2 + e)^4  *)
  (* for NExpo, w(d^2) = exp(-a(d^2))  where a is in the order of 1/j^2 where j
                                   = avg distance between data points *)
  begin
    Result := Exp( {x * x* }x) / ( x *  x * x + FRegControl2);
  end;
    { you might want to leave the third power of x in - under some circumstances
      it actually improves the fit }
begin
  { there's a problem with this algorythm - only works for data spread
    over a x range of 1. also suffers severely around x = 0. So the data is
    internally mapped to the range 1 to 2. Not wonderful - but has the advantage
    of standardising the meaning of regcontrol2 across different ranges }
  vx := (x - xvMin + xDiff) / xDiff;
  j := 0;
  sumX := 0;
  sumY := 0;
  sumX2 := 0;
  sumXY := 0;
  pp := FData;
  while pp <> nil do
  begin
    cx := (pp^.xv - xvMin + xDiff) / xDiff;
    cy := pp^.yv;
    Weight := w(Abs(vx - cx));
    j := j + Weight;
    sumX := sumX + (Weight * cx);
    sumY := sumY + (Weight * cy);
    sumX2 := sumX2 + (cx * cx);
    sumXY := sumXY + Weight * (cx * cy);
    pp := pp^.next;
  end;
  Result := (((j * sumXY) - (sumX * sumY)) / ((j * sumX2) - (sumX * sumX))) * vx
            + (sumY / j) - (((j * sumXY) - (sumX * sumY)) / ((j * sumX2) - (sumX * sumX))) * (sumX / j);
end;

procedure TDataSeries.doDWLS;
var pp, pp1: pgPoint;
  x, xs, Xmax, xDiff: Double;
begin
  EmptySeriesData(FRegrData);
  FRegrData := nil;
  If FRegControl1 = 0 then exit;
  if xvMin = xvMax then Exit;
  xDiff := (xvMax - xvMin);
  xs := xDiff / RegControl1;
 Xmax := xvMax + 0.00001*xs; {OK for < 100,000 points}
  x := xvMin;
  while x <= Xmax do
  begin
    if FRegrData = nil then
    begin
      New(pp);
      pp^.next := nil;
      FRegrData := pp;
    end
    else
    begin
      New(pp1);
      pp1^.next := nil;
      pp^.next := pp1;
      pp := pp1;
    end;
    pp^.xv := x;
    pp^.rv := GetDWLS(x, xDiff);
    x := x + xs;
  end;
end;

{ spline algorithm - adapted from the "Numerical Recipes in Pascal"}
function GetSplineResult(x: Double; var pp, ps: pgPoint): Double;
var pp1: pgPoint;
  h, a, b: Double;
begin
  pp := ps;
  pp1 := nil;
  while (pp^.next <> nil) and (pp^.xv < x) do
  begin
    pp1 := pp;
    pp := pp^.next;
  end;
  if pp1 = nil then pp1 := pp;
  h := pp1^.next^.xv - pp1^.xv;
  a := (pp1^.next^.xv - x) / h;
  b := (x - pp1^.xv) / h;
  Result := a * pp1^.yv + b * pp1^.next^.yv +
            ((a * a * a - a) * pp1^.rv + (b * b * b - b) * pp1^.next^.rv) * h * h / 6;
  pp := pp1;
end;

procedure TDataSeries.DoSplineRegression;
var
  pp, pp1, pp2: pgPoint;
  pn, qn, sig, un, x, xs, Xmin, Xmax: Double;
  u, u1: pgPoint;
begin
  EmptySeriesData(FRegrData);
  FRegrData := nil;
  if FAllowDuplicates or (FRegControl1 = 0) then exit;
  pp := FData;
  New(u);
  u^.next := nil;
  if FRegControl2 > 0.99e30 then
  begin
    pp^.rv := 0.0;
    u^.rv := 0.0;
  end
  else
  begin
    pp^.rv := - 0.5;
    u^.rv := (3.0 / (pp^.next^.xv - pp^.xv))
             * ((pp^.next^.yv - pp^.yv) / (pp^.next^.xv - pp^.xv) - FRegControl2)
  end;
  pp1 := pp;
  pp := pp^.next;
  pp1^.next := nil; { we're going to reverse the whole list so we can
                    traverse it backwards later }
  while pp^.next <> nil do
  begin
    sig := (pp^.xv - pp1^.xv) / (pp^.next^.xv - pp1^.xv);
    pn := sig * pp1^.rv + 2;
    pp^.rv := (sig - 1) / pn;
    New(u1);
    u1^.next := u;
    u := u1;
    u^.rv := (pp^.next^.yv - pp^.yv) / (pp^.next^.xv - pp^.xv)
             - (pp^.yv - pp1^.yv) / (pp^.xv - pp1^.xv);
    u^.rv := (6 * u^.rv / (pp^.next^.xv - pp1^.xv) - sig * u^.next^.rv) / pn;
    { continuing the reversal of the list ....}
    pp2 := pp1;
    pp1 := pp;
    pp := pp^.next;
    pp1^.next := pp2;
  end;
  if (RegControl2 > 0.99e30) then
  begin
    qn := 0.0;
    un := 0.0;
  end
  else
  begin
    qn := 0.5;
    un := (3 / (pp^.xv - pp1^.xv)) * (FRegControl2 - (pp^.yv - pp1^.yv)
                                      / (pp^.xv - pp1^.xv));
  end;
  pp^.rv := (un - qn * u^.rv) / (qn * pp1^.rv + 1);
  { now, pp points at the end of the list. pp1 points at the list backwards,
    starting at the second last point }
  while pp1 <> nil do
  begin
    pp2 := pp1^.next;
    pp1^.rv := (pp1^.rv * pp^.rv + u^.rv);
    u1 := u;
    u := u^.next;
    Dispose(u1);
    pp1^.next := pp;
    pp := pp1;
    pp1 := pp2;
  end;
  { build regression }
  Xmin := xvMin;
  xs := (xvMax - Xmin) / RegControl1;
  Xmax := xvMax + 0.00001*xs; {OK for < 100,000 points}
  x := xvMin;
  pp := nil;
  while x <= Xmax do
  begin
    if FRegrData = nil then
    begin
      New(pp2);
      pp2^.next := nil;
      FRegrData := pp2;
    end
    else
    begin
      New(pp1);
      pp1^.next := nil;
      pp2^.next := pp1;
      pp2 := pp1;
    end;
    pp2^.xv := x;
    pp2^.rv := GetSplineResult(x, pp, FData);
    x := x + xs;
  end;
end;

procedure TDataSeries.DoRegression;
begin
  if not NeedRegreCalc then exit;
  NeedRegreCalc := False;
  case FRegType of
    rg_None: raise Exception.Create('internal error: regression');
    rg_Linear: DoLinearRegression;
    rg_passingBablok: DoPassingBablok;
    rg_RunningAverage: DoRunningAverage;
    rg_Spline: DoSplineRegression;
    rg_dwls: doDWLS;
    rg_quadratic: DoQuadraticRegression;
   else raise Exception.Create('Unknown regression type');
  end;
end;

procedure TDataSeries.CalcStats;
var pp:pgpoint;
    psp,p1,p2:psPoint;
    i:longint;
begin
 Stats.current := true;
 if fData = nil then
   begin
   Stats.Count := 0;
   Stats.Total := 0;
   exit;
   end;
 {the list may have to be ordered by y, not x. So a list is constructed
  and sorted using the same routines as passing-Bablok}
 pp := FData;
 psP := nil;
 i := 0;
 while pp <> nil do
   begin
   inc(i);
   new(p1);
   p1^.next := nil;
   if isxaxis then p1^.x := pp^.xv else p1^.x := pp^.yv;
   if psp = nil then
     begin
     psp := p1;
     p2 := psp;
     end
   else
     begin
     p2^.next := p1;
     p2 := p2^.next;
     end;
   pp := pp^.next;
   end;
 if not isxaxis then sort(psP);
 DoCalcStats(psp,i,Stats);
 empty(psp);
end;

procedure TDataSeries.correlate;
{this code for Pearson Correlation from Numerical Recipes in Pascal}
   function betacf(a,b,x: double): double;
   const
     itmax=100;
     eps=3.0e-7;
   var
     tem,qap,qam,qab,em,d,bz,bpp,bp,bm,az,app,am,aold,ap: double;
     m: integer;
   begin
     am := 1.0;
     bm := 1.0;
     az := 1.0;
     qab := a+b;
     qap := a+1.0;
     qam := a-1.0;
     bz := 1.0-qab*x/qap;
     for m := 1 to itmax do
      begin
      em := m;
      tem := em+em;
      d := em*(b-m)*x/((qam+tem)*(a+tem));
      ap := az+d*am;
      bp := bz+d*bm;
      d := -(a+em)*(qab+em)*x/((a+tem)*(qap+tem));
      app := ap+d*az;
      bpp := bp+d*bz;
      aold := az;
      am := ap/bpp;
      bm := bp/bpp;
      az := app/bpp;
      bz := 1.0;
      if ((abs(az-aold)) < (eps*abs(az))) THEN break;
      end;
     if ((abs(az-aold)) >= (eps*abs(az))) then
        raise exception.create('internal data error in correlation');
     result := az;
   end;
   function gammln(xx: double): double;
   const
     stp = 2.50662827465;
     half = 0.5;
     one = 1.0;
     fpf = 5.5;
   var
    x,tmp,ser: double;
    j: integer;
    cof: array [1..6] of double;
   begin
    cof[1] := 76.18009173;
    cof[2] := -86.50532033;
    cof[3] := 24.01409822;
    cof[4] := -1.231739516;
    cof[5] := 0.120858003e-2;
    cof[6] := -0.536382e-5;
    x := xx-one;
    tmp := x+fpf;
    tmp := (x+half)*ln(tmp)-tmp;
    ser := one;
    for j := 1 to 6 do
      begin
      x := x+one;
      ser := ser+cof[j]/x
      end;
    result := tmp+ln(stp*ser)
   end;
   function betai(a,b,x: double): double;
   var bt: double;
   begin
    if ((x < 0.0) or (x > 1.0)) then
       raise exception.create('Invalid internal data, Correlation');
    bt := exp(gammln(a+b)-gammln(a)-gammln(b) +a*ln(x)+b*ln(1.0-x));
    if (x < ((a+1.0)/(a+b+2.0)))
     then result := bt*betacf(a,b,x)/a
     else result := 1.0-bt*betacf(b,a,1.0-x)/b;
   end;

var yt,xt,t,sumyy,sumxy,sumxx,df,z,zl,zu,YAverage,XAverage: double;
    j:longint;
    pp:pgpoint;
begin
  FCompstats.current := true;
  FCompStats.Error := false;
  if (fdata = nil) or (fdata^.next = nil) or (fdata^.next^.next = nil) then
    begin
    FCompStats.error := true;
    FCompStats.Comment := 'No Correlation available unless at least 3 points exist';
    exit;
    end;
  XAverage := 0.0;
  YAverage := 0.0;
  pp := FData;
  j := 0;
  while pp <> nil do
    begin
    XAverage := XAverage + pp^.xv;
    YAverage := YAverage + pp^.yv;
    inc(j);
    pp := pp^.next;
    end;
  XAverage := XAverage/j;
  YAverage := YAverage/j;
  sumxx := 0.0;
  sumyy := 0.0;
  sumxy := 0.0;
  pp := FData;
  while pp <> nil do
    begin
    xt := pp^.xv-XAverage;
    yt := pp^.yv-YAverage;
    sumxx := sumxx+sqr(xt);
    sumyy := sumyy+sqr(yt);
    sumxy := sumxy+xt*yt;
    pp := pp^.next;
    end;
  with FCompStats do
    begin
    if (sumxx = 0) or (sumyy = 0) then
      begin
      Comment := 'Invalid Correlation due to lack of variance in a variable';
      exit;
      end;
    PearsonR := sumxy/sqrt(sumxx*sumyy);
    RSquared := PearsonR * PearsonR;
    if j < 10 then
      begin
      PRMin := 0;
      PRMax := 0;
      end
    else
      begin
      z := 0.5 * ln((1 + PearsonR) / (1 - PearsonR));
      zl := z - 1.96 / sqrt(j-3);
      zu := z + 1.96 / sqrt(j-3);
      PRMin := (exp(2 * zl) - 1) / (exp(2 * zl) + 1);
      PRMax := (exp(2 * zu) - 1) / (exp(2 * zu) + 1);
      end;
    RSqMin := PRMin * PRMin;
    RSqMax := PRMax * PRMax;
    z := 0.5*ln(((1.0+PearsonR)+tiny)/((1.0-PearsonR)+tiny));
    df := j-2;
    t := PearsonR*sqrt(df/(((1.0-PearsonR)+tiny)*((1.0+PearsonR)+tiny)));
    PValue := betai(0.5*df,0.5,df/(df+sqr(t)))
    end;
end;

function TDataSeries.GetYStats;
begin
 if not FYStats.current then
   calcstats(FYStats,false);
 result := FYStats;
end;

function TDataSeries.GetXStats;
begin
 if not FXStats.current then
  calcstats(FXStats,true);
 result := FXStats;
end;

function TDataSeries.GetCompStats;
begin
 if not FCompStats.current then
   correlate;
 if (FRegType <> rg_none) and NeedRegRecalc then doregression;
 result := FCompStats;
end;

procedure TDataSeries.getintercepts;
  procedure addintercept(x:double);
  var pp:psPoint;
  begin
  inc(count);
  result := ir_found;
  new(pp);
  pp^.next := intercepts;
  pp^.x := x;
  intercepts := pp;
  end;

var p:pgpoint;
    xdiff:double;
begin
 count := 0;
 intercepts := nil;
 if (FRegType = rg_none) or (FRegType = rg_RunningAverage) then
   begin
   result := ir_Invalid;
   exit;
   end;
 if NeedRegRecalc then doregression;
 case FRegType of
  rg_Linear,rg_passingBablok:
     if (FCompStats.RegSlope = 0)
       then result := ir_invalid
       else AddIntercept((y - FCompStats.regintercept) / FCompStats.RegSlope);
  rg_quadratic:
     if (FCompStats.RegSlope = 0) and (FCompStats.RegSlope2 = 0)
       then result := ir_invalid
       else {get intercept};
  rg_Spline,rg_dwls:
     begin
     p := FRegrData;
     if (p = nil) or (p^.next = nil) then
       begin
       result := ir_invalid;
       exit;
       end;
     xdiff := p^.next^.xv - p^.xv;  {in FRegData xdiff is constant}
     while p^.next <> nil do
       begin
       if (p^.rv = y) and (p^.rv = p^.next^.rv)  then
         begin
         result := ir_invalid;
         exit;
         end;
       if ((p^.rv <= y) and (p^.next^.rv > y))
              or ((p^.rv > y) and (p^.next^.rv <= y)) then
         AddIntercept(p^.xv + xdiff * ((y - p^.rv)/(p^.next^.rv - p^.rv)));
       p := p^.next;
    end;
   end;
 end;
 sort(intercepts);
end;

procedure TDataSeries.disposeintercepts(var intercepts:psPoint);
begin
 empty(intercepts);
end;

{$ENDIF} {STATISTICS}


end.
