unit mStrGrid;

{Version 2.2c,  20.2.1997,  PostCardWare, Albrecht Mengel, mengel@stat-econ.uni-kiel.de
 See mStrGrid.txt

If you have problems/new ideas with mStrGrid or mStrList, please feel free to email me!

New: Method ReorderCompleteRows(KeyCol:integer; OrderValues:TStrings);
         Just call with the wished order of the KeyColumn as StringList.
         This sets all columns into that same order.
         Known Problems: No duplicate values allowed in OrderValues nor in the key column
                         OderValues must be a true permutation of the key column
         If UseFixed=hfSmart (default), the fixed rows are excluded from reordering and so
         the number of entries in OrderValues must be the same as the non-fixed entries in
         the grid column.

New: Multi-Key sorting is possible now with SortCompletRows (and SortCompleteColumns):
     First sort the least important column(s), then sort the most important column(s).
     (The order of equal-valued entries is kept).
     As result, the whole table is sorted after the most important column. If any values there
     are equal, the table is sorted by the next less important value, and so on.
     (There is an expample supplied to test this property)

Tip for coloring mStrGrid: see mStrGrid.txt
Tip for sorting by clicking at a title cell: see mStrGrid.txt

New date sorting Properties: DateTimeFormat:string; ShortYearExpand:boolean; ShortYearBorder:(0..99)
     When sorting dates, they are interpreted in the kind as DateTimeFormat is set to.
     Default is the value of (the global) ShortDateFormat+' '+LongTimeFormat.
     See mStrGrid.txt for more description

For ever: Unsolvable problem with Click into fixed cells: The (programmer of the) OnClick procedure
whishes to use COL and ROW to determine which cell was clicked. But the assigment of COL
and ROW to a fixed cell does not work (I cannot change that). To implement the wished behavior
I set temporarily fixedcols/fixedrows to zero, while the OnClick is running.
The published methods of this unit know the original values and work correct. But
You as normal programmer find fixedcols/fixedrows in OnClick to be zero, when clicked a
fixed cell. As cold comfort you may read the properties SavedFixedRows and SavedFixedCols,
which contain the original values (I published them in this version). If you wish to alter
fixedcols/fixedrows in OnClick, just assign new values to SavedFixedCols/SavedFixedRows.
They set fixedcols/fixedrows when OnClick finishes}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Grids, Clipbrd, mStrList;

type
  HandleFixed = (hfNot,hfSmart,hfYes);
  mSortState = (Unsorted,CompleteRows,CompleteCols,SingleCol,SingleRow,AllRows,AllCols);
  TMSortType = (soString,soStringCaseSensitive,soNumeric,soDate);
  TmStrGrid = class(TStringgrid)
  private
    { Private-Deklarationen }
    fKeyType:TMSortType;
    fKeyPos:Integer;
    fKeyLen:Integer;
    fUseFixed:HandleFixed;
    SearchKey:String;
    FirstSearchRow,FirstSearchCol,ActSearchRow,ActSearchCol,LastSearchRow,LastSearchCol:Integer;
    SearchRowWise,SearchIsActive:Boolean;
    fSearchSubstring:Boolean;
    fEnableClipboardShortcuts:Boolean;
    fPasteToCursor:Boolean;
    fClickFixed:Boolean;
    fSortDescending:Boolean;
    fReverseSearch:Boolean;
    fSearchExists:Boolean;
    SortState:mSortState;
    SortArg:integer;
    fSortIndex:TmStrList;
    ColAnfang:integer;                     {Sort complete rows: Which col to begin}
    RowAnfang:integer;                     {Sort complete cols: Which row to begin}
    clicked:boolean;                       {While in Clicking on fixed cells, the fixed rows/cols ...}
    fSavedFixedRows:Integer;               {... must be set to zero. The mStrGrid methods called in OnClick ...}
    fSavedFixedCols:Integer;               {... have to remember the actual fixedrows/fixedcols}
    fDateTimeFormat:String;
    fShortYearExpand:Boolean;
    fShortYearBorder:Integer;
    function GetFixedRows:Integer;
    function GetFixedCols:Integer;
    procedure CalcDrawInfo(var DrawInfo: TGridDrawInfo);
    procedure CalcSizingState(X, Y: Integer; var State: TGridState;
      var Index: Longint; var SizingPos, SizingOfs: Integer;
      var FixedInfo: TGridDrawInfo);
    function CalcCoordFromPoint(X, Y: Integer;
      const DrawInfo: TGridDrawInfo): TGridCoord;
    procedure CalcDrawInfoXY(var DrawInfo: TGridDrawInfo;
      UseWidth, UseHeight: Integer);
    procedure CalcFixedInfo(var DrawInfo: TGridDrawInfo);
    function GetColWidths(Index: Longint): Integer;
    function GetRowHeights(Index: Longint): Integer;
    procedure SetReverseSearch(value:Boolean);
    procedure SetDateTimeFormat(value:String);
    function GetDateTimeFormat:String;
    function GetShortYearExpand:Boolean;
    procedure SetShortYearExpand(value:Boolean);
    function GetShortYearBorder:Integer;
    procedure SetShortYearBorder(value:Integer);
  protected
    { Protected-Deklarationen }
    procedure MouseDown(Button:TMouseButton; Shift:TShiftState; X,Y:Integer); override;
    procedure MouseUp(Button:TMouseButton; Shift:TShiftState; X,Y:Integer); override;
    procedure KeyPress(var Key:Char); override;
    procedure UndoCompRowSort;
    procedure UndoCompColSort;
    procedure UndoRowSort;
    procedure UndoColSort;
    procedure UndoAllRowsSort;
    procedure UndoAllColsSort;
  public
    { Public-Deklarationen }
    constructor Create(AOwner: TComponent);override;
    procedure CutToClipboard;
    procedure CopyToClipboard;
    procedure PasteFromClipboard;
    destructor destroy;override;
  published
    { Published-Deklarationen }
    procedure InsertCols(where,howmuch:Integer);
    procedure InsertRows(where,howmuch:Integer);
    procedure DeleteCols(where,howmuch:Integer);
    procedure DeleteRows(where,howmuch:Integer);
    procedure AddRow(contents:String; delimiter:Char);
    procedure AddCol(contents:String;  delimiter:Char);
    function ModifyRow(which:integer; contents,delimiter:string):integer;
    function ModifyCol(which:integer; contents,delimiter:string):integer;
    function FindFirst(Key:String; RowWise:Boolean; VAR ResultCol,ResultRow:Integer):Boolean;
    function FindFirstInRow(Key:String; searchRow:Integer; VAR ResultCol,ResultRow:Integer):Boolean;
    function FindFirstInCol(Key:String; searchCol:Integer; VAR ResultCol,ResultRow:Integer):Boolean;
    function FindNext(VAR ResultCol,ResultRow:Integer):Boolean;
    procedure SortCompleteColumns(KeyRow:integer);
    procedure SortCompleteRows(KeyCol:integer);
    procedure SortRow(ThisRow:integer);
    procedure SortCol(ThisCol:integer);
    procedure SortAllRows;
    procedure SortAllCols;
    procedure ReorderCompleteRows(KeyCol:integer; OrderValues:TStrings);
    procedure UndoSort;
    procedure LoadFromFile(FileName,ColSep:string);
    procedure SaveToFile(FileName,ColSep:String);
    property KeyType:TMSortType read fKeyType write fKeyType;
    property KeyPos:Integer read fKeyPos write fKeyPos;
    property KeyLen:Integer read fKeyLen write fKeyLen;
    property UseFixed:HandleFixed read fUseFixed write fUseFixed;
    property SearchSubstring:Boolean read fSearchSubstring write fSearchSubstring;
    property EnableClipboardShortcuts:Boolean read fEnableClipboardShortcuts write fEnableClipboardShortcuts;
    property OnKeyPress;
    property OnMouseDown;
    property OnMouseUp;
    property PasteToCursor:Boolean read fPasteToCursor write fPasteToCursor;
    property ClickFixed:Boolean read fClickFixed write fClickFixed;
    property SortDescending:Boolean read fSortDescending write fSortDescending;
    property ReverseSearch:Boolean read fReverseSearch write SetReverseSearch;
    property SearchExists:Boolean read fSearchExists;
    property SortIndex:TmStrList read fSortIndex;
    property SavedFixedRows:Integer read GetFixedRows write fSavedFixedRows;
    property SavedFixedCols:Integer read GetFixedCols write fSavedFixedCols;
    property DateTimeFormat:String read GetDateTimeFormat write SetDateTimeFormat;
    property ShortYearExpand:Boolean read GetShortYearExpand write SetShortYearExpand;
    property ShortYearBorder:Integer read GetShortYearBorder write SetShortYearBorder;
  end;

  MFileError = class(Exception);

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Additional', [TmStrGrid]);
end;

FUNCTION rVAL(CONST von:String):Real;
  {Reads a real value out of the string without an error message.
   Returns 0 if no numeric value}
  VAR bis,err:Integer; nach:Real;
  BEGIN VAL(von,nach,bis);
        IF bis>0 THEN VAL(copy(von,1,bis-1),nach,err);
        rVAL:=nach
  END;{rVAL}

constructor TmStrGrid.Create(AOwner: TComponent);
begin inherited Create(AOwner);
      fKeyType:=soString;
      fKeyPos:=1;
      fKeyLen:=MaxInt;
      clicked:=false;
      fSearchExists:=false;
      SortState:=Unsorted;
      fSortIndex:=TMStrList.create;
      fDateTimeFormat:=fSortIndex.DateTimeFormat;
      fShortYearExpand:=fSortIndex.ShortYearExpand;
      fShortYearBorder:=fSortIndex.ShortYearBorder
end;

destructor TmStrGrid.destroy;
begin fSortIndex.free;
      inherited
end;

function TmStrGrid.GetDateTimeFormat:String;
begin result:=fSortIndex.DateTimeFormat end;

procedure TmStrGrid.SetDateTimeFormat(value:String);
begin if value<>fDateTimeFormat then
       begin fDateTimeFormat:=value;
             fSortIndex.DateTimeFormat:=value
end    end;

function TmStrGrid.GetShortYearExpand:Boolean;
begin result:=fSortIndex.ShortYearExpand end;

procedure TmStrGrid.SetShortYearExpand(value:Boolean);
begin if value<>fShortYearExpand then
      begin fShortYearExpand:=value;
            fSortIndex.ShortYearExpand:=value
end   end;

function TmStrGrid.GetShortYearBorder:Integer;
begin result:=fSortIndex.ShortYearBorder end;

procedure TmStrGrid.SetShortYearBorder(value:Integer);
begin if value<>fShortYearBorder then
      begin fShortYearBorder:=value;
            fSortIndex.ShortYearBorder:=value
end   end;

procedure TmStrGrid.SortCompleteColumns(KeyRow:integer);
{Spalten sortieren nach der Zeile KeyRow}
var i,colanfang:integer;
begin case fUseFixed of
       hfNot  :begin rowanfang:=GetFixedRows; colanfang:=GetFixedCols end;
       hfSmart:begin rowanfang:=0;            colanfang:=GetFixedCols end;
       {hfYes} else  rowanfang:=0;            colanfang:=0
      end;
      fSortIndex.clear;
      fSortIndex.KeyPos:=fKeyPos;
      fSortIndex.KeyLen:=fKeyLen;
      fSortIndex.KeyType:=TLSortType(fKeyType);
      fSortIndex.ScipFirst:=colanfang;
      fSortIndex.SortDescending:=fSortDescending;
      {get key row}
      fSortIndex.addstrings(Rows[KeyRow]);
      {Sortierindizees bilden}
      for i:=0 to colcount-1 do
       fSortIndex.objects[i]:=TObject(i);
      {Liste sortieren}
      fSortIndex.sorted:=true;
      fSortIndex.sorted:=false;
      fSortIndex.RestoreOrderInGroups;
      {Indizees lesbar machen}
      with fSortIndex do
       for i:=0 to count-1 do
        strings[i]:=inttostr(integer(objects[i]));
      {Spalten vertauschen}
      UndoCompColSort;
      SortState:=CompleteCols;
end;

procedure TmStrGrid.UndoCompColSort;
var puffer,index:TStringList; i,j,k:integer;
begin index:=TStringList.create;
      index.addstrings(fSortIndex);
      {Resort the Sort Index, so that the index leads to unsort}
      with Index do
      for i:=0 to count-1 do
        begin fSortIndex.objects[integer(objects[i])]:=TObject(i);
              fSortIndex.strings[integer(objects[i])]:=inttostr(i);
        end;
      puffer:=TStringList.create;
      repeat {Suche einen Zyklus}
             i:=0;
             with Index do
             begin while (i<count) and (i=integer(objects[i])) do inc(i);
                   if i=count then break;
                   {Tausche Zyklus}
                   puffer.clear;
                   puffer.addstrings(cols[i]);
                   repeat j:=Integer(objects[i]);
                          objects[i]:=TObject(i);
                          if integer(objects[j])=j then break;
                          for k:=rowanfang to RowCount-1 do
                            cells[i,k]:=cells[j,k];
                          i:=j;
                   until false;
             end;
             for k:=rowanfang to RowCount-1 do
               begin cells[i,k]:=puffer.strings[k];
                     objects[i,k]:=puffer.objects[k]
               end;
      until false;
      puffer.free
end;

procedure TmStrGrid.SortCompleteRows(KeyCol:integer);
{Zeilen sortieren nach der Spalte KeyCol}
var i,rowanfang:integer;
begin case fUseFixed of
       hfNot  :begin rowanfang:=GetFixedRows; colanfang:=GetFixedCols end;
       hfSmart:begin rowanfang:=GetFixedRows; colanfang:=0 end;
       {hfYes} else  rowanfang:=0;            colanfang:=0
      end;
      fSortIndex.clear;
      fSortIndex.KeyPos:=fKeyPos;
      fSortIndex.KeyLen:=fKeyLen;
      fSortIndex.KeyType:=TLSortType(fKeyType);
      fSortIndex.ScipFirst:=rowanfang;
      fSortIndex.SortDescending:=fSortDescending;
      {get key col}
      fSortIndex.addstrings(cols[KeyCol]);
      {Sortierindizees bilden}
      for i:=0 to rowcount-1 do
       fSortIndex.objects[i]:=TObject(i);
      {Liste sortieren}
      fSortIndex.sorted:=true;
      fSortIndex.sorted:=false;
      fSortIndex.RestoreOrderInGroups;
      {Indizees lesbar machen}
      with fSortIndex do
       for i:=0 to count-1 do
        strings[i]:=inttostr(integer(objects[i]));
      {Zeilen vertauschen}
      UndoCompRowSort;
      SortState:=CompleteRows;
end;

procedure TmStrGrid.UndoCompRowSort;
var puffer,index:TStringList; i,j,k:integer;
begin index:=TStringList.create;
      index.addstrings(fSortIndex);
      {Resort the Sort Index, so that the index leads to unsort}
      with index do
      for i:=0 to count-1 do
        begin fSortIndex.objects[integer(objects[i])]:=TObject(i);
              fSortIndex.strings[integer(objects[i])]:=inttostr(i);
        end;
      puffer:=TStringList.create;
      repeat {Suche einen Zyklus}
             i:=0;
             with index do
             begin while (i<count) and (i=integer(objects[i])) do inc(i);
                   if i=count then break;
                   {Tausche Zyklus}
                   puffer.clear;
                   puffer.addstrings(rows[i]);
                   repeat j:=Integer(objects[i]);
                          objects[i]:=TObject(i);
                          if integer(objects[j])=j then break;
                          for k:=ColAnfang to ColCount-1 do
                            cells[k,i]:=cells[k,j];
                          i:=j;
                   until false;
             end;
             for k:=ColAnfang to ColCount-1 do
               begin cells[k,i]:=puffer.strings[k];
                     objects[k,i]:=puffer.objects[k]
               end;
      until false;
      puffer.free
end;

procedure TmStrGrid.ReorderCompleteRows(KeyCol:integer; OrderValues:TStrings);
{Zeilen umsortieren nach der Spalte KeyCol}
var i,rowanfang:integer;
begin case fUseFixed of
       hfNot  :begin rowanfang:=GetFixedRows; colanfang:=GetFixedCols end;
       hfSmart:begin rowanfang:=GetFixedRows; colanfang:=0 end;
       {hfYes} else  rowanfang:=0;            colanfang:=0
      end;
      if OrderValues.count<>RowCount-rowanfang then
        raise MFileError.Create('Incorrect list length of Order Values');
      with fSortIndex do
      begin clear;
            {get key col}
            addstrings(cols[KeyCol]);
            {Sortierindizees fuer fixed}
            for i:=0 to rowanfang-1 do
             objects[i]:=TObject(i);
            {Sortierindizees bilden durch Umordnung}
            for i:=rowanfang to RowCount-1 do
              objects[i]:=TObject(IndexOf(OrderValues.strings[i-rowanfang]));
      end;
      {Zeilen vertauschen}
      UndoCompRowSort;
      SortState:=CompleteRows;
end;

procedure TmStrGrid.SortRow(ThisRow:integer);
var i,colanfang:integer;
begin case fUseFixed of
       hfNot  :colanfang:=GetFixedCols;
       hfSmart:colanfang:=GetFixedCols; else
       {hfYes} colanfang:=0
      end;
      SortArg:=ThisRow;
      fSortIndex.clear;
      fSortIndex.KeyPos:=fKeyPos;
      fSortIndex.KeyLen:=fKeyLen;
      fSortIndex.KeyType:=TLSortType(fKeyType);
      fSortIndex.ScipFirst:=colanfang;
      fSortIndex.SortDescending:=fSortDescending;
      {Go!}
      fSortIndex.addstrings(Rows[ThisRow]);
      {Sortierindizees bilden}
      for i:=0 to colcount-1 do
       fSortIndex.objects[i]:=TObject(i);
      {Liste sortieren}
      fSortIndex.sorted:=true;
      fSortIndex.sorted:=false;
      {Indizees lesbar machen}
      with fSortIndex do
       for i:=0 to count-1 do
        strings[i]:=inttostr(integer(objects[i]));
      {Werte vertauschen}
      UndoRowSort;
      SortState:=SingleRow;
end;

procedure TmStrGrid.UndoRowSort;
var index:TStringList; i,j:integer; puffer:String; ptr:TObject;
begin index:=TStringList.create;
      index.addstrings(fSortIndex);
      {Resort the Sort Index, so that the index leads to unsort}
      with Index do
      for i:=0 to count-1 do
        begin fSortIndex.objects[integer(objects[i])]:=TObject(i);
              fSortIndex.strings[integer(objects[i])]:=inttostr(i);
        end;
      repeat {Suche einen Zyklus}
             i:=0;
             with Index do
             while (i<count) and (i=integer(objects[i])) do inc(i);
             if i=Index.count then break;
             {Tausche Zyklus}
             puffer:=cells[i,SortArg]; ptr:=objects[i,SortArg];
             with index do
             repeat j:=Integer(objects[i]);
                    objects[i]:=TObject(i);
                    if integer(objects[j])=j then break;
                    cells[i,SortArg]:=cells[j,SortArg];
                    i:=j;
             until false;
             cells[i,SortArg]:=puffer;
             objects[i,SortArg]:=ptr
      until false;
end;

procedure TmStrGrid.SortCol(ThisCol:integer);
var i,rowanfang:integer;
begin case fUseFixed of
       hfNot  :rowanfang:=GetFixedRows;
       hfSmart:rowanfang:=GetFixedRows; else
       {hfYes} rowanfang:=0;
      end;
      SortArg:=ThisCol;
      fSortIndex.clear;
      fSortIndex.KeyPos:=fKeyPos;
      fSortIndex.KeyLen:=fKeyLen;
      fSortIndex.KeyType:=TLSortType(fKeyType);
      fSortIndex.ScipFirst:=rowanfang;
      fSortIndex.SortDescending:=fSortDescending;
      {Go!}
      fSortIndex.addstrings(Cols[ThisCol]);
      {Sortierindizees bilden}
      for i:=0 to rowcount-1 do
       fSortIndex.objects[i]:=TObject(i);
      {Liste sortieren}
      fSortIndex.sorted:=true;
      fSortIndex.sorted:=false;
      {Indizees lesbar machen}
      with fSortIndex do
       for i:=0 to count-1 do
        strings[i]:=inttostr(integer(objects[i]));
      {Werte vertauschen}
      UndoColSort;
      SortState:=SingleCol;
end;

procedure TmStrGrid.UndoColSort;
var index:TStringList; i,j:integer; puffer:String; ptr:TObject;
begin index:=TStringList.create;
      index.addstrings(fSortIndex);
      {Resort the Sort Index, so that the index leads to unsort}
      with Index do
      for i:=0 to count-1 do
        begin fSortIndex.objects[integer(objects[i])]:=TObject(i);
              fSortIndex.strings[integer(objects[i])]:=inttostr(i);
        end;
      repeat {Suche einen Zyklus}
             i:=0;
             with Index do
             while (i<count) and (i=integer(objects[i])) do inc(i);
             if i=Index.count then break;
             {Tausche Zyklus}
             puffer:=cells[SortArg,i]; ptr:=objects[SortArg,i];
             with index do
             repeat j:=Integer(objects[i]);
                    objects[i]:=TObject(i);
                    if integer(objects[j])=j then break;
                    cells[SortArg,i]:=cells[SortArg,j];
                    i:=j;
             until false;
             cells[SortArg,i]:=puffer;
             objects[SortArg,i]:=ptr
      until false;
end;

procedure TmStrGrid.SortAllRows;
{Sorting all rows independent}
var i,r,colanfang:integer; index:TmStrList;
begin case fUseFixed of
       hfNot  :begin rowanfang:=GetFixedRows; colanfang:=GetFixedCols end;
       hfSmart:begin rowanfang:=0;            colanfang:=GetFixedCols end;
       {hfYes} else  rowanfang:=0;            colanfang:=0
      end;
      fSortIndex.clear;
      index:=TmStrList.create;
      with index do
      begin KeyPos:=fKeyPos;
            KeyLen:=fKeyLen;
            KeyType:=TLSortType(fKeyType);
            ScipFirst:=colanfang;
            SortDescending:=fSortDescending;
            for r:=Rowanfang to RowCount-1 do
            begin clear;
                  addstrings(Rows[r]);
                  {Sortierindizees bilden}
                  for i:=0 to colcount-1 do
                   objects[i]:=TObject(i);
                  {Liste sortieren}
                  sorted:=true;
                  sorted:=false;
                  {Sortierindizees speichern}
                  for i:=0 to count-1 do
                   fSortIndex.addobject(inttostr(integer(objects[i])),objects[i]);
            end;
      SortState:=AllRows;
      UndoAllRowsSort;
end   end;

procedure TmStrGrid.UndoAllRowsSort;
var fullindex,index:TStringList; i,j,aktrow,offset:integer; puffer:String; ptr:TObject;
begin fullindex:=TStringList.create;
      fullindex.addstrings(fSortIndex);
      index:=TStringList.create;
      offset:=0;
      for aktrow:=Rowanfang to RowCount-1 do
      begin with index do
            begin {get the indizees of the row i}
                  clear;
                  for i:=0 to Colcount-1 do
                    Index.addobject('',fullindex.objects[offset+i]);
                  {Resort the Sort Index, so that the index leads to unsort}
                  for i:=0 to count-1 do
                    begin fSortIndex.objects[offset+integer(objects[i])]:=TObject(i);
                          fSortIndex.strings[offset+integer(objects[i])]:=inttostr(i);
                    end;
                  offset:=offset+count;
            end;
            repeat {Suche einen Zyklus}
                   i:=0;
                   with Index do
                   while (i<count) and (i=integer(objects[i])) do inc(i);
                   if i=Index.count then break;
                   {Tausche Zyklus}
                   puffer:=cells[i,aktrow]; ptr:=objects[i,aktrow];
                   with index do
                   repeat j:=Integer(objects[i]);
                          objects[i]:=TObject(i);
                          if integer(objects[j])=j then break;
                          cells[i,aktrow]:=cells[j,aktrow];
                          i:=j;
                   until false;
                   cells[i,aktrow]:=puffer;
                   objects[i,aktrow]:=ptr
            until false;
end   end;

procedure TmStrGrid.SortAllCols;
{Sort all cols independend}
var i,c,rowanfang:integer; index:TmStrList;
begin case fUseFixed of
       hfNot  :begin rowanfang:=GetFixedRows; colanfang:=GetFixedCols end;
       hfSmart:begin rowanfang:=GetFixedRows; colanfang:=0 end;
       {hfYes} else  rowanfang:=0;            colanfang:=0
      end;
      fSortIndex.clear;
      index:=TmStrList.create;
      with index do
      begin KeyPos:=fKeyPos;
            KeyLen:=fKeyLen;
            KeyType:=TLSortType(fKeyType);
            ScipFirst:=rowanfang;
            SortDescending:=fSortDescending;
            for c:=colanfang to ColCount-1 do
            begin clear;
                  addstrings(Cols[c]);
                  {Sortierindizees bilden}
                  for i:=0 to Rowcount-1 do
                   objects[i]:=TObject(i);
                  {Liste sortieren}
                  sorted:=true;
                  sorted:=false;
                  {Sortierindizees speichern}
                  for i:=0 to count-1 do
                   fSortIndex.addobject(inttostr(integer(objects[i])),objects[i]);
            end;
      SortState:=AllCols;
      UndoAllColsSort;
end   end;

procedure TmStrGrid.UndoAllColsSort;
var fullindex,index:TStringList; i,j,aktcol,offset:integer; puffer:String; ptr:TObject;
begin fullindex:=TStringList.create;
      fullindex.addstrings(fSortIndex);
      index:=TStringList.create;
      offset:=0;
      for aktcol:=Colanfang to ColCount-1 do
      begin with index do
            begin {get the indizees of the col i}
                  clear;
                  for i:=0 to Rowcount-1 do
                    Index.addobject('',fullindex.objects[offset+i]);
                  {Resort the Sort Index, so that the index leads to unsort}
                  for i:=0 to count-1 do
                    begin fSortIndex.objects[offset+integer(objects[i])]:=TObject(i);
                          fSortIndex.strings[offset+integer(objects[i])]:=inttostr(i);
                    end;
                  offset:=offset+count;
            end;
            repeat {Suche einen Zyklus}
                   i:=0;
                   with Index do
                   while (i<count) and (i=integer(objects[i])) do inc(i);
                   if i=Index.count then break;
                   {Tausche Zyklus}
                   puffer:=cells[aktcol,i]; ptr:=objects[aktcol,i];
                   with index do
                   repeat j:=Integer(objects[i]);
                          objects[i]:=TObject(i);
                          if integer(objects[j])=j then break;
                          cells[aktcol,i]:=cells[aktcol,j];
                          i:=j;
                   until false;
                   cells[aktcol,i]:=puffer;
                   objects[aktcol,i]:=ptr
            until false;
end   end;

procedure TmStrGrid.UndoSort;
begin case SortState of
      CompleteRows:UndoCompRowSort;
      CompleteCols:UndoCompColSort;
      SingleCol:UndoColSort;
      SingleRow:UndoRowSort;
      AllRows:UndoAllRowsSort;
      AllCols:UndoAllColsSort;
end   end;

function TmStrGrid.ModifyRow(which:integer; contents,delimiter:string):integer;
var p,c,len:integer;
begin c:=0;
      len:=length(delimiter)-1;
      repeat p:=pos(delimiter,contents);
             if p=0 then break;
             cells[c,which]:=copy(contents,1,p-1);
             system.delete(contents,1,p+len);
             inc(c)
      until false;
      cells[c,which]:=contents;
      ModifyRow:=c+1
end;

function TmStrGrid.ModifyCol(which:integer; contents,delimiter:string):integer;
var p,r,len:integer;
begin r:=0;
      len:=length(delimiter)-1;
      repeat p:=pos(delimiter,contents);
             if p=0 then break;
             cells[which,r]:=copy(contents,1,p-1);
             system.delete(contents,1,p+len);
             inc(r)
      until false;
      cells[which,r]:=contents;
      ModifyCol:=r+1
end;

procedure TmStrGrid.LoadFromFile(FileName,ColSep:string);
var f:textfile; zeile:string; MaxCol,z,s:integer;
begin assignfile(f,FileName);
      {$I-}reset(f);{$I+}
      if IOResult <> 0 then
        raise MFileError.Create('File '+Filename+' not found');
      z:=-1;
      MaxCol:=0;
      while not eof(f) do
        begin readln(f,zeile);
              if zeile='' then continue;
              inc(z);
              if z>GetFixedRows then RowCount:=z+1;
              s:=ModifyRow(z,zeile,ColSep);
              if s>MaxCol then MaxCol:=s;
        end;
      RowCount:=z+1;
      ColCount:=MaxCol;
      closefile(f)
end;

procedure TmStrGrid.SaveToFile(FileName,ColSep:String);
var f:textfile; z,s:integer;
begin assignfile(f,FileName);
      rewrite(f);
      for z:=0 to RowCount-1 do
        begin for s:=0 to ColCount-1 do
                begin if s>0 then write(f,ColSep);
                      write(f,cells[s,z]);
                end;
              writeln(f)
        end;
      closefile(f)
end;

procedure TmStrGrid.CutToClipboard;
var s,z:integer;
begin CopyToClipboard;
with selection do
begin for s:=Left to Right do
       for z:=Top to Bottom do
        cells[s,z]:=''
end end;

procedure TmStrGrid.CopyToClipboard;
var s,z:integer; len:integer; puffer,ptr:PChar;
begin with selection do
begin {Get Length}
      len:=1; {incl #0}
      for z:=Top to Bottom do
       begin for s:=Left to Right do
              len:=len+length(cells[s,z])+1;{tab}
             len:=len+1{-tab+cr&lf}
       end;
      puffer:=nil; {Just to prevent useless warning}
      try GetMem(puffer,len);
      {fill buffer}
      puffer^:=#0;
      ptr:=puffer;
      for z:=Top to Bottom do
       begin for s:=Left to Right do
               ptr:=StrEnd(StrPCopy(ptr,cells[s,z]+#9));
             dec(ptr); {Aufs Tab gehen}
             ptr:=StrEnd(StrPCopy(ptr,#13+#10)); {cr & lf}
       end;
      ptr:=ptr-2; {Vor das letzte Cr}
      ptr^:=#0; {Weg damit}
      ClipBoard.SetTextBuf(puffer)
      finally FreeMem(puffer,len);
      end;
end end;

procedure TmStrGrid.PasteFromClipboard;
var Inhalt,Ende,Cr,Tab:PChar;
    Eintrag:Array[0..255] of char;
    s,z:integer;
    Data: THandle;
begin if not Clipboard.HasFormat(CF_TEXT) then exit;
      Clipboard.Open;
      Data:=GetClipboardData(CF_TEXT);
      try if Data<>0
           then inhalt:=PChar(GlobalLock(Data))
           else inhalt:=nil
      finally
           if Data<>0 then GlobalUnlock(Data);
           ClipBoard.Close;
      end;
    if inhalt=nil then exit;
    if fPasteToCursor
    then begin z:=row;
               s:=col
         end
    else begin z:=selection.Top;
               s:=selection.Left
         end;
    Ende:=StrScan(Inhalt,#0);
    repeat {Zeile holen}
           Cr:=StrScan(inhalt,#13);
           if Cr=nil then Cr:=Ende;
           repeat Tab:=StrScan(inhalt,#9);
                  if (Tab>Cr) or (Tab=nil) then Tab:=Cr;
                  StrLCopy(Eintrag,Inhalt,Tab-Inhalt);
                  cells[s,z]:=StrPas(Eintrag);
                  inc(s);
                  if s=colCount then ColCount:=s;
                  Inhalt:=Tab+1
           until Tab=Cr;
           inc(Inhalt); {lf weg}
           if fPasteToCursor
           then s:=col
           else s:=selection.Left;
           inc(z);
           if z=RowCount then RowCount:=z+1
    until Cr=Ende
end;

procedure TmStrGrid.InsertCols(where,howmuch:Integer);
var i:Integer;
begin ColCount:=ColCount+howmuch;
      for i:=ColCount-1 downto where+howmuch do
        Cols[i]:=Cols[i-howmuch];
      cols[where].clear;
      for i:=where+1 to where+howmuch-1 do
        Cols[i]:=Cols[where];
end;

procedure TmStrGrid.InsertRows(where,howmuch:Integer);
var i:Integer;
begin RowCount:=RowCount+howmuch;
      for i:=RowCount-1 downto where+howmuch do
        Rows[i]:=Rows[i-howmuch];
      Rows[where].clear;
      for i:=where+1 to where+howmuch-1 do
        Rows[i]:=Rows[where];
end;

procedure TmStrGrid.DeleteCols(where,howmuch:Integer);
var i:Integer;
begin if ColCount-where<howmuch then howmuch:=ColCount-where;
      for i:=where to ColCount-1 do
        Cols[i]:=Cols[i+howmuch];
      ColCount:=ColCount-howmuch;
end;

procedure TmStrGrid.DeleteRows(where,howmuch:Integer);
var i:Integer;
begin if RowCount-where<howmuch then howmuch:=RowCount-where;
      for i:=where to RowCount-1 do
        Rows[i]:=Rows[i+howmuch];
      RowCount:=RowCount-howmuch;
end;

procedure TmStrGrid.AddRow(contents:String; delimiter:Char);
begin RowCount:=RowCount+1;
      ModifyRow(RowCount-1,contents,delimiter);
end;

procedure TmStrGrid.AddCol(contents:String;  delimiter:Char);
begin ColCount:=ColCount+1;
      ModifyCol(ColCount-1,contents,delimiter);
end;

function TmStrGrid.FindFirst(Key:String; RowWise:Boolean; VAR ResultCol,ResultRow:Integer):Boolean;
begin SearchKey:=Key; SearchRowWise:=RowWise; SearchIsActive:=true;
      case fUseFixed of
       hfNot:begin FirstSearchRow:=GetFixedRows; FirstSearchCol:=GetFixedCols end;
       hfSmart:if rowwise
               then begin FirstSearchRow:=GetFixedRows; FirstSearchCol:=0; end
               else begin FirstSearchRow:=0; FirstSearchCol:=GetFixedCols; end;
       hfYes:begin FirstSearchRow:=0; FirstSearchCol:=0 end;
      end;
      LastSearchRow:=RowCount-1;
      LastSearchCol:=ColCount-1;
      if fReverseSearch
      then begin ActSearchRow:=LastSearchRow;
                 ActSearchCol:=LastSearchCol
           end
      else begin ActSearchRow:=FirstSearchRow;
                 ActSearchCol:=FirstSearchCol
           end;
      fSearchExists:=true;
      FindFirst:=FindNext(ResultCol,ResultRow);
end;

function TmStrGrid.FindFirstInRow(Key:String; searchRow:Integer; VAR ResultCol,ResultRow:Integer):Boolean;
begin SearchKey:=Key; SearchRowWise:=true; SearchIsActive:=true;
      FirstSearchRow:=searchRow; ActSearchRow:=searchRow; LastSearchRow:=searchRow;
      if fUseFixed=hfNot
      then FirstSearchCol:=GetFixedCols
      else FirstSearchCol:=0;
      if fReverseSearch
      then ActSearchCol:=LastSearchCol
      else ActSearchCol:=FirstSearchCol;
      LastSearchCol:=ColCount-1;
      fSearchExists:=true;
      FindFirstInRow:=FindNext(ResultCol,ResultRow)
end;

function TmStrGrid.FindFirstInCol(Key:String; searchCol:Integer; VAR ResultCol,ResultRow:Integer):Boolean;
begin SearchKey:=Key; SearchRowWise:=false; SearchIsActive:=true;
      FirstSearchCol:=searchCol; ActSearchCol:=searchCol; LastSearchCol:=searchCol;
      if fUseFixed=hfNot
      then FirstSearchRow:=GetFixedRows
      else FirstSearchRow:=0;
      if fReverseSearch
      then ActSearchRow:=LastSearchRow
      else ActSearchRow:=FirstSearchRow;
      LastSearchRow:=RowCount-1;
      fSearchExists:=true;
      FindFirstInCol:=FindNext(ResultCol,ResultRow);
end;

function TmStrGrid.FindNext(VAR ResultCol,ResultRow:Integer):Boolean;
var SearchReal:Real; SearchDate:TDateTime; found:boolean;
  function next:boolean;
  begin next:=false;
        if fReverseSearch
        then if SearchRowWise
             then begin dec(ActSearchCol);
                        if ActSearchCol<FirstSearchCol then
                           begin ActSearchCol:=LastSearchCol;
                                 dec(ActSearchRow);
                                 if ActSearchRow<FirstSearchRow then exit
                  end      end
             else begin dec(ActSearchRow);
                        if ActSearchRow<FirstSearchRow then
                           begin ActSearchRow:=LastSearchRow;
                                 dec(ActSearchCol);
                                 if ActSearchCol<FirstSearchCol then exit
                  end      end
        else if SearchRowWise
             then begin inc(ActSearchCol);
                        if ActSearchCol>LastSearchCol then
                           begin ActSearchCol:=FirstSearchCol;
                                 inc(ActSearchRow);
                                 if ActSearchRow>LastSearchRow then exit
                  end      end
             else begin inc(ActSearchRow);
                        if ActSearchRow>LastSearchRow then
                           begin ActSearchRow:=FirstSearchRow;
                                 inc(ActSearchCol);
                                 if ActSearchCol>LastSearchCol then exit
                  end      end;
        next:=true
  end;
begin if not SearchIsActive then
        begin FindNext:=false;
              ResultRow:=-1; ResultCol:=-1;
              exit
        end;
      SearchReal:=0; SearchDate:=0;
     {Suchwort konvertieren}
      case fKeyType of
       soNumeric:SearchReal:=Rval(SearchKey);
       soString:if fSearchSubstring then SearchKey:=UpperCase(SearchKey);
       soDate:SearchDate:=StrToDateTime(SearchKey);
      end;
      found:=false;
      if fSearchSubstring then
      repeat {Suche Passenden}
             case fKeyType of
              soString:if pos(SearchKey,UpperCase(cells[ActSearchCol,ActSearchRow]))>0 then
                        begin found:=true; break end;
              soStringCaseSensitive:if pos(SearchKey,cells[ActSearchCol,ActSearchRow])>0 then
                        begin found:=true; break end;
              soNumeric:if Rval(copy(cells[ActSearchCol,ActSearchRow],fKeyPos,fKeyLen))=SearchReal then
                        begin found:=true; break end;
              soDate:if StrToDateTime(copy(cells[ActSearchCol,ActSearchRow],fKeyPos,fKeyLen))=SearchDate then
                        begin found:=true; break end;
             end;
      until found or not next
      else
      repeat {Suche Passenden}
             case fKeyType of
              soString:if AnsiCompareText(copy(cells[ActSearchCol,ActSearchRow],fKeyPos,fKeyLen),SearchKey)=0 then
                        begin found:=true; break end;
              soStringCaseSensitive:if copy(cells[ActSearchCol,ActSearchRow],fKeyPos,fKeyLen)=SearchKey then
                        begin found:=true; break end;
              soNumeric:if Rval(copy(cells[ActSearchCol,ActSearchRow],fKeyPos,fKeyLen))=SearchReal then
                        begin found:=true; break end;
              soDate:if StrToDateTime(copy(cells[ActSearchCol,ActSearchRow],fKeyPos,fKeyLen))=SearchDate then
                        begin found:=true; break end;
             end;
      until found or not next;
      if found
       then begin ResultRow:=ActSearchRow; ResultCol:=ActSearchCol;
                  SearchIsActive:=next
            end
       else begin SearchIsActive:=false;
                  ResultRow:=-1; ResultCol:=-1
            end;
      FindNext:=found;
end;

procedure TmStrGrid.SetReverseSearch(value:Boolean);
var dummycol,dummyrow:integer;
begin if value<>fReverseSearch then
      begin fReverseSearch:=value;
            if SearchIsActive
            then findnext(dummycol,dummyrow)
            else if fSearchExists then
                   begin SearchIsActive:=true;
                         if fReverseSearch
                         then begin ActSearchRow:=LastSearchRow;
                                    ActSearchCol:=LastSearchCol
                              end
                         else begin ActSearchRow:=FirstSearchRow;
                                    ActSearchCol:=FirstSearchCol
end   end          end        end;

procedure TmStrGrid.KeyPress(var Key: Char);
begin if NOT EditorMode and EnableClipboardShortcuts then
      case key of
       ^c:CopyToClipboard;
       ^v:PasteFromClipboard;
       ^x:CutToClipboard;
      end;
      inherited KeyPress(key);
end;

{All following procedures/functions implement a new click behaviour:
 MouseDown is heavy modified, MouseUp only a little, and the rest is just copied from the
 original GRID.PAS}

function TmStrGrid.GetFixedRows:Integer;
{This is to get the correct Fixedcols/rows value if the user click procedure calls mStrGrid methods}
begin if clicked then result:=fSavedFixedRows
                 else result:=FixedRows
end;

function TmStrGrid.GetFixedCols:Integer;
{This is to get the correct Fixedcols/rows value if the user click procedure calls mStrGrid methods}
begin if clicked then result:=fSavedFixedCols
                 else result:=FixedCols
end;

procedure TmStrGrid.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{This is to get the correct Fixedcols/rows value if the user click procedure calls mStrGrid methods}
begin clicked:=false;
      inherited;
end;

{This is the only modified procedure from TCustomGrid}

procedure TmStrGrid.MouseDown(Button:TMouseButton; Shift:TShiftState; X,Y:Integer);
var
  CellHit: TGridCoord;
  DrawInfo: TGridDrawInfo;
  DummySizingIndex:LongInt;
  DummySizingPos,DummySizingOfs:integer;
begin
  if not (csDesigning in ComponentState) and CanFocus then
  begin
    SetFocus;
    if ValidParentForm(Self).ActiveControl <> Self then
    begin
      inherited;
      Exit;
    end;
  end;
  if (Button = mbLeft) and (ssDouble in Shift) then
    DblClick
  else if Button = mbLeft then
  begin
    CalcDrawInfo(DrawInfo);
    { Check grid sizing }
    CalcSizingState(X, Y, FGridState, DummySizingIndex, DummySizingPos, DummySizingOfs,
      DrawInfo);
    if FGridState <> gsNormal then
    begin
      inherited;
      Exit;
    end;
    CellHit := CalcCoordFromPoint(X, Y, DrawInfo);
    with CellHit do
    if fClickFixed and
       (((X < FixedCols) and (X >=0)) or
        ((Y < FixedRows) and (Y >=0)))
    then begin {Alter Fixed Col/Row Barrier}
               {This is to get the correct Fixedcols/rows value
                if the user click procedure calls mStrGrid methods}
               fSavedFixedRows:=FixedRows;
               fSavedFixedCols:=FixedCols;
               FixedRows:=0;
               FixedCols:=0;
               clicked:=true;
               {The following two assignments call in the three stages inherited CustomGrid
                the methods SetCol which calls FocusCell which calls MoveCurrent.
                As it is not allowed to set COL or ROW into the fixed cells,
                the assignment is not done, if Y <=FixedRows (the same with X.)}
               row:=Y;
               col:=X;
               FixedRows:=fSavedFixedRows;
               FixedCols:=fSavedFixedCols;
         end
    else inherited
  end
end;

{The following procedures/functions are just copied from TCustomGrid}

procedure TmStrGrid.CalcDrawInfoXY(var DrawInfo: TGridDrawInfo;
  UseWidth, UseHeight: Integer);
  procedure CalcAxis(var AxisInfo: TGridAxisDrawInfo; UseExtent: Integer);
  var
    I: Integer;
  begin
    with AxisInfo do
    begin
      GridExtent := UseExtent;
      GridBoundary := FixedBoundary;
      FullVisBoundary := FixedBoundary;
      LastFullVisibleCell := FirstGridCell;
      for I := FirstGridCell to GridCellCount - 1 do
      begin
        Inc(GridBoundary, GetExtent(I) + EffectiveLineWidth);
        if GridBoundary > GridExtent + EffectiveLineWidth then
        begin
          GridBoundary := GridExtent;
          Break;
        end;
        LastFullVisibleCell := I;
        FullVisBoundary := GridBoundary;
      end;
    end;
  end;

begin
  CalcFixedInfo(DrawInfo);
  CalcAxis(DrawInfo.Horz, UseWidth);
  CalcAxis(DrawInfo.Vert, UseHeight);
end;

procedure TmStrGrid.CalcDrawInfo(var DrawInfo: TGridDrawInfo);
begin
  CalcDrawInfoXY(DrawInfo, ClientWidth, ClientHeight);
end;


function TmStrGrid.CalcCoordFromPoint(X, Y: Integer;
  const DrawInfo: TGridDrawInfo): TGridCoord;

  function DoCalc(const AxisInfo: TGridAxisDrawInfo; N: Integer): Integer;
  var
    I, Start, Stop: Longint;
    Line: Integer;
  begin
    with AxisInfo do
    begin
      if N < FixedBoundary then
      begin
        Start := 0;
        Stop :=  FixedCellCount - 1;
        Line := 0;
      end
      else
      begin
        Start := FirstGridCell;
        Stop := GridCellCount - 1;
        Line := FixedBoundary;
      end;
      Result := -1;
      for I := Start to Stop do
      begin
        Inc(Line, GetExtent(I) + EffectiveLineWidth);
        if N < Line then
        begin
          Result := I;
          Exit;
        end;
      end;
    end;
  end;

begin
  Result.X := DoCalc(DrawInfo.Horz, X);
  Result.Y := DoCalc(DrawInfo.Vert, Y);
end;

procedure TmStrGrid.CalcFixedInfo(var DrawInfo: TGridDrawInfo);

  procedure CalcFixedAxis(var Axis: TGridAxisDrawInfo; LineOptions: TGridOptions;
    FixedCount, FirstCell, CellCount: Integer; GetExtentFunc: TGetExtentsFunc);
  var
    I: Integer;
  begin
    with Axis do
    begin
      if LineOptions * Options = [] then
        EffectiveLineWidth := 0
      else
        EffectiveLineWidth := GridLineWidth;

      FixedBoundary := 0;
      for I := 0 to FixedCount - 1 do
        Inc(FixedBoundary, GetExtentFunc(I) + EffectiveLineWidth);

      FixedCellCount := FixedCount;
      FirstGridCell := FirstCell;
      GridCellCount := CellCount;
      GetExtent := GetExtentFunc;
    end;
  end;

begin
  CalcFixedAxis(DrawInfo.Horz, [goFixedVertLine, goVertLine], FixedCols,
    LeftCol, ColCount, GetColWidths);
  CalcFixedAxis(DrawInfo.Vert, [goFixedHorzLine, goHorzLine], FixedRows,
    TopRow, RowCount, GetRowHeights);
end;

function TmStrGrid.GetColWidths(Index: Longint): Integer;
begin Result:=ColWidths[index]
end;

function TmStrGrid.GetRowHeights(Index: Longint): Integer;
begin Result:=RowHeights[index]
end;

procedure TmStrGrid.CalcSizingState(X, Y: Integer; var State: TGridState;
  var Index: Longint; var SizingPos, SizingOfs: Integer;
  var FixedInfo: TGridDrawInfo);

  procedure CalcAxisState(const AxisInfo: TGridAxisDrawInfo; Pos: Integer;
    NewState: TGridState);
  var
    I, Line, Back, Range: Integer;
  begin
    with AxisInfo do
    begin
      Line := FixedBoundary;
      Range := EffectiveLineWidth;
      Back := 0;
      if Range < 7 then
      begin
        Range := 7;
        Back := (Range - EffectiveLineWidth) shr 1;
      end;
      for I := FirstGridCell to GridCellCount - 1 do
      begin
        Inc(Line, GetExtent(I));
        if Line > GridExtent then Break;
        if (Pos >= Line - Back) and (Pos <= Line - Back + Range) then
        begin
          State := NewState;
          SizingPos := Line;
          SizingOfs := Line - Pos;
          Index := I;
          Exit;
        end;
        Inc(Line, EffectiveLineWidth);
      end;
      if (Pos >= GridExtent - Back) and (Pos <= GridExtent) then
      begin
        State := NewState;
        SizingPos := GridExtent;
        SizingOfs := GridExtent - Pos;
        Index := I;
      end;
    end;
  end;

var
  EffectiveOptions: TGridOptions;
begin
  State := gsNormal;
  Index := -1;
  EffectiveOptions := Options;
  if csDesigning in ComponentState then
  begin
    Include(EffectiveOptions, goColSizing);
    Include(EffectiveOptions, goRowSizing);
  end;
  if [goColSizing, goRowSizing] * EffectiveOptions <> [] then
    with FixedInfo do
    begin
      Vert.GridExtent := ClientHeight;
      Horz.GridExtent := ClientWidth;
      if (X > Horz.FixedBoundary) and (goColSizing in EffectiveOptions) then
      begin
        if Y >= Vert.FixedBoundary then Exit;
        CalcAxisState(Horz, X, gsColSizing);
      end
      else if (Y > Vert.FixedBoundary) and (goRowSizing in EffectiveOptions) then
      begin
        if X >= Horz.FixedBoundary then Exit;
        CalcAxisState(Vert, Y, gsRowSizing);
      end;
    end;
end;

end.
