{*********************************************************}
{                                                         }
{ The Jerek Component Library                             }
{ Version: 1.0                                            }
{ Author: Jere Kpyaho (jere@mikrobitti.fi)               }
{ Last modified: 1999-11-26                               }
{                                                         }
{*********************************************************}

unit Jerek;

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Graphics;


{*********************************************************}
{  TVersionInfo: Encapsulates the Win32 version           }
{                information resource                     }
{*********************************************************}

type
  ENoVersionInfoException = class(Exception);

  TVariableFieldNumber =
    (vfComments, vfCompany, vfFileDescription,
     vfFileVersion, vfInternalName, vfLegalCopyright,
     vfLegalTrademarks, vfOriginalFilename, vfPrivateBuild,
     vfProductName, vfProductVersion, vfSpecialBuild);

  { The version information class. }
  TVersionInfo = class(TComponent)
  private
    // The following are the variable fields of the version information.
    FComments: string;
    FCompany: string;
    FFileDescription: string;
    FFileVersion: string;
    FInternalName: string;
    FLegalCopyright: string;
    FLegalTrademarks: string;
    FOriginalFilename: string;
    FPrivateBuild: string;
    FProductName: string;
    FProductVersion: string;
    FSpecialBuild: string;
    // End of variable fields.

    FUnknownString: string;
    Data: Pointer;       { pointer to version information data }
    BaseString: string;  { base of version query string }

  protected
    procedure ReadVersionInfo;
    procedure SetVariableFields;
    function GetVariableString(FieldNumber: TVariableFieldNumber): string;

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure Loaded; override;

    // The variable version information fields as read-only properties:
    property Comments: string read FComments stored False;
    property Company: string read FCompany stored False;
    property FileDescription: string read FFileDescription stored False;
    property FileVersion: string read FFileVersion stored False;
    property InternalName: string read FInternalName stored False;
    property LegalCopyright: string read FLegalCopyright stored False;
    property LegalTrademarks: string read FLegalTrademarks stored False;
    property OriginalFilename: string read FOriginalFilename stored False;
    property PrivateBuild: string read FPrivateBuild stored False;
    property ProductName: string read FProductName stored False;
    property ProductVersion: string read FProductVersion stored False;
    property SpecialBuild: string read FSpecialBuild stored False;
    // End of variable field properties.

  published
    property UnknownString: string read FUnknownString write FUnknownString;
  end;


{*********************************************************}
{  TDigitalDisplay: emulates a digital readout            }
{*********************************************************}

type
  TDigitalDisplay = class(TGraphicControl)
  private
    FDigits: TBitmap;
    FDigitCount: Smallint;
    FValue: Longint;
    DigitWidth: Integer;
    procedure SetDigits(Value: TBitmap);
    procedure SetDigitCount(Value: Smallint);
    procedure SetValue(Value: Longint);
    procedure AdjustSize;
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop,
                    AWidth, AHeight: Integer); override;
  published
    property Digits: TBitmap read FDigits write SetDigits;
    property DigitCount: Smallint read FDigitCount
        write SetDigitCount default 1;
    property Value: Longint read FValue write SetValue
        default 0;
    property Enabled;
    property Visible;
    property Height default 16;
    property Width default 16;
  end;


{*********************************************************}
{ TLed: emulates a LED (Light Emitting Diode)             }
{*********************************************************}
const
  DEFAULT_LED_WIDTH = 16;
  DEFAULT_LED_HEIGHT = 16;
  DEFAULT_LED_BLINK_RATE = 1000; { in milliseconds }
  MIN_LED_BLINK_RATE = 100;
  MAX_LED_BLINK_RATE = 5000;

type
  TLedColor = (lcRed, lcYellow, lcGreen, lcBlue); { LED color values }
  TLedState = (lsOn, lsOff, lsBlink);

  TLed = class(TCustomControl)
  private
    FState: TLedState;
    FColor: TLedColor;
    FIsOn: Boolean;
    FBlink: Boolean;
    FBlinkRate: Integer;
    FTimer: UINT;
    procedure SetColor(Value: TLedColor);
    procedure SetBlinkRate(Value: Integer);
    procedure SetBlinkState(Value: Boolean);
    procedure Toggle;
    procedure SetState(Value: TLedState);
  protected
    procedure Paint; override;
    procedure WMTimer(var Msg: TWMTimer); message WM_TIMER;
    procedure WMDestroy(var Msg: TWMDestroy); message WM_DESTROY;
  public
    constructor Create(AOwner: TComponent); override;
    procedure TurnOn;
    procedure TurnOff;
  published
    property State: TLedState read FState write SetState default lsOff;
    property Color: TLedColor read FColor write SetColor default lcRed;
    property IsOn: Boolean read FIsOn default False; { read-only }
    property Blink: Boolean read FBlink write SetBlinkState default False;
    property BlinkRate: Integer read FBlinkRate write SetBlinkRate
                                default DEFAULT_LED_BLINK_RATE;
    property Height default DEFAULT_LED_HEIGHT;
    property Width default DEFAULT_LED_WIDTH;
    property Visible;
    property Enabled;
  end;


procedure Register;

///////////////////////////////////////////////////////////////////////////

implementation

const
  SUnknown = '(unknown)';

  VariableFieldCount = 12;

  VariableFieldNames : array [0..VariableFieldCount-1] of string =
    ('Comments', 'CompanyName', 'FileDescription', 'FileVersion',
     'InternalName', 'LegalCopyright', 'LegalTrademarks', 'OriginalFilename',
     'PrivateBuild', 'ProductName', 'ProductVersion', 'SpecialBuild');

{ Construct the version information object. }
constructor TVersionInfo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FUnknownString := SUnknown;

  Data := nil;
end;

{ Destroy the version information object }
destructor TVersionInfo.Destroy;
begin
  if Assigned(Data) then
    FreeMem(Data);

  inherited Destroy;
end;


procedure TVersionInfo.Loaded;
begin
  inherited Loaded;

  if not (csDesigning in ComponentState) then
  begin
    ReadVersionInfo;    // read the version information from the executable
    SetVariableFields;  // initialize the fields based on the data that was read
  end;
end;


procedure TVersionInfo.SetVariableFields;
begin
  FComments        := GetVariableString(vfComments);
  FCompany         := GetVariableString(vfCompany);
  FFileDescription := GetVariableString(vfFileDescription);
  FFileVersion     := GetVariableString(vfFileVersion);
  FInternalName    := GetVariableString(vfInternalName);
  FLegalCopyright  := GetVariableString(vfLegalCopyright);
  FLegalTrademarks := GetVariableString(vfLegalTrademarks);
  FOriginalFilename:= GetVariableString(vfOriginalFilename);
  FPrivateBuild    := GetVariableString(vfPrivateBuild);
  FProductName     := GetVariableString(vfProductName);
  FProductVersion  := GetVariableString(vfProductVersion);
  FSpecialBuild    := GetVariableString(vfSpecialBuild);
end;


{ Get the contents of a variable file information field based on the
  field ID number }
function TVersionInfo.GetVariableString(FieldNumber: TVariableFieldNumber): string;
var
  StringLength: UINT;
  StringData: Pointer;
  VerName: string;
begin
  if not Assigned(Data) then { no data, can't continue }
    Exit;

  { Construct a version information query for the translation/field }
  VerName := BaseString + VariableFieldNames[Ord(FieldNumber)];

  { Get the value of the variable file information field: }
  if not VerQueryValue(Data, PChar(VerName), 
                       StringData, StringLength) then
    Exit;

  { If the query was OK and the string length was non-zero, return
    the string data. Otherwise, return the default "unknown" string. }
  if StringLength <> 0 then
    Result := PChar(StringData)
  else
    Result := FUnknownString;
end;


procedure TVersionInfo.ReadVersionInfo;
var
  Filename: array [0..259] of Char;  { the name of our executable }
  Handle: DWORD;      { dummy handle to receive zero from GetFileVersionInfo(Size) }
  InfoSize: DWORD;    { length of version information from GetFileVersionInfoSize }
  ValueLength: UINT;  { length of version field from VerQueryValue }
  TranslationInfoPointer: Pointer;
  LangID, CharSetID: UINT;
  LangIDStr, CharSetIDStr: string[8];
  OK: Boolean;
begin
  { Get the filename of our executable. }
  GetModuleFileName(HInstance, Filename, SizeOf(Filename));

  { Find out the size of the version information block and
    allocate memory for it: }
  InfoSize := GetFileVersionInfoSize(Filename, Handle);
  if InfoSize = 0 then 
    raise ENoVersionInfoException.Create('No version info found in executable file');

  GetMem(Data, InfoSize);
  if Data = nil then
    Exit;

  { Copy the version information to our data block: }
  OK := GetFileVersionInfo(Filename, Handle, InfoSize, Data);
  if not OK then
    Exit;

  { Get the current translation (language and character set ID: }
  VerQueryValue(Data, '\VarFileInfo\Translation', TranslationInfoPointer, ValueLength);
  LangID := LoWord(DWORD(TranslationInfoPointer^));
  CharSetID := HiWord(DWORD(TranslationInfoPointer^));

  { Convert language and character set IDs to hexadecimal strings: }
  LangIDStr    := IntToHex(LangID, 4);
  CharSetIDStr := IntToHex(CharSetID, 4);

  { Construct a version query string for the relevant translation block: }
  FmtStr(BaseString, '\StringFileInfo\%s%s\', [LangIDStr, CharSetIDStr]);

end;

{*********************************************************}

constructor TDigitalDisplay.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FDigits := TBitmap.Create;
  FDigitCount := 1;
  FValue := 0;
  DigitWidth := 16;
  Width := FDigitCount * DigitWidth;
  Height := 16;
end;

destructor TDigitalDisplay.Destroy;
begin
  FDigits.Free;

  inherited Destroy;
end;

procedure TDigitalDisplay.AdjustSize;
begin
  Width := FDigitCount * DigitWidth;
  Height := FDigits.Height;
end;

procedure TDigitalDisplay.SetBounds(ALeft, ATop,
                                AWidth, AHeight: Integer);
begin
  if csDesigning in ComponentState then
  begin
    AWidth := FDigitCount * DigitWidth;
    AHeight := FDigits.Height;
  end;
  inherited;
end;

procedure TDigitalDisplay.SetDigits(Value: TBitmap);
begin
  FDigits.Assign(Value);
  if Assigned(FDigits) then
  begin
    DigitWidth := FDigits.Width div 10;
    AdjustSize;
    Invalidate;
  end;
end;

procedure TDigitalDisplay.SetDigitCount(Value: Smallint);
begin
  if Value <> FDigitCount then
  begin
    FDigitCount := Value;
    AdjustSize;
    Invalidate;
  end;
end;

procedure TDigitalDisplay.SetValue(Value: Longint);
begin
  if Value <> FValue then
  begin
    FValue := Value;
    Invalidate;
  end;
end;

procedure TDigitalDisplay.Paint;
var
  ValueString: string;
  I: Integer;
  FromRect, ToRect: TRect;
  DigitValue: Integer;
begin
  inherited Paint;

  if Assigned(FDigits) then
  begin
    // Get the color of the top left pixel
    // and use it as the background color.
    Canvas.Brush.Color := FDigits.Canvas.Pixels[0, 0];
    Canvas.FillRect(Rect(0, 0, Width, Height));

    ValueString := Format(
      '%.' + IntToStr(FDigitCount) + 'd', [FValue]);

    for I := 1 to FDigitCount do
    begin
      DigitValue := Ord(ValueString[I]) - Ord('0');

      FromRect := Rect(DigitValue * DigitWidth, 0,
        0, Self.Height);
      FromRect.Right := FromRect.Left + DigitWidth;

      ToRect := Rect((I - 1) * DigitWidth, 0,
        0, Self.Height);
      ToRect.Right := ToRect.Left + DigitWidth;

      Canvas.CopyRect(ToRect, FDigits.Canvas, FromRect);
    end;
  end;
end;


{*********************************************************}

{ Create the LED and set default attribute values. }
constructor TLed.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  Width := DEFAULT_LED_WIDTH;
  Height := DEFAULT_LED_HEIGHT;

  FBlinkRate := DEFAULT_LED_BLINK_RATE;
  FColor := lcRed;
  FState := lsOff;
  FTimer := 1;
end;

procedure TLed.WMDestroy(var Msg: TWMDestroy);
begin
  KillTimer(Handle, FTimer);
  FTimer := 0;
  inherited;
end;

{ Turn the LED on. }
procedure TLed.TurnOn;
begin
  FIsOn := True;
  Invalidate;
end;

{ Turn the LED off. }
procedure TLed.TurnOff;
begin
  FIsOn := False;
  Invalidate;
end;

procedure TLed.SetState(Value: TLedState);
begin
  if FState <> Value then
  begin
    FState := Value;
    Invalidate;

  end;
end;

{ Set the LED color. }
procedure TLed.SetColor(Value: TLedColor);
begin
  { Check previous property value to avoid unnecessary updating }
  if FColor <> Value then
  begin
    FColor := Value;
    Invalidate;
  end;
end;

{ Toggle the LED state between on and off. }
procedure TLed.Toggle;
begin
  if FIsOn then TurnOff
  else TurnOn
end;

{ Set the LED's blink rate. If the LED is currently blinking, }
{ kill the old timer and create a new one. }
procedure TLed.SetBlinkRate(Value: Integer);
begin
  if Value <> FBlinkRate then
  begin
    { Validate rate value }
    if Value < MIN_LED_BLINK_RATE then
      Value := MIN_LED_BLINK_RATE
    else if Value > MAX_LED_BLINK_RATE then
      Value := MAX_LED_BLINK_RATE;

    FBlinkRate := Value;
    if FBlink then
    begin
      KillTimer(Handle, FTimer);
      if SetTimer(Handle, FTimer, FBlinkRate, nil) = 0 then
        FBlink := False;
    end;
  end;
end;

{ Set the blink state (yes/no) of the LED. }
procedure TLed.SetBlinkState(Value: Boolean);
begin
  if Value <> FBlink then
  begin
    FBlink := Value;
    if FBlink then
    begin
      if SetTimer(Handle, FTimer, FBlinkRate, nil) = 0 then
        FBlink := False;
    end
    else
      KillTimer(Handle, FTimer);
  end;
end;

{ Respond to the WM_TIMER message by toggling the LED state. }
procedure TLed.WMTimer(var Msg: TWMTimer);
begin
  if FBlink then Toggle;
end;

{ Paint the LED in the on or off state using the Color property. }
procedure TLed.Paint;
const
  OnColors: array [TLedColor] of TColor =
    (clRed, clYellow, clLime, clBlue);
  OffColors: array [TLedColor] of TColor =
    ($0000008B, $00006464, clGreen, $008B0000);
var
  BrushColor: TColor;
begin
  { Determine which color to use (on or off). } 
  if FIsOn then
    BrushColor := OnColors[FColor]
  else
    BrushColor := OffColors[FColor];

  { Draw the LED }
  Canvas.Pen.Color := clBlack;
  Canvas.Brush.Color := BrushColor;
  Canvas.Ellipse(0, 0, Width, Height);

  { Draw a highlight on the LED }
  Canvas.Pen.Color := clWhite;
  Canvas.Arc(Width div 4, Height div 4,
             (3 * Width) div 4, (3 * Height) div 4,
             Width div 2, Height div 4,
             Width div 4, Height div 2);
end;


{*********************************************************}


procedure Register;
begin
  RegisterComponents('Jerek', [TVersionInfo, TDigitalDisplay, TLed]);
end;


end.

