{ -------------------------------------------------------------------------------------}
{ An horizontally scrolling listbox component for Delphi32.                            }
{ Copyright 1996, Fulvio Senore.  All Rights Reserved.                                 }
{ This component can be freely used and distributed in commercial and private          }
{ environments, provided this notice is not modified in any way.                       }
{ -------------------------------------------------------------------------------------}
{ Feel free to contact me if you have any questions, comments or suggestions at        }
{ fsenore@avigliana.alpcom.it (Fulvio Senore)                                          }
{ -------------------------------------------------------------------------------------}
{ Date last modified:  28 Dec 1996                                                        }
{ -------------------------------------------------------------------------------------}

{ -------------------------------------------------------------------------------------}
{ Description: This unit implements a listbox with horizontal                          }
{ scrolling capabilities                                                               }
{ -------------------------------------------------------------------------------------}
{ The idea is simple: we must inform the listbox about the maximum length              }
{ of the stored strings. Before adding every string we compute its                     }
{ length in pixel and, if the length is greater then the current maximum,              }
{ we send a LB_SETHORIZONTALEXTENT message to set the new maximum length.              }
{ -------------------------------------------------------------------------------------}
{ This is a simple example: it can only correctly add new strings or clear             }
{ the listbox. I really don't need more, but you can extend it.                        }
{ For example, to correctly delete a line you need to know if you just                 }
{ deleted the longest line: in this case send a LB_SETHORIZONTALEXTENT                 }
{ message to set the maximum length of the remaining lines.                            }
{ -------------------------------------------------------------------------------------}
{ Added methods to the listbox:                                                        }
{   procedure AddString( s: string );                                                  }
{   procedure ClearList( s: string );                                                  }
{                                                                                      }
{ Use the first to add strings to the listbox and the second to clear                  }
{ the listbox.                                                                         }
{                                                                                      }
{  I realize that it would be better to use the standard Items.Add and                 }
{  Items.Clear methods, but I don't know how to override them (they are                }
{  methods of a property of the listbox). If anybody knows how to do it                }
{  I will be glad to receive an explanation.                                           }
{ -------------------------------------------------------------------------------------}


unit HListBox;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls;

type
  THListBox = class(TListBox)
  private
    { Private declarations }
    MaxExtent: integer;     { maximum length of the strings in the listbox (in pixel) }
    function GetStringExtent( s: string ): integer;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create( AOwner: TComponent ); override;
    procedure AddString( s: string ); { adds a string to the listbox }
    procedure ClearList( s: string ); { deletes all string from the listbox }
  published
    { Published declarations }
  end;

procedure Register;

implementation

constructor THListBox.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  MaxExtent := 0;
end;


function THListBox.GetStringExtent( s: string ): integer;
var
  dwExtent:           DWORD;
  hDCListBox:         HDC;
  hFontOld, hFontNew: HFONT;
  tm:                 TTextMetric;
  Size:               TSize;
begin
  hDCListBox := GetDC( Handle );
  hFontNew := SendMessage( Handle, WM_GETFONT, 0, 0 );
  hFontOld := SelectObject( hDCListBox, hFontNew );
  GetTextMetrics( hDCListBox, tm );

  { the following two lines should be modified for Delphi 1.0: call GetTextExtent }
  GetTextExtentPoint32( hDCListBox, PChar(s), Length(s), Size );
  dwExtent := Size.cx + tm.tmAveCharWidth;

  SelectObject( hDCListBox, hFontOld );
  ReleaseDC( Handle, hDCListBox );

  GetStringExtent := LOWORD( dwExtent );
end;


procedure THListBox.AddString( s: string );
var
  StrExtent: integer;
begin
  StrExtent := GetStringExtent( s );
  if StrExtent > MaxExtent then
  begin
    MaxExtent := StrExtent;
    SendMessage( Handle, LB_SETHORIZONTALEXTENT, MaxExtent, 0 );
  end;

  { adds the string to the listbox }
  Items.Add( s );
end;

procedure THListBox.ClearList( s: string );
begin
  MaxExtent := 0;
	SendMessage( Handle, LB_SETHORIZONTALEXTENT, 0, 0 );

	{ scrolls the listbox to the left }
	SendMessage( Handle, WM_HSCROLL, SB_TOP, 0 );

  { clears the listbox }
  Items.Clear;
end;




procedure Register;
begin
  RegisterComponents('Tools', [THListBox]);
end;

end.
