unit Gradbtn;
{
  Graduated-Color Fill Button Component
  John Baumbach                         Delphi 1.0 or 2.0
  email: mantis@vcnet.com                1-2-96
  http://www.vcnet.com/mantis

  Also check out:

      TMovingButton: button that travels around in the parent container or form
      ScrollingBackground: Form Gallery Component that has a background that
          scrolls smoothly down the form (behind the components...)

  Available on my home page (see above).

  Installation Instructions:
  To check out a sample application without installing
  the component, load the project file "grad.dpr" with
  the Delphi editor and run.  If you have an error like
  "Can't execute a DLL", make sure the sample form is loaded
  by "View | Forms | GradTest", then clicking "OK".

  If you have an error like "Invalid File Name" during compile time
  change your "Options | Project | Directories-Conditionals |
  Output Directory" to blanks (if Delphi is set up to produce an
  output file).

  Installing Component to Toolbar:
  This component is currently set up to be added to the "Samples"
  palette of the toolbar.  If you wish to add it to a different
  one, change the word "Samples" to the name of the desired
  palette in the "Register" procedure below.

  Select "Options | Install Components" and select "Add", then
  "gradbtn.pas" from the directory where you unzipped this file.
  Delphi will then recompile the toolbar and the "TGradButton"
  component will be added to the palette!

  If you want to keep this component permanently, copy the "gradbtn.pas"
  and "gradbtn.dcr" files to the "Delphi\LIB" subdirectory and follow the
  above installation step.

  You can remove the component at any time the same way, except choose
  "Options | Install Components" and choose "GradButton" to remove.

  The icon for this component is in the file "gradbtn.dcr".  If you use this
  component under Delphi 2.0, load it into the resource editor and then save
  it.  Otherwise, Delphi won't be able to compile it correctly.

  Using the GradButton:
  Once installed in toolbar, you can use it just like it's ancestor,
  the "TSpeedButton" component.  However, there are a few new features and
  properties that you can set.

  The new properties are:

      BevelWidth      width in pixels of the button's bevel
      BorderWidth     the black background width around the button
      FaceColor       the top graduated color of the button
      GradColor       the bottom graduated color of the button
      HighlightColor  the highlight color of the bevel
      ShadowColor     the shadow color of the bevel

  Limitations:

      Glyph:    The "Glyph" property of the button doesn't work.  You can add
                it yourself in the "Paint" routine if you wish.
      Hot Key:  You can't use a hot key with the button (ie. "&OK")

  Colors:

  If you are using a 16 or 256 color system, I would recommend using only
  dark colors for the graduated fill and keeping the button size small.
  This is to keep the dithering effect down to a minimum.  On systems with
  more colors (32K+), the graduated color effect looks fantastic on any
  size button or with any colors.
}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Buttons, ExtCtrls;

type

  TGradButton = class(TSpeedButton)
  private
    { Private Declarations }
    BackGround: TBitmap;
    fGradColor: TColor; {clBlue}
    fFaceColor: TColor; {clBlack}
    fHighlightColor: TColor; {clWhite}
    fBottomColor: TColor; {clBlack}
    fBevWidth: integer;   {1}
    fRectWidth: integer;  {1}
    procedure GradFill(Clr1, Clr2: TColor; TheBitmap: TBitmap);
    procedure SetGradColor(AColor: TColor);
    procedure SetFaceColor(AColor: TColor);
    procedure SetHighlightColor(AColor: TColor);
    procedure SetBottomColor(AColor: TColor);
    procedure SetBevWidth(ANumber: integer);
    procedure SetRectWidth(ANumber: integer);
  protected
    { Protected Declarations }
    procedure Paint; override;
    destructor Destroy;
  public
    { Public Declarations }
    constructor Create(AOwner: TComponent); override;
  published
    { Published Declarations }
    property GradColor: TColor read fGradcolor
     write SetGradColor default clBlue;
    property FaceColor: TColor read fFacecolor
     write SetFaceColor default clBlack;
    property HighlightColor: TColor read fHighlightColor
     write SetHighlightColor default clWhite;
    property ShadowColor: TColor read fBottomColor
     write SetBottomColor default clBlack;
    property BevelWidth: integer read fBevWidth
     write SetBevWidth default 1;
    property BorderWidth: integer read fRectWidth
     write SetRectWidth default 1;
  end;

procedure Register;

implementation

{The "Set..." procedures redraw the button after a property is changed}
procedure TGradButton.SetGradColor(AColor: TColor);
begin
    if AColor <> fGradcolor then begin
        fGradColor:= AColor;
        GradFill(fGradColor, fFaceColor, BackGround);
        Paint;
    end;
end;

procedure TGradButton.SetFaceColor(AColor: TColor);
begin
    if AColor <> fFacecolor then begin
        fFaceColor:= AColor;
        GradFill(fGradColor, fFaceColor, BackGround);
        Paint;
    end;
end;

procedure TGradButton.SetHighlightColor(AColor: TColor);
begin
    if AColor <> fHighlightcolor then begin
        fHighlightColor:= AColor;
        Paint;
    end;
end;

procedure TGradButton.SetBottomColor(AColor: TColor);
begin
    if AColor <> fBottomcolor then begin
        fBottomColor:= AColor;
        Paint;
    end;
end;

procedure TGradButton.SetBevWidth(ANumber: integer);
begin
    if ANumber <> fBevWidth then begin
        fBevWidth:= ANumber;
        Paint;
    end;
end;

procedure TGradButton.SetRectWidth(ANumber: integer);
begin
    if ANumber <> fRectWidth then begin
        fRectWidth:= ANumber;
        Paint;
    end;
end;

{This procedure draws the gradient colors onto the bitmap}
procedure TGradButton.GradFill(Clr1, Clr2: TColor; TheBitmap: TBitmap);
var
  RGBFrom: array[0..2] of Byte;    { from RGB values                     }
  RGBDiff: array[0..2] of integer; { difference of from/to RGB values    }
  ColorBand: TRect;                  { color band rectangular coordinates  }
  I: Integer;                { color band index                    }
  R: Byte;                   { a color band's R value              }
  G: Byte;                   { a color band's G value              }
  B: Byte;                   { a color band's B value              }
begin
   { extract from RGB values}
   RGBFrom[0] := GetRValue(ColorToRGB(Clr1));
   RGBFrom[1] := GetGValue(ColorToRGB(Clr1));
   RGBFrom[2] := GetBValue(ColorToRGB(Clr1));
   { calculate difference of from and to RGB values}
   RGBDiff[0] := GetRValue(ColorToRGB(Clr2)) - RGBFrom[0];
   RGBDiff[1] := GetGValue(ColorToRGB(Clr2)) - RGBFrom[1];
   RGBDiff[2] := GetBValue(ColorToRGB(Clr2)) - RGBFrom[2];
   { set pen sytle and mode}
   TheBitmap.Canvas.Pen.Style:= psSolid;
   TheBitmap.Canvas.Pen.Mode:= pmCopy;
   { set color band's left and right coordinates}
   ColorBand.Left:= 0;
   ColorBand.Right:= TheBitmap.Width;
   for I := 0 to $ff do
   begin
       { calculate color band's top and bottom coordinates}
       ColorBand.Top:= MulDiv (I, TheBitmap.Height, $100);
       ColorBand.Bottom:= MulDiv (I + 1, TheBitmap.Height, $100);
       { calculate color band color}
       R := RGBFrom[0] + MulDiv(I, RGBDiff[0], $ff);
       G := RGBFrom[1] + MulDiv(I, RGBDiff[1], $ff);
       B := RGBFrom[2] + MulDiv(I, RGBDiff[2], $ff);
       { select brush and paint color band}
       TheBitmap.Canvas.Brush.Color := RGB(R, G, B);
       TheBitmap.Canvas.FillRect(ColorBand);
   end;
end;

{This procedure draws the button onto the form}
procedure TGradButton.Paint;
var
  x, y, W, H: integer;
  R, InnerRect: TRect;
begin
   {Get button's total rectangle area}
   R:=ClientRect;
   {Get the part we'll draw the gradient color to.  It's the total rectangle }
   {area minus the border width                                              }
   with InnerRect do begin
      Top:=R.Top + fRectWidth;
      Left:=R.Left + fRectWidth;
      Right:=R.Right - fRectWidth;
      Bottom:=R.Bottom - fRectWidth;
   end;
   {Set the font}
   canvas.font.assign(font);
   {Get coordinates for centering the caption onto the button}
   x:=(width - canvas.textwidth(caption)) div 2;
   y:=(height - canvas.textheight(caption)) div 2;
   {Test to see if the size has changed.  If so, we need to redraw }
   {the gradient colors onto the background bitmap.  If not, we    }
   {won't redraw because doing it is slow...                       }
   W:=Width - (2 * fRectWidth);
   H:=Height - (2 * fRectWidth);
   if (W <> BackGround.Width) or (H <> BackGround.Height) then begin
       BackGround.Width:= W;
       BackGround.Height:= H;
       GradFill(fGradColor, fFaceColor, BackGround);
   end;
   {We draw the button differently depending on if it is pressed or not}
   if fState = bsDown then with canvas do begin
      {If button is pressed...}
      Brush.Color:=clBlack;        {Set border color}
      FillRect(R);                 {Draw border     }
      Brush.Color:=fFaceColor;     {Set face color  }
      {Draw the gradient colors, but offset to simulate that the button}
      {is pressed }
      Draw(fRectWidth + 1, fRectWidth + 1, BackGround);
      Frame3D(Canvas, InnerRect, fBottomColor, fFaceColor, fBevWidth);
                                   {Draw a 3D frame like a normal button, but pressed}
      SetBKMode(canvas.handle, transparent);
                                   {Set transparency mode for text output}
      Textout(x + 1, y + 1, caption);   {Draw the caption }
   end
   else with canvas do begin
      {If button not pressed...}
      Brush.Color:=clBlack;
      FillRect(R);
      Brush.Color:=fFaceColor;
      Draw(fRectWidth, fRectWidth, BackGround);
      Frame3D(Canvas, InnerRect, fHighlightColor, fBottomcolor, fBevWidth);
      SetBKMode(canvas.handle, transparent);
      textout(x,y,caption);
   end;
   {Reset the background mode}
   SetBKMode(canvas.handle,opaque);
end;

{Create the button}
constructor TGradButton.Create(AOwner: TComponent);
begin
   {Get all methods of ancestor (TSpeedButton) }
   inherited Create(AOwner);
   {Set the default properties.  This is the way the button appears     }
   {when you first place it on a form or create it at runtime.  You can }
   {change these if you don't like the default properties!              }
   Width:=100;
   Height:=30;
   fGradColor:= clBlue;
   fFaceColor:= clBlack;
   fHighlightColor:= clWhite;
   fBottomColor:= clBlack;
   fBevWidth:= 1;
   fRectWidth:= 1;
   {Create the background bitmap.  It is freed in the "destroy" procedure}
   BackGround:=TBitmap.Create;
   BackGround.Width:=Width;
   BackGround.Height:=Height;
   {Draw the gradient colors to the background bitmap for future painting}
   GradFill(fGradColor, fFaceColor, BackGround);
   Font.Color:=clWhite;
   Caption:='TGradButton';
end;

destructor TGradButton.Destroy;
begin
   {Free the bitmap because Windows can't do it on it's own...}
   Background.Free;
end;

procedure Register;
begin
   {Write component to Component Palette if installing}
   RegisterComponents('Samples', [TGradButton]);
end;

end.
