{ Custom Open Dialog Component -- Williams }
{ Most of this code is taken from the
  standard open dialog component }
unit CustDlg;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
   CommDlg,FiltEdit,DsgnIntf;


type
  TOpenNotifyEvent = procedure(Sender : TObject; Wnd,Parent : HWnd; nfy : POpenFilename ) of Object;
  TShareNotifyEvent = procedure(Sender : TObject; Wnd,Parent : HWnd; nfy : POpenFilename; fn : String) of Object;
  TCommandNotifyEvent=procedure(Sender : TObject; Wnd,Parent : HWnd; cmd : Integer) of Object;
  TOpenOption = (ofReadOnly, ofOverwritePrompt, ofHideReadOnly,
    ofNoChangeDir, ofShowHelp, ofNoValidate, ofAllowMultiSelect,
    ofExtensionDifferent, ofPathMustExist, ofFileMustExist, ofCreatePrompt,
    ofShareAware, ofNoReadOnlyReturn, ofNoTestFileCreate, ofNoNetworkButton,
    ofNoLongNames, ofOldStyleDialog, ofNoDereferenceLinks);
  TOpenOptions = set of TOpenOption;

  TFileEditStyle = (fsEdit, fsComboBox);

  TCustomOpenDialog = class(TCommonDialog)
  protected
    FHwnd : Hwnd;            { Window handle for hook }
    FCenter : Bool;          { Center dialog? }
    FTemplateName: string;   { Dialog template name }
    FHistoryList: TStrings;
    FOptions: TOpenOptions;
    FFilter: String;
    FFilterIndex: Integer;
    FInitialDir: string;
    FTitle: string;
    FDefaultExt: string;
    FFileName: TFileName;
    FFiles: TStrings;
    FFileEditStyle: TFileEditStyle;
{ Custom event handlers }
    FOnInitDone : TOpenNotifyEvent;
    FOnFileOK : TOpenNotifyEvent;
    FOnFolderChange : TOpenNotifyEvent;
    FOnHelp : TOpenNotifyEvent;
    FOnSelChange : TOpenNotifyEvent;
    FOnTypeChange : TOpenNotifyEvent;
    FOnShareViolation : TShareNotifyEvent;
    FOnCommand : TCommandNotifyEvent;
    procedure SetHistoryList(Value: TStrings);
    procedure SetInitialDir(const Value: string);
    function DoExecute(Func: Pointer): Bool;
    function FoldPath : String;
    function FilePth : String;
    procedure SetDefExt(s : String);
    function GetSpec : String;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: Boolean; virtual;
    property Files: TStrings read FFiles;
{ New read only properties }
    property Wnd : HWnd read FHWnd write FHWnd;
    property FolderPath : String read FoldPath;
    property FilePath : String read FilePth;
    property Spec : String read GetSpec;
{ New write only property }
    property DefExt : String write SetDefExt;
{ New procedures }
    procedure HideControl(id : Integer);
    procedure SetControlText(id : Integer;s : String);
  published
    property OnInitDone : TOpenNotifyEvent read FOnInitDone write FOnInitDone;
    property OnFileOK : TOpenNotifyEvent read FOnFileOK write FOnFileOK;
    property OnFolderChange : TOpenNotifyEvent read FOnFolderChange write FOnFolderChange;
    property OnHelp : TOpenNotifyEvent read FOnHelp write FOnHelp;
    property OnSelChange : TOpenNotifyEvent read FOnSelChange write FOnSelChange;
    property OnTypeChange : TOpenNotifyEvent read FOnTypeChange write FOnTypeChange;
    property OnShareViolation : TShareNotifyEvent read FOnShareViolation write FOnShareViolation;
    property OnCommand: TCommandNotifyEvent read FOnCommand write FOnCommand;
    property DefaultExt: string read FDefaultExt write FDefaultExt;
    property FileEditStyle: TFileEditStyle read FFileEditStyle write FFileEditStyle default fsEdit;
    property FileName: TFileName read FFileName write FFileName;
    property Filter: String read FFilter write FFilter;
    property FilterIndex: Integer read FFilterIndex write FFilterIndex default 1;
    property HistoryList: TStrings read FHistoryList write SetHistoryList;
    property InitialDir: string read FInitialDir write SetInitialDir;
    property Options: TOpenOptions read FOptions write FOptions default [];
    property Title: string read FTitle write FTitle;
    property TemplateName : string read FTemplateName write FTemplateName;
    property Center : Bool read FCenter write FCenter default True;
  end;


procedure Register;

implementation


procedure Register;
begin
  RegisterComponents('Dialogs', [TCustomOpenDialog]);
{ Need filter property editor }
  RegisterPropertyEditor(TypeInfo(String), TCustomOpenDialog, 'Filter',
    TFilterProperty);

end;
procedure CenterWindow(Wnd: HWnd);
var
  Rect: TRect;
begin
  GetWindowRect(Wnd, Rect);
  SetWindowPos(Wnd, 0,
    (GetSystemMetrics(SM_CXSCREEN) - Rect.Right + Rect.Left) div 2,
    (GetSystemMetrics(SM_CYSCREEN) - Rect.Bottom + Rect.Top) div 3,
    0, 0, SWP_NOACTIVATE or SWP_NOSIZE or SWP_NOZORDER);
end;

{ Explorer hook. Centers the dialog on the screen in response to
  the WM_INITDIALOG message also distributes events}

function ExplorerHook(Wnd: HWnd; Msg, WParam, LParam: Longint): Longint; stdcall;
var
 Parent : HWnd;
 ofn : ^TOpenFileName;
 obj : TCustomOpenDialog;
begin
  Result := 0;
  Parent:=GetParent(Wnd);
  if (Msg = WM_INITDIALOG) then
    begin
      ofn:=Pointer(LParam);    {remember object pointer }
      SetProp(Wnd,'TCOD',ofn.lCustData);
      obj:=Pointer(ofn.lCustData);
      obj.FHWnd:=Wnd;
    end;
{ Get pointer to object }
  obj:=Pointer(GetProp(Wnd,'TCOD'));
  if (Msg = WM_NCDESTROY) then  { clean up }
    RemoveProp(Wnd,'TCOD');
{ route notifications }
  if (Msg = WM_NOTIFY) then
  begin
{ Center after INIT if requested }
    if (obj.FCenter and (POFNotify(LParam)^.hdr.code = CDN_INITDONE)) then
      CenterWindow(GetWindowLong(Wnd, GWL_HWNDPARENT));
{ Dispatch each event }
  if (Assigned(obj.FOnInitDone) and (POFNotify(LParam)^.hdr.code = CDN_INITDONE)) then
      obj.FOnInitDone(obj,Wnd,Parent,POpenFileName(LParam));
  if (Assigned(obj.FOnFileOK) and (POFNotify(LParam)^.hdr.code = CDN_FILEOK)) then
      obj.FOnFileOK(obj,Wnd,Parent,POpenFileName(LParam));
  if (Assigned(obj.FOnFolderChange) and (POFNotify(LParam)^.hdr.code = CDN_FOLDERCHANGE)) then
      obj.FOnFolderChange(obj,Wnd,Parent,POpenFileName(LParam));
  if (Assigned(obj.FOnHelp) and (POFNotify(LParam)^.hdr.code = CDN_HELP)) then
      obj.FOnHelp(obj,Wnd,Parent,POpenFileName(LParam));
  if (Assigned(obj.FOnSelChange) and (POFNotify(LParam)^.hdr.code = CDN_SELCHANGE)) then
      obj.FOnSelChange(obj,Wnd,Parent,POpenFileName(LParam));
  if (Assigned(obj.FOnTypeChange) and (POFNotify(LParam)^.hdr.code = CDN_TYPECHANGE)) then
      obj.FOnTypeChange(obj,Wnd,Parent,POpenFileName(LParam));
  if (Assigned(obj.FOnShareViolation) and (POFNotify(LParam)^.hdr.code = CDN_SHAREVIOLATION)) then
      obj.FOnShareViolation(obj,Wnd,Parent,POpenFileName(LParam),
        POFNotify(LParam)^.pszFile);


  end;
{ dispatch WM_COMMAND }
 if (Msg=WM_COMMAND) and Assigned(obj.FOnCommand) then
     obj.FOnCommand(obj,Wnd,Parent,LOWORD(WParam));

end;



{ TCustOpenDialog }

constructor TCustomOpenDialog.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FHistoryList := TStringList.Create;
  FFiles := TStringList.Create;
  FFilterIndex := 1;
  FFileEditStyle := fsEdit;
  FCenter:=True;
end;

destructor TCustomOpenDialog.Destroy;
begin
  FFiles.Free;
  FHistoryList.Free;
  inherited Destroy;
end;

function TCustomOpenDialog.DoExecute(Func: Pointer): Bool;
const
  MultiSelectBufferSize = 8192;
  OpenOptions: array [TOpenOption] of Longint = (
    OFN_READONLY, OFN_OVERWRITEPROMPT, OFN_HIDEREADONLY,
    OFN_NOCHANGEDIR, OFN_SHOWHELP, OFN_NOVALIDATE, OFN_ALLOWMULTISELECT,
    OFN_EXTENSIONDIFFERENT, OFN_PATHMUSTEXIST, OFN_FILEMUSTEXIST,
    OFN_CREATEPROMPT, OFN_SHAREAWARE, OFN_NOREADONLYRETURN,
    OFN_NOTESTFILECREATE, OFN_NONETWORKBUTTON, OFN_NOLONGNAMES,
    OFN_EXPLORER, OFN_NODEREFERENCELINKS);
var
  Option: TOpenOption;
  P: PChar;
  CDefaultExt: array[0..3] of Char;
  OpenFilename: TOpenFilename;

  function AllocFilterStr(const S: string): PChar;
  var
    P: PChar;
  begin
    Result := nil;
    if S <> '' then
    begin
      Result := StrCopy(StrAlloc(Length(S) + 2), PChar(S));
      P := Result;
      while P^ <> #0 do
      begin
        if P^ = '|' then P^ := #0;
        Inc(P);
      end;
      Inc(P);
      P^ := #0;
    end;
  end;

  function ExtractFileName(P: PChar; var S: string): PChar;
  var
    Separator: Char;
  begin
    Separator := #0;
    if ofOldStyleDialog in FOptions then Separator := ' ';
    Result := P;
    while (Result[0] <> #0) and (Result[0] <> Separator) do Inc(Result);
    SetString(S, P, Result - P);
    if Result[0] = Separator then Inc(Result);
  end;

  procedure ExtractFileNames(P: PChar);
  var
    DirName, FileName: string;
  begin
    P := ExtractFileName(P, DirName);
    P := ExtractFileName(P, FileName);
    if FileName = '' then
      FFiles.Add(DirName)
    else
      repeat
        if (FileName[1] <> '\') and ((Length(FileName) <= 3) or
          (FileName[2] <> ':') or (FileName[3] <> '\')) then
          FileName := DirName + '\' + FileName;
        FFiles.Add(FileName);
        P := ExtractFileName(P, FileName);
      until FileName = '';
  end;

begin
  FFiles.Clear;
  FillChar(OpenFileName, SizeOf(OpenFileName), 0);
  with OpenFilename do
  try
    lStructSize := SizeOf(TOpenFilename);
    hInstance := System.HInstance;
    lpstrFilter := AllocFilterStr(FFilter);
    nFilterIndex := FFilterIndex;
    if ofAllowMultiSelect in FOptions then
      nMaxFile := MultiSelectBufferSize else
      nMaxFile := MAX_PATH;
    GetMem(lpstrFile, nMaxFile + 2);
    FillChar(lpstrFile^, nMaxFile + 2, 0);
    StrLCopy(lpstrFile, PChar(FFileName), nMaxFile);
    lpstrInitialDir := PChar(FInitialDir);
    lpstrTitle := PChar(FTitle);
{ Always enable hook }
    Flags := OFN_ENABLEHOOK;
    for Option := Low(Option) to High(Option) do
      if Option in FOptions then
        Flags := Flags or OpenOptions[Option];
{ Always do new style open box }
      Flags := Flags or OFN_EXPLORER;
    if FDefaultExt <> '' then
    begin
      P := PChar(FDefaultExt);
      lpstrDefExt := StrLCopy(CDefaultExt, P, 3)
    end;
        { add custom callback }
        lpfnHook := ExplorerHook;
        { add custom resource  }
        if FTemplateName<>'' then
        begin
          lpTemplateName:=PChar(FTemplateName);
          Flags:=Flags or OFN_ENABLETEMPLATE;
        end;
        { allow callback to find object }
        lCustData:=LongInt(Self);
    hWndOwner := Application.Handle;
    Result := TaskModalDialog(Func, OpenFileName);
    if Result then
    begin
      if ofAllowMultiSelect in FOptions then
      begin
        ExtractFileNames(lpstrFile);
        FFileName := FFiles[0];
      end else
      begin
        ExtractFileName(lpstrFile, FFileName);
        FFiles.Add(FFileName);
      end;
      if (Flags and OFN_EXTENSIONDIFFERENT) <> 0 then
        Include(FOptions, ofExtensionDifferent) else
        Exclude(FOptions, ofExtensionDifferent);
      if (Flags and OFN_READONLY) <> 0 then
        Include(FOptions, ofReadOnly) else
        Exclude(FOptions, ofReadOnly);
      FFilterIndex := nFilterIndex;
    end;
  finally
    if lpstrFile <> nil then FreeMem(lpstrFile, nMaxFile + 2);
    if lpstrFilter <> nil then StrDispose(lpstrFilter);
  end;
end;

procedure TCustomOpenDialog.SetHistoryList(Value: TStrings);
begin
  FHistoryList.Assign(Value);
end;

procedure TCustomOpenDialog.SetInitialDir(const Value: string);
var
  L: Integer;
begin
  L := Length(Value);
  if (L > 1) and (Value[L] = '\') and (Value[L - 1] <> ':') then Dec(L);
  FInitialDir := Copy(Value, 1, L);
end;

function TCustomOpenDialog.Execute: Boolean;
begin
  Result := DoExecute(@GetOpenFileName);
end;

{ Helper function to read folder path }
function TCustomOpenDialog.FoldPath : String;
var
  s : String;
  n : Integer;
begin
  SetLength(s,256);
  n:=SendMessage(GetParent(FHWnd),CDM_GetFolderPath,256,LongInt(PChar(s)));
  SetLength(s,n-1);
  result:=s;
end;

{ Helper function to read File Path }
function TCustomOpenDialog.FilePth : String;
var
  s:String;
  n:Integer;
begin
   SetLength(s,256);
   n:=SendMessage(GetParent(FHWnd),CDM_GETFILEPATH,256,LongInt(PChar(s)));
   SetLength(s,n-1);
   result:=s;
end;

{ Helper function to read Spec }
function TCustomOpenDialog.GetSpec : String;
var
  s:String;
  n:Integer;
begin
   SetLength(s,256);
   n:=SendMessage(GetParent(FHWnd),CDM_GETSPEC,256,LongInt(PChar(s)));
   SetLength(s,n-1);
   result:=s;
end;

{ Helper procedure to set default ext }
procedure TCustomOpenDialog.SetDefExt(s : String);
begin
  SendMessage(GetParent(FHWnd),CDM_SETDEFEXT,0,LongInt(PChar(s)));
end;

{ Hide a specific control }
procedure TCustomOpenDialog.HideControl(id : Integer);
begin
  SendMessage(GetParent(FHWnd),CDM_HIDECONTROL,id,0);
end;

{ Set a control's text }
procedure TCustomOpenDialog.SetControlText(id : Integer;s : String);
begin
  SendMessage(GetParent(FHWnd),CDM_SETCONTROLTEXT,id,LongInt(PChar(s)));
end;

end.
