unit CCopydlg;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, FileCtrl;

type
  {custom type to hold file information}
  TFileInfo = record
    Date: longint;
    Size: longint;
  end;

  TCCopyBoxDlg = class(TComponent)
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
  private
    FDir: string;        {stores current directory}
    FOkToAll: boolean;   {stores initial value passed for overwrite prompt}
    FCancel: boolean;
    procedure CustInitialise(strSourceDir_l,strTargetDir_l: string; bOverwritePrompt: boolean);
    procedure SetUpFiles;
    procedure CustCopyFiles(sSrce, sDest: string);
    function CheckDir(sDir: string): string;
    function IsDir(sDrive: string): boolean;
    function DiskInDrive(i: integer): boolean;
    function GetFileInfo(sFile: string): TFileInfo;
    function IsSpace(sDestination: string): longint;
  public
    strSourceFile: string;
    strTargetFile: string;
    strSourceDir: string;
    strTargetDir: string;
    FList: TStringList;
    procedure CopySingleFile;
    procedure CopyMultipleFiles;
  published
    constructor Create(AOwner: TComponent); override;
  end;

var
  CCopyBoxDlg: TCCopyBoxDlg;
  iErrorMode : word;
  OkToAll    : boolean;

procedure Register;

implementation

uses CCopy;

constructor TCCopyBoxDlg.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  strSourceFile := '';
  strTargetFile := '';
  strSourceDir := '';
  strTargetDir := '';
  FList := TStringList.Create
end;

{set environment}
procedure TCCopyBoxDlg.CustInitialise(strSourceDir_l,strTargetDir_l: string; bOverwritePrompt: boolean);
var
  s: string;
begin

  {store current dir}
  GetDir(0,FDir);

  {turn off DOS error reporting}
  iErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);

  {set source dir}
  strSourceDir := CheckDir(strSourceDir_l);

  {set destination dir}
  strTargetDir := CheckDir(strTargetDir_l);

  {set bools}
  FCancel := false;
  FOkToAll := bOverwritePrompt; {store value, since OkToAll could be set to false by SetupFiles proc}
  if bOverwritePrompt then
     OkToAll := false
  else
     OkToAll := true;

end;

{check directory - this routine amends the string passed so that all directory
 labels don't end in a backslash, then checks that the directory is valid by
 calling isDir}
function TCCopyBoxDlg.CheckDir(sDir: string): string;
begin

 case length(sDir) of
 {case of sDir being just a drive letter, add ':'}
  1:
   begin
    if isDir(sDir) then
     result := sDir + ':'
    else
     result := FDir;
   end;
 else
  begin
   {case of sDir ending in '\', then remove}
   if sDir[length(sDir)] = '\' then
    begin
     delete(sDir,length(sDir),1);
     if isDir(sDir) then
      result := sDir
     else
      result := FDir;
    end
   else
    {text is okay, so check if directory exists}
    if isDir(sDir) then
     result := sDir
    else
     result := FDir;
  end;
 end;
end;

{return free space, plus size of existing file. This routine is only called
 by CustCopyFiles}
function TCCopyBoxDlg.IsSpace(sDestination: string): longint;
var
 c: char;
 i: integer;
 li: longint;
 fExists: TFileInfo;
begin

 {get drive letter}
 c := sDestination[1];

 {check that drive letter is valid}
 if c in ['a'..'z'] then Dec(c,($20));
 if not (c in ['A'..'Z']) then
  begin
   messageDlg('Invalid drive ID',mtWarning,[mbOK],0);
   result := 0;
  end;

 {get alphabet index of character - ie: A is 1. Remember, it's now uppercase}
 i := Ord(c)-$40 ;

 li := 0;
 li := DiskFree(i);

 {if the file exists, then add the existing file's size from value returned by diskFree}
 if FileExists(sDestination) then
   begin
    fExists := GetFileInfo(sDestination);
    li := li + fExists.size;
   end;

 result := li;

end;

{check for directory, or drive}
function TCCopyBoxDlg.IsDir(sDrive: string): boolean;
var
 c: char;
 i: integer;
begin

 {get drive letter}
 c := sDrive[1];

 {check that drive letter is valid}
 if c in ['a'..'z'] then Dec(c,($20));
 if not (c in ['A'..'Z']) then
  begin
   messageDlg('Invalid drive ID',mtWarning,[mbOK],0);
   result := false;
  end;

 {get alphabet index of character - ie: A is 1. Remember, it's now uppercase}
 i := Ord(c)-$40;


 if GetDriveType(i -1) = DRIVE_REMOVABLE then
  {floppy}
  begin

   {ensure floppy in drive}
    while not DiskInDrive(i) do
     begin
      DiskInDrive(i);
     end;

   {floppy in drive, now check for directory}
   if (length(sDrive) > 3) then {where 3 would be the size of 'a:\'}
    begin
     {check floppy for sub-dir}
     if DirectoryExists(sDrive) then
      result := true
     else
      result := false;
    end
   else
    {user trying to copy to root of floppy drive}
    result := true
  end
 else
  {hard disk}
  begin
   {first, if sDrive is less than 3 characters then the user is trying to copy to a
    root, in which case DirectoryExists will fail. If the routine has reached this stage
    we should be sure that the drive is legal so return true}
   if length(sDrive) <= 3 then
    result := true
   else
   if DirectoryExists(sDrive) then
    result := true
   else
    begin
     messageDlg('"'+sDrive+'" Directory not found',mtWarning,[mbOK],0);
     result := false;
    end;
  end;

 {finally, check that drive is a legal drive}
 if DiskSize(i) = -1 then Result := False;

end;

{check for floppy disk in drive}
function TCCopyBoxDlg.DiskInDrive(i: integer): boolean;
begin
  if DiskSize(i) = -1 then
  begin
    messageDlg('Please insert a floppy disk into the floppy drive',mtInformation,[mbOK],0);
    result := false
  end
  else
  result := true;
end;

{routine returns file information - called in the case of overwrites}
function TCCopyBoxDlg.GetFileInfo(sFile: string): TFileInfo;
var
  f: file;
  fInfo: TFileInfo;
begin

  if not FileExists(sFile) then exit;

  {Set file access mode to readonly in case file is in use.}
  System.FileMode := fmOpenRead;
  {assign and open files}
  AssignFile(f,sFile);
  {$I-}
  Reset(f,1);
  {$I+}
  {Set file access mode back to normal default for other processes}
  System.Filemode := fmOpenReadWrite;
  if IOResult <> 0  then
  begin
    messageDlg('Could not open: '+sFile,mtWarning,[mbOK],0);
    fInfo.size := 0;
    fInfo.date := 0;
  end
  else
  begin
    fInfo.size := FileSize(f);
    fInfo.date := FileGetDate(TFileRec(f).Handle);
  end;
  result := fInfo;
  system.closeFile(f);

end;

{***Copy procs******************************************************************}

{setup copying}
procedure TCCopyBoxDlg.SetUpFiles;
var
  iNum: integer;
  sSrce, sDest: ^string;
  li, liFree: longint;
  f: File;
  fSrce, fDest: TFileInfo;
  i: integer;
  c: char;
begin
  try
    New(sSrce);
    New(sDest);

    {initialise}
    iNum := 0;
    {ensure that directories exists - actually, at this stage both labels should be valid}
    if not isDir(strSourceDir) then
    begin
      exit;
    end;
    if not isDir(strTargetDir) then
    begin
      exit;
    end;

    {check that the user is not trying to copy over source files}
    if CompareText(strSourceDir,strTargetDir) = 0 then
    begin
      messageDlg('Can not overwrite source files.',mtWarning,[mbOK],0);
      exit;
    end;

    {ensure that there are items in the file-list box}
    if (FList.Count) = 0 then
    begin
      exit;
    end;

    {check space on target}
    li := 0;
    liFree := 0;
    {increment through file list, adding up file sizes}
    for iNum := 0 to (FList.Count-1) do
    begin
      System.FileMode := fmOpenRead;
      {assign and open files}
      AssignFile(f, strSourceDir +'\'+ ExtractFileName(FList.Strings[iNum]));
      {$I-}
      Reset(f,1);
      {$I+}
      if IOResult = 0 then
      begin
        {increment var holding the total size of source files}
        li := li + FileSize(f);
        {if target file exists, find its size}
        if FileExists(strTargetDir +'\' + ExtractFileName(FList.Strings[iNum])) then
        begin
          fDest := GetFileInfo(strTargetDir +'\' + ExtractFileName(FList.Strings[iNum]));
          liFree := liFree + fDest.size;
        end;
        CloseFile(f);
      end;
      System.FileMode := fmOpenReadWrite;
    end;
    {get drive letter}
    c := strTargetDir[1];
    if c in ['a'..'z'] then Dec(c,($20));
    i :=  ord (c) -$40;
    {find target disk size}
    if li > DiskSize(i) then
    begin
      messageDlg('Insufficient space on target for all of the selected files',mtWarning,[mbOK],0);
      exit;
    end;
    {find target free - we add, to free space, the size of the existing files since the user
    is probably going to overwrite them}
    liFree := liFree + DiskFree(i);
    if li > liFree then
    begin
      messageDlg('Insufficient free space on target for selected files',mtWarning,[mbOK],0);
      exit;
    end;

    {init for loop}
    for iNum := 0 to (FList.Count -1) do
    begin

      {get source file name}
      sSrce^ := strSourceDir +'\'+ ExtractFileName(FList.Strings[iNum]);

      {get destination file name}
      sDest^ := strTargetDir + '\' + (ExtractFileName(FList.Strings[iNum]));

      application.processMessages;

      {break?}
      if CCopyBox.bCancel = true then
      begin
        if MessageDlg('Do you really want to interrupt the copy process?',mtConfirmation,[mbYes, mbNo],0)
           = mrYes
        then
        begin
          FCancel := true;
          break;
        end
        else CCopyBox.bCancel := false;
      end
      else CCopyBox.bCancel := false;

      {check to see if file exists}
      if not OkToAll then
      begin
        if FileExists(sDest^) then
        begin
          {since file exists, we must get info of both source and target files}
          fSrce := GetFileInfo(sSrce^);
          fDest := GetFileInfo(sDest^);

          case messageDlg('Overwrite '+sDest^+ #13#10 +
                          'Size: '+IntToStr(fSrce.size)+' bytes  Date:'
                          +dateTimeToStr(FileDateToDateTime(fSrce.date))+ #13#10 + #13#10 +
                          'with: '+sSrce^+ #13#10 +
                          'Size: '+IntToStr(fDest.size)+' bytes  Date:'+
                          DateTimeToStr(FileDateToDateTime(fDest.date))+ #13#10 + #13#10,
                          mtConfirmation,[mbYes,mbAll,mbNo],0)
          of
          idYes:
            custCopyFiles(sSrce^,sDest^);

          (idNo+1): {mrAll}
            begin
              OkToAll := true;
              custCopyFiles(sSrce^,sDest^);
            end;

          idNo:
          {do nothing}

          end;
        end
        else
        {file doesn't already exist - so copy}
        custCopyFiles(sSrce^,sDest^);
      end
      else
      {file does already exist, but overwrite is true}
      custCopyFiles(sSrce^,sDest^);

      Application.ProcessMessages;

    end;

    {cleanup}
    if FOkToAll then
       okToAll := false
    else
       okToAll := true;

  finally
    Dispose(sSrce);
    Dispose(sDest);
  end;
end;

{copy routine}
procedure TCCopyBoxDlg.CustCopyFiles(sSrce,sDest: string);
type
  iobufptr = ^iobufr; {allowate a LARGE buffer to speed up copies}
  iobufr   = array[0..32767] of char; {MAX=65535}
var
 fSrce, fDest: file;
 wRead, wWritten: word;
 p: iobufptr;
 FDate, FSize, wSumWritten: Longint;
 f: TFileInfo;
begin
  {initialise}
  wRead := 0;
  wWritten := 0;
  wSumWritten := 0;
  CCopyBox.Gauge.Progress := 0;
  CCopyBox.SourceLabel.Caption := sSrce;
  CCopyBox.TargetLabel.Caption := sDest;

  Application.ProcessMessages;

  {Set file access mode to readonly in case file is in use.}
  System.FileMode := fmOpenRead;
   {assign and open files}
   AssignFile(fSrce,sSrce);
   {$I-}
   Reset(fSrce,1);
   {$I+}
  {Set file access mode back to normal default for other processes}
  System.Filemode := fmOpenReadWrite;
  if IOResult <> 0  then
   begin
    messageDlg('Could not open: '+sSrce,mtWarning,[mbOK],0);
    exit;
   end;
  {Store file Date & Time for later use}
  FDate := FileGetDate(TFileRec(fSrce).Handle);

  {before creating new file, check that there is sufficient free space}
  if isSpace(sDest) > FileSize(fSrce) then
   begin
    {Set file access mode to allow Exclusive Creation }
    System.Filemode := fmOpenWrite and fmShareExclusive;
    AssignFile(fDest,sDest);
    {$I-}
    Rewrite(fDest, 1);
    {$I+}
    {Set file access mode back to normal default for other processes}
    System.Filemode := fmOpenReadWrite;

    if IOResult <> 0  then
     begin
      {Close the Source file we already have open.}
      System.CloseFile(fSrce);
      messageDlg('Could not create: '+sDest,mtWarning,[mbOK],0);
      exit;
     end;
   end
  else
   begin
    {this message should only ever be seen if the CopyIndivFile call is used to
     open the DLL}
    if messageDlg('There is insufficient space on the target drive'+#13#10+
                  'for: '+ sSrce +#13#10+#13#10+
                  'Do you wish to cancel the copy process?',mtConfirmation,[mbYes,mbNo],0)
     = mrYes then FCancel := true;
    exit;
   end;

  {allocate a file iobuffer on Heap to avoid stack overflow error}
  new(p);

  {copy loop}
   FSize := FileSize(fSrce);

   repeat
    BlockRead(fSrce, p^, SizeOf(p^), wRead);
    BlockWrite(fDest, p^, wRead, wWritten);
    wSumWritten := wSumWritten + wWritten;
    CCopyBox.Gauge.Progress := Round(wSumWritten/FSize * 100);
   until (wRead = 0) or (wWritten <> wRead);

   {release heap space for iobuffer }
   dispose(p);

   {restore Source file date & time to Destination file }
   Reset(fDest,1);
   FileSetDate(TFileRec(fDest).Handle,FDate);
   System.CloseFile(fDest);

   {clean up}
   System.CloseFile(fSrce);
end;

{************************************************************}

procedure TCCopyBoxDlg.CopySingleFile;
begin
  { Create dialog in memory }
  CCopyBox := TCCopyBox.Create(application);

  try
    { do something }
    CCopyBox.bCancel := false;

    if (StrSourceDir <> '') and (strSourceFile <> '') and
       (StrTargetDir <> '') and (strTargetFile <> '')
    then
    begin
      strSourceDir := CheckDir(strSourceDir);
      strTargetDir := CheckDir(strTargetDir);

      CCopyBox.SourceLabel.Caption := StrSourceDir + '\' + StrSourceFile;
      CCopyBox.TargetLabel.Caption := StrTargetDir + '\' + StrTargetFile;

      CCopyBox.Show;
      CCopyBox.Update;

      CustCopyFiles(strSourceDir + '\' + strSourceFile,
                    strTargetDir + '\' + strTargetFile);

  { procedure TCCopyBoxDlg.CustInitialise(pSource,pDestination: pChar; bOverwritePrompt: boolean); }
  { procedure TCCopyBoxDlg.SetUpFiles; }
    end
    else
    messageDlg('No files to copy!',mtWarning,[mbOK],0);
  finally
    CCopybox.Free;
  end;
end;

procedure TCCopyBoxDlg.CopyMultipleFiles;
var i : integer;
begin
  { Create dialog in memory }
  CCopyBox := TCCopyBox.Create(application);

  try
    { do something }
    CCopyBox.bCancel := false;

    if (StrSourceDir <> '') and (strTargetDir <> '')
    then
    begin
      CCopyBox.Show;
      CCopyBox.Update;

      if (FList.Count) <> 0 then
      CustInitialise(StrSourceDir, strTargetDir, true);
      begin
        SetupFiles;
      end;
    end
    else
    messageDlg('No files to copy!',mtWarning,[mbOK],0);

  finally
    CCopybox.Free;
  end;
end;

{************************************************************}

procedure TCCopyBoxDlg.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  {reset system}
  SetErrorMode(iErrorMode);
  FList.Free;
end;

procedure Register;
begin
  RegisterComponents('Custom', [TCCopyBoxDlg]);
end;

end.
