/*
  Maze editor: dialog box definitions
  Copyright (C) 1998 by Jorrit Tyberghein
  Written by Andrew Zabolotny <bit@eltech.ru>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "sysdef.h"
#include "me_res.h"
#include "me_text.h"
#include "me_app.h"
#include "me_util.h"
#include "csutil/scanstr.h"

// Private class: model manager dialog
class csModelManagerDialog : public csDialog
{
  csModelType ModelType;
public:
  csModelManagerDialog (csWindow *iParent) : csDialog (iParent)
  { ModelType = csmtUnknown; }
  virtual bool HandleEvent (csEvent &Event);
  void RefreshList ();
};

// Center a dialog so that mouse is in center
static inline void CenterOnMouse (csApp *app, csComponent *c)
{
  int x = System->Mouse->GetLastX () - c->bound.Width () / 2;
  int y = System->Mouse->GetLastY () - c->bound.Height () / 2;
  c->SetPos (x, y);
  ((MazeEditor *)app)->EnsureFullyVisible (c);
}

void ShowAbout (csComponent *app)
{
  char msg [400];
  sprintf (msg,
    "Crystal Space Maze Editor\n"
    "Version %s\n"
    "Released %s\n"
    "\n"
    "Design & Implementation:\n"
    "        Andrew Zabolotny\n"
    "Crystal Space Windowing System:\n"
    "        Andrew Zabolotny\n"
    "Moral support:\n"
    "        My wife Ulea\n"
    "        Jorrit Tyberghein\n",
    VERSION, RELEASE_DATE);
  MessageBox (app, "About Maze Editor", msg);
}

#define MZWID_VERTICES		0xc0000000
#define MZWID_FACES		0xc0000001
#define MZWID_LIGHTS		0xc0000002
#define MZWID_FRAMES		0xc0000003
#define MZWID_ACTIONS		0xc0000004

// [x]=====Clear what?===[x][x]
// |                          |
// | [x] Remove all ~vertices |
// | [x] Remove all ~faces    |
// | [x] Remove all ~lights   |
// | [x] Remove all f~rames   |
// | [x] Remove all ~actions  |
// |                          |
// |    [ Ok ] [ Cancel ]     |
// +--------------------------+
int ClearModelDialog (csApp *iApp, bool &ioRemoveVertices, bool &ioRemoveFaces,
  bool &ioRemoveLights, bool &ioRemoveFrames, bool &ioRemoveActions)
{
  CHK (csWindow *w = new csWindow (iApp, TEXT_CLEARMODELTITLE,
    CSWS_BUTSYSMENU | CSWS_BUTCLOSE | CSWS_TITLEBAR));
  CHK (csDialog *d = new csDialog (w));
  w->SetDragStyle (w->GetDragStyle () & ~CS_DRAG_SIZEABLE);

  CHK (csComponent *c = new csCheckBox (d, MZWID_VERTICES));
  c->SetPos (5, 6); c->SetState (CSS_GROUP, true);
  c->SendCommand (cscmdCheckBoxSet, (void *)ioRemoveVertices);

  CHK (c = new csStatic (d, c, TEXT_REMOVEVERTICES));
  c->SetPos (21, 8);

  CHK (c = new csCheckBox (d, MZWID_FACES));
  c->SetPos (5, 22);
  c->SendCommand (cscmdCheckBoxSet, (void *)ioRemoveFaces);

  CHK (c = new csStatic (d, c, TEXT_REMOVEFACES));
  c->SetPos (21, 24);

  CHK (c = new csCheckBox (d, MZWID_LIGHTS));
  c->SetPos (5, 38);
  c->SendCommand (cscmdCheckBoxSet, (void *)ioRemoveLights);

  CHK (c = new csStatic (d, c, TEXT_REMOVELIGHTS));
  c->SetPos (21, 40);

  CHK (c = new csCheckBox (d, MZWID_FRAMES));
  c->SetPos (5, 54);
  c->SendCommand (cscmdCheckBoxSet, (void *)ioRemoveFrames);

  CHK (c = new csStatic (d, c, TEXT_REMOVEFRAMES));
  c->SetPos (21, 56);

  CHK (c = new csCheckBox (d, MZWID_ACTIONS));
  c->SetPos (5, 70);
  c->SendCommand (cscmdCheckBoxSet, (void *)ioRemoveActions);

  CHK (c = new csStatic (d, c, TEXT_REMOVEACTIONS));
  c->SetPos (21, 72);

  CHK (csButton *b = new csButton (d, cscmdOK, CSBS_DEFAULTVALUE | CSBS_DISMISS | CSBS_DEFAULT));
  b->SetText (TEXT_OKAYBUTTON);
  b->SetSuggestedSize (+16, +2);
  b->SetPos (20, 88);
  b->SetState (CSS_GROUP, true);

  CHK (b = new csButton (d, cscmdCancel, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b->SetText (TEXT_CANCELBUTTON);
  b->SetSuggestedSize (+16, +2);
  b->SetPos (80, 88);

  // and now set window size and center it.
  w->SetSize (180, 136);
  CenterOnMouse (iApp, w);

  int retcode = iApp->Execute (w);
  if (retcode == cscmdOK)
  {
    ioRemoveVertices = (d->GetChild (MZWID_VERTICES)->SendCommand (cscmdCheckBoxQuery) != NULL);
    ioRemoveFaces = (d->GetChild (MZWID_FACES)->SendCommand (cscmdCheckBoxQuery) != NULL);
    ioRemoveLights = (d->GetChild (MZWID_LIGHTS)->SendCommand (cscmdCheckBoxQuery) != NULL);
    ioRemoveFrames = (d->GetChild (MZWID_FRAMES)->SendCommand (cscmdCheckBoxQuery) != NULL);
    ioRemoveActions = (d->GetChild (MZWID_ACTIONS)->SendCommand (cscmdCheckBoxQuery) != NULL);
  } /* endif */
  delete w;
  return retcode;
}

#define MZWID_TOPLEFT		0xc0000000
#define MZWID_TOPCENTER		0xc0000001
#define MZWID_TOPRIGHT		0xc0000002
#define MZWID_LEFT		0xc0000003
#define MZWID_CENTER		0xc0000004
#define MZWID_RIGHT		0xc0000005
#define MZWID_BOTTOMLEFT	0xc0000006
#define MZWID_BOTTOMCENTER	0xc0000007
#define MZWID_BOTTOMRIGHT	0xc0000008
#define MZWID_USERDEFINED	0xc0000009
#define MZWID_USERDEFVALUE	0xc000000a
#define MZWID_VIEWPLANE		0xc000000b
#define MZWID_NEAR		0xc000000c
#define MZWID_CENTERED		0xc000000d
#define MZWID_FAR		0xc000000e
#define MZWID_ONLYSELECTION	0xc000000f

// [x]=======Select modification axis location=====[x][x]
// |                                                    |
// |+-~2D location-------------------------------------+|
// ||(*) Top-Left    (*)~Top-Center    (*) Top-Right   ||
// ||(*) ~Left       (*)~Center        (*) ~Right      ||
// ||(*) Bottom-Left (*)~Bottom-Center (*) Bottom-Right||
// ||(*) ~User-defined: (100,10.23,15.34)              ||
// |+--------------------------------------------------+|
// |+-~3D Location-------------------------------------+|
// ||(*) ~Viewplane  (*) ~Near  (*) Centere~d  (*) ~Far||
// |+--------------------------------------------------+|
// | [x] Consider only ~selection                       |
// |                                                    |
// |            [ ~Ok ] [ Cancel ] [ ~Define ]          |
// +----------------------------------------------------+
int SetupAxisDialog (csApp *iApp, mzModificationAxis &Axis)
{
  CHK (csWindow *w = new csWindow (iApp, TEXT_MODIFYAXISTITLE,
    CSWS_BUTSYSMENU | CSWS_BUTCLOSE | CSWS_TITLEBAR));
  CHK (csDialog *d = new csDialog (w));
  w->SetDragStyle (w->GetDragStyle () & ~CS_DRAG_SIZEABLE);

  csComponent *c;
  csStatic *s;

  CHK (s = new csStatic (d, NULL, TEXT_AXIS2DLOCATION, csscsFrameLabel));
  s->SetRect (5, 6, 390, 84);

  CHK (c = new csRadioButton (d, MZWID_TOPLEFT));
  c->SetPos (15, 16); c->SetState (CSS_GROUP, true);

  CHK (s = new csStatic (d, c, TEXT_AXISTOPLEFT));
  s->SetPos (31, 18);

  CHK (c = new csRadioButton (d, MZWID_TOPCENTER));
  c->SetPos (135, 16);

  CHK (s = new csStatic (d, c, TEXT_AXISTOPCENTER));
  s->SetPos (151, 18);

  CHK (c = new csRadioButton (d, MZWID_TOPRIGHT));
  c->SetPos (255, 16);

  CHK (s = new csStatic (d, c, TEXT_AXISTOPRIGHT));
  s->SetPos (271, 18);

  CHK (c = new csRadioButton (d, MZWID_LEFT));
  c->SetPos (15, 32);

  CHK (s = new csStatic (d, c, TEXT_AXISLEFT));
  s->SetPos (31, 34);

  CHK (c = new csRadioButton (d, MZWID_CENTER));
  c->SetPos (135, 32);

  CHK (s = new csStatic (d, c, TEXT_AXISCENTER));
  s->SetPos (151, 34);

  CHK (c = new csRadioButton (d, MZWID_RIGHT));
  c->SetPos (255, 32);

  CHK (s = new csStatic (d, c, TEXT_AXISRIGHT));
  s->SetPos (271, 34);

  CHK (c = new csRadioButton (d, MZWID_BOTTOMLEFT));
  c->SetPos (15, 48);

  CHK (s = new csStatic (d, c, TEXT_AXISBOTTOMLEFT));
  s->SetPos (31, 50);

  CHK (c = new csRadioButton (d, MZWID_BOTTOMCENTER));
  c->SetPos (135, 48);

  CHK (s = new csStatic (d, c, TEXT_AXISBOTTOMCENTER));
  s->SetPos (151, 50);

  CHK (c = new csRadioButton (d, MZWID_BOTTOMRIGHT));
  c->SetPos (255, 48);

  CHK (s = new csStatic (d, c, TEXT_AXISBOTTOMRIGHT));
  s->SetPos (271, 50);

  CHK (c = new csRadioButton (d, MZWID_USERDEFINED));
  c->SetPos (15, 64);

  CHK (s = new csStatic (d, c, TEXT_AXISUSERDEFINED));
  s->SetPos (31, 66);

  char strbuff [201];
  CHK (c = new csInputLine (d, sizeof (strbuff) - 1, csifsThinRect));
  c->SetRect (135, 64, 380, 76);
  c->id = MZWID_USERDEFVALUE;
  ftoa (strbuff, Axis.Origin.x);
  strcat (strbuff, ", ");
  ftoa (&strbuff [strlen (strbuff)], Axis.Origin.y);
  strcat (strbuff, ", ");
  ftoa (&strbuff [strlen (strbuff)], Axis.Origin.z);
  c->SetText (strbuff);

  CHK (s = new csStatic (d, NULL, TEXT_AXIS3DLOCATION, csscsFrameLabel));
  s->SetRect (5, 84, 390, 114);

  CHK (c = new csRadioButton (d, MZWID_VIEWPLANE));
  c->SetPos (15, 94); c->SetState (CSS_GROUP, true);

  CHK (s = new csStatic (d, c, TEXT_AXISVIEWPLANE));
  s->SetPos (31, 96);

  CHK (c = new csRadioButton (d, MZWID_NEAR));
  c->SetPos (115, 94);

  CHK (s = new csStatic (d, c, TEXT_AXISNEAR));
  s->SetPos (131, 96);

  CHK (c = new csRadioButton (d, MZWID_CENTERED));
  c->SetPos (215, 94);

  CHK (s = new csStatic (d, c, TEXT_AXISCENTERED));
  s->SetPos (231, 96);

  CHK (c = new csRadioButton (d, MZWID_FAR));
  c->SetPos (315, 94);

  CHK (s = new csStatic (d, c, TEXT_AXISFAR));
  s->SetPos (331, 96);

//---

  CHK (c = new csCheckBox (d, MZWID_ONLYSELECTION));
  c->SetPos (15, 116); c->SetState (CSS_GROUP, true);

  CHK (s = new csStatic (d, c, TEXT_AXISONLYSELECTION));
  s->SetPos (31, 118);

  CHK (csButton *b1 = new csButton (d, cscmdOK, CSBS_DEFAULTVALUE | CSBS_DISMISS | CSBS_DEFAULT));
  b1->SetText (TEXT_OKAYBUTTON);
  b1->SetSuggestedSize (+16, +2);
  b1->SetPos (80, 132);
  b1->SetState (CSS_GROUP, true);

  CHK (csButton *b2 = new csButton (d, cscmdCancel, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b2->SetText (TEXT_CANCELBUTTON);
  b2->SetSuggestedSize (+16, +2);
  b2->SetPos (b1->bound.xmax + 16, b1->bound.ymin);

  CHK (csButton *b3 = new csButton (d, cscmdRetry, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b3->SetText (TEXT_DEFINEBUTTON);
  b3->SetSuggestedSize (+16, +2);
  b3->SetPos (b2->bound.xmax + 16, b1->bound.ymin);

  // and now set window size and center it.
  w->SetSize (400, 180);
  CenterOnMouse (iApp, w);

  int activeid = 0;
  switch (Axis.Type)
  {
    case axUserDefined:  activeid = MZWID_USERDEFINED; break;
    case axTopLeft:      activeid = MZWID_TOPLEFT; break;
    case axTopCenter:    activeid = MZWID_TOPCENTER; break;
    case axTopRight:     activeid = MZWID_TOPRIGHT; break;
    case axLeft:         activeid = MZWID_LEFT; break;
    case axCenter:       activeid = MZWID_CENTER; break;
    case axRight:        activeid = MZWID_RIGHT; break;
    case axBottomLeft:   activeid = MZWID_BOTTOMLEFT; break;
    case axBottomCenter: activeid = MZWID_BOTTOMCENTER; break;
    case axBottomRight:  activeid = MZWID_BOTTOMRIGHT; break;
  } /* endswitch */
  csComponent *activecomp = d->GetChild (activeid);
  if (activecomp)
  {
    activecomp->SendCommand (cscmdRadioButtonSet, (void *)true);
    activecomp->Select ();
  } /* endif */

  activeid = 0;
  switch (Axis.Pos)
  {
    case axViewPlane: activeid = MZWID_VIEWPLANE; break;
    case axNear:      activeid = MZWID_NEAR; break;
    case axCentered:  activeid = MZWID_CENTERED; break;
    case axFar:       activeid = MZWID_FAR; break;
  } /* endswitch */
  d->GetChild (activeid)->SendCommand (cscmdRadioButtonSet, (void *)true);

  d->GetChild (MZWID_ONLYSELECTION)->SendCommand (cscmdCheckBoxSet,
    (void *)Axis.OnlySelection);

  int retcode = iApp->Execute (w);

  if (retcode != cscmdCancel)
  {
    if (d->GetChild (MZWID_TOPLEFT)->SendCommand (cscmdRadioButtonQuery))
      Axis.Type = axTopLeft;
    else if (d->GetChild (MZWID_TOPCENTER)->SendCommand (cscmdRadioButtonQuery))
      Axis.Type = axTopCenter;
    else if (d->GetChild (MZWID_TOPRIGHT)->SendCommand (cscmdRadioButtonQuery))
      Axis.Type = axTopRight;
    else if (d->GetChild (MZWID_LEFT)->SendCommand (cscmdRadioButtonQuery))
      Axis.Type = axLeft;
    else if (d->GetChild (MZWID_CENTER)->SendCommand (cscmdRadioButtonQuery))
      Axis.Type = axCenter;
    else if (d->GetChild (MZWID_RIGHT)->SendCommand (cscmdRadioButtonQuery))
      Axis.Type = axRight;
    else if (d->GetChild (MZWID_BOTTOMLEFT)->SendCommand (cscmdRadioButtonQuery))
      Axis.Type = axBottomLeft;
    else if (d->GetChild (MZWID_BOTTOMCENTER)->SendCommand (cscmdRadioButtonQuery))
      Axis.Type = axBottomCenter;
    else if (d->GetChild (MZWID_BOTTOMRIGHT)->SendCommand (cscmdRadioButtonQuery))
      Axis.Type = axBottomRight;
    else
      Axis.Type = axUserDefined;

    d->GetChild (MZWID_USERDEFVALUE)->GetText (strbuff, sizeof (strbuff));
    ScanStr (strbuff, "%f, %f, %f", &Axis.Origin.x, &Axis.Origin.y, &Axis.Origin.z);

    if (d->GetChild (MZWID_VIEWPLANE)->SendCommand (cscmdRadioButtonQuery))
      Axis.Pos = axViewPlane;
    else if (d->GetChild (MZWID_NEAR)->SendCommand (cscmdRadioButtonQuery))
      Axis.Pos = axNear;
    else if (d->GetChild (MZWID_CENTERED)->SendCommand (cscmdRadioButtonQuery))
      Axis.Pos = axCentered;
    else if (d->GetChild (MZWID_FAR)->SendCommand (cscmdRadioButtonQuery))
      Axis.Pos = axFar;

    Axis.OnlySelection = (d->GetChild (MZWID_ONLYSELECTION)->SendCommand (cscmdCheckBoxQuery) != NULL);
  } /* endif */

  if (retcode == cscmdRetry)
    MazeEditor::fDraftModificationAxis = true;

  delete w;
  return retcode;
}

#define MZWID_NSIDESVALUE	0xc0000000

// [x]===Number of sides?[x][x]
// |                          |
// | # sides                  |
// | 12+/- |
// |                          |
// |    [ Ok ] [ Cancel ]     |
// +--------------------------+
int NumberOfSidesDialog (csApp *iApp, char *iTitle, int &iSides)
{
  CHK (csWindow *w = new csWindow (iApp, iTitle,
    CSWS_BUTSYSMENU | CSWS_BUTCLOSE | CSWS_TITLEBAR));
  CHK (csDialog *d = new csDialog (w));
  w->SetDragStyle (w->GetDragStyle () & ~CS_DRAG_SIZEABLE);

  CHK (csSpinBox *sb = new csSpinBox (d));
  sb->SetRect (5, 16, 169, 32);
  sb->id = MZWID_NSIDESVALUE;
  sb->SetLimits (3, 256);
  sb->SetValue (iSides);

  CHK (csStatic *s = new csStatic (d, sb, TEXT_ENTERSIDES));
  s->SetPos (5, 6);

  CHK (csButton *b1 = new csButton (d, cscmdOK, CSBS_DEFAULTVALUE | CSBS_DISMISS | CSBS_DEFAULT));
  b1->SetText (TEXT_OKAYBUTTON);
  b1->SetSuggestedSize (+16, +2);
  b1->SetPos (16, 36);

  CHK (csButton *b2 = new csButton (d, cscmdCancel, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b2->SetText (TEXT_CANCELBUTTON);
  b2->SetSuggestedSize (+16, +2);
  b2->SetPos (b1->bound.xmax + 16, b1->bound.ymin);

  // and now set window size and center it.
  w->SetSize (180, 84);
  CenterOnMouse (iApp, w);

  int retcode = iApp->Execute (w);
  if (retcode == cscmdOK)
    iSides = (int)d->GetChild (MZWID_NSIDESVALUE)->SendCommand (cscmdSpinBoxQueryValue);
  delete w;
  return retcode;
}

#define MZWID_RADIALVALUE	0xc0000000
#define MZWID_CROSSSECTIONVALUE	0xc0000001

// [x]=Torus setup dialog[x][x]
// |                          |
// | Radial       CrossSection|
// | .12.... +/-  .8..... +/- |
// |                          |
// |    [ Ok ] [ Cancel ]     |
// +--------------------------+
int TorusParmDialog (csApp *iApp, int &iRadialDetail, int &iCrossSectionDetail)
{
  CHK (csWindow *w = new csWindow (iApp, TEXT_TORUSDETAILTITLE,
    CSWS_BUTSYSMENU | CSWS_BUTCLOSE | CSWS_TITLEBAR));
  CHK (csDialog *d = new csDialog (w));
  w->SetDragStyle (w->GetDragStyle () & ~CS_DRAG_SIZEABLE);

  CHK (csSpinBox *sb = new csSpinBox (d));
  sb->SetRect (5, 16, 84, 32);
  sb->id = MZWID_RADIALVALUE;
  sb->SetLimits (3, 256);
  sb->SetValue (iRadialDetail);

  CHK (csStatic *s = new csStatic (d, sb, TEXT_RADIALDETAILVALUE));
  s->SetPos (5, 6);

  CHK (sb = new csSpinBox (d));
  sb->SetRect (90, 16, 169, 32);
  sb->id = MZWID_CROSSSECTIONVALUE;
  sb->SetLimits (3, 256);
  sb->SetValue (iCrossSectionDetail);

  CHK (s = new csStatic (d, sb, TEXT_CROSSDETAILVALUE));
  s->SetPos (90, 6);

  CHK (csButton *b1 = new csButton (d, cscmdOK, CSBS_DEFAULTVALUE | CSBS_DISMISS | CSBS_DEFAULT));
  b1->SetText (TEXT_OKAYBUTTON);
  b1->SetSuggestedSize (+16, +2);
  b1->SetPos (16, 36);

  CHK (csButton *b2 = new csButton (d, cscmdCancel, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b2->SetText (TEXT_CANCELBUTTON);
  b2->SetSuggestedSize (+16, +2);
  b2->SetPos (b1->bound.xmax + 16, b1->bound.ymin);

  // and now set window size and center it.
  w->SetSize (180, 84);
  CenterOnMouse (iApp, w);

  int retcode = iApp->Execute (w);
  if (retcode == cscmdOK)
  {
    iRadialDetail = (int)d->GetChild (MZWID_RADIALVALUE)->SendCommand (cscmdSpinBoxQueryValue);
    iCrossSectionDetail = (int)d->GetChild (MZWID_CROSSSECTIONVALUE)->SendCommand (cscmdSpinBoxQueryValue);
  } /* endif */
  delete w;
  return retcode;
}

#define MZWID_DETAILVALUE	0xc0000000

// [x]Sphere detail level?[x][x]
// |                           |
// | Detail level              |
// | 12+/-  |
// |                           |
// |     [ Ok ] [ Cancel ]     |
// +---------------------------+
int SphereParmDialog (csApp *iApp, int &iSphereDetail)
{
  CHK (csWindow *w = new csWindow (iApp, TEXT_SPHEREDETAILTITLE,
    CSWS_BUTSYSMENU | CSWS_BUTCLOSE | CSWS_TITLEBAR));
  CHK (csDialog *d = new csDialog (w));
  w->SetDragStyle (w->GetDragStyle () & ~CS_DRAG_SIZEABLE);

  CHK (csSpinBox *sb = new csSpinBox (d));
  sb->SetRect (5, 16, 169, 32);
  sb->id = MZWID_DETAILVALUE;
  sb->SetLimits (3, 60);
  sb->SetValue (iSphereDetail);

  CHK (csStatic *s = new csStatic (d, sb, TEXT_SPHEREDETAILVALUE));
  s->SetPos (5, 6);

  CHK (csButton *b1 = new csButton (d, cscmdOK, CSBS_DEFAULTVALUE | CSBS_DISMISS | CSBS_DEFAULT));
  b1->SetText (TEXT_OKAYBUTTON);
  b1->SetSuggestedSize (+16, +2);
  b1->SetPos (16, 36);

  CHK (csButton *b2 = new csButton (d, cscmdCancel, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b2->SetText (TEXT_CANCELBUTTON);
  b2->SetSuggestedSize (+16, +2);
  b2->SetPos (b1->bound.xmax + 16, b1->bound.ymin);

  // and now set window size and center it.
  w->SetSize (180, 84);
  CenterOnMouse (iApp, w);

  int retcode = iApp->Execute (w);
  if (retcode == cscmdOK)
    iSphereDetail = (int)d->GetChild (MZWID_DETAILVALUE)->SendCommand (cscmdSpinBoxQueryValue);
  delete w;
  return retcode;
}

// [x]====Please wait=[x][x]
// |                       |
// | Please stand by while |
// | saving world file ... |
// |                       |
// +-----------------------+
csWindow *PleaseWaitBox (csApp *iApp, char *iText)
{
  CHK (csWindow *w = new csWindow (iApp, TEXT_PLEASEWAITTITLE,
    CSWS_BUTSYSMENU | CSWS_BUTCLOSE | CSWS_TITLEBAR));
  CHK (csDialog *d = new csDialog (w));
  w->SetDragStyle (w->GetDragStyle () & ~CS_DRAG_SIZEABLE);

  CHK (csStatic *s = new csStatic (d, NULL, TEXT_PLEASESTANDBY));
  s->SetPos (5, 10);
  int dw = s->bound.Width ();
  CHK (s = new csStatic (d, NULL, iText));
  s->SetPos (5, 20);
  if (dw < s->bound.Width ())
    dw = s->bound.Width ();

  // and now set window size and center it.
  w->Select ();
  w->SetSize (dw + 16, 58);
  w->Center ();

  return w;
}

#define MZWID_MODELLIST		0xc0000000
#define MZWID_ROOMS			0xc0000001
#define	MZWID_SECTORS		0xc0000002
#define MZWID_THINGS		0xc0000003
#define MZWID_SPRITES		0xc0000004

enum
{
  cscmdSelectModel = cscmdOK,
  cscmdCreateModel = 0x80000000,
  cscmdPropertiesModel,
  cscmdDeleteModel
};

bool csModelManagerDialog::HandleEvent (csEvent &Event)
{
  switch (Event.Type)
  {
    case csevCommand:
      switch (Event.Command.Code)
      {
        case cscmdRadioButtonSelected:
          RefreshList ();
          return true;
        case cscmdListBoxItemDoubleClicked:
        case cscmdSelectModel:
        {
          csListBoxItem *lbi = (csListBoxItem *)GetChild (MZWID_MODELLIST)->SendCommand (cscmdListBoxQueryFirstSelected);
          if (lbi)
          {
            ((MazeEditor *)app)->SelectModel (lbi->id - 1);
            if ((GLOBAL_MODEL->Type != csmtRoom) || (GLOBAL_MODEL->Type != csmtSector)
             || !MazeEditor::fDraftShowInactiveRooms)
              GLOBAL_FITALL_DELAYED;
            app->Dismiss (Event.Command.Code);
          } /* endif */
          return true;
        } /* endif */
        case cscmdCreateModel:
          ((MazeEditor *)app)->CreateModel (ModelType);
          ModelType = csmtUnknown;
          RefreshList ();
          return true;
        case cscmdPropertiesModel:
          return true;
        case cscmdDeleteModel:
        {
          csListBoxItem *lbi = (csListBoxItem *)GetChild (MZWID_MODELLIST)->SendCommand (cscmdListBoxQueryFirstSelected);
          if (lbi
           && (MessageBox (app, TEXT_WARNING, TEXT_SUREDELETEMODEL,
              CSMBS_WARNING | CSMBS_OKCANCEL) == cscmdOK))
          {
            ((MazeEditor *)app)->DeleteModel (lbi->id - 1);
            ModelType = csmtUnknown;
            RefreshList ();
          } /* endif */
          return true;
        }
      } /* endswitch */
      break;
  } /* endswitch */
  return csDialog::HandleEvent (Event);
}

void csModelManagerDialog::RefreshList ()
{
  csModelType Type;
  if (GetChild (MZWID_ROOMS)->SendCommand (cscmdRadioButtonQuery))
    Type = csmtRoom;
  else if (GetChild (MZWID_SECTORS)->SendCommand (cscmdRadioButtonQuery))
    Type = csmtSector;
  else if (GetChild (MZWID_THINGS)->SendCommand (cscmdRadioButtonQuery))
    Type = csmtThing;
  else if (GetChild (MZWID_SPRITES)->SendCommand (cscmdRadioButtonQuery))
    Type = csmtSprite;
  else
    return;

  if (ModelType == Type)
    return;
  ModelType = Type;

  csListBox *list = (csListBox *)GetChild (MZWID_MODELLIST);
  list->SendCommand (cscmdListBoxClear, NULL);
  int i, count = 0, Models = GLOBAL_MODELS.Length ();
  csListBoxItem *act = NULL;
  for (i = 0; i < Models; i++)
  {
    mz3DModel *m = (mz3DModel *)GLOBAL_MODELS [i];
    if (m->Type == ModelType)
    {
      CHK (csListBoxItem *lbi = new csListBoxItem (list, m->Name, i + 1,
        m == GLOBAL_MODEL ? cslisEmphasized : cslisNormal));
      if (m == GLOBAL_MODEL)
        act = lbi;
      count++;
    } /* endif */
  } /* endfor */
  GetChild (cscmdSelectModel)->SetState (CSS_DISABLED, !count);
  GetChild ((unsigned int)cscmdPropertiesModel)->SetState (CSS_DISABLED, !count);
  GetChild ((unsigned int)cscmdDeleteModel)->SetState (CSS_DISABLED, !count);
  if (act)
    act->SendCommand (cscmdListBoxItemSet, (void *)true);
  list->PlaceItems ();
}

// [x]=====Model manager==[x][x]
// |+-------------+            |
// ||Big hall     | (*)Rooms   |
// ||Left corridor| ( )Things  |
// ||Street       | ( )Sprites |
// ||             |            |
// ||             | [ Select ] |
// ||             | [ Create ] |
// ||             | [ Proper ] |
// ||             | [ Delete ] |
// ||             | [ Cancel ] |
// |+-------------+            |
// +---------------------------+
void ModelManagerDialog (csApp *iApp)
{
  CHK (csWindow *w = new csWindow (iApp, TEXT_MODELMANAGERTITLE,
    CSWS_BUTSYSMENU | CSWS_BUTCLOSE | CSWS_TITLEBAR));
  CHK (csModelManagerDialog *d = new csModelManagerDialog (w));
  w->SetDragStyle (w->GetDragStyle () & ~CS_DRAG_SIZEABLE);

  CHK (csListBox *lb = new csListBox (d, CSLBS_VSCROLL, cslfsThickRect));
  lb->SetRect (5, 16, 155, 206);
  lb->id = MZWID_MODELLIST;
  lb->SetState (CSS_GROUP, true);

  CHK (csStatic *st = new csStatic (d, lb, TEXT_MODELLIST));
  st->SetPos (lb->bound.xmin, 6);

  CHK (csRadioButton *rb = new csRadioButton (d, MZWID_ROOMS));
  rb->SetPos (lb->bound.xmax + 10, 16);
  rb->SetState (CSS_GROUP, true);

  CHK (st = new csStatic (d, rb, TEXT_ROOMSRADIOBUT));
  st->SetPos (rb->bound.xmax, rb->bound.ymin + 2);

  CHK (rb = new csRadioButton (d, MZWID_SECTORS));
  rb->SetPos (lb->bound.xmax + 10, 32);

  CHK (st = new csStatic (d, rb, TEXT_SECTORSRADIOBUT));
  st->SetPos (rb->bound.xmax, rb->bound.ymin + 2);

  CHK (rb = new csRadioButton (d, MZWID_THINGS));
  rb->SetPos (lb->bound.xmax + 10, 48);

  CHK (st = new csStatic (d, rb, TEXT_THINGSRADIOBUT));
  st->SetPos (rb->bound.xmax, rb->bound.ymin + 2);

  CHK (rb = new csRadioButton (d, MZWID_SPRITES));
  rb->SetPos (lb->bound.xmax + 10, 64);

  CHK (st = new csStatic (d, rb, TEXT_SPRITESRADIOBUT));
  st->SetPos (rb->bound.xmax, rb->bound.ymin + 2);

  CHK (csButton *b1 = new csButton (d, cscmdSelectModel, CSBS_DEFAULTVALUE | CSBS_DEFAULT));
  b1->SetText (TEXT_SELECTBUTTON);
  b1->SetSuggestedSize (+32, +2);
  b1->SetPos (lb->bound.xmax + 10, 90);
  b1->SetState (CSS_GROUP, true);

  CHK (csButton *b2 = new csButton (d, cscmdCreateModel, CSBS_DEFAULTVALUE));
  b2->SetText (TEXT_CREATEBUTTON);
  b2->SetRect (b1->bound.xmin, b1->bound.ymax + 4,
               b1->bound.xmax, b1->bound.ymax + 4 + b1->bound.Height ());

  CHK (csButton *b3 = new csButton (d, cscmdPropertiesModel, CSBS_DEFAULTVALUE));
  b3->SetText (TEXT_PROPERTIESBUTTON);
  b3->SetRect (b2->bound.xmin, b2->bound.ymax + 4,
               b2->bound.xmax, b2->bound.ymax + 4 + b2->bound.Height ());

  CHK (csButton *b4 = new csButton (d, cscmdDeleteModel, CSBS_DEFAULTVALUE));
  b4->SetText (TEXT_DELETEBUTTON);
  b4->SetRect (b3->bound.xmin, b3->bound.ymax + 4,
               b3->bound.xmax, b3->bound.ymax + 4 + b3->bound.Height ());

  CHK (csButton *b5 = new csButton (d, cscmdCancel, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b5->SetText (TEXT_CANCELBUTTON);
  b5->SetRect (b4->bound.xmin, b4->bound.ymax + 4,
               b4->bound.xmax, b4->bound.ymax + 4 + b4->bound.Height ());

  // and now set window size and center it.
  w->SetSize (276, 232);
  CenterOnMouse (iApp, w);

  MazeEditor *app = (MazeEditor *)iApp;		// for GLOBAL_XXX macros to work
  int actrb = MZWID_ROOMS;
  if (GLOBAL_MODEL)
    actrb = GLOBAL_MODEL->Type == csmtThing ? MZWID_THINGS :
            GLOBAL_MODEL->Type == csmtSprite ? MZWID_SPRITES :
            MZWID_ROOMS;
  d->GetChild (actrb)->SendCommand (cscmdRadioButtonSet, (void *)true);
  d->RefreshList ();

  iApp->Execute (w);
  delete w;
}

#define MZWID_COORDAXES		0xc0000000
#define MZWID_GRID		0xc0000001
#define MZWID_MODIFYCENTER	0xc0000002
#define MZWID_VERTCROSSES	0xc0000003
#define MZWID_INACTIVEROOMS	0xc0000004
#define MZWID_SNAPTOGRID	0xc0000005
#define MZWID_PICKDISTANCE	0xc0000007
#define MZWID_GRIDSPACING	0xc0000006

// [x]================Draft View Setup============[x][x]
// |+-Gadgets---------------+ +-Draw------------------+|
// ||[x] Coordinate axes    | |[x] Vertices as crosses||
// ||[x] Grid               | |[x] Inactive rooms     ||
// ||[x] Modification center| |[x] Snap to grid       ||
// |+-----------------------+ +-----------------------+|
// | Pick distance             Grid spacing            |
// | .3.....................   .0.25.................. |
// |                                                   |
// |                [ Ok ]     [ Cancel ]              |
// +---------------------------------------------------+
int DraftSetupDialog (csApp *iApp)
{
  CHK (csWindow *w = new csWindow (iApp, TEXT_DRAFTSETUPTITLE,
    CSWS_BUTSYSMENU | CSWS_BUTCLOSE | CSWS_TITLEBAR));
  CHK (csDialog *d = new csDialog (w));
  w->SetDragStyle (w->GetDragStyle () & ~CS_DRAG_SIZEABLE);

  csCheckBox *cb;
  csStatic *st;
  csInputLine *il;
  char strbuff [201];

  // Frame "Gadgets"
  CHK (st = new csStatic (d, NULL, TEXT_DRAFTSETUPGADGETS, csscsFrameLabel));
  st->SetRect (5, 6, 170, 68);

  // Coordinate axes
  CHK (cb = new csCheckBox (d, MZWID_COORDAXES));
  cb->SetPos (9, 16); cb->SetState (CSS_GROUP, true);
  cb->SendCommand (cscmdCheckBoxSet, (void *)MazeEditor::fDraftAxis);

  CHK (st = new csStatic (d, cb, TEXT_DRAFTSETUPCOORDAXIES));
  st->SetPos (cb->bound.xmax, cb->bound.ymin + 2);

  // Grid
  CHK (cb = new csCheckBox (d, MZWID_GRID));
  cb->SetPos (9, 32);
  cb->SendCommand (cscmdCheckBoxSet, (void *)MazeEditor::fDraftGrid);

  CHK (st = new csStatic (d, cb, TEXT_DRAFTSETUPGRID));
  st->SetPos (cb->bound.xmax, cb->bound.ymin + 2);

  // Modification center
  CHK (cb = new csCheckBox (d, MZWID_MODIFYCENTER));
  cb->SetPos (9, 48);
  cb->SendCommand (cscmdCheckBoxSet, (void *)MazeEditor::fDraftModificationAxis);

  CHK (st = new csStatic (d, cb, TEXT_DRAFTSETUPMODIFYCENTER));
  st->SetPos (cb->bound.xmax, cb->bound.ymin + 2);

  // Grid "Draw"
  CHK (st = new csStatic (d, NULL, TEXT_DRAFTSETUPDRAW, csscsFrameLabel));
  st->SetRect (175, 6, 340, 68);

  // Vertices as crosses
  CHK (cb = new csCheckBox (d, MZWID_VERTCROSSES));
  cb->SetPos (179, 16); cb->SetState (CSS_GROUP, true);
  cb->SendCommand (cscmdCheckBoxSet, (void *)MazeEditor::fCrossVertices);

  CHK (st = new csStatic (d, cb, TEXT_DRAFTSETUPVERTCROSSES));
  st->SetPos (cb->bound.xmax, cb->bound.ymin + 2);

  // Show inactive rooms
  CHK (cb = new csCheckBox (d, MZWID_INACTIVEROOMS));
  cb->SetPos (179, 32);
  cb->SendCommand (cscmdCheckBoxSet, (void *)MazeEditor::fDraftShowInactiveRooms);

  CHK (st = new csStatic (d, cb, TEXT_DRAFTSETUPINACTIVEROOMS));
  st->SetPos (cb->bound.xmax, cb->bound.ymin + 2);

  // Snap to grid
  CHK (cb = new csCheckBox (d, MZWID_SNAPTOGRID));
  cb->SetPos (179, 48);
  cb->SendCommand (cscmdCheckBoxSet, (void *)MazeEditor::fSnapToGrid);

  CHK (st = new csStatic (d, cb, TEXT_DRAFTSETUPSNAPTOGRID));
  st->SetPos (cb->bound.xmax, cb->bound.ymin + 2);

  // Pick distance
  CHK (il = new csInputLine (d, sizeof (strbuff) - 1, csifsThickRect));
  il->SetRect (9, 78, 166, 92); il->SetState (CSS_GROUP, true);
  il->id = MZWID_PICKDISTANCE;
  ftoa (strbuff, sqrt (MazeEditor::DraftPickDist));
  il->SetText (strbuff);

  CHK (st = new csStatic (d, il, TEXT_DRAFTSETUPPICKDISTANCE));
  st->SetPos (il->bound.xmin, il->bound.ymin - 10);

  // Grid spacing
  CHK (il = new csInputLine (d, sizeof (strbuff) - 1, csifsThickRect));
  il->SetRect (179, 78, 336, 92); il->SetState (CSS_GROUP, true);
  il->id = MZWID_GRIDSPACING;
  ftoa (strbuff, MazeEditor::DraftGridSpacing);
  il->SetText (strbuff);

  CHK (st = new csStatic (d, il, TEXT_DRAFTSETUPGRIDSPACING));
  st->SetPos (il->bound.xmin, il->bound.ymin - 10);

  // Okay button
  CHK (csButton *b1 = new csButton (d, cscmdOK, CSBS_DEFAULTVALUE | CSBS_DISMISS | CSBS_DEFAULT));
  b1->SetText (TEXT_OKAYBUTTON); b1->SetState (CSS_GROUP, true);
  b1->SetSuggestedSize (+16, +2);
  b1->SetPos (170 - 16 - b1->bound.Width (), 98);

  // Cancel button
  CHK (csButton *b2 = new csButton (d, cscmdCancel, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b2->SetText (TEXT_CANCELBUTTON);
  b2->SetSuggestedSize (+16, +2);
  b2->SetPos (170 + 16, b1->bound.ymin);

  // and now set window size and center it.
  w->SetSize (354, 144);
  CenterOnMouse (iApp, w);

  int retcode = iApp->Execute (w);
  if (retcode == cscmdOK)
  {
    if (MazeEditor::fDraftAxis != (bool)d->GetChild (MZWID_COORDAXES)->SendCommand (cscmdCheckBoxQuery))
    {
      csEvent *ev = new csEvent (0, csevCommand, (unsigned int)cscmdMzToggleAxis);
      iApp->PutEvent (ev);
    } /* endif */
    if (MazeEditor::fDraftGrid != (bool)d->GetChild (MZWID_GRID)->SendCommand (cscmdCheckBoxQuery))
    {
      csEvent *ev = new csEvent (0, csevCommand, (unsigned int)cscmdMzToggleGrid);
      iApp->PutEvent (ev);
    } /* endif */
    MazeEditor::fDraftModificationAxis = (d->GetChild (MZWID_MODIFYCENTER)->SendCommand (cscmdCheckBoxQuery) != NULL);
    MazeEditor::fCrossVertices = (d->GetChild (MZWID_VERTCROSSES)->SendCommand (cscmdCheckBoxQuery) != NULL);
    MazeEditor::fDraftShowInactiveRooms = (d->GetChild (MZWID_INACTIVEROOMS)->SendCommand (cscmdCheckBoxQuery) != NULL);
    MazeEditor::fSnapToGrid = (d->GetChild (MZWID_SNAPTOGRID)->SendCommand (cscmdCheckBoxQuery) != NULL);

    d->GetChild (MZWID_PICKDISTANCE)->GetText (strbuff, sizeof (strbuff));
    ScanStr (strbuff, "%f", &MazeEditor::DraftPickDist);
    MazeEditor::DraftPickDist *= MazeEditor::DraftPickDist;

    d->GetChild (MZWID_GRIDSPACING)->GetText (strbuff, sizeof (strbuff));
    ScanStr (strbuff, "%f", &MazeEditor::DraftGridSpacing);
  }
  delete w;
  return retcode;
}
#define	MZWID_RADIUS		0x0000000
#define MZWID_REDCOLORVALUES	0x0000001
#define MZWID_GREENCOLORVALUES	0x0000002
#define MZWID_BLUECOLORVALUES	0x0000003

// [x]=Light setup dialog[x][x]
// |                          |
// | Radius		   Color   	  |
// | .8..... +/-   .12.... +/-|  
// |			   .12.... +/-|
// |			   .12.... +/-|
// |                          |
// |    [ Ok ] [ Cancel ]     |
// +--------------------------+
int LightParmDialog (csApp *iApp, float &iRadius, csColor &iColorValues)
{
  CHK (csWindow *w = new csWindow (iApp, TEXT_LIGHTDETAILTITLE,
    CSWS_BUTSYSMENU | CSWS_BUTCLOSE | CSWS_TITLEBAR));
  CHK (csDialog *d = new csDialog (w));
  w->SetDragStyle (w->GetDragStyle () & ~CS_DRAG_SIZEABLE);

  CHK (csSpinBox *sb = new csSpinBox (d));
  sb->SetRect (5, 16, 84, 32);
  sb->id = MZWID_RADIUS;
  sb->SetLimits (0, 100);
  sb->SetValue (int(iRadius));

  CHK (csStatic *s = new csStatic (d, sb, TEXT_LIGHTRADIUSVALUE));
  s->SetPos (5, 6);

  CHK (sb = new csSpinBox (d));
  sb->SetRect (90, 16, 169, 32);
  sb->id = MZWID_REDCOLORVALUES;
  sb->SetLimits (3, 255);
  sb->SetValue (int(iColorValues.red*255));

  CHK (s = new csStatic (d, sb, TEXT_REDCOLORVALUES));
  s->SetPos (90, 6);

  CHK (sb = new csSpinBox (d));
  sb->SetRect (90, 44, 169, 64);
  sb->id = MZWID_GREENCOLORVALUES;
  sb->SetLimits (3, 255);
  sb->SetValue (int(iColorValues.green*255));

  CHK (s = new csStatic (d, sb, TEXT_GREENCOLORVALUES));
  s->SetPos (90, 36);

  CHK (sb = new csSpinBox (d));
  sb->SetRect (90, 76, 169, 96);
  sb->id = MZWID_BLUECOLORVALUES;
  sb->SetLimits (3, 255);
  sb->SetValue (int(iColorValues.blue*255));

  CHK (s = new csStatic (d, sb, TEXT_BLUECOLORVALUES));
  s->SetPos (90, 66);

  CHK (csButton *b1 = new csButton (d, cscmdOK, CSBS_DEFAULTVALUE | CSBS_DISMISS | CSBS_DEFAULT));
  b1->SetText (TEXT_OKAYBUTTON);
  b1->SetSuggestedSize (+16, +2);
  b1->SetPos (16, 100);

  CHK (csButton *b2 = new csButton (d, cscmdCancel, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b2->SetText (TEXT_CANCELBUTTON);
  b2->SetSuggestedSize (+16, +2);
  b2->SetPos (b1->bound.xmax + 16, b1->bound.ymin);

  // and now set window size and center it.
  w->SetSize (180, 160);
  CenterOnMouse (iApp, w);

  int retcode = iApp->Execute (w);
  if (retcode == cscmdOK)
  {
    iRadius = (int)d->GetChild (MZWID_RADIUS)->SendCommand (cscmdSpinBoxQueryValue);
    iColorValues.red = (int)d->GetChild (MZWID_REDCOLORVALUES)->SendCommand (cscmdSpinBoxQueryValue) / (float) 255;
    iColorValues.blue = (int)d->GetChild (MZWID_BLUECOLORVALUES)->SendCommand (cscmdSpinBoxQueryValue) / (float) 255;
    iColorValues.green = (int)d->GetChild (MZWID_GREENCOLORVALUES)->SendCommand (cscmdSpinBoxQueryValue) / (float) 255;
  } /* endif */
  delete w;
  return retcode;
}




#define	MZWID_POLYGONLIST	0x00000000
#define	MZWID_SECTORLIST	0x00000001

  // [x]==Portal Generation==[x][x]
// |+-------------+            |
// ||Big hall     | (*)Rooms   |
// ||Left corridor| ( )Things  |
// ||Street       | ( )Sprites |
// ||             |            |
// ||             | [ Select ] |
// ||             | [ Create ] |
// ||             | [ Proper ] |
// ||             | [ Delete ] |
// ||             | [ Cancel ] |
// |+-------------+            |
// +---------------------------+
void PortalManagerDialog (csApp *app)
{
	int m_ind=0, p_ind=0;

  CHK (csWindow *w = new csWindow (app, TEXT_PORTALTITLE,
    CSWS_BUTSYSMENU | CSWS_BUTCLOSE | CSWS_TITLEBAR));

  CHK (csDialog *d = new csDialog (w));
  w->SetDragStyle (w->GetDragStyle () & ~CS_DRAG_SIZEABLE);

  CHK (csListBox *lb1 = new csListBox (d, CSLBS_VSCROLL, cslfsThickRect));
  lb1->SetRect (5, 16, 120, 186);
  lb1->id = MZWID_POLYGONLIST;

  // List all the selected polygons in the world
  lb1->SendCommand (cscmdListBoxClear, NULL);
  int i, count1 = 0, Models = GLOBAL_MODELS.Length ();
  csListBoxItem *act = NULL;
  //for (i = 0; i < Models; i++)
  {
	  // currently we only display selected polygons in current model
//    mz3DModel *m = (mz3DModel *)GLOBAL_MODELS [i];
	mz3DModel *m = GLOBAL_MODEL;
	int j, NumPolys = m->Polygons();
	for (j=0; j<NumPolys; j++)
    {
		mz3DPolygon p = m->Polygon(j);
		if (p.selected) {
			if (count1 == 0)
			// now we only select the current model, so the index must be current
				m_ind = GLOBAL_MODELINDEX, p_ind = j;
			CHK (csListBoxItem *lbi = new csListBoxItem (lb1, p.Name, j + 1,
				m == GLOBAL_MODEL ? cslisEmphasized : cslisNormal));
			if (m == GLOBAL_MODEL)
			act = lbi;
			count1++;
		} /* endif */
	} /* end 2nd for */
  } /* endfor */

  if (count1 <= 0) {
  // no selected polygon, exit
//	System->Printf (MSG_INITIALIZATION, "No polygons being selected!");
	printf ("No polygons being selected!\n");
	return;
  }
  lb1->PlaceItems();
  
  lb1->SetState (CSS_GROUP, true);

  CHK (csStatic *st1 = new csStatic (d, lb1, TEXT_POLYGONLIST));
  st1->SetPos (lb1->bound.xmin, 6);

  CHK (csListBox *lb2 = new csListBox (d, CSLBS_VSCROLL, cslfsThickRect));
  lb2->SetRect (130, 16, 245, 186);
  lb2->id = MZWID_SECTORLIST;

  // List all the models/sectors in the world
  lb2->SendCommand (cscmdListBoxClear, NULL);
  Models = GLOBAL_MODELS.Length ();
  int count2 = 0;
  act = NULL;
  for (i = 0; i < Models; i++)
  {
    mz3DModel *m = (mz3DModel *)GLOBAL_MODELS [i];
    if (m->Type == csmtSector || m->Type == csmtRoom)
    {
      CHK (csListBoxItem *lbi = new csListBoxItem (lb2, m->Name, i + 1,
        m == GLOBAL_MODEL ? cslisEmphasized : cslisNormal));
      if (m == GLOBAL_MODEL)
        act = lbi;
      count2++;
    } /* endif */
  } /* endfor */

  lb2->PlaceItems();
  lb2->SetState (CSS_GROUP, true);

  CHK (csStatic *st2 = new csStatic (d, lb2, TEXT_SECTORLIST));
  st2->SetPos (lb2->bound.xmin, 6);

  CHK (csButton *b5 = new csButton (d, cscmdOK, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b5->SetText (TEXT_OKAYBUTTON);
  b5->SetRect (50, 190, 110, 210);

  CHK (csButton *b6 = new csButton (d, cscmdCancel, CSBS_DEFAULTVALUE | CSBS_DISMISS));
  b6->SetText (TEXT_CANCELBUTTON);
  b6->SetRect (120, 190, 180, 210);

  // and now set window size and center it.
  w->SetSize (276, 232);
  CenterOnMouse (app, w);

  int retcode = app->Execute (w);
  if (retcode == cscmdOK)
  {
      csListBoxItem *lbi = (csListBoxItem *)(d->GetChild (MZWID_SECTORLIST)->SendCommand (cscmdListBoxQueryFirstSelected));
      if (lbi)
      {
		      mz3DModel *m = (mz3DModel *)GLOBAL_MODELS [lbi->id-1], *m2 = (mz3DModel*)GLOBAL_MODELS[m_ind];
			
			  char tmp[80];
			  // follow the original spec. of CS
			  sprintf(tmp, " %s  ", /*m2->Polygon(p_ind).Name,*/ m->Name);
			m2->Polygon(p_ind).portal = strnew(tmp);
	  }
  } /* endif */
  delete w;
}




