/* -----------------------------------------------------------------
                               OS2PAL.CPP
     This class is for manipulation of the OS/2 palette.

     Sample code for the article "Gearing Up For Games" in EDM/2.

                 Article and code by Michael T. Duffy

   ----------------------------------------------------------------- */

// ***********************************************************************
//  Header Files
// ***********************************************************************

#define __IBMC__
#define INCL_WIN
#define INCL_GPI
#define INCL_DOS
#define INCL_ERRORS

// OS/2 API functions
#include <os2.h>
#define  _MEERROR_H_
#include <mmioos2.h>                   /* DIVE  */
#include <dive.h>
#include <fourcc.h>


// ANSI C functions
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "newtypes.hpp"
#include "os2pal.hpp"

#include "errcodes.hpp"

// ***********************************************************************
//  Constant Palette Definitions
// ***********************************************************************

// Note that OS2 palettes are stored with their colors in the order
//  Blue, Greeen, Red, whereas palettes normally used with VGA from DOS
//  (such as PCX palettes) are stored as Red, Green, Blue.

BYTE               abyOS2Default16Palette[] =
                                    {
//  B    G    R       O           B    G    R       O
    0,   0,   0, PC_RESERVED,   171,   0,   0, PC_RESERVED,
    0, 171,   0, PC_RESERVED,   171, 171,   0, PC_RESERVED,
    0,   0, 171, PC_RESERVED,   171,   0, 171, PC_RESERVED,
    0,  85, 171, PC_RESERVED,   171, 171, 171, PC_RESERVED,
   85,  85,  85, PC_RESERVED,   255,  85,  85, PC_RESERVED,
   85, 255,  85, PC_RESERVED,   255, 255,  85, PC_RESERVED,
   85,  85, 255, PC_RESERVED,   255,  85, 255, PC_RESERVED,
   85, 255, 255, PC_RESERVED,   255, 255, 255, PC_RESERVED
                                    };

const BYTE         abyDefault16Palette[] =
                                    {
//   R    G   B          R     G    B
     0,   0,  0,         0,    0, 171,
     0, 171,  0,         0,  171, 171,
   171,   0,  0,       171,    0, 171,
   171,  85,  0,       171,  171, 171,
    85,  85, 85,        85,   85, 255,
    85, 255, 85,        85,  255, 255,
   255,  85, 85,       255,   85, 255,
   255, 255, 85,       255,  255, 255
                                    };

// ***********************************************************************
//  Code
// ***********************************************************************


//........................................................................
OS2Palette::OS2Palette
//........................................................................
  (
  VOID
  )
{
usLastErrorCode  = 0;
usCurrentFormat  = PAL_FMT_UNKNOWN;
SetFlag (0);

hab  = NULLHANDLE;
hdc  = NULLHANDLE;
hps  = NULLHANDLE;
hwnd = NULLHANDLE;
hpal = NULLHANDLE;
};

//........................................................................
OS2Palette::OS2Palette
//........................................................................
  (
  HAB              habIn,
  HWND             hwndIn,
  HDC              hdcIn,
  HPS              hpsIn
  )
{
SetFlag (0);

hab  = habIn;
hdc  = hdcIn;
hps  = hpsIn;
hwnd = hwndIn;
hpal = NULLHANDLE;

InitSystemPalette (habIn, hwndIn, hdcIn, hpsIn);
};

//........................................................................
OS2Palette::~OS2Palette
//........................................................................
  (
  VOID
  )
{


UninitSystemPalette ();

SetFlag (0);

hab  = NULLHANDLE;
hdc  = NULLHANDLE;
hps  = NULLHANDLE;
hwnd = NULLHANDLE;
};

//........................................................................
VOID OS2Palette::Default16
//........................................................................
  (
  VOID
  )
{

memcpy (abyData, abyDefault16Palette, sizeof (abyDefault16Palette));
usCurrentFormat = PAL_FMT_8BIT;
};


//........................................................................
VOID OS2Palette::Convert
//........................................................................
  (
  ULONG            ulFormat
  )
{
LONG               lIndex;
BYTE               byRed;
BYTE               byBlue;
BYTE               byGreen;


switch (ulFormat)
  {
  case PAL_FMT_8BIT:
    {
    if (usCurrentFormat == PAL_FMT_OS2)
      {
      lIndex = 0;
      do
        {
        byRed   = abyData [(lIndex * 4) + 2];
        byGreen = abyData [(lIndex * 4) + 1];
        byBlue  = abyData [(lIndex * 4) + 0];

        abyData [(lIndex * 3) + 0] = byRed;
        abyData [(lIndex * 3) + 1] = byGreen;
        abyData [(lIndex * 3) + 2] = byBlue;

        ++lIndex;
        } while (lIndex < 256);
      usCurrentFormat = PAL_FMT_8BIT;
      }
    };
    break;
  case PAL_FMT_OS2:
    {
    if (usCurrentFormat == PAL_FMT_8BIT)
      {
      lIndex = 255;
      do
        {
        byRed   = abyData [(lIndex * 3) + 0];
        byGreen = abyData [(lIndex * 3) + 1];
        byBlue  = abyData [(lIndex * 3) + 2];

        abyData [(lIndex * 4) + 3] = byFlag;
        abyData [(lIndex * 4) + 2] = byRed;
        abyData [(lIndex * 4) + 1] = byGreen;
        abyData [(lIndex * 4) + 0] = byBlue;

        --lIndex;
        } while (lIndex >= 0);
      usCurrentFormat = PAL_FMT_OS2;
      }
    else if (usCurrentFormat == PAL_FMT_OS2)
      {
      // Just duplicate the flag into all the flag bytes.
      lIndex = 255;
      do
        {
        abyData [(lIndex * 4) + 3] = byFlag;
        --lIndex;
        } while (lIndex >= 0);
      };
    };
    break;
  case PAL_FMT_UNKNOWN:
    {
    // Can't convert an unknown format (blank) palette to anything, so just
    //  don't do anything.
    };
    break;
  default:
    {
    // Bad format.  Possibly signal error here.
    };
    break;
  };
};

//........................................................................
VOID OS2Palette::InitSystemPalette
//........................................................................
  (
  HAB              habIn,
  HWND             hwndIn,
  HDC              hdcIn,
  HPS              hpsIn
  )
{
ULONG              ulcPaletteColors = 256;
SIZEL              sizl;


UninitSystemPalette ();

if (habIn  != NULLHANDLE) hab  = habIn;
if (hdcIn  != NULLHANDLE) hdc  = hdcIn;
if (hpsIn  != NULLHANDLE) hps  = hpsIn;
if (hwndIn != NULLHANDLE) hwnd = hwndIn;

sizl.cx = sizl.cy = 0;

Convert (PAL_FMT_OS2);

hpal = GpiCreatePalette (hab, LCOL_PURECOLOR, LCOLF_CONSECRGB,
                         ulcPaletteColors, (PULONG) abyData);

GpiSelectPalette  (hps, hpal);
WinRealizePalette (hwnd, hps, &ulcPaletteColors);
};

//........................................................................
VOID OS2Palette::UninitSystemPalette
//........................................................................
  (
  VOID
  )
{
if (hps  != NULLHANDLE)  GpiSelectPalette (hps, NULLHANDLE);
if (hpal != NULLHANDLE)  GpiDeletePalette (hpal);

hps  = NULLHANDLE;
hwnd = NULLHANDLE;
};


//........................................................................
VOID OS2Palette::SuspendSystemPalette
//........................................................................
  (
  VOID
  )
{
ULONG              ulcPaletteColors = 256;

GpiSelectPalette  (hps, NULLHANDLE);
WinRealizePalette (hwnd, hps, &ulcPaletteColors);
};

//........................................................................
VOID OS2Palette::RestoreSystemPalette
//........................................................................
  (
  VOID
  )
{
ULONG              ulcPaletteColors = 256;

GpiSelectPalette  (hps, hpal);
WinRealizePalette (hwnd, hps, &ulcPaletteColors);
};


//........................................................................
VOID OS2Palette::SetAsDiveSource
//........................................................................
  (
  HDIVE            hdive
  )
{
Convert (PAL_FMT_OS2);
DiveSetSourcePalette (hdive, 0, 256L, (PBYTE) abyData);
};

//........................................................................
VOID OS2Palette::SetAsDiveDestination
//........................................................................
  (
  HDIVE            hdive
  )
{
Convert (PAL_FMT_OS2);
DiveSetDestinationPalette (hdive, 0, 256L, (PBYTE) abyData);
};

//........................................................................
VOID OS2Palette::LoadActualPalette
//........................................................................
  (
  HPS              hpsIn
  )
{
Convert (PAL_FMT_OS2);

if (hpsIn  != NULLHANDLE) hps = hpsIn;

memset (abyData, 0, sizeof (abyData));
GpiQueryRealColors (hps, 0, 0, 256, (PLONG) abyData);
usCurrentFormat = PAL_FMT_OS2;
};

//........................................................................
VOID OS2Palette::SetValuesFromRaw8
//........................................................................
  (
  PBYTE            pbyValuesIn,
  ULONG            ulStart,
  ULONG            ulNumber
  )
{
ULONG              ulSavedFormat;


ulSavedFormat = usCurrentFormat;

// Get ready to receive 8 bit values
Convert (PAL_FMT_8BIT);
// Copy over the range of values requested
memcpy (&abyData [ulStart * 3], pbyValuesIn, ulNumber * 3);
// return to the original format
Convert (ulSavedFormat);
};

//........................................................................
ERRORSTATUS OS2Palette::PaletteFromBlock
//........................................................................
  (
  PBYTE            pbyBlock,
  ULONG            ulBlockSize
  )
{
ULONG              ulDataSize;
PPALLIBHDR         ppalhdr;


// Set a pointer to the header information ...
ppalhdr = (PPALLIBHDR) pbyBlock;
// ... and move the Block pointer to the beginning of the data.
pbyBlock += sizeof (PALLIBHDR);

// Data is saved as an 8-bit palette regardless of the format
usCurrentFormat = PAL_FMT_8BIT;

// if the format is saved as an OS/2 format, set the byte to use as the flag.
if (ppalhdr->usFormat == PAL_FMT_OS2)
  {
  SetFlag (ppalhdr->byFlag);
  }
else
  {
  SetFlag (0);
  };

// Get the size of the data.
ulDataSize = ulBlockSize - sizeof (PALLIBHDR);

// It should be 768 bytes (256 colors at 3 bytes/entry).
if (ulDataSize != 768)
  {
  usLastErrorCode = ERR_PAL_DATA_CORRUPTED;
  return (ES_ERROR);
  };

memset (abyData, 0, 1024);
memcpy (abyData, pbyBlock, ulDataSize);

// Convert the data to the format it was saved as.
Convert (ppalhdr->usFormat);
return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS OS2Palette::RequestSaveBlock
//........................................................................
  (
  PBYTE *          ppbyBlockOut,
  PULONG           pulSizeOut
  )
{
PPALLIBHDR         ppalhdr;
PBYTE              pbyBlock;
USHORT             usSavedFormat;
ULONG              ulBlockSize;


ulBlockSize = sizeof (PALLIBHDR) + 768;

// Allocate memory for the block of saved data.
if ((pbyBlock = (PBYTE) malloc(ulBlockSize)) == NULL)
  {
  usLastErrorCode = ERR_GEN_ALLOC_FAILURE;
  return (ES_ERROR);
  };

// save the current format so you can return to it when you are done.
usSavedFormat = usCurrentFormat;

// Convert to 8 bit format, because that is how all palettes are to be saved.
Convert (PAL_FMT_8BIT);

// Set the return values.
*ppbyBlockOut    = pbyBlock;
*pulSizeOut      = ulBlockSize;

// set a pointer to the first part of the block.  This will be the header.
ppalhdr          = (PPALLIBHDR) pbyBlock;
memset (ppalhdr, 0, sizeof (PALLIBHDR));

// Fill in and move past header.
ppalhdr->usFormat = usSavedFormat;
ppalhdr->byFlag   = byFlag;
pbyBlock += sizeof (PALLIBHDR);

// Write data.
memcpy (pbyBlock, abyData, 768);

// restore original format.
Convert (usSavedFormat);

return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS OS2Palette::ReleaseSaveBlock
//........................................................................
  (
  PBYTE *          ppbyBlockOut
  )
{

if ((ppbyBlockOut != NULL) &&
    (*ppbyBlockOut != NULL))
  {
  free (*ppbyBlockOut);
  *ppbyBlockOut = NULL;
  return (ES_NO_ERROR);
  }
else
  {
  usLastErrorCode = ERR_GEN_BAD_PARAM;
  return (ES_ERROR);
  };
};

