/* -----------------------------------------------------------------
                               BITMAP.CPP

     This is the base class from which bitmap decompressors will be
  derived.  It should provide all the basic functionality so that
  new compression/decompression methods may be added without affecting
  the main code.

     BitmapPainter is an abstract class, so only derived types may be
  created and used.

     A BitmapPainter will take a buffer which holds the compressed
  graphic information, and write the decompressed image to a Canvas.

     Sample code for the article "Gearing Up For Games" in EDM/2.

                 Article and code by Michael T. Duffy

   ----------------------------------------------------------------- */

// ***********************************************************************
//  Header Files
// ***********************************************************************

// OS/2 type definitions

#define __IBMC__
#include <os2def.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "newtypes"
#include "canvas.hpp"
#include "bitmap.hpp"

#include "errcodes.hpp"

// ***********************************************************************
//  Error Codes
// ***********************************************************************


// ***********************************************************************
//  Code
// ***********************************************************************

//........................................................................
BitmapPainter::BitmapPainter
//........................................................................
  (
  VOID
  )
{


MonoSetWhite (0xff);
MonoSetBlack (0x00);
memset (abyPalette, 0, 768);
pbySourceBuffer   = NULL;
pcnvDestCanvas    = NULL;
ulBufferSize      = 0;
ulBufferOffset    = 0;
usLastErrorCode   = 0;
ulPainterType     = 0;

pbyEncodeBuffer   = NULL;
ulEncBufferSize   = 0;
ulEncBufferOffset = 0;
};

//........................................................................
BitmapPainter::~BitmapPainter
//........................................................................
  (
  VOID
  )
{
};

//........................................................................
ERRORSTATUS BitmapPainter::AssociateBuffer
//........................................................................
  (
  PBYTE            pbyBufferIn,
  ULONG            ulBufferSizeIn
  )
{

// Make sure the buffer is of a valid size
if (ulBufferSizeIn == 0)
  {
  usLastErrorCode = ERR_GEN_BAD_PARAM;
  return (ES_ERROR);
  };

// Set internal variables to work on the requested buffer.
pbySourceBuffer = pbyBufferIn;
ulBufferSize    = ulBufferSizeIn;
ulBufferOffset  = 0;
return (ES_NO_ERROR);
};

//........................................................................
VOID BitmapPainter::DissociateBuffer
//........................................................................
  (
  VOID
  )
{

// Clear internal variables that point to any existing buffer.
pbySourceBuffer = NULL;
ulBufferSize    = 0;
ulBufferOffset  = 0;
};

//........................................................................
ERRORSTATUS BitmapPainter::AssociateCanvas
//........................................................................
  (
  PCANVAS          pcnvCanvasIn
  )
{
// Canvases should not be assigned a NULL value with this function.
//  Use DissociateCanvas
if (pcnvCanvasIn == NULL)
  {
  usLastErrorCode = ERR_GEN_BAD_PARAM;
  return (ES_ERROR);
  };

pcnvDestCanvas = pcnvCanvasIn;
return (ES_NO_ERROR);
};

//........................................................................
VOID BitmapPainter::DissociateCanvas
//........................................................................
  (
  VOID
  )
{
pcnvDestCanvas = NULL;
};


//........................................................................
ERRORSTATUS BitmapPainter::InitializeEncodeBuffer
//........................................................................
  (
  VOID
  )
{


// Allocate initial memory for the buffer.
ulEncBufferSize = BMP_WORK_BUFFER_SIZE;
if ((pbyEncodeBuffer = (PBYTE) malloc (ulEncBufferSize)) == NULL)
  {
  usLastErrorCode = ERR_GEN_ALLOC_FAILURE;
  return (ES_ERROR);
  };
ulEncBufferOffset = 0;
return (ES_NO_ERROR);
};


//........................................................................
ERRORSTATUS BitmapPainter::AssureEncodeMemory
//........................................................................
  (
  ULONG            ulAmount
  )
// This routine makes sure there is enough memory in the encode buffer for
//   storage of a piece of encoded info.  If there is not enough room in the
//   current buffer, a new buffer is created, the old buffer is copied over,
//   and all internal pointers are updated.  If a new buffer cannot be
//   allocated, this routine returns an error.
{
PBYTE              pbyNewBuffer;


if ((ulEncBufferOffset + ulAmount) >= ulEncBufferSize)
  {
  // You need a larger encode buffer
  if ((pbyNewBuffer =
      (PBYTE) malloc (ulEncBufferSize + BMP_WORK_BUFF_INCREMENT)) == NULL)
    {
    usLastErrorCode = ERR_GEN_ALLOC_FAILURE;
    return (ES_ERROR);
    };

  memcpy (pbyNewBuffer, pbyEncodeBuffer, ulEncBufferSize);
  free (pbyEncodeBuffer);
  ulEncBufferSize += BMP_WORK_BUFF_INCREMENT;
  pbyEncodeBuffer  = pbyNewBuffer;
  };
return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS BitmapPainter::TrimEncodeBuffer
//........................................................................
  (
  VOID
  )
// This routine reduces the size of the encode buffer to fit the contained
//   data exactly.
{
PBYTE              pbyNewBuffer;


// Allocate final buffer the same size as existing information
if ((pbyNewBuffer = (PBYTE) malloc (ulEncBufferOffset)) == NULL)
  {
  usLastErrorCode = ERR_GEN_ALLOC_FAILURE;
  return (ES_ERROR);
  };

memcpy (pbyNewBuffer, pbyEncodeBuffer, ulEncBufferSize);
free (pbyEncodeBuffer);

ulEncBufferSize  = ulEncBufferOffset;
pbyEncodeBuffer  = pbyNewBuffer;

return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS BitmapPainter::FreeEncodeBuffer
//........................................................................
  (
  VOID
  )
{

if (pbyEncodeBuffer == NULL)
  {
  usLastErrorCode = ERR_GEN_BAD_PARAM;
  return (ES_ERROR);
  };

free (pbyEncodeBuffer);
pbyEncodeBuffer   = NULL;
ulEncBufferOffset = 0;
ulEncBufferSize   = 0;

return (ES_NO_ERROR);
};


