//-------------------------------------------------------------------------
// The following is part of the VIOWIN user-interface library source code.
// The code is copyright (c) 1994-1995 by Larry Salomon, Jr. and may not
// be used in a for-profit application without the expressed, written
// consent of Larry Salomon, Jr.  All rights reserved.
//-------------------------------------------------------------------------
#define INCL_WINDIALOGS
#define INCL_WININPUT
#define INCL_WINMESSAGEMGR
#define INCL_WINWINDOWMGR
#include "vwdefs.h"
#include <string.h>

HVWWND EXPENTRY vwCreateWindow(PCHAR pchClass,
                               USHORT usId,
                               ULONG ulStyle,
                               PCHAR pchText,
                               LONG lX,
                               LONG lY,
                               ULONG ulCx,
                               ULONG ulCy,
                               LONG lForeClr,
                               LONG lBackClr)
//-------------------------------------------------------------------------
// This function creates a window.
//
// Input:  pchClass - points to the name of the class of the window
//         usId - specifies the window id.  If -1 is specified, the check
//                for other windows with the same id is not performed.
//         ulStyle - specifies the style of the window
//         pchText - points to the text of the window
//         lX, lY - specifies the lower left corner of the window, relative
//                  to the desktop
//         ulCx, ulCy - specifies the width and height of the window
//         lForeClr - specifies the VWCLR_* constant for the foreground color
//         lBackClr - specifies the VWCLR_* constant for the background color
// Returns:  window handle if successful, NULLHANDLE otherwise
//-------------------------------------------------------------------------
{
   PVWCLASSINFO pciClass;
   VWWND vwWnd;
   HVWWND hwndNew;
   BOOL bDesktop;

   if (hmqQueue==NULL) {
      return NULL;
   } /* endif */

   //----------------------------------------------------------------------
   // Make sure the class was registered
   //----------------------------------------------------------------------
   pciClass=(PVWCLASSINFO)CmnLstQueryRecord(habAnchor->hclClasses,0);
   pciClass=(PVWCLASSINFO)CmnLstSearchRecord(pciClass,
                                             pchClass,
                                             (PFNRECCOMP)_findClass);
   if (pciClass==NULL) {
      return NULL;
   } /* endif */

   memset(&vwWnd,0,sizeof(VWWND));
   vwWnd.ulSzStruct=sizeof(VWWND);
   vwWnd.pciClass=pciClass;

   //----------------------------------------------------------------------
   // If the window id is not -1, check for another window with this id.
   //----------------------------------------------------------------------
   if ((usId!=-1) && (vwWindowFromID(usId)!=NULL)) {
      return NULL;
   } /* endif */

   //----------------------------------------------------------------------
   // If there are no other windows, then this window's id *must* be
   // VWWID_DESKTOP.
   //----------------------------------------------------------------------
   if ((CmnLstQueryRecordCount(habAnchor->hclWindows)==0) &&
       (usId!=VWWID_DESKTOP)) {
      return NULL;
   } /* endif */

   vwWnd.usId=usId;
   vwWnd.ulStyle=ulStyle;
   vwWnd.swpSwp.lX=lX;
   vwWnd.swpSwp.lY=lY;
   vwWnd.swpSwp.ulCx=ulCx;
   vwWnd.swpSwp.ulCy=ulCy;
   vwWnd.lForeClr=lForeClr;
   vwWnd.lBackClr=lBackClr;

   if (!CmnLstAddRecord(habAnchor->hclWindows,
                        &vwWnd,
                        LAR_TAIL,
                        (PPVOID)&hwndNew)) {
      return NULL;
   } /* endif */

   //----------------------------------------------------------------------
   // If this is the desktop, set a flag so that we post it a WM_PAINT
   // instead of sending it the message.
   //----------------------------------------------------------------------
   bDesktop=(usId==VWWID_DESKTOP);

   if (bDesktop) {
      habAnchor->ulStatus|=VW_HABST_CREATINGDESK;
   } /* endif */

   //----------------------------------------------------------------------
   // Check for valid initialization
   //----------------------------------------------------------------------
   if (LONGFROMMR(vwSendMsg(hwndNew,WM_CREATE,0,0))==TRUE) {
      CmnLstDeleteRecord(habAnchor->hclWindows,hwndNew);
      habAnchor->ulStatus&=~VW_HABST_CREATINGDESK;
      return NULL;
   } /* endif */

   if (bDesktop) {
      habAnchor->ulStatus&=~VW_HABST_CREATINGDESK;
   } /* endif */

   vwWnd.pchText=NULL;
   vwSetWindowText(hwndNew,pchText);

   //----------------------------------------------------------------------
   // Paint
   //----------------------------------------------------------------------
   if ((habAnchor->ulStatus & VW_HABST_CREATINGDESK)==0) {
      vwSendMsg(hwndNew,WM_PAINT,0,0);
   } else {
      vwPostMsg(hwndNew,WM_PAINT,0,0);
   } /* endif */

   return hwndNew;
}

BOOL EXPENTRY vwDestroyWindow(HVWWND hwndWnd)
//-------------------------------------------------------------------------
// This function destroys a window
//
// Input:  hwndWnd - handle to the window
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (hwndWnd==VWHWND_DESKTOP) {
      hwndWnd=vwWindowFromID(VWWID_DESKTOP);
   } /* endif */

   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   //----------------------------------------------------------------------
   // If this is the desktop window, traverse the window list and destroy
   // all of the children first.
   //----------------------------------------------------------------------
   if (vwQueryWindowUShort(hwndWnd,QWS_ID)==VWWID_DESKTOP) {
      CmnLstTraverseList(habAnchor->hclWindows,(PFNRECFUNC)destroyWindow);
      destroyWindow(hwndWnd);
   } else {
      destroyWindow(hwndWnd);
   } /* endif */

   return TRUE;
}

BOOL EXPENTRY vwIsWindow(HVWWND hwndWnd)
//-------------------------------------------------------------------------
// This function determines if the specified handle is a valid window
// handle or not.
//
// Input:  hwndWnd - handle to query
// Returns:  TRUE if the handle is valid, FALSE otherwise
//-------------------------------------------------------------------------
{
   ULONG ulNumItems;
   ULONG ulIndex;
   HVWWND hwndList;

   if (hmqQueue==NULL) {
      return FALSE;
   } /* endif */

   ulNumItems=CmnLstQueryRecordCount(habAnchor->hclWindows);

   for (ulIndex=0; ulIndex<ulNumItems; ulIndex++) {
      if (ulIndex==0) {
         hwndList=(HVWWND)CmnLstQueryRecord(habAnchor->hclWindows,0);
      } else {
         hwndList=(HVWWND)CmnLstQueryRelative(hwndList,LQR_NEXT);
      } /* endif */

      if (hwndList==hwndWnd) {
         return TRUE;
      } /* endif */
   } /* endfor */

   return FALSE;
}

BOOL EXPENTRY vwIsWindowEnabled(HVWWND hwndWnd)
//-------------------------------------------------------------------------
// This function determines if the window is enabled or not.
//
// Input:  hwndWnd - handle to query
// Returns:  TRUE if the window is enabled, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   return ((hwndWnd->ulStyle & WS_DISABLED)==0);
}

HVWWND EXPENTRY vwWindowFromID(USHORT usId)
//-------------------------------------------------------------------------
// This function returns the handle of the window with the specified id.
//
// Input:  usId - specifies the id to search for
// Returns:  handle of the window with the specified id if found, NULLHANDLE
//           otherwise
//-------------------------------------------------------------------------
{
   HVWWND hwndFound;

   if (hmqQueue==NULL) {
      return NULL;
   } /* endif */

   //----------------------------------------------------------------------
   // If the id is -1, then the caller doesn't care about the window id,
   // so we should return NULL to allow for multiple windows with this
   // id can be used multiple times.
   //----------------------------------------------------------------------
   if (usId==(USHORT)-1) {
      return NULL;
   } /* endif */

   hwndFound=(HVWWND)CmnLstQueryRecord(habAnchor->hclWindows,0);
   hwndFound=(HVWWND)CmnLstSearchRecord(hwndFound,
                                        &usId,
                                        (PFNRECCOMP)_findWindowId);
   return hwndFound;
}

BOOL EXPENTRY vwEnableWindow(HVWWND hwndWnd,BOOL bEnable)
//-------------------------------------------------------------------------
// This function sets or resets the WS_DISABLED bit for the window.
//
// Input:  hwndWnd - specifies the window handle
//         bEnable - flag indicating the desired enabled state
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   if (bEnable) {
      hwndWnd->ulStyle&=~WS_DISABLED;
   } else {
      hwndWnd->ulStyle|=WS_DISABLED;
   } /* endif */

   return TRUE;
}

BOOL EXPENTRY vwSetWindowText(HVWWND hwndWnd,PCHAR pchText)
//-------------------------------------------------------------------------
// This function attempts to set the window text by sending the window
// a WM_SETWINDOWPARAMS message.  It is the responsibility of the window
// class to copy the text to its own workspace.
//
// Input:  hwndWnd - specifies the window handle
//         pchText - points to the new text
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   WNDPARAMS wpParms;

   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   if (pchText!=NULL) {
      wpParms.pszText=pchText;
   } else {
      wpParms.pszText="";
   } /* endif */

   wpParms.fsStatus=WPM_TEXT;
   wpParms.cchText=strlen(pchText)+1;

   if (SHORT1FROMMP(vwSendMsg(hwndWnd,
                              WM_SETWINDOWPARAMS,
                              MPFROMP(&wpParms),
                              0))) {
      return TRUE;
   } else {
      return FALSE;
   } /* endif */
}

BOOL EXPENTRY vwQueryWindowText(HVWWND hwndWnd,ULONG ulSzBuf,PCHAR pchBuf)
//-------------------------------------------------------------------------
// This function returns the window text by sending the window a
// WM_QUERYWINDOWPARAMS message.
//
// Input:  hwndWnd - specifies the window handle
//         ulSzBuf - specifies the size of the buffer
//         pchBuf - points to the buffer to receive the text
// Output:  pchBuf - points to the buffer containing the text
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   WNDPARAMS wpParms;

   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   if (pchBuf==NULL) {
      return FALSE;
   } /* endif */

   wpParms.fsStatus=WPM_TEXT;
   wpParms.cchText=ulSzBuf;
   wpParms.pszText=pchBuf;

   if (SHORT1FROMMP(vwSendMsg(hwndWnd,
                              WM_QUERYWINDOWPARAMS,
                              MPFROMP(&wpParms),
                              0))) {
      return TRUE;
   } else {
      return FALSE;
   } /* endif */
}

ULONG EXPENTRY vwQueryWindowTextLength(HVWWND hwndWnd)
//-------------------------------------------------------------------------
// This function returns the length of the window text.
//
// Input:  hwndWnd - specifies the window handle
// Returns:  length if successful, 0 otherwise
//-------------------------------------------------------------------------
{
   WNDPARAMS wpParms;

   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   wpParms.fsStatus=WPM_CCHTEXT;

   if (SHORT1FROMMP(vwSendMsg(hwndWnd,
                              WM_QUERYWINDOWPARAMS,
                              MPFROMP(&wpParms),
                              0))) {
      return wpParms.cchText;
   } else {
      return 0;
   } /* endif */
}

BOOL EXPENTRY vwQueryWindowPos(HVWWND hwndWnd,PVWSWP pswpSwp)
//-------------------------------------------------------------------------
// This function returns the size and position of the window.
//
// Input:  hwndWnd - specifies the window handle
//         pswpSwp - points to the VWSWP structure to receive the result
// Output:  pswpSwp - points to the VWSWP structure containing the result
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   (*pswpSwp)=hwndWnd->swpSwp;
   return TRUE;
}

BOOL EXPENTRY vwQueryWindowRect(HVWWND hwndWnd,PRECTL prclRect)
//-------------------------------------------------------------------------
// This function returns the size of the window.
//
// Input:  hwndWnd - specifies the window handle
//         prclRect - points to the RECTL structure to receive the result
// Output:  prclRect - points to the RECTL structure containing the result
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   prclRect->xLeft=0;
   prclRect->yBottom=0;
   prclRect->xRight=hwndWnd->swpSwp.ulCx;
   prclRect->yTop=hwndWnd->swpSwp.ulCy;
   return TRUE;
}

PVOID EXPENTRY vwQueryWindowPtr(HVWWND hwndWnd,ULONG ulPtr)
//-------------------------------------------------------------------------
// This function returns the specified window pointer.
//
// Input:  hwndWnd - specifies the window handle
//         ulPtr - specifies 0 or 1
// Returns:  specified window pointer if successful, NULL otherwise
//-------------------------------------------------------------------------
{
   if (!vwIsWindow(hwndWnd)) {
      return NULL;
   } /* endif */

   if (ulPtr>1) {
      return NULL;
   } /* endif */

   return hwndWnd->pvData[ulPtr];
}

BOOL EXPENTRY vwSetWindowPtr(HVWWND hwndWnd,ULONG ulPtr,PVOID pvData)
//-------------------------------------------------------------------------
// This function set the specified window pointer.
//
// Input:  hwndWnd - specifies the window handle
//         ulPtr - specifies 0 or 1
//         pvData - specifies the window pointer
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   if (ulPtr>1) {
      return FALSE;
   } /* endif */

   hwndWnd->pvData[ulPtr]=pvData;
   return TRUE;
}

SHORT EXPENTRY vwQueryWindowUShort(HVWWND hwndWnd,LONG lIndex)
//-------------------------------------------------------------------------
// This function returns the value of the specified window variable.
//
// Input:  hwndWnd - specifies the window handle
//         lIndex - specifies a QWS_* constant
// Returns:  value of the variable if successful, -1 otherwise
//-------------------------------------------------------------------------
{
   if (!vwIsWindow(hwndWnd)) {
      return -1;
   } /* endif */

   switch (lIndex) {
   case QWS_ID:
      return hwndWnd->usId;
   default:
      return -1;
   } /* endswitch */
}

BOOL EXPENTRY vwSetWindowUShort(HVWWND hwndWnd,LONG lIndex,USHORT usValue)
//-------------------------------------------------------------------------
// This function sets the value of the specified window variable.
//
// Input:  hwndWnd - specifies the window handle
//         lIndex - specifies a QWS_* constant
//         usValue - specifies the new value
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   switch (lIndex) {
   default:
      return FALSE;
   } /* endswitch */

   return TRUE;
}

LONG EXPENTRY vwQueryWindowULong(HVWWND hwndWnd,LONG lIndex)
//-------------------------------------------------------------------------
// This function returns the value of the specified window variable.
//
// Input:  hwndWnd - specifies the window handle
//         lIndex - specifies a QWL_* constant
// Returns:  value of the variable if successful, -1 otherwise
//-------------------------------------------------------------------------
{
   if (!vwIsWindow(hwndWnd)) {
      return -1;
   } /* endif */

   switch (lIndex) {
   case QWL_STYLE:
      return hwndWnd->ulStyle;
   default:
      return -1;
   } /* endswitch */
}

BOOL EXPENTRY vwSetWindowULong(HVWWND hwndWnd,LONG lIndex,ULONG ulValue)
//-------------------------------------------------------------------------
// This function sets the value of the specified window variable.
//
// Input:  hwndWnd - specifies the window handle
//         lIndex - specifies a QWL_* constant
//         ulValue - specifies the new value
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   switch (lIndex) {
   case QWL_STYLE:
      hwndWnd->ulStyle=ulValue;
      break;
   default:
      return FALSE;
   } /* endswitch */

   return TRUE;
}

HVWWND EXPENTRY vwQueryWindow(HVWWND hwndWnd,LONG lCmd)
//-------------------------------------------------------------------------
// This function returns the handle of a window relative to the specified
// window.
//
// Input:  hwndWnd - specifies the window handle
//         lCmd - specifies a QW_* constant
// Returns:  window handle if successful, NULLHANDLE otherwise
//-------------------------------------------------------------------------
{
   ULONG ulNumRecs;
   HVWWND hwndReturn;

   if (!vwIsWindow(hwndWnd)) {
      return NULL;
   } /* endif */

   ulNumRecs=CmnLstQueryRecordCount(habAnchor->hclWindows);

   switch (lCmd) {
   case QW_NEXT:
      hwndReturn=CmnLstQueryRelative(hwndWnd,LQR_NEXT);
      break;
   case QW_PREV:
      hwndReturn=CmnLstQueryRelative(hwndWnd,LQR_PREVIOUS);

      if (hwndReturn==vwWindowFromID(VWWID_DESKTOP)) {
         hwndReturn=NULL;
      } /* endif */
      break;
   case QW_TOP:
      hwndReturn=CmnLstQueryRecord(habAnchor->hclWindows,1);
      break;
   case QW_BOTTOM:
      hwndReturn=CmnLstQueryRecord(habAnchor->hclWindows,ulNumRecs-1);
      break;
   default:
      return NULL;
   } /* endswitch */

   return hwndReturn;
}

LONG EXPENTRY vwQueryForeColor(HVWWND hwndWnd)
//-------------------------------------------------------------------------
// This function returns the foreground color of the window.
//
// Input:  hwndWnd - specifies the window handle
// Returns:  VWCLR_* constant if successful, -1 otherwise
//-------------------------------------------------------------------------
{
   if (hwndWnd==VWHWND_DESKTOP) {
      hwndWnd=vwWindowFromID(VWWID_DESKTOP);
   } /* endif */

   if (!vwIsWindow(hwndWnd)) {
      return -1;
   } /* endif */

   return hwndWnd->lForeClr;
}

LONG EXPENTRY vwQueryBackColor(HVWWND hwndWnd)
//-------------------------------------------------------------------------
// This function returns the background color of the window.
//
// Input:  hwndWnd - specifies the window handle
// Returns:  VWCLR_* constant if successful, -1 otherwise
//-------------------------------------------------------------------------
{
   if (hwndWnd==VWHWND_DESKTOP) {
      hwndWnd=vwWindowFromID(VWWID_DESKTOP);
   } /* endif */

   if (!vwIsWindow(hwndWnd)) {
      return -1;
   } /* endif */

   return hwndWnd->lBackClr;
}

BOOL EXPENTRY vwSetForeColor(HVWWND hwndWnd,LONG lColor)
//-------------------------------------------------------------------------
// This function sets the foreground color of the window.
//
// Input:  hwndWnd - specifies the window handle
//         lColor - specifies the new color
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (hwndWnd==VWHWND_DESKTOP) {
      hwndWnd=vwWindowFromID(VWWID_DESKTOP);
   } /* endif */

   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   hwndWnd->lForeClr=lColor;
   return TRUE;
}

LONG EXPENTRY vwSetBackColor(HVWWND hwndWnd,LONG lColor)
//-------------------------------------------------------------------------
// This function sets the background color of the window.
//
// Input:  hwndWnd - specifies the window handle
//         lColor - specifies the new color
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (hwndWnd==VWHWND_DESKTOP) {
      hwndWnd=vwWindowFromID(VWWID_DESKTOP);
   } /* endif */

   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   hwndWnd->lBackClr=lColor;
   return TRUE;
}

BOOL EXPENTRY vwUpdateWindow(HVWWND hwndWnd)
//-------------------------------------------------------------------------
// This function updates the specified window.  If the window is the
// desktop, it updates all of the children also.
//
// Input:  hwndWnd - specifies the window handle
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   if (hwndWnd==VWHWND_DESKTOP) {
      hwndWnd=vwWindowFromID(VWWID_DESKTOP);
   } /* endif */

   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   if (vwQueryWindowUShort(hwndWnd,QWS_ID)==VWWID_DESKTOP) {
      //-------------------------------------------------------------------
      // Since the desktop is first in the list, traverse the list and
      // send each window a WM_PAINT message.
      //-------------------------------------------------------------------
      CmnLstTraverseList(habAnchor->hclWindows,(PFNRECFUNC)paintWindow);
   } else {
      paintWindow(hwndWnd);
   } /* endif */

   return TRUE;
}

MRESULT EXPENTRY vwDefWindowProc(HVWWND hwndWnd,
                                 ULONG ulMsg,
                                 MPARAM mpParm1,
                                 MPARAM mpParm2)
//-------------------------------------------------------------------------
// This is the default window procedure for windows.  It processes certain
// messages only.  Others, it passes to the desktop window, unless it is
// the desktop window which is calling this function.
//-------------------------------------------------------------------------
{
   HVWWND hwndDesktop;

   if (!vwIsWindow(hwndWnd)) {
      return MRFROMLONG(0);
   } /* endif */

   hwndDesktop=vwWindowFromID(VWWID_DESKTOP);

   switch (ulMsg) {
   case WM_QUERYWINDOWPARAMS:
      {
         PWNDPARAMS pwpParms;

         pwpParms=(PWNDPARAMS)PVOIDFROMMP(mpParm1);

         switch (pwpParms->fsStatus) {
         case WPM_CCHTEXT:
            pwpParms->cchText=strlen(hwndWnd->pchText)+1;
            return MRFROMLONG(TRUE);
         case WPM_TEXT:
            *pwpParms->pszText=0;
            strncat(pwpParms->pszText,hwndWnd->pchText,pwpParms->cchText-1);
            return MRFROMLONG(TRUE);
         default:
            break;
         } /* endswitch */
      }
      break;
   case WM_SETWINDOWPARAMS:
      {
         PWNDPARAMS pwpParms;

         pwpParms=(PWNDPARAMS)PVOIDFROMMP(mpParm1);

         switch (pwpParms->fsStatus) {
         case WPM_TEXT:
            {
               PCHAR pchText;

               CmnMemAllocate(habAnchor->hcmWork,
                              pwpParms->cchText,
                              (PPVOID)&pchText);
               if (pchText==NULL) {
                  return FALSE;
               } /* endif */

               strcpy(pchText,pwpParms->pszText);

               if (hwndWnd->pchText!=NULL) {
                  CmnMemFree(habAnchor->hcmWork,hwndWnd->pchText);
               } /* endif */

               hwndWnd->pchText=pchText;
               return MRFROMLONG(TRUE);
            }
         default:
            break;
         } /* endswitch */
      }
      break;
   case WM_CLOSE:
      vwPostMsg(hwndWnd,WM_QUIT,0,0);
      return MRFROMLONG(0);
   default:
      break;
   } /* endswitch */

   if (hwndWnd!=hwndDesktop) {
      //-------------------------------------------------------------------
      // If the message wasn't processed and the window isn't the desktop,
      // pass the message up the chain.  Some messages get sent; the rest
      // get posted.
      //-------------------------------------------------------------------
      switch (ulMsg) {
      case WM_COMMAND:
      case WM_CONTROL:
      case WM_CHAR:
         if (vwInSendMsg()) {
            return vwSendMsg(hwndDesktop,ulMsg,mpParm1,mpParm2);
         } else {
            vwPostMsg(hwndDesktop,ulMsg,mpParm1,mpParm2);
            return MRFROMLONG(0);
         } /* endif */
      default:
         return MRFROMLONG(0);
      } /* endswitch */
   } else {
      return MRFROMLONG(0);
   } /* endif */
}
