/***************************************************************************
 *
 * MODULE:	RMX Starter application
 * SOURCE:	$Source$
 * OVERVIEW:	This file contains the implementation of rmxstart.exe,
 *              a application that starts remote applications using
 *              rmxstrtr.exe running on the other computer.
 *
 * Copyright (c) 1995 Johan Wikman (johan.wikman@ntc.nokia.com)
 *
 * $Log$
 *
 ****************************************************************************/

#define INCL_NOCOMMON
#define INCL_DOSERRORS
#define INCL_DOSFILEMGR
#define INCL_DOSPROCESS
#include "rmxstart.h"
#include <rmxcomms.h>

#include <iostream.h>
#include <new.h>
#include <string.h>
#include <os2.h>



/****************************************************************************
 * MODULE VARIABLES AND CONSTANTS
 ****************************************************************************/

const char* const USAGE  = "usage: rmxstart display cpu "
                           "application [arguments]";
const ULONG MAX_ATTEMPTS = 5;
const ULONG SIZE_BUFFER  = 4096;


/****************************************************************************
 * MODULE FUNCTIONS
 ****************************************************************************/

static HCONNECTION OpenConnection(PCSZ pcszHost, PCSZ pcszPort);
static VOID        ReadReply     (HCONNECTION hConn);
static BOOL        SendRequest   (HCONNECTION hConn,
				  PCSZ        pcszDisplay, 
				  PCSZ        pcszApplication,
				  PCSZ        pcszArguments);


/****************************************************************************
 * MAIN
 ****************************************************************************/

int main(int argc, char* argv[])
{
  if ((argc != 4) && (argc != 5))
    {
      cerr << USAGE << endl;
      return EXIT_FLAGS;
    }

  PCSZ
    pcszDisplay     = argv[1],
    pcszHost        = argv[2],
    pcszApplication = argv[3],
    pcszArguments   = 0;
  
  if (argc == 5)
    pcszArguments = argv[4];

  // Out of memory handler.
  
  set_new_handler(OutOfMem);

  PSZ
    pszName = GetStarterName();
  
  if (!pszName)
    {
      cerr << "rmxstart: Failed to get the starter port. Is the "
	   << "initialization file of the communications DLL you "
	   << "are using up to date?"
	   << endl;
      
      return EXIT_INIT;
    }
  
  ULONG
    rc = EXIT_OK;
  HCONNECTION
    hConn = OpenConnection(pcszHost, pszName);

  delete [] pszName;

  if (hConn)
    {
      if (SendRequest(hConn, pcszDisplay, pcszApplication, pcszArguments))
	ReadReply(hConn);
      else
	rc = EXIT_COMM;

      RmxClose(hConn);
    }
  else
    rc = EXIT_COMM;

  return rc;
}


/****************************************************************************
 * MODULE FUNCTIONS (IMPLEMENTATION)
 ****************************************************************************
 *
 * FUNCTION: HCONNECTION OpenConnection(PCSZ pcszHost, PCSZ pcszPort);
 *
 * INPUT:
 *    pcszHost: The name of the remote computer.
 *    pcszPort: The port the starter is listening on.
 *
 * RETURN:
 *    A handle to the starter connection.
 *
 * OVERVIEW:
 *    We do not know (nor are we interested in knowing) what the actual
 *    port is.
 *
 ****************************************************************************/

static HCONNECTION OpenConnection(PCSZ pcszHost, PCSZ pcszPort)
{
  HCONNECTION
    hConn = 0;
  ULONG
    ulAttempts = 1,
    rc;

  do
    {
      rc = RmxOpen(pcszHost, pcszPort, &hConn);
      
      if (rc == ERROR_PIPE_BUSY)
	// When the connection busy, we sleep longer and longer periods. 
	DosSleep(ulAttempts * 200);
      
    }
  while ((rc == ERROR_PIPE_BUSY) && (ulAttempts++ < MAX_ATTEMPTS));

  switch (rc)
    {
    case NO_ERROR:
      break;
      
    case ERROR_PIPE_BUSY:
      cerr << "rmxstart: The starter is busy. Retry." << endl;
      break;

    case ERROR_INVALID_NAME:
      cerr << "rmxstart: The starter is not found. "
	      "Is the network and starter running?" << endl;
      break;

    case ERROR_PATH_NOT_FOUND:
      cerr << "rmxstart: Cannot find the starter. Is the network running."
	   << endl;
      break;

    default:
      cerr << "rmxstart: A non-specified error (" << rc 
	   << ") occurred. Retry." << endl;
    }
  
  return hConn;
}


/****************************************************************************
 *
 * FUNCTION: VOID ReadReply(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: Handle to the starter connection.
 *
 * OVERVIEW:
 *    This function reads the starter reply and prints an appropriate message.
 *
 ****************************************************************************/

VOID ReadReply(HCONNECTION hConn)
{
  ULONG
    ulBytesRead,
    ulSize = SIZE_BUFFER;
  BYTE
    *pbReply = 0;
  ULONG
    rc;
  
  do
    {
      pbReply = new BYTE [ulSize];
      
      rc = RmxRead(hConn, pbReply, ulSize, &ulBytesRead);
      
      if (rc == RMXERR_ENLARGE_BUFFER)
	{
	  delete [] pbReply;
	  ulSize = ulBytesRead;
	}
    }
  while (rc == RMXERR_ENLARGE_BUFFER);

  if (rc || (ulBytesRead != sizeof(ULONG)))
    {
      delete [] pbReply;
      cerr << "rmxstart: Failed to read reply from starter." << endl;
      return;
    }

  switch (*((ULONG*) pbReply))
    {
    case RMXAPP_EXEC_FAILED:
      cout << "rmxstart: The application could not be started." << endl;
      break;
      
    case RMXAPP_INVALID_REQUEST:
      // Should never happen
      cout << "rmxstart: The starter could not recognize the request." << endl;
      break;
      
    case RMXAPP_NOT_FOUND:
      cout << "rmxstart: The application or a DLL it depends on "
	   << "was not found." << endl;
      break;
      
    case RMXAPP_NOT_RMXABLE:
      cout << "rmxstart: The application has not been marked for RMX." << endl;
      break;
      
    case RMXAPP_RESOURCE_EXHAUSTED:
      cout << "rmxstart: The application could not be started due to "
	      "resource exhaustion." << endl;
      break;
      
    case RMXAPP_STARTED:
      break;
      
    default:
      // Should never happen.
      cout << "rmxstart: Unknown code returned from starter." << endl;
    }

  delete [] pbReply;
}


/****************************************************************************
 *
 * FUNCTION: VOID SendRequest(HCONNECTION hConn,
 *		 	      PCSZ        pcszDisplay, 
 *			      PCSZ        pcszApplication,
 *			      PCSZ        pcszArguments)
 *
 * INPUT:
 *    hConn:           Handle to the starter connection.
 *    pcszDisplay:     The display variable of this machine.
 *    pcszApplication: The application to be started.
 *    pcszArguments:   The arguments of the application.
 *
 * OVERVIEW:
 *    This function builds a proper request and sends it to the starter.
 *
 ****************************************************************************/

static BOOL SendRequest(HCONNECTION hConn, 
			PCSZ        pcszDisplay, 
			PCSZ        pcszApplication,
			PCSZ        pcszArguments)
{
  ULONG
    lenDisplay     = strlen(pcszDisplay) + 1,
    lenApplication = strlen(pcszApplication) + 1,
    lenArguments   = pcszArguments ? strlen(pcszArguments) + 1 : 0,
    lenRequest     = lenDisplay + lenApplication + lenArguments + 1;
  PSZ
    pszRequest = new CHAR[lenRequest],
    p          = pszRequest;

  strcpy(p, pcszDisplay);
  p += lenDisplay;
  strcpy(p, pcszApplication);
  p += lenApplication;
  
  if (pcszArguments)
    {
      strcpy(p, pcszArguments);
      p += lenArguments;
    }
  
  *p = 0; // Double NULL at last.
  
  ULONG
    rc = RmxWrite(hConn, pszRequest, lenRequest);
  
  delete [] pszRequest;
  
  if (rc == NO_ERROR)
    return TRUE;
  else
    {
      cerr << "rmxstart: Failed to send request to starter." << endl;
      return FALSE;
    }
}


