/*****************************************************************************
 * Server.exe
 ***************************************************************************** 
 *
 * Source:   $Source$
 *
 * Overview: This module contain the source for the server test
 *           program of RMX.   
 *
 * $Log$
 *
 ***************************************************************************** 
 *
 * Copyright (c) 1995 Johan Wikman
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appear in all copies and 
 * that both that copyright notice and this permission notice appear in 
 * supporting documentation.
 *
 * THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY
 * APPLICABLE LAW. THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
 * IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
 * ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 
 *
 *****************************************************************************/

/*****************************************************************************
 * INCLUDE FILES
 *****************************************************************************/

#define INCL_NOCOMMON
#define INCL_NOPMAPI
#define INCL_DOSERRORS
#define INCL_DOSMEMMGR
#define INCL_DOSPROCESS
#define INCL_DOSSEMAPHORES
#include <common.h>
#include <rmxcomms.h>

#include <iostream.h>
#include <process.h>
#include <stdio.h>
#include <string.h>
#include <os2.h>


/*****************************************************************************
 * EXTERNAL VARIABLES
 *****************************************************************************/

PSZ pszAppName = "Server"; // User by common


/****************************************************************************
 * MODULE VARIABLES
 ****************************************************************************/

const ULONG SIZE_NUMBER = 12; // Characters required to display any LONG.

const ULONG TIME_10SEC  = 10000;


/*****************************************************************************
 * MODULE FUNCTIONS
 *****************************************************************************/

PVOID AllocSharedMem(PSZ pszName);
void  ClientThread  (void*);
HEV   CreateEventSem(PSZ pszName);
void  HandleClient  (HCONNECTION hConn);
void  SendError     (HCONNECTION hConn, ULONG ulCode);
void  StartEngine   (HCONNECTION hConn, BOOL bVerbose);


/*****************************************************************************
 * MAIN
 *****************************************************************************/

int main()
{
  PSZ
    pszServer = GetPortName();
  BOOL
    bError = FALSE;

  Print("Server started");

  while (!bError)
    {
      ULONG
	rc;
      HCONNECTION
	hConn;

      rc = RmxCreate(pszServer, &hConn);
      
      if (rc)
	{
	  Print("RmxCreate:", ErrorAsString(rc));
	  Print("Fatal error, exiting");
	  
	  bError = TRUE;
	  continue;
	}

      rc = RmxConnect(hConn);

      if (rc)
	{
	  Print("RmxConnect:", ErrorAsString(rc));
	  Print("Ignoring error");
	  
	  RmxClose(hConn);
	  continue;
	}

      Print("Connection opened, spawning handler thread");

      int
	tid = _beginthread(ClientThread, SIZE_STACK, (void*) hConn);

      if (tid == -1)
	{
	  Print("Failed to create handler thread, ignoring client request");
	  
	  RmxDisConnect(hConn);
	  RmxClose(hConn);
	}
    }
  
  return 1;
}


/*****************************************************************************
 * MODULE FUNCTIONS (IMPLEMENTATION)
 *****************************************************************************
 * 
 * FUNCTION: PVOID AllocSharedMem(PSZ pszMemName);
 *
 * OUTPUT:
 *    pszMemName: Upon successful return the buffer pointed to by this
 *                argument will contain the name of the shared memory object.
 *
 * RETURN:
 *    Pointer to shared memory.
 *
 * OVERVIEW:
 *    This function allocates a shared memory object. Failure is
 *    reported by returning NULL. It is the callers responsibility to
 *    deallocate the memory.
 *
 *****************************************************************************/

PVOID AllocSharedMem(PSZ pszMemName)
{
  PPIB
    ppib;
  PTIB
    ptib;
  
  DosGetInfoBlocks(&ptib, &ppib);
  
  sprintf(pszMemName, "\\SHAREMEM\\RMX\\SERVER\\%u\\%u",
	  ppib->pib_ulpid, ptib->tib_ptib2->tib2_ultid);

  PVOID
    pvMem;
  ULONG
    rc = DosAllocSharedMem(&pvMem, pszMemName, SIZE_PAGE, 
			   PAG_READ | PAG_WRITE| PAG_COMMIT);
  
  if (rc)
    pvMem = 0;
  else
    memset(pvMem, SIZE_PAGE, 0);
  
  return pvMem;
}


/*****************************************************************************
 * 
 * FUNCTION: void ClientThread(void* pvArg);
 *
 * INPUT:
 *    pvArg: The thread argument.
 *
 * OVERVIEW:
 *    The thread function.
 *
 *****************************************************************************/

void ClientThread(void* pvArg)
{
  HCONNECTION
    hConn = (HCONNECTION) pvArg;

  HandleClient(hConn);
}


/*****************************************************************************
 * 
 * FUNCTION: HEV CreateEventSem(PSZ pszSemName);
 *
 * INPUT:
 *    pszSemName: Upon successful return the buffer pointed to by this
 *                argument will contain the name of the event semaphore.
 *
 * OVERVIEW:
 *    This function creates a named event semaphore and returns its
 *    handle. Failure is reported by returning 0.
 *
 *****************************************************************************/

HEV CreateEventSem(PSZ pszSemName)
{
  PPIB
    ppib;
  PTIB
    ptib;
  
  DosGetInfoBlocks(&ptib, &ppib);
  
  sprintf(pszSemName, "\\SEM32\\RMX\\SERVER\\%u\\%u",
	  ppib->pib_ulpid, ptib->tib_ptib2->tib2_ultid);
  
  HEV
    hev;
  ULONG
    rc = DosCreateEventSem(pszSemName, &hev, DC_SEM_SHARED, FALSE);
  
  if (rc)
    hev = 0;
  
  return hev;
}


/*****************************************************************************
 * 
 * FUNCTION: void HandleClient(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The connection handle.
 *
 * OVERVIEW:
 *    This function reads the request sent by the client, and, if it is
 *    correct, starts an engine that provides services to that
 *    particular client.
 *
 *****************************************************************************/

void HandleClient(HCONNECTION hConn)
{
  PSZ
    pbBuffer = 0;
  ULONG
    ulBufferSize = 0,
    ulMessageSize;
  
  ulMessageSize = ReadMessage(hConn, &pbBuffer, &ulBufferSize);
  
  if (ulMessageSize == (2 * sizeof(ULONG)))
    {
      ULONG
	ulMessage = *(ULONG*)pbBuffer;
      BOOL
	bVerbose = *(((ULONG*) pbBuffer) + 1);
      
      if (ulMessage == REQUEST_CONNECT)
	StartEngine(hConn, bVerbose);
      else
	SendError(hConn, REPLY_INVALID_REQUEST);
    }
  else
    SendError(hConn, REPLY_INVALID_REQUEST);
	
  delete [] pbBuffer;

  ULONG
    rc;

  rc = RmxDisConnect(hConn);

  if (rc)
    Print("RmxDisConnect:", ErrorAsString(rc));

  rc = RmxClose(hConn);

  if (rc)
    Print("RmxClose:", ErrorAsString(rc));
}


/*****************************************************************************
 * 
 * FUNCTION: void StartEngine(HCONNECTION hConn, BOOL bVerbose);
 *
 * INPUT:
 *    hConn:    The connection handle.
 *    bVerbose: The required verbosity mode of the started engine.
 *
 * OVERVIEW:
 *    This function starts a dedicated engine process for providing
 *    services to a particular client process. The synchronization of
 *    the current thread and the engine process is handled with an
 *    event semaphore and shared memory. If the engine is successfully
 *    started the name is listens to is transferred back to the client.
 *
 *****************************************************************************/

void StartEngine(HCONNECTION hConn, BOOL bVerbose)
{
  // First we create an event semaphore for synchronizing this server
  // thread and the engine it will spawn.
  
  CHAR
    acSemName[CCHMAXPATH];
  HEV 
    hev = CreateEventSem(acSemName);

  if (!hev)
    {
      Print("Failed to create event semaphore");
      
      SendError(hConn, REPLY_SPAWNING_FAILED);
      return;
    }
  
  // Then we create an shared memory object to allow the engine and
  // the server to communicate with one another.

  CHAR
    acMemName[CCHMAXPATH];
  PVOID
    pvMem = AllocSharedMem(acMemName);

  if (!pvMem)
    {
      Print("Failed to allocate shared memory");

      SendError(hConn, REPLY_SPAWNING_FAILED);
      DosCloseEventSem(hev);
      return;
    }

  CHAR
    acVerbose[SIZE_NUMBER];
  
  sprintf(acVerbose, "%ld", bVerbose);
  
  // All set, time to start the engine.

  int
    pid = spawnl(P_NOWAITO, "engine.exe", "engine.exe", 
		 acSemName, acMemName, acVerbose, 0);

  if (pid == -1)
    {
      Print("Failed to spawn engine.exe");

      SendError(hConn, REPLY_SPAWNING_FAILED);
      DosCloseEventSem(hev);
      DosFreeMem(pvMem);
      return;
    }

  // Then we have to wait for the engine's notification. But only maximum 10
  // seconds. 
  
  ULONG
    rc = DosWaitEventSem(hev, TIME_10SEC);

  DosCloseEventSem(hev);

  switch (rc)
    {
    case NO_ERROR:
      break;

    case ERROR_TIMEOUT:
      Print("Timeout occurred while waiting for engine notification");
      break;
      
    default:
      Print("Waiting for event semaphore failed");
    }

  if (rc)
    {
      SendError(hConn, REPLY_SPAWNING_FAILED);
      DosFreeMem(pvMem);
      return;
    }

  // Now the shared memory should contain the name of the engine
  // connection. The engine writes the connectio name if it succeeds
  // in initializing itself.
  
  PSZ
    pszName = (PSZ) pvMem;

  if (*pszName == 0)
    {
      Print("Engine failed to initialize itself");

      SendError(hConn, REPLY_ENGINE_FAILED);
      DosFreeMem(pvMem);
    }
  else
    {
      ULONG
	ulSize = strlen(pszName) + sizeof(ULONG) + 1;
      PSZ
	pszReply = new CHAR [ulSize];

      *(ULONG*)pszReply = REPLY_OK;

      strcpy(pszReply + sizeof(ULONG), pszName);

      DosFreeMem(pvMem);

      rc = RmxWrite(hConn, pszReply, ulSize);

      if (rc)
	Print("RmxWrite:", ErrorAsString(rc));
    }
}


/*****************************************************************************
 * 
 * FUNCTION: void SendError(HCONNECTION hConn, ULONG ulReply
 *
 * INPUT:
 *    hConn: The connection handle.
 *
 * OVERVIEW:
 *    This functions sends an error code to the client.
 *
 *****************************************************************************/

void SendError(HCONNECTION hConn, ULONG ulReply)
{
  WriteMessage(hConn, (PBYTE) &ulReply, sizeof(ULONG));
}
