/*****************************************************************************
 * Engine.exe
 ***************************************************************************** 
 *
 * Source:   $Source$
 *
 * Overview: This module contain the source for the engine test
 *           program of RMX.   
 *
 * $Log$
 *
 ***************************************************************************** 
 *
 * Copyright (c) 1995 Johan Wikman
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appear in all copies and 
 * that both that copyright notice and this permission notice appear in 
 * supporting documentation.
 *
 * THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY
 * APPLICABLE LAW. THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
 * IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
 * ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 
 *
 *****************************************************************************/

/*****************************************************************************
 * INCLUDE FILES
 *****************************************************************************/

#define INCL_NOCOMMON
#define INCL_NOPMAPI
#define INCL_DOSMEMMGR
#define INCL_DOSPROCESS
#define INCL_DOSSEMAPHORES
#include <common.h>
#include <rmxcomms.h>

#include <iostream.h>
#include <process.h>
#include <stdio.h>
#include <string.h>
#include <os2.h>


/*****************************************************************************
 * EXTERNAL VARIABLES
 *****************************************************************************/

PSZ pszAppName = "Engine"; // Used by common


/****************************************************************************
 * MODULE VARIABLES
 ****************************************************************************/

static CHAR usage[]      = "usage: engine sem-name mem-name verbose";

static PSZ  pszName      = 0;
static HEV  hev          = 0;
static BOOL bVerbose     = FALSE;


/*****************************************************************************
 * MODULE FUNCTIONS
 *****************************************************************************/

static void EngineLoop   (HCONNECTION hConn);
static void Thread       (void*);
static void WaitForClient(HCONNECTION hConn);


/*****************************************************************************
 * MAIN
 *****************************************************************************/

int main(int argc, char* argv[])
{
  if (argc != 4)
    {
      cerr << usage << endl;
      return 1;
    }

  PSZ
    pszSemName = argv[1],
    pszMemName = argv[2],
    pszVerbose = argv[3];

  ULONG 
    rc;
  
  rc = DosOpenEventSem(pszSemName, &hev);
  
  if (rc)
    {
      Print("Failed to open event semaphore");
      return 1;
    }

  PVOID
    pvMem;
  
  rc = DosGetNamedSharedMem(&pvMem, pszMemName, PAG_READ | PAG_WRITE);
  
  if (rc)
    {
      Print("Failed to get named shared memory");
      
      DosPostEventSem(hev);
      DosCloseEventSem(hev);
      
      return 1;
    }

  bVerbose = atoi(pszVerbose) ? TRUE : FALSE;

  HCONNECTION
    hConn;
  ULONG
    ulSize = 0;
  
  do
    {
      rc = RmxCreateUnique(&ulSize, pszName, &hConn);
      
      if (rc == RMXERR_ENLARGE_BUFFER)
	{
	  delete [] pszName;
	  pszName = new CHAR [ulSize];
	}
    }
  while (rc == RMXERR_ENLARGE_BUFFER);

  if (rc)
    Print("RmxCreateUnique:", ErrorAsString(rc));
  else
    {
      Print("Engine connection is:", pszName);
      
      strcpy((PSZ) pvMem, pszName);
      
      DosFreeMem(pvMem);
      DosPostEventSem(hev);
      DosCloseEventSem(hev);
      
      WaitForClient(hConn);
    }

  return 1;
}


/*****************************************************************************
 * MODULE FUNCTIONS (IMPLEMENTATION)
 *****************************************************************************
 * 
 * FUNCTION: void EngineLoop(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The connection handle.
 *
 * OVERVIEW:
 *    This function simply reads a message from the client and sends
 *    the message unaltered back.
 *
 *****************************************************************************/

void EngineLoop(HCONNECTION hConn)
{
  ULONG
    ulSize = 0;
  PBYTE
    pbMessage = 0;

  BOOL
    bContinue = TRUE;
  
  while (bContinue)
    {
      ULONG
	ulMessageSize = ReadMessage(hConn, &pbMessage, &ulSize);

      if (bVerbose)
	{
	  CHAR
	    acReport[256]; // More than enough

	  sprintf(acReport, "Received message of %lu bytes", ulMessageSize);
	  
	  Print(acReport);
	}
      
      if (ulMessageSize == 0)
	{
	  Print("No data read. Client has closed connection");
	  bContinue = FALSE;
	}
      else
	{
	  // To simulate that we are doing something.
	  
	  // DosSleep(200);
	  bContinue = WriteMessage(hConn, pbMessage, ulMessageSize);
	}
    }
  
  Print("Exiting");
}


/*****************************************************************************
 * 
 * FUNCTION: VOID Thread(VOID* pvArgument);
 *
 * INPUT:
 *    pvArgument: Not used.
 *
 * OVERVIEW:
 *    The function creates a connection and starts waiting for a new
 *    client thread. 
 *
 *****************************************************************************/

void Thread(void*)
{
  ULONG
    rc;
  HCONNECTION
    hConn;

  rc = RmxCreate(pszName, &hConn);

  if (rc)
    {
      Print("RmxCreate:", ErrorAsString(rc));
      return;
    }
  
  WaitForClient(hConn);
}


/*****************************************************************************
 * 
 * FUNCTION: void WaitForClient(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The connection.
 *
 * OVERVIEW:
 *    The functions waits for a client thread to connect. Immediately
 *    when that happens, the function starts another thread that will
 *    create a new connection. This way there will always be a
 *    connection ready and waiting for a new client thread. 
 *
 *****************************************************************************/

static void WaitForClient(HCONNECTION hConn)
{
  ULONG
    rc;

  if (bVerbose)
    Print("Waiting for client");
  
  rc = RmxConnect(hConn);

  if (rc)
    {
      Print("RmxConnect:", ErrorAsString(rc));
      return;
    }

  int
    tid = _beginthread(Thread, SIZE_STACK, (void*) pszName);

  if (tid == -1)
    Print("Failed to spawn thread");
      
  EngineLoop(hConn);
  
  rc = RmxDisConnect(hConn);
  
  if (rc)
    Print("RmxDisConnect:", ErrorAsString(rc));
}
