/*****************************************************************************
 * common
 ***************************************************************************** 
 *
 * Source:   $Source$
 *
 * Overview: This file implements the common stuff of client, engine
 *           and server. 
 *
 * $Log$
 *
 ***************************************************************************** 
 *
 * Copyright (c) 1995 Johan Wikman
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appear in all copies and 
 * that both that copyright notice and this permission notice appear in 
 * supporting documentation.
 *
 * THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY
 * APPLICABLE LAW. THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
 * IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
 * ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 
 *
 *****************************************************************************/

/*****************************************************************************
 * Include Files
 *****************************************************************************/

#define INCL_NOCOMMON
#define INCL_NOPMAPI
#define INCL_DOSERRORS
#define INCL_DOSSEMAPHORES
#define INCL_DOSPROCESS
#include <rmxcomms.h>
#include <common.h>

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <os2.h>


/****************************************************************************
 * EXTERNAL FUNCTIONS
 ****************************************************************************
 * 
 * FUNCTION: PCSZ ErrorAsString(ULONG rc);
 *
 * INPUT:
 *    rc: An error code as returned by some RMX function.
 *
 * RETURN:
 *    Corresponding message.
 *
 * OVERVIEW:
 *    This function returns, given a RMX error code, a corresponding
 *    printable message. 
 *
 ****************************************************************************/

PCSZ ErrorAsString(ULONG rc)
{
  switch (rc)
    {
    case RMXAPI_OK:
      return "Ok";

    case RMXERR_UNKNOWN_SERVICE:
      return "The service required is not known";

    case RMXERR_ENLARGE_BUFFER:
      return "The provided buffer was too small";

    case RMXERR_GENERAL:
      return "Unrecoverable error occurred";

    case RMXERR_BAD_NAME:
      return "The name provided cannot be used for creating a connection";
      
    case RMXERR_BAD_NETWORK:
      return "The network or the server is down";

    case RMXERR_CONNECTION_NOT_CONNECTED:
      return "The connection is not connected";

    case RMXERR_BROKEN_CONNECTION:
      return "The connection is broken";

    case RMXERR_CONNECTION_BUSY:
      return "The connection is busy";

    case RMXERR_INVALID_HANDLE:
      return "The provided handle was invalid";

    case RMXERR_OUT_OF_MEM:
      return "Out of memory";
      
    case RMXERR_OUT_OF_RESOURCES:
      return "Out of resources";
      
    default:
      return "Unknown error returned";
    }
}


/****************************************************************************
 * 
 * FUNCTION: PSZ GetPortName();
 *
 * RETURN:
 *    The name of the server port. 
 *
 * OVERVIEW:
 *    This function returns the name of the server. The server process
 *    uses this name when it creates the connection and the client process
 *    uses it when it opens the connection.
 *
 ****************************************************************************/

PSZ GetPortName()
{
  ULONG
    rc;
  
  ULONG
    ulSize = SIZE_NAME;
  PSZ
    pszName = 0;
  
  do
    {
      pszName = new CHAR [ulSize];
  
      rc = RmxGetServiceName("rmxtester", &ulSize, pszName);

      if (rc == RMXERR_ENLARGE_BUFFER)
	{
	  // If the buffer is too small, the required size is returned
	  // in ulSize.
	  
	  delete [] pszName;
	}
    }
  while (rc == RMXERR_ENLARGE_BUFFER);

  if (rc)
    {
      Print("RmxGetServiceName:", ErrorAsString(rc));
      delete [] pszName;
      return 0;
    }

  return pszName;
}


/****************************************************************************
 * 
 * FUNCTION: void Print(PCSZ pcszMessage1, PCSZ pcszMessage2);
 *
 * INPUT:
 *    pcszMessage1: The first part of the message.
 *    pcszMessage2: The second part of the message.
 *
 * OVERVIEW:
 *    This function concatenates the two strings (actually it places a
 *    space in between) and prints the message together with process and
 *    thread identification on standard out. If pcszMessage2 is NULL it
 *    is ignored. 
 *
 ****************************************************************************/

void Print(PCSZ pcszMessage1, PCSZ pcszMessage2)
{
  PTIB
    ptib;
  PPIB
    ppib;

  DosGetInfoBlocks(&ptib, &ppib);

  // Yes, we assume there is an upper bound on the message sizes.

  CHAR
    acMessage[1024];

  if (pcszMessage2)
    sprintf(acMessage, "%s [%d, %d]: %s %s.\n", 
	    pszAppName, ppib->pib_ulpid, ptib->tib_ptib2->tib2_ultid,
	    pcszMessage1, pcszMessage2);
  else
    sprintf(acMessage, "%s [%d, %d]: %s.\n", 
	    pszAppName, ppib->pib_ulpid, ptib->tib_ptib2->tib2_ultid,
	    pcszMessage1);

  // Trust me, even if the text is in one single string and we are
  // using MT libraries the printing has to be protected. The
  // semaphore is created the first time the function is called. 
  
  static HMTX 
    hmtx;
  
  if (hmtx == 0)
    {
      DosEnterCritSec();
      
      // We have to check the variable once more. In pathological
      // conditions two threads might simultaneously enter the
      // function and simultanously end up in this block.

      if (hmtx == 0)
	{
	  // We create an anonymous mutex semaphore whose initial
	  // state is unowned.

	  if (DosCreateMutexSem(0, &hmtx, 0, FALSE) != NO_ERROR)
	    hmtx = 0; 
	}
      
      DosExitCritSec();
      
      if (hmtx == 0)
	return;
    }
  
  if (DosRequestMutexSem(hmtx, SEM_INDEFINITE_WAIT) == NO_ERROR)
    {
      cout << acMessage << flush;
      
      DosReleaseMutexSem(hmtx);
    }
}


/****************************************************************************
 * 
 * FUNCTION: ULONG ReadMessage(HCONNECTION hConn,
 *                             PBYTE* ppbMessage, ULONG* pulBufferSize);
 *
 * INPUT:
 *    hConnection:   Handle of connection.
 *
 * INPUT/OUTPUT:
 *    ppbMessage:    Address of pointer to buffer where message will be
 *                   read. If the size is not sufficient this function
 *                   will deallocate the buffer and allocate a new
 *                   one. Hence, the address of the buffer pointed to may
 *                   change as a result of this call.
 *    pulBufferSize: Upon input should point to a variable that
 *                   contains the size of the buffer. Upon return the
 *                   variable pointed to will contain the new size.
 *
 * RETURN:
 *    The actual message size.
 *
 * OVERVIEW:
 *    This function reads a message from the given connection. It does
 *    not return until a message has been read, or an error has
 *    occurred. An error is reported as 0 size message. Even in that
 *    case the buffer may have been reallocated. It is always the
 *    callers responsibility to eventually free the buffer.
 *
 ****************************************************************************/

ULONG ReadMessage(HCONNECTION hConn, PBYTE* ppbMessage, ULONG* pulBufferSize)
{
  ULONG
    rc;
  ULONG
    ulBytesRead;

  // The following loop is necessary. We can never know in advance
  // exactly how big a message is on its way.

  do
    {
      rc = RmxRead(hConn, *ppbMessage, *pulBufferSize, &ulBytesRead);
      
      if (rc == RMXERR_ENLARGE_BUFFER)
	{
	  // If the buffer was too small, the required size is
	  // returned in ulBytesRead.
	  
	  delete [] *ppbMessage;
	  *ppbMessage = new BYTE [ulBytesRead];
	  
	  *pulBufferSize = ulBytesRead;
	}
    }
  while (rc == RMXERR_ENLARGE_BUFFER);
  
  if (rc)
    {
      Print("RmxRead:", ErrorAsString(rc));
      ulBytesRead = 0;
    }

  return ulBytesRead;
}


/****************************************************************************
 * 
 * FUNCTION: BOOL WriteMessage(HCONNECTION hConn,
 *                             PBYTE pbMessage, ULONG ulMessageSize);
 *
 * INPUT:
 *    hConnection:   Handle of connection.
 *
 * INPUT:
 *    pbMessage:     Pointer to a buffer that contains the message to
 *                   be sent.
 *    ulMessageSize: The message size.
 *
 * RETURN:
 *    Success status.
 *
 * OVERVIEW:
 *    This function sends a message over the connection to the peer.
 *    If the function fails it most likely means that the connection
 *    has broken.
 *
 ****************************************************************************/

BOOL WriteMessage(HCONNECTION hConn, PBYTE pbMessage, ULONG
		  ulMessageSize)
{
  ULONG
    rc = RmxWrite(hConn, pbMessage, ulMessageSize);

  if (rc != RMXAPI_OK)
    {
      Print("RmxWrite:", ErrorAsString(rc));
      return FALSE;
    }
  else
    return TRUE;
}
