/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/objects.c
 * Summary:
 *	Manage a list of memory objects. Allow the object that a pointer
 *	points to to be determined.
 * Other notes:
 *	The list of memory objects is stored as a binary tree. Clearly, we
 *	need to be able to allocate and free parts of this tree but without
 *	causing recursion. So, separate functions __bounds_malloc and
 *	__bounds_free are implemented in `lib/malloc.c' for this.
 * Author      	Date		Notes
 * RWMJ		5/12/94		Initial implementation.
 * RWMJ		23/1/95	       	Now using separate trees for static, etc.
 * RWMJ		9/2/95		Added splay trees. Removed hash table.
 * RWMJ		12/2/95		Put all the objects into a single tree.
 * RWMJ		15/3/95		Enter/leave critical section code.
 * RWMJ		4/4/95		`new_object' doesn't call `memset'.
 * RWMJ		25/4/95		Cache code added and removed!
 * RWMJ		27/5/95		Code to collect locality stats.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "bounds-lib.h"

#include "mutex.h"			/* for 'enter|leave_critical_section'*/

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

#define SPLAY_COUNT_OPERATIONS	0	/* Keep splay-tree statistics. */
#define SPLAY_STRICT_CHECKING	0	/* Stringent checks (slow). */
#define SPLAY_COUNT_STAT	0	/* Do splay statictics. */

#if SPLAY_COUNT_STAT
#define       MAX_STEPS       256

static unsigned max_steps = 0;
static unsigned max_no_steps = 0;
static unsigned *hits = NULL;
static unsigned *no_hits = NULL;
#endif

/* The following flag is no longer needed. Used `-no-warn-unaligned' in
 * GCC_BOUNDS_OPTS to get this behaviour.
 */
#if 0
/* This flag is especially for Stuart Kemp <skemp@crow.bmc.com>, who wants
 * to be able to access his arrays of int as chars if needs be. Set this
 * to `1' to enable the appropriate patches, Stuart.		-RWMJ
 */
#define NO_ALIGNMENT_CHECKS	0
#endif

static tree hist_object = { NULL };

/*----------------------------------------------------------------------
 *	Allocate/free object structures.
 *----------------------------------------------------------------------*/

static inline object *
new_object (void)
{
  object *p;

  /* check if a object is in the hist_object list */
  if (hist_object.root) {
      p = hist_object.root;
      hist_object.root = hist_object.root->left;
      return(p);
  }
  p = __bounds_malloc (sizeof (struct object));
  if (!p) {
    __bounds_internal_error ("out of memory allocating object structure",
			     __FILE__, __LINE__);
  }
  /* The following call to `memset' resulted in considerable overhead. I've
   * altered all the calls to new_object so that clearing this area is no
   * longer necessary. Be careful when you add or change the contents of
   * the object structure.
   */
/*  memset (p, 0, sizeof (struct object)); */
  return p;
}

static inline void
free_object (object *p)
{
  /* never realy free object. Put it in the hist_object list */
  p->left = hist_object.root;
  hist_object.root = p;
#if 0
  __bounds_free (p);
#endif
}

/*----------------------------------------------------------------------
 *	This is the splay tree used to keep track of memory locations.
 *	The root node is called `root'. If it is non-NULL, then the
 *	tree is arranged as in the following example:
 *			root ----> [100..199]
 *				    /      \
 *			       [75..90]   [200..299]
 *			        /
 *			   [0..74]
 *	This represents the following memory usage:
 *		[0..74]		Allocated
 *		[75..90]    	Allocated
 *		[91..99]	Free
 *		[100..199]	Allocated
 *		[200..299]	Allocated
 *	In general, that is, objects at lower addresses go on the left
 *	side of the tree. The inline function `tree_lookup' turns a memory
 *	location into the correct object pointer. The inline function
 *	`tree_add' inserts a new object into the tree. The inline function
 *	`tree_del' deletes an object from the tree.
 *----------------------------------------------------------------------*/

static tree object_tree = {NULL};

#if SPLAY_COUNT_OPERATIONS

/* Keep track of the number of operations performed, so we can work out if
 * we are successfully doing O(log n) work.
 */
static unsigned add_operations = 0;	/* Calls to tree_add. */
static unsigned look_up_operations = 0;	/* Calls to tree_lookup(_base). */
static unsigned look_up_steps = 0;	/* Steps taken in the above. */
static unsigned nlook_up_operations = 0;/* Failed calls to tree_lookup*. */
static unsigned nlook_up_steps = 0;	/* Steps taken in the above. */
static unsigned del_operations = 0;	/* Calls to tree_del. */

/* We break down calls to `tree_splay' according to where they come
 * from.
 */
static unsigned splay_look_up_operations = 0;
static unsigned splay_look_up_steps = 0;
static unsigned splay_add_operations = 0;
static unsigned splay_add_steps = 0;
static unsigned splay_del_operations = 0;
static unsigned splay_del_steps = 0;

/* The following variables tell where we are coming from in tree_splay.
 */
static unsigned *splay_operations;
static unsigned *splay_steps;

/* Place the frequency with which you want dumps here. The tree will be saved
 * after this number of look-up operations.
 */
#define DUMP_EVERY	50000

static void tree_dump_statistics (tree *);

#endif

/*----------------------------------------------------------------------
 * Operations on splay trees.
 *----------------------------------------------------------------------*/

/* SPLAY is the fundamental operation on splay trees. It takes node `n'
 * (assumed to exist within the splay tree) and pulls it up to the root
 * of the tree. The node returned (for convenience) is the root node, which
 * will be `n' and `t->root'.
 */
static inline object *
tree_splay (tree *t, object *n)
{
#if SPLAY_COUNT_OPERATIONS
  ++(*splay_operations);
#endif

  while (n != t->root) {
    object *parent = n->parent, *grandparent = parent->parent;

#if SPLAY_COUNT_OPERATIONS
    ++(*splay_steps);
#endif

    /* Determine which transformation to perform. Each of these transformations
     * will move the node nearer to the root, and has the side effect of
     * making the tree more balanced.
     */
    /* 1a,b. Parent is the root, so perform `zig' rotation. */
    if (grandparent == NULL) {
      if (n->left_of_parent) {
	object *b = n->right;

	n->right = parent;
	n->parent = parent->parent;
	n->left_of_parent = parent->left_of_parent;
	parent->left = b;
	parent->parent = n; parent->left_of_parent = 0;
	if (b) b->parent = parent, b->left_of_parent = 1;
      }
      else {
	object *b = n->left;

	n->left = parent;
	n->parent = parent->parent;
	n->left_of_parent = parent->left_of_parent;
	parent->right = b;
	parent->parent = n; parent->left_of_parent = 1;
	if (b) b->parent = parent, b->left_of_parent = 0;
      }
    }
    /* 2a,b. `n', its parent and grandparent are in a straight line, so perform
     * the `zig-zig' rotation. */
    else if (n->left_of_parent == parent->left_of_parent) {
      if (n->left_of_parent) {
	object *b = n->right;

	n->right = grandparent;
	n->parent = grandparent->parent;
	n->left_of_parent = grandparent->left_of_parent;
	parent->left = b;
	grandparent->parent = n; grandparent->left_of_parent = 0;
	if (b) b->parent = parent, b->left_of_parent = 1;
      } else {
	object *b = n->left;

	n->left = grandparent;
	n->parent = grandparent->parent;
	n->left_of_parent = grandparent->left_of_parent;
	parent->right = b;
	grandparent->parent = n; grandparent->left_of_parent = 1;
	if (b) b->parent = parent, b->left_of_parent = 0;
      }
    }
    /* 3a,b. `n', its parent and grandparent aren't in a straight line, so
     * perform the `zig-zag' rotation. */
    else {
      if (n->left_of_parent) {
	object *b = n->right, *c = n->left;

	n->right = parent;
	n->left = grandparent;
	n->parent = grandparent->parent;
	n->left_of_parent = grandparent->left_of_parent;
	parent->left = b;
	parent->parent = n; parent->left_of_parent = 0;
	grandparent->right = c;
	grandparent->parent = n; grandparent->left_of_parent = 1;
	if (b) b->parent = parent, b->left_of_parent = 1;
	if (c) c->parent = grandparent, c->left_of_parent = 0;
      } else {
	object *b = n->left, *c = n->right;

	n->left = parent;
	n->right = grandparent;
	n->parent = grandparent->parent;
	n->left_of_parent = grandparent->left_of_parent;
	parent->right = b;
	parent->parent = n; parent->left_of_parent = 1;
	grandparent->left = c;
	grandparent->parent = n; grandparent->left_of_parent = 0;
	if (b) b->parent = parent, b->left_of_parent = 0;
	if (c) c->parent = grandparent, c->left_of_parent = 1;
      }
    }
    if (n->parent == NULL)      t->root = n;
    else if (n->left_of_parent) n->parent->left = n;
    else 			n->parent->right = n;
  } /* while */

#if SPLAY_STRICT_CHECKING
  __bounds_tree_check (t);
#endif

  return n;
} /* tree_splay */

/*----------------------------------------------------------------------
 * Add, look-up and delete operations. All these are done using the
 * fundamental `splay' operation.
 *----------------------------------------------------------------------
 */

#if SPLAY_COUNT_STAT

/* calculate the splay statistics */

static inline void
add_hits(index)
unsigned index;
{
  if (index >= max_steps) {
    int i;
    int lastmax = max_steps;

    while (index >= max_steps)
      max_steps += MAX_STEPS;
    if (hits == NULL) {
      hits = __bounds_malloc((max_steps + 1) * sizeof(unsigned));
    }
    else {
      hits = __bounds_realloc(hits,(max_steps + 1) * sizeof(unsigned));
    }
    if (hits == NULL)
      __bounds_internal_error ("out of memory allocating hits data",
			     __FILE__, __LINE__);
    for (i = lastmax+1 ; i <= max_steps ; i++) hits[i] = 0;
  }
  hits[index]++;
}

static inline void
add_no_hits(index)
unsigned index;
{
  if (index >= max_no_steps) {
    int i;
    int lastmax = max_no_steps;

    while (index >= max_no_steps)
      max_no_steps += MAX_STEPS;
    if (no_hits == NULL) {
      no_hits = __bounds_malloc((max_no_steps + 1) * sizeof(unsigned));
    }
    else {
      no_hits = __bounds_realloc(no_hits,(max_no_steps + 1) * sizeof(unsigned));
    }
    if (no_hits == NULL)
      __bounds_internal_error ("out of memory allocating hits data",
			     __FILE__, __LINE__);
    for (i = lastmax+1 ; i <= max_no_steps ; i++) no_hits[i] = 0;
  }
  no_hits[index]++;
}
#endif

/* print the splay statistics */

void
__bound_print_statistics()
{
#if SPLAY_COUNT_STAT
  int i;
  double total1_hits = 0.0;
  double total1_steps = 0.0;
  double total2_hits = 0.0;
  double total2_steps = 0.0;
  double depth;
  unsigned n1;
  unsigned n2;
  unsigned max = max_steps > max_no_steps ? max_steps : max_no_steps;

  for (i = 1 ; i <= max ; i++) {
	n1 = i <= max_steps ? hits[i] : 0;
	n2 = i <= max_no_steps ? no_hits[i] : 0;
	if (n1 || n2)
		printf("%4d %10u %10u\n",i,n1,n2);
	if (n1) {
		total1_hits += (double)n1;
		total1_steps += (double)n1 * (double)i;
	}
	if (n2) {
		total2_hits += (double)n2;
		total2_steps += (double)n2 * (double)i;
	}
  }
  depth = total1_hits ? ((double)total1_steps*100.0)/(double)total1_hits : 0.0;
  n1 = (unsigned)(depth / 100.0);
  n2 = (unsigned)(depth - 100.0 * (double)n1);
  printf ("Hits    %10u. Steps %10u. Average steps hits    %4u.%02u\n",
		(unsigned)total1_hits, (unsigned)total1_steps, n1, n2);
  depth = total2_hits ? ((double)total2_steps*100.0)/(double)total2_hits : 0.0;
  n1 = (unsigned)(depth / 100.0);
  n2 = (unsigned)(depth - 100.0 * (double)n1);
  printf ("No Hits %10u. Steps %10u. Average steps no hits %4u.%02u\n",
		(unsigned)total2_hits, (unsigned)total2_steps, n1, n2);
#if 0
  __bounds_debug_memory (NULL,NULL);
#endif
#endif
}

/* Return leftmost object below the current object node.
 */
static inline object *
leftmost (object *n)
{
  while (n->left) n = n->left;
  return n;
}

/* Look-up a node in the splay tree. Notice that the act of looking up a
 * node changes the tree. (In particular, the node, if found, moves to
 * the root of the tree). Returns the node, or NULL if not found.
 */
static inline object *
tree_lookup (tree *t, void *ptr)
{
  object *n = t->root;

#if SPLAY_COUNT_STAT
  int hits_steps;
#endif
#if SPLAY_COUNT_OPERATIONS
  int steps = 0;
  look_up_operations ++;
  if (look_up_operations == DUMP_EVERY)
    tree_dump_statistics (t);
  splay_operations = &splay_look_up_operations;
  splay_steps = &splay_look_up_steps;
#endif

/* Handle the toplevel of the splay tree with this inline code. It it
 * faster than using the while loop and the tree_splay function.
 * If the FAST_UPDATE is set to 0 the tree is not balanced anymore. This
 * is faster than keeping the tree balanced.
 */
#ifdef FAST
#undef FAST
#endif
#define	FAST 1
#define	FAST_UPDATE 0
#if FAST
  if (n) {
    if (ptr < n->base) {
      n = n->left;
      if (n) {
        if (ptr < n->base)
	  n = n->left;
	else if (ptr > n->extent)
	  n = n->right;
	else if (ptr != n->extent) {
#if FAST_UPDATE
	  object *parent = n->parent;
	  object *b = n->right;

	  n->right = parent;
	  n->parent = NULL;
	  parent->left = b;
	  parent->parent = n; parent->left_of_parent = 0;
	  if (b) b->parent = parent, b->left_of_parent = 1;
          t->root = n;
#endif
#if SPLAY_COUNT_STAT
          add_hits(2);
          n->hits ++;
          n->steps += 2;
#endif
#if SPLAY_COUNT_OPERATIONS
          n->hits ++;
          look_up_steps += 2;
#endif
          return n;
	}
      }
    }
    else if (ptr > n->extent) {
      n = n->right;
      if (n) {
        if (ptr < n->base)
	  n = n->left;
	else if (ptr > n->extent)
	  n = n->right;
	else if (ptr != n->extent) {
#if FAST_UPDATE
	  object *parent = n->parent;
	  object *b = n->left;

	  n->left = parent;
	  n->parent = NULL;
	  parent->right = b;
	  parent->parent = n; parent->left_of_parent = 1;
	  if (b) b->parent = parent, b->left_of_parent = 0;
          t->root = n;
#endif
#if SPLAY_COUNT_STAT
          add_hits(2);
          n->hits ++;
          n->steps += 2;
#endif
#if SPLAY_COUNT_OPERATIONS
          n->hits ++;
          look_up_steps += 2;
#endif
          return n;
	}
      }
    }
    else if (ptr != n->extent) {
#if SPLAY_COUNT_STAT
      add_hits(1);
      n->hits ++;
      n->steps += 1;
#endif
#if SPLAY_COUNT_OPERATIONS
      n->hits ++;
      look_up_steps += 1;
#endif
      return n;
    }
  }
#endif
#if SPLAY_COUNT_STAT
  hits_steps = 2;
#endif

  while (n) {
#if SPLAY_COUNT_STAT
    hits_steps ++;
#endif
#if SPLAY_COUNT_OPERATIONS
    steps ++;
#endif

    if (ptr < n->base) n = n->left;
    else
      {
      /* This is tricky. Normally, a pointer refers to an object if the
       * pointer lies between base and extent inclusively. We add extra
       * padding between checked objects to ensure this is true. However,
       * if we aren't able to add this padding, then the next object may
       * lie immediately after the previous object in memory, ie. extent
       * and the base of the next object may be the same.
       */
	if (ptr > n->extent)
	  n = n->right;
	else if (ptr == n->extent)
	  {
	    /* This is the tricky boundary condition. Use the hints given. If
	     * `no_padding' is 0, there can be no immediate object on the
	     * right, so stop the search. If `no_padding' is true, we must
	     * look rightwards for the adjacent object, and go that way if the
	     * right base == this extent.
	     */
	    if (n->no_padding
		&& n->right
		&& leftmost (n->right)->base == n->extent)
	      n = n->right;
	    else {
	      /* if there could be a object next to this one. Do not use the
	       * current object
	       */
	      if (n->no_padding) n = NULL;
	      break;
	    }
	  }
	else break;
      }
  }

  if (n)
    {
#if SPLAY_COUNT_STAT
      add_hits(hits_steps);
      n->hits ++;
      n->steps += hits_steps;
#endif
#if SPLAY_COUNT_OPERATIONS
      n->hits ++;
      look_up_steps += steps;
#endif
      return tree_splay (t, n);
    }
  else
    {
#if SPLAY_COUNT_STAT
      add_no_hits(hits_steps);
#endif
#if SPLAY_COUNT_OPERATIONS
      nlook_up_steps += steps;
      nlook_up_operations ++;
#endif
      return NULL;
    }
}

/* Look-up a node in the splay tree, but only match on an object with an
 * identical base address.
 */
static inline object *
tree_lookup_base (tree *t, void *base)
{
  object *n = t->root;

#if SPLAY_COUNT_OPERATIONS
  int steps = 0;
  look_up_operations ++;
  if (look_up_operations == DUMP_EVERY)
    tree_dump_statistics (t);
  splay_operations = &splay_look_up_operations;
  splay_steps = &splay_look_up_steps;
#endif
#if FAST
  if (n) {
    if (base < n->base) {
      n = n->left;
      if (n) {
	if (base < n->base)
	  n = n->left;
	else if (base >= n->extent)
	  n = n->right;
	else if (base == n->base) {
#if FAST_UPDATE
	  object *parent = n->parent;
	  object *b = n->right;

	  n->right = parent;
	  n->parent = NULL;
	  parent->left = b;
	  parent->parent = n; parent->left_of_parent = 0;
	  if (b) b->parent = parent, b->left_of_parent = 1;
          t->root = n;
#endif
	  return n;
	}
	else
	  return NULL;
      }
    }
    else if (base >= n->extent) {
      n = n->right;
      if (n) {
	if (base < n->base)
	  n = n->left;
	else if (base >= n->extent)
	  n = n->right;
	else if (base == n->base) {
#if FAST_UPDATE
	  object *parent = n->parent;
	  object *b = n->left;

	  n->left = parent;
	  n->parent = NULL;
	  parent->right = b;
	  parent->parent = n; parent->left_of_parent = 1;
	  if (b) b->parent = parent, b->left_of_parent = 0;
          t->root = n;
#endif
	  return n;
	}
	else
	  return NULL;
      }
    }
    else if (base == n->base)
      return n;
    else
      return NULL;
  }
#endif

  while (n) {
#if SPLAY_COUNT_OPERATIONS
    steps ++;
#endif

    if (base < n->base) n = n->left;
    else if (base >= n->extent) n = n->right;
    else break;
  }
  if (n && n->base == base)
    {
#if SPLAY_COUNT_OPERATIONS
      n->hits ++;
      look_up_steps += steps;
#endif
      return tree_splay (t, n);
    }
  else
    {
#if SPLAY_COUNT_OPERATIONS
      nlook_up_steps += steps;
      nlook_up_operations ++;
#endif
      return NULL;
    }
}

/* Add a node to the splay tree. We assume the node isn't already in the
 * tree, and that no node of the same value is in the tree. The node will
 * rise to the root of the tree (by splay), and we return it for convenience.
 */
static inline object *
tree_add (tree *t, object *n)
{
  object *tmp = t->root, *prev;

  n->parent = n->left = n->right = NULL;

#if SPLAY_COUNT_OPERATIONS
  add_operations++;
  splay_operations = &splay_add_operations;
  splay_steps = &splay_add_steps;
#endif

  if (tmp == NULL) {
    /* Tree is empty. Add the node at root and return immediately. */
    t->root = n;
    return n;
  }

  while (tmp) {
    prev = tmp;
    if (n->base < tmp->base)
      tmp = tmp->left;
    else
      tmp = tmp->right;
  }

  /* `prev' points to the last node before we fell off the bottom of the
   * tree.
   */
  if (prev->parent == NULL) {
    /* `prev' is the root of the tree, so we can just add the node here with
     * no problem.
     */
    if (n->base < prev->base) {
      prev->left = n;
      n->parent = prev;
      n->left_of_parent = 1;
    } else {
      prev->right = n;
      n->parent = prev;
      n->left_of_parent = 0;
    }
    return tree_splay (t, n);
  }

  /* If parent to prev and prev to new node are in the same direction, we
   * can just add the new node.
   */
  if (n->base < prev->base && prev->left_of_parent) {
    prev->left = n;
    n->parent = prev;
    n->left_of_parent = 1;
    return tree_splay (t, n);
  }
  if (n->base > prev->base && !prev->left_of_parent) {
    prev->right = n;
    n->parent = prev;
    n->left_of_parent = 0;
    return tree_splay (t, n);
  }

  /* If parent to prev and prev to new node are not in a straight line, we
   * insert the new node between parent and prev.
   */
  if (prev->left_of_parent) {
    object *parent = prev->parent;

    parent->left = n;
    n->left = prev;
    n->parent = parent; n->left_of_parent = 1;
    prev->parent = n; prev->left_of_parent = 1;
  } else {
    object *parent = prev->parent;

    parent->right = n;
    n->right = prev;
    n->parent = parent; n->left_of_parent = 0;
    prev->parent = n; prev->left_of_parent = 0;
  }
  return tree_splay (t, n);
}

/* Delete a node from the tree. We assume that the node to be deleted exists
 * in the tree.
 */
static inline void
tree_del (tree *t, object *n)
{
  object *left, *right, *largest;
  tree tmp_left;

#if SPLAY_COUNT_OPERATIONS
  del_operations ++;
  splay_operations = &splay_del_operations;
  splay_steps = &splay_del_steps;
#endif

  if (n->left == NULL) {
    right = n->right;
    if (n->parent == NULL) {
      t->root = right;
      if (right) right->parent = NULL;
    }
    else if (n->left_of_parent) {
      n->parent->left = right;
      if (right) right->parent = n->parent, right->left_of_parent = 1;
    }
    else {
      n->parent->right = right;
      if (right) right->parent = n->parent, right->left_of_parent = 0;
    }
    return;
  }

  if (n->right == NULL) {
    left = n->left;
    if (n->parent == NULL) {
      t->root = left;
      if (left) left->parent = NULL;
    }
    else if (n->left_of_parent) {
      n->parent->left = left;
      if (left) left->parent = n->parent, left->left_of_parent = 1;
    }
    else {
      n->parent->right = left;
      if (left) left->parent = n->parent, left->left_of_parent = 0;
    }
    return;
  }
 
  right = n->right;
  left = n->left;

  /* Search the left orphaned subtree for the largest value it contains. This
   * will necessarily be smaller than the deleted node, and all the nodes in
   * the right subtree.
   * (Notice that the largest value is just found by going right as far as
   * possible in the left subtree).
   */
  largest = left;
  while (largest->right) {
    largest = largest->right;
  }

  /* Splay the left tree again so that the largest node is brought to the top.
   */
  left->parent = NULL;
  tmp_left.root = left;
  tree_splay (&tmp_left, largest);

  /* Now, since the largest valued node in the left tree is at the top, we
   * know that its right pointer is NULL, and we can just attach the right
   * tree there.
   */
  if (n->parent == NULL) {
    t->root = largest;
    largest->parent = NULL;
  }
  else if (n->left_of_parent) {
    n->parent->left = largest;
    largest->parent = n->parent;
    largest->left_of_parent = 1;
  }
  else {
    n->parent->right = largest;
    largest->parent = n->parent;
    largest->left_of_parent = 0;
  }
  largest->right = right;
  right->parent = largest;
  right->left_of_parent = 0;
}

/*----------------------------------------------------------------------
 *	Perform a strict check on the contents of the splay tree. This
 *	check is done after each tree operation if 'SPLAY_STRICT_CHECKING'
 *	is defined as 1. However, we include this code anyway so that
 *	we can call it from the debugger or app. if necessary.
 *----------------------------------------------------------------------*/

static void tc_error (tree *t, object *n, char *s) __attribute__((noreturn));
static void tc_test (tree *t, object *n, object *parent, int left_of_parent,
		     void *min, void *max);

void
__bounds_tree_check (tree *t)
{
  object *root = t->root;

  if (root == NULL) return;

  if (root->parent != NULL)
    tc_error (t, root, "root node has a parent (should be NULL)");
  if (root->left)
    tc_test (t, root->left, root, 1, (void *) 0, root->base - 1);
  if (root->right)
    tc_test (t, root->right, root, 0, root->extent, (void *) ~0);
}

static void
tc_test (tree *t, object *n, object *parent, int left_of_parent,
	 void *min, void *max)
{
  if (n->parent != parent)
    tc_error (t, n, "node has parent different from true parent");
  if (n->left_of_parent != left_of_parent)
    tc_error (t, n, "node has left_of_parent different from true direction");
  if (n->base >= n->extent)
    tc_error (t, n, "node has zero or negative size");
  if (n->base < min || n->extent - 1 > max)
    tc_error (t, n, "node is out of order within the tree");
  if (n->left)
    tc_test (t, n->left, n, 1, min, n->base - 1);
  if (n->right)
    tc_test (t, n->right, n, 0, n->extent, max);
}

static void
tc_error (tree *t, object *n, char *msg)
{
  printf ("tree_check: %s\n"
	  "  tree root at %p\n"
	  "  object struct at %p\n"
	  "  object refers to %p..%p\n",
	  msg, t->root, n, n->base, n->extent-1);
  ABORT ();
}

/*----------------------------------------------------------------------
 *	If `SPLAY_COUNT_OPERATIONS' is set to 1, then we keep statistics
 *	on the current state of the splay tree, and we dump the splay
 *	tree every N look-ups so we can draw pictures and measure
 *	stats on the tree, etc.
 *----------------------------------------------------------------------*/

#if SPLAY_COUNT_OPERATIONS

#include "ext-tree.h"	/* Describes external format of splay tree files. */
#include <fcntl.h>

static void tree_dump_tree (int fd, object *n);
static void tree_write_file (int fd, void *, size_t);
static void tree_close_file (int fd);
static void tree_next_name (void);
static char dumpfile [128] = "splaytree.aa";
static int dump_index = 0;

static void
tree_dump_statistics (tree *t)
{
  int fd;
  struct ext_tree et;

  printf ("Dump tree after %d look-up operations.\n", DUMP_EVERY);

  /* Notice that we can't make C library calls here, which would simplify
   * this code substantially.
   */
  fd = creat (dumpfile, 0644);
  if (fd < 0)
    {
      printf ("Dump failed at: open\n");
      return;
    }
  strcpy (et.magic, EXT_TREE_MAGIC);
  et.index = dump_index; dump_index ++;
  et.dump_every = DUMP_EVERY;
  et.add_operations = add_operations;
  et.del_operations = del_operations;
  et.look_up_operations = look_up_operations - nlook_up_operations;
  et.look_up_steps = look_up_steps;
  et.nlook_up_operations = nlook_up_operations;
  et.nlook_up_steps = nlook_up_steps;
  et.splay_operations = splay_look_up_operations
    + splay_add_operations + splay_del_operations;
  et.splay_steps = splay_look_up_steps
    + splay_add_steps + splay_del_steps;
  et.splay_look_up_operations = splay_look_up_operations;
  et.splay_look_up_steps = splay_look_up_steps;
  et.splay_add_operations = splay_add_operations;
  et.splay_add_steps = splay_add_steps;
  et.splay_del_operations = splay_del_operations;
  et.splay_del_steps = splay_del_steps;
  tree_write_file (fd, &et, sizeof et);

  add_operations = del_operations = look_up_operations = look_up_steps =
    nlook_up_operations = nlook_up_steps =
    splay_look_up_operations = splay_look_up_steps =
    splay_add_operations = splay_add_steps =
    splay_del_operations = splay_del_steps = 0;

  tree_dump_tree (fd, t->root);

  tree_close_file (fd);
  tree_next_name ();
}

static void
tree_dump_tree (int fd, object *n)
{
  /* Dump the current subtree to the dump file.
   */
  int zero_int = 0, one_int = 1;

  if (n == NULL)
    tree_write_file (fd, &zero_int, sizeof (int));
  else
    {
      tree_write_file (fd, &one_int, sizeof (int));
      tree_write_file (fd, n, sizeof (object));
      tree_dump_tree (fd, n->left);
      tree_dump_tree (fd, n->right);
      n->hits = 0;
    }
}

static void
tree_next_name (void)
{
  /* Increment the name `splaytree.aa', `splaytree.ab', etc.
   */
  int sl = strlen (dumpfile);
  char p;

  p = dumpfile [sl-1];
  p++;
  if (p == 'z' + 1)
    {
      p = 'a';
      (dumpfile [sl-2]) ++;
    }
  dumpfile [sl-1] = p;
}

#define BUFSIZE		4096
char wr_buffer [BUFSIZE];
int wr_pos = 0;

static void
tree_write_file (int fd, void *data, size_t size)
{
  /* Efficiently write data to given file, buffering nK internally. We can
   * safely assume that none of the writes will be larger than the buffer
   * size.
   */
  if (wr_pos + size > BUFSIZE)
    {
      if (write (fd, wr_buffer, wr_pos) != wr_pos)
	printf ("Dump failed at: write\n");
      wr_pos = 0;
    }
  memcpy (wr_buffer + wr_pos, data, size);
  wr_pos += size;
}

static void
tree_close_file (int fd)
{
  /* Flush what's in our buffer and close the file.
   */
  if (wr_pos > 0)
    {
      if (write (fd, wr_buffer, wr_pos) != wr_pos)
	printf ("Dump failed at: write\n");
      wr_pos = 0;
    }
  close (fd);
}

#endif /* SPLAY_COUNT_OPERATIONS */

/*----------------------------------------------------------------------
 *	If the cache is configured (set `USE_CACHE' in `bounds-lib.h')
 *	then we implement a pointer cache here, so we can hopefully
 *	look up pointers very quickly in the common case when they are
 *	reused.
 *	We cache pointers on the lower n bits (ignoring the least 2 bits
 *	which are invariant). The cache look-up is thus very quick compared
 *	to calling tree_lookup. We invalidate the cache in various ways
 *	when we add or delete objects to the main tree.
 *	NOTE (25/4/95): The cache is not worthwhile for two reasons:
 *	(a) We need to invalidate the whole cache frequently, since objects
 *	may appear in several cache locations at once.
 *	(b) The splay tree effectively caches frequently-used objects near
 *	the root.
 *----------------------------------------------------------------------*/

#if USE_CACHE

#define CACHE_BITS   	5		/* Cache size is 2^CACHE_BITS. */
#define CACHE_SIZE	(1<<(CACHE_BITS))

static struct {
  void *pointer;
  object *obj;
} cache [CACHE_SIZE];

void
__bounds_initialize_cache (void)
{
  memset (cache, 0, sizeof cache);
}

static inline int
cache_hashfn (void *pointer)
{
  /* Ignore the bottom 2 bits. Generate hash function on the next bits up.
   */
  return ((unsigned) pointer / sizeof (int)) & (CACHE_SIZE - 1);
}

static inline object *
cache_lookup (void *pointer)
{
  int offset = cache_hashfn (pointer);

  if (cache[offset].pointer == pointer)
    return cache[offset].obj;
  else
    return NULL;
}

static inline void
cache_update_on_add (void)
{
  /* This is called when we call tree_add. */
  memset (cache, 0, sizeof cache);
}

static inline void
cache_update_on_del (void)
{
  /* This is called when we call tree_del. */
  memset (cache, 0, sizeof cache);
}

static inline void
cache_add (void *pointer, object *obj)
{
  int offset = cache_hashfn (pointer);
  cache[offset].pointer = pointer;
  cache[offset].obj = obj;
}

#endif /* USE_CACHE */

/*----------------------------------------------------------------------
 *	A general function that looks up the object referred to by
 *	a pointer.
 *----------------------------------------------------------------------*/

inline object *
__bounds_find_object (void *pointer)
{
  object *result;

  enter_critical_section ();
#if USE_CACHE
  result = cache_lookup (pointer);
  if (result) return result;
#endif
  result = tree_lookup (&object_tree, pointer);
#if USE_CACHE
  if (result != NULL) cache_add (pointer, result);
#endif
  leave_critical_section ();
  return result;
}

inline object *
__bounds_find_object_by_base (void *base)
{
  object *result;

  enter_critical_section ();
#if USE_CACHE
  result = cache_lookup (base);
  if (result) return result;
#endif
  result = tree_lookup_base (&object_tree, base);
#if USE_CACHE
  if (result != NULL) cache_add (base, result);
#endif
  leave_critical_section ();
  return result;
}

/*----------------------------------------------------------------------
 *	Add, delete and lookup heap objects specifically.
 *----------------------------------------------------------------------*/

/* Check for power of 2 of aligment.
 * It is much faster doing an and than doing a mod function.
 */

static inline int
check_align(int n)
{
        return(((n - 1) & n) == 0);
}

void
__bounds_add_heap_object (void *base, size_t count, size_t align)
{
  object *obj;

#if COLLECT_STATS
  ++__bounds_stats_add_heap;
#endif
#if DEBUG_FEATURES
  if (__bounds_debug_print_calls) {
    printf ("__bounds_add_heap_object(p=%p, sz=%u, align=%u)\n",
	    base, count, align);
  }
#endif

  if (count == 0 || align == 0)
    __bounds_internal_error ("attempted to add a zero-sized heap object",
			     NULL, 0);

  obj = new_object ();
  obj->left = obj->right = obj->parent = NULL;
  obj->base = base;
  obj->size = count*align;
  obj->extent = base + count*align;
  obj->align = align;
  obj->sclass = obj_sclass_heap;
  obj->filename = NULL;
  obj->line = 0;
  obj->name = NULL;
  obj->alloca_function = NULL;
  /* Padding can be added after every heap object by the run-time malloc
   * library.
   */
  obj->no_padding = 0;
  obj->align_mask = check_align(align);
  obj->hits = 0;
  obj->steps = 0;
  enter_critical_section ();
  tree_add (&object_tree, obj);
#if USE_CACHE
  cache_update_on_add ();
#endif
  leave_critical_section ();
}

void
__bounds_delete_heap_object (void *base)
{
  object *obj;

#if COLLECT_STATS
  ++__bounds_stats_delete_heap;
#endif
#if DEBUG_FEATURES
  if (__bounds_debug_print_calls) {
    printf ("__bounds_delete_heap_object(p=%p)\n", base);
  }
#endif

  obj = __bounds_find_object_by_base (base);
  if (obj == NULL ||
      obj->sclass != obj_sclass_heap) {
    __bounds_internal_error ("__bounds_del_heap_obj passed pointer to non-heap object", __FILE__, __LINE__);
  }

  enter_critical_section ();
  tree_del (&object_tree, obj);
#if USE_CACHE
  cache_update_on_del ();
#endif
  leave_critical_section ();
  free_object (obj);
}

int
__bounds_is_heap_object (void *base)
{
  /* This function returns true if and only if the pointer passed is the base
   * of a heap object.
   */
  object *obj;

  return
    ((obj = __bounds_find_object_by_base (base)) != NULL &&
     obj->sclass == obj_sclass_heap);
}

/*----------------------------------------------------------------------
 *	Add a static object to the list of objects. This function may be
 *	called more than once for each object.
 *----------------------------------------------------------------------*/

void
__bounds_note_constructed_object (void *base, int size, size_t align,
				  char *filename, int line, char *name)
{
  object *obj = __bounds_find_object_by_base (base);
  obj_sclass sclass = size < 0 ? obj_sclass_external : obj_sclass_static;

  if (size < 0) size = -size;
#if DEBUG_FEATURES
  if (__bounds_debug_print_calls) {
    printf ("__bounds_note_constructed_object(p=%p, sz=%u, align=%u, file=\"%s\", ln=%d, name=\"%s\")\n",
	    base, size, align, filename, line, name);
  }
#endif

  if (size == 0 || align == 0)
    __bounds_internal_error ("attempted to add a zero-sized static object",
			     filename, line);

  if (obj && obj->sclass == obj_sclass_static && sclass == obj_sclass_static)
    {
      /* Object has already constructed. Just check that it is declared the
       * same way here.
       */
      if (obj->size != size ||
	  obj->align != align)
	{
	  __bounds_error ("static object redeclared with different size or alignment",
			  filename, line, base, obj);
	  return;
	}
      return;
    }

  /* Allow a new static declaration to unconditionally overwrite an
   * external one, since external ones are tentative.
   * Also allow an external one to overwrite another external one.
   */
  if (obj && obj->sclass == obj_sclass_external) {
    obj->size = size;
    obj->extent = base+size;
    obj->align = align;
    obj->sclass = sclass;
    obj->filename = filename;
    obj->name = name;
    obj->line = line;
    obj->no_padding = sclass == obj_sclass_external ? 1 : 0;
    obj->align_mask = check_align(align);
    return;
  }

  /* Ignore an external declaration if we already have a firm static
   * declaration for this object.
   */
  if (obj && sclass == obj_sclass_external)
    return;

  /* Create the object anew. */
  obj = new_object ();
  obj->left = obj->right = obj->parent = NULL;
  obj->base = base;
  obj->size = size;
  obj->extent = base + size;
  obj->align = align;
  obj->sclass = sclass;
  obj->filename = filename;
  obj->line = line;
  obj->name = name;
  obj->alloca_function = NULL;
  /* Padding is added after every static object now by the compiler.
   */
  obj->no_padding = sclass == obj_sclass_external ? 1 : 0;
  obj->align_mask = check_align(align);
  obj->hits = 0;
  obj->steps = 0;
  enter_critical_section ();
  tree_add (&object_tree, obj);
#if USE_CACHE
  cache_update_on_add ();
#endif
  leave_critical_section ();
}

/*----------------------------------------------------------------------
 *	This is used to build a list of private static data. The table
 *	is built by GCC, and a pointer to it is passed here.
 *----------------------------------------------------------------------*/

void
__bounds_note_constructed_private_table (private_table *table, char *filename,
					 int line)
{
  char *name;
  char *unnamed = "(unnamed static)";

  while (table->ptr) {
    if (table->name != NULL)
      name = table->name;
    else
      name = unnamed;
    if (table->size > 0)
      __bounds_note_constructed_object (table->ptr, table->size, 1,
					NULL, 0, name);
    else
      /* GCC with bounds checking sometimes generates zero-sized entries
       * in the table, even though these ought to give warnings and be
       * ignored at compile time. Generate warnings here. If these warnings
       * appear, the compiler will need to be fixed.
       * (4/4/95: These should be fixed now).
       */
      __bounds_warning (filename, line, NULL,
			"zero-sized object `%s' generated in private_statics table",
			name);
    table++;
  }
}

/*----------------------------------------------------------------------
 *	Create a record of a stack object. This function is called once
 *	for each local variable when we enter a function or block.
 *----------------------------------------------------------------------*/

inline object *
__bounds_internal_add_stack_object (void *base, size_t size, size_t align,
				    char *filename, int line, char *name,
				    int no_padding)
{
  object *obj;
#if COLLECT_STATS
  ++__bounds_stats_add_stack;
#endif
#if 0
  /* Check that we don't create duplicates of stack objects. There is currently
   * (19/7/95) a bug whereby we occasionally don't call '__bounds_pop_function'
   * This can cause duplicate objects to be created here, so we must check for
   * this. However, this is slow, so hopefully it can be deleted in future
   * when we fix the other bug.
   */
  obj = __bounds_find_object_by_base (base);
  if (obj != NULL)
    __bounds_internal_error ("duplicate stack object created in `__bounds_internal_add_stack_object'", filename, line);
#endif

  obj = new_object ();

  if (size == 0 || align == 0)
    __bounds_internal_error ("attempted to add a zero-sized stack object",
			     filename, line);

  /* Create a new object record. */
  obj->left = obj->right = obj->parent = NULL;
  obj->base = base;
  obj->size = size;
  obj->extent = base + size;
  obj->align = align;
  obj->sclass = obj_sclass_stack;
  obj->filename = filename;
  obj->line = line;
  obj->name = name;
  obj->alloca_function = NULL;
  /* For parameters, no padding can be adding between adjacent objects. Setting
   * this flag allows you to increment a pointer beyond a parameter and get
   * to the next parameter in the list.
   */
  obj->no_padding = no_padding;
  obj->align_mask = check_align(align);
  obj->hits = 0;
  obj->steps = 0;
  enter_critical_section ();
  tree_add (&object_tree, obj);
#if USE_CACHE
  cache_update_on_add ();
#endif
  leave_critical_section ();

  return obj;
}

inline void
__bounds_internal_delete_stack_object (object *obj)
{
#if COLLECT_STATS
  ++__bounds_stats_delete_stack;
#endif
  if (obj == NULL ||
      obj->sclass != obj_sclass_stack) {
    __bounds_internal_error ("__bounds_delete_stack_object passed pointer to non-stack object", __FILE__, __LINE__);
  }
  enter_critical_section ();
  tree_del (&object_tree, obj);
#if USE_CACHE
  cache_update_on_del ();
#endif
  leave_critical_section ();
  free_object (obj);
}

void
__bounds_add_stack_object (void *base, size_t size, size_t align,
			   char *filename, int line, char *name)
{
#if DEBUG_FEATURES
  if (__bounds_debug_print_calls) {
    printf ("__bounds_add_stack_object(p=%p, sz=%u, align=%u, file=\"%s\", ln=%d, name=\"%s\")\n",
	    base, size, align, filename, line, name);
  }
#endif

  __bounds_internal_add_stack_object (base, size, align,
				      filename, line, name,
				      0);
}

void
__bounds_delete_stack_object (void *base)
{
  object *obj;

#if DEBUG_FEATURES
  if (__bounds_debug_print_calls) {
    printf ("__bounds_delete_stack_object(p=%p)\n", base);
  }
#endif

  obj = __bounds_find_object_by_base (base);
  if (obj == NULL
      || obj->sclass != obj_sclass_stack)
    __bounds_internal_error ("unknown or non-stack object passed to __bounds_delete_stack_object",
			     __FILE__, __LINE__);
  __bounds_internal_delete_stack_object (obj);
}

/*----------------------------------------------------------------------
 *	Note the location of the arguments to main.
 *	(31/5/95: Made these into stack objects.)
 *----------------------------------------------------------------------*/

void
__bounds_note_main_args (int argc, char **argv)
{
  int i;

  if (__bounds_checking_on) {
#if COLLECT_STATS
    ++__bounds_stats_environment;
#endif
    __bounds_internal_add_stack_object (argv, (argc+1) * sizeof (char *),
					sizeof (char *), "main", 0, "argv", 1);
    for (i = 0; i < argc; ++i) {
#if COLLECT_STATS
      ++__bounds_stats_environment;
#endif
      __bounds_internal_add_stack_object (argv[i],
					  strlen (argv[i]) + 1, 1,
					  "main", 0, "argv", 1);
    }
  }
}

/*----------------------------------------------------------------------
 *	Produce a debugging memory map on stderr. Only lists objects
 *	between minbase and maxbase. If either of these are NULL, then
 *	they default sensibly.
 *----------------------------------------------------------------------*/

static double total_hits = 0.0;
static double total_steps = 0.0;

static inline char *
sclass_name (object *obj)
{
  switch (obj->sclass) {
  case obj_sclass_heap:
    return "heap";
  case obj_sclass_static:
    return "static";
  case obj_sclass_external:
    return "external";
  case obj_sclass_stack:
    return "stack";
  }
  return NULL;
}

static unsigned
print_map (object *obj, int level, void *minbase, void *maxbase)
{
  unsigned count;
  double depth;
  unsigned n1;
  unsigned n2;

  if (obj && minbase <= obj->base && obj->base <= maxbase) {
    count = print_map (obj->left, level+1, minbase, maxbase);
    total_hits += (double)obj->hits;
    total_steps += (double)obj->steps;
    depth = obj->hits ? ((double)obj->steps * 100.0) / (double)obj->hits : 0.0;
    n1 = (unsigned)(depth / 100.0);
    n2 = (unsigned)(depth - 100.0 * (double)n1);
    printf ("%10u %2u.%02u %p-%p %4u %4u (%2u) %s %s\n",
	    obj->hits,
	    n1,
	    n2,
	    obj->base,
	    obj->extent - 1,
	    obj->size,
	    obj->align,
	    level,
	    sclass_name (obj),
	    obj->name);
    count += print_map (obj->right, level+1, minbase, maxbase);
    return count + 1;
  }
  else
    return 0;
}

void
__bounds_debug_memory (void *minbase, void *maxbase)
{
  unsigned count;
  double depth;
  unsigned n1;
  unsigned n2;

  if (maxbase == NULL) maxbase = (void *) (-1);

  enter_critical_section ();

  printf ("Memory map:\n");
  count = print_map (object_tree.root, 0, minbase, maxbase);

  depth = total_hits ? ((double)total_steps * 100.0) / (double)total_hits : 0.0;
  n1 = (unsigned)(depth / 100.0);
  n2 = (unsigned)(depth - 100.0 * (double)n1);
  printf ("Number of objects: %u  average steps %u.%02u\n", count, n1, n2);

  leave_critical_section ();
}
