/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.system.classes;

import java.io.*;
import java.lang.*;
import java.util.*;

/** This class encapsulates both String & Object
  * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
  * @version 0.7, 12/12/97
*/
class StringListItem implements Serializable {
  String FString;
  Object FObject;

  public StringListItem(String FString, Object FObject) {
    this.FString = FString;
    this.FObject = FObject;
  }

  public StringListItem() {
    FString = "";
    FObject = null;
  }
}

/** <code>StringList</code> is a subclass of Strings class for creating a collection of String & Object pairs.
  * It Does Not override writeObject() or readObject() methods.
  * So it is your responsobilities to inherit from it and do that it it is neccessary.
  * This object is some Java variant of the Borland's Delphi TStringList class. Many of methods defined
  * in TStrings are translated from Object Pascal to Java.
  *
  * @author	George Petkov <pgeorge@mail.techno-link.com>,<gpetkov@hotmail.com>, <gpetkov@usa.net>
  * @version 0.7, 12/12/97
*/

public class StringList extends Strings {
  Vector Elements;
  Vector ChangeListeners;

  public StringList(int initialCapacity, int capacityIncrement) {
    super();
    Elements = new Vector(initialCapacity, capacityIncrement);
    Init();
  }

  public StringList() {
    super();
    Elements = new Vector();
    Init();
  }

  /** This function is called by the constructors to make the remaining part */
  protected void Init() {
    ChangeListeners = new Vector();
  }

  /** Returns the Listener at this Index */
  private StringListItem getVectorObjectAt(int Index) throws ArrayIndexOutOfBoundsException {
    return (StringListItem)Elements.elementAt(Index);  // This function is synchronized
  }

  /** This function is called just befor some Changes are beeing performed, and after all of them */
  private void CallChangeAdapter(boolean IsBeforeChange) {
    int Cntr;
    StringsListener Adapter;

    try {
      for (Cntr = 0; Cntr < ChangeListeners.size(); Cntr++) {
        Adapter = (StringsListener)(ChangeListeners.elementAt(Cntr));
        if (IsBeforeChange) Adapter.BeforeChange(this);
        else                Adapter.AfterChange (this);
      }
    }
    catch(ArrayIndexOutOfBoundsException e) {} // this will never happend, but I do not want to write everywhere this exception.
  }

  /** Check to call BeforeChange() Listener's methods */
  protected void Changing() {
    if (FUpdateCount == 0) CallChangeAdapter(true);
  }

  /** Check to call AfterChange() Listener's methods */
  protected void Changed () {
    if (FUpdateCount == 0) CallChangeAdapter(false);
  }

  protected void   SetUpdateState(boolean Updating) {
    if (Updating) Changing();
    else          Changed ();
  }

  protected int    GetCapacity() {
    return Elements.capacity();
  }

  public synchronized Object clone() {
    int        ItemsCntr;
    try {
      StringList StrL      = (StringList)super.clone();
      StrL.Elements        = new Vector();
      StrL.ChangeListeners = new Vector();

      for (ItemsCntr = 0; ItemsCntr < getCount(); ItemsCntr++) {
        StrL.Add(getString(ItemsCntr));
      }

      return StrL;
    }
    catch (CloneNotSupportedException e) { // this shouldn't happen, since we are Cloneable
      throw new InternalError();
    }
  }

  /** Adds a listener which is activated before any Changes and after all of them */
  public void AddChangeListener(StringsListener Listener) {
    if (Listener == null) return;
    ChangeListeners.addElement(Listener);
  }

  /** Adds a listener which is activated before any Changes and after all of them */
  public void RemoveChangeListener(StringsListener Listener) {
    ChangeListeners.removeElement(Listener);
  }

  public boolean isChangeListeners() {
    return ! ChangeListeners.isEmpty();
  }

  /** returns number items in collection */
  public int getCount() {
    return Elements.size();
  }

  /** returns String at Index */
  public String getString(int Index) throws ArrayIndexOutOfBoundsException {
    return getVectorObjectAt(Index).FString;
  }

  /** returns Object at Index, or null */
  public Object getObject (int Index) throws ArrayIndexOutOfBoundsException {
    return getVectorObjectAt(Index).FObject;
  }

  public Object getObject (String Name) {
    int Index = IndexOf(Name);
    if (Index < 0) return null;
    try {
      return getObject(Index);
    }
    catch (ArrayIndexOutOfBoundsException e) {
      return null;
    }
  }

  /** Changes the String at Index */
  public synchronized void setString(int Index, final String S) throws ArrayIndexOutOfBoundsException {
    CheckBounds(Index);
    Changing();
    getVectorObjectAt(Index).FString = S;
    Changed();
  }

  /** Changes the Object at Index */
  public synchronized void setObject(int Index, Object AObject) throws ArrayIndexOutOfBoundsException {
    CheckBounds(Index);
    Changing();
    getVectorObjectAt(Index).FObject = AObject;
    Changed();
  }

  /** Inserts String & Object at Index */
  public synchronized void Insert(int Index, final String S, Object AObject) throws ArrayIndexOutOfBoundsException {
    Changing();
    Elements.insertElementAt(new StringListItem(S,AObject), Index);
    Changed();
  }

  /** Delete all elements */
  public synchronized void Clear() {
    if (getCount() == 0) return;
    Changing();
    Elements.removeAllElements();
    Changed();
  }

  /** Deletes element at Index */
  public synchronized void Delete(int Index) throws ArrayIndexOutOfBoundsException {
    Changing();
    Elements.removeElementAt(Index);
    Changed();
  }

  public synchronized void   Delete(String ItemName) {
    int ItemIndex = IndexOf(ItemName);
    if (ItemIndex < 0) return;
    try {
      Delete(ItemIndex);
    }
    catch (ArrayIndexOutOfBoundsException e) {};
  }

  public synchronized void   Delete(Object Item) {
    int ItemIndex = IndexOf(Item);
    if (ItemIndex < 0)
      return;
    try {
      Delete(ItemIndex);
    }
    catch (ArrayIndexOutOfBoundsException e) {}
  }
}
