/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.system.classes.options.manager;

import java.io.*;
import java.lang.*;
import org.freebuilder.system.debug.*;
import org.freebuilder.system.ideengine.*;
import org.freebuilder.system.classes.*;
import org.freebuilder.system.classes.options.*;
import org.freebuilder.system.classes.options.opened.*;
import org.freebuilder.system.classes.events.*;
import org.freebuilder.system.classes.events.engine.IdeEventsEngine;

/** There are 3 kinds of configuration : default, main, project and current.
  * The first time you start the application, the DEFAULT configuration is made. All the
  * tools, dialogs .. are "hard-coded" configured by the application. When you began to
  * change these options, they are written like main configuration. Thise main configuration
  * will be the project default configuration for every new project. For every project you
  * can change some options. This configuration will be saved like a project configuration.
  * If there is not opened project, the Current configuration is main configuration,
  * otherwise project configuration.
  * Options can be changed :
  * 1. By Dialogs you can customize them, or you can set default options (hard coded in program).
  * 2. When open a new project file, the main options are assigned.
  * 3. No special tasks are required ! IF you change a place and size for some windows,
  * or open, close files or projects.
  *
  * The class <code>OptionsMngr</code> is responsible for doing the above tasks.
  * Every project have a corresponding configuration file. When there is not open project
  * the main configuration is changed. It also save history of last 20th open project,
  * files and browsed files. So you can quickly reopen one of them. if a project have been
  * opened on application close, it opens it after start of the application and all the
  * options for that project (including even Beans palette configuration) for that project
  * is restored. The name of the project configuration file is the name of the project and ".cfg"
  *
  * There is two memeber field Objects : MainConfig MainOptions which holds and manage
  * MainConfiguration and ProjectConfig   ProjectOptions for project configuration.
  * You can get only Current configuration via calling getOptionsColl()
  *
  * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
  * @version 0.7, 02-12-97 */

public class OptionsMngr {
  public static final String cProjectConfigFileExt = ".cfg";
  private MainConfig             MainOptions       = null;
  private ProjectConfig          ProjectOptions    = null;
  private CreateDOptionsListener DOptions          = null;

  /** Have to pass a EventsEngine and the name for main configuration file */
  public OptionsMngr(String AConfigFileName, CreateDOptionsListener ADOptions) {
    DOptions     = ADOptions;
    try {
      MainOptions = (MainConfig)MainConfig.Load(Globals.EventsEngine, AConfigFileName);
      MainOptions.setDOptionsListener(DOptions);
    }
    catch (Exception e) {
      MainOptions = null;
    }

    if (MainOptions == null) {
      MainOptions = new MainConfig(Globals.EventsEngine, AConfigFileName, DOptions);
      MainOptions.CreateDefaultConfig();
    }
  }

  /** We will change first the Project Options if there is an opened project */
  protected ProjectConfig getOptionsColl() {
    if (ProjectOptions != null) return ProjectOptions;
    return MainOptions;
  }

  public ProjectConfig getProjectOptions() {
    return ProjectOptions;
  }  

  /** Enable or disable save main and project options. After enable it a RefreshOptions
    * method is called */
  public void EnableSaveOptions(boolean Enable) {
    MainOptions.EnableSaveOptions(Enable);
    if (ProjectOptions != null)
      ProjectOptions.EnableSaveOptions(Enable);
  }

  /** Can OptionsMngr send events ? */
  public boolean IsEventsEnabled() {
    return MainOptions.IsEventsEnabled();
  }

  /** Is current configuration (main or project) loaded from some configuration file */
  public boolean IsLoaded() {
    return getOptionsColl().IsLoaded();
  }

  /** Enables sending the events from main and project options */
  public void EnableEvents(boolean Enable) {
    if (MainOptions.IsEventsEnabled() == Enable) return;
    if (Enable) {
      MainOptions.EnableEvents(Enable);
      if (ProjectOptions != null)
        ProjectOptions.EnableEvents(Enable);
    }
    else {
      if (ProjectOptions != null)
        ProjectOptions.EnableEvents(Enable);
      MainOptions.EnableEvents(Enable);
    }
  }

  /** Refresh (sends msgOptionsRefresh) the options which exist in both configurations */
  public synchronized void Refresh() {
    int Cntr, To;
    OptionsSuperI Options;

    if (ProjectOptions == null) {
      MainOptions.RefreshOptions();
      return;
    }

    ProjectOptions.RefreshOptions();

    To = MainOptions.getCount();
    try {
      for (Cntr = 0; Cntr < To; Cntr++) {
        Options = MainOptions.getOptions(Cntr);
        if (! ProjectOptions.IsOptions(Options))
          MainOptions.SendOptionsRefresh(Options);
      }
    }
    catch (ArrayIndexOutOfBoundsException e) {
      if (Debug.isDebug) {
        Debug.Assert(e,this, "IndexOutOfBounds");
      }
    }
  }

  /** Search in Project (if it is opened) & Main Options Collections */
  public OptionsSuperI getOptions(String OptionsName) {
    OptionsSuperI Result = null;
    if (ProjectOptions != null)
      Result = ProjectOptions.getOptions(OptionsName);

    if (Result != null) return Result;
    Result = MainOptions.getOptions(OptionsName);
    return Result;
  }

  public void setOptions(String OptionsName, OptionsSuperI NewOptions) { // Changed
    ProjectConfig CrntOptions = getOptionsColl();
    if (CrntOptions == null) return;
    CrntOptions.setOptions(OptionsName, NewOptions);
  }

  public void setOptions(OptionsSuperI NewOptions) {
    ProjectConfig CrntOptions = getOptionsColl();
    if (CrntOptions == null) return;
    CrntOptions.setOptions(NewOptions);
  }

  public boolean IsOptions (String OptionsName) {
    if (ProjectOptions != null)
      if (ProjectOptions.IsOptions(OptionsName))
        return true;

    return MainOptions.IsOptions(OptionsName);
  }

  /** Is options with the name of the NewOptions ? */
  public boolean IsOptions (OptionsSuperI NewOptions) {
    String OptionsName;
    if (NewOptions == null) return false;
    OptionsName = NewOptions.getName();
    return IsOptions(OptionsName);
  }

  public boolean addOptions(OptionsSuperI NewOptions) throws DuplicateOptionNameException {
    if (ProjectOptions != null)
      return ProjectOptions.addOptions(NewOptions);

    return MainOptions.addOptions(NewOptions);
  }

  /** if Project is opened This method will remove required options from project's options
    * only. Otherwise will remove them from Main Options ! */
  public boolean removeOptions(String OptionsName) {
    ProjectConfig CrntOptions = getOptionsColl();
    if (CrntOptions == null) return true;

    return CrntOptions.removeOptions(OptionsName);
  }

  /** if Project is opened This method will remove required options from project's options
    * only. otherwise will remove them from Main Options ! */
  public boolean removeOptions(OptionsSuperI AOptions) {
    String OptionsName;
    if (AOptions == null) return true;
    OptionsName = AOptions.getName();
    return removeOptions(OptionsName);
  }

  /** Restore main options for current configuration */
  public void RestoreDefaults() {
    if (ProjectOptions != null) {
      ProjectOptions.RestoreDefaults();
      return;
    }
    MainOptions.RestoreDefaults();
  }

  //=========================================================================================
  // Project Load, New, Close .... functions 
  //=========================================================================================
  /** Open a coresponding options for the new project with AFileName. Adds this name to the
    * collection of the last opened files, and make it first. if IsEventsEnabled() for a
    * main configuration set it true and for project configuration. if IsSaveEnabled() for
    * a main configuration set it true and for project configuration. The last means that
    * after all the RefreshOptions is called for project configuration, and all the objects
    * which have a options will change them depending on these which are in project configuration. */
  public boolean OpenProject(String AFileName) {
    String ProjectConfigFileName = AFileName;
    if (ProjectConfigFileName.length() == 0) return false;

    if (ProjectOptions != null) {
      Debug.Assert(this," Have to Close Project before open the other one");
      CloseLastProject();
    }

    try {
      ProjectOptions = (ProjectConfig)ProjectConfig.Load(Globals.EventsEngine, ProjectConfigFileName);
      ProjectOptions.setDOptionsListener(DOptions);
    }
    catch (Exception e) {
      ProjectOptions = null;
      return false;
    }

    CallAfterOpenCreateProject(ProjectConfigFileName);
    return true;
  }

  public void afterRenameProject (String OldProjectName) {
    if ((OldProjectName == null) || (OldProjectName.length() == 0))
      return;
    StrLLastProjects LastOpenedProjects = (StrLLastProjects) MainOptions.getOptions (MainConfig.cRecentProjectsOName);
    if (LastOpenedProjects != null) {
      LastOpenedProjects.Delete  (OldProjectName);
      MainOptions.setOptions (LastOpenedProjects);
    }
  }

  public void onRenameProject (String AProjectFileName) {
    updateProjectName (AProjectFileName);
    Globals.EventsEngine.PostEvent (new SystemEvent (SystemEvent.msgProjectNameChanged, this, new String (AProjectFileName)));
  }

  private void updateProjectName (String AProjectFileName) {
    Globals.setProjectName  (AProjectFileName);
    MainOptions.OpenProject (AProjectFileName); // Add the new project name in the Last Opened Projects Collection
  }

  private void CallAfterOpenCreateProject(String AProjectFileName) {
    Debug.Assert(ProjectOptions != null, this, "CallAfterOpenCreateProject  ProjectOptions = null");

    if (MainOptions.IsEventsEnabled())
      ProjectOptions.EnableEvents(true);

    if (MainOptions.IsSaveEnabled())
      ProjectOptions.EnableSaveOptions(true);

    updateProjectName(AProjectFileName);
  }

  /** Creates a new Project with Project name in function parameter */
  public void CreateNewProject(String AProjectName) {
    if (isOpenedProject()) CloseLastProject();
    ProjectOptions = new ProjectConfig(Globals.EventsEngine, AProjectName, DOptions);
    ProjectOptions.CreateDefaultConfig();
    ProjectOptions.UpdateOptionsFrom(MainOptions);

    CallAfterOpenCreateProject(AProjectName);
  }

  /** Close the project configuration, and leave only the main configuration. If you will
    * not open a new project file, after that, have to call RefreshOptions, to refresh them
    * coresponding to the main configuration. This method designates that there is not an
    * open project file, so if you close the application, the next time you start, there
    * will be not an open project. */
  public void    CloseLastProject() { // Only 1 project will be opened at a time.
    if (ProjectOptions == null) {
      if (Debug.isDebug)
        Debug.Assert(ProjectOptions != null,this," Project Options are already closed");
      return;
    }

   ProjectOptions.EnableEvents     (false);
   ProjectOptions.EnableSaveOptions(false);
   ProjectOptions.RemoveChangeListener(ProjectOptions);
   ProjectOptions = null;
   MainOptions.CloseLastProject();
   Globals.setProjectName("");
  }

  public boolean isOpenedProject() { return ProjectOptions != null; }
  public boolean CanSaveProject()  { return isOpenedProject(); }

  public boolean SaveProjectAs(String NewFileName) {
    if ((! isOpenedProject()) || (NewFileName == null) ) return false;
    if (NewFileName.length() == 0) return false;

    ProjectOptions.setConfigFileName(NewFileName);
    ProjectOptions.Save();
    onRenameProject(NewFileName);
    return true;
  }

  public boolean SaveProject() {
    return SaveProjectAs(Globals.getProjectName());
  }
  //=========================================================================================
  //=========================================================================================


  /** Adds file with that name to the list of last opened files, and make it first */
  public void    OpenFile   (String AFileName) {
    ProjectConfig CrntOptions = getOptionsColl();
    if (CrntOptions == null) return;
    CrntOptions.OpenFile(AFileName);
  }

  public void    CloseFile  (String AFileName) {
    ProjectConfig CrntOptions = getOptionsColl();
    if (CrntOptions == null) return;
    CrntOptions.CloseFile(AFileName);
  }

  /** Was an open project the last time you close the application */
  public boolean ShallOpenProjectFile() {
    return MainOptions.WasLastProjectOpened();
  }

  /** Returns the name of the last opened project file you have worked on it the last time */
  public String getLastProjectFileName() {
    String LastOpenProject = MainOptions.getLastProjectFileName();
    return LastOpenProject;
  }

  /** for testing */
  public void PrintOptions() {
    System.out.println("Main Options");
    MainOptions.PrintOptions();
    System.out.println("");
    System.out.println("");

    if (isOpenedProject()) {
      System.out.println("Project : " + ProjectOptions.getConfigFileName() + " Options");
      ProjectOptions.PrintOptions();
    }
  }
}
