/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.gui.ide;

import pizza.compiler.*;
import sun.tools.java.*;
import sun.tools.javac.*;
import sun.tools.debug.*;
import sun.tools.javadoc.*;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.util.*;
import java.beans.*;
import java.io.*;
import java.net.*;

import org.freebuilder.gui.beans.*;
import org.freebuilder.gui.config.*;
import org.freebuilder.gui.idemenu.*;
import org.freebuilder.gui.editors.*;
import org.freebuilder.system.paths.*;
import org.freebuilder.system.debug.*;
import org.freebuilder.system.packages.*;
import org.freebuilder.system.projectman.*;
import org.freebuilder.system.classes.events.*;
import org.freebuilder.system.classes.events.engine.*;

import org.freebuilder.tools.editor.*;
import org.freebuilder.tools.output.*;

import org.freebuilder.gui.ideoptions.*;

import org.freebuilder.system.classes.*;
import org.freebuilder.system.ideengine.*;
import org.freebuilder.system.classes.options.*;
import org.freebuilder.system.classes.options.manager.*;

import com.sun.java.swing.*;
import com.sun.java.swing.tree.*;
import com.sun.java.swing.event.*;
import com.sun.java.swing.plaf.basic.*;

/**
  * @author	<a href=mailto:wreck@nat.bg> Peter Radkov </a>
    @version 0.7, 07-01-98
*/
class Finder {
  private int     FindFrom;
  private String  FindWhat;
  private String  FindWhere;
  private boolean WholeWordsOnly;
  private boolean isForward;
  private JTextArea TheTextArea;
  private int Start;
  private boolean OkToFindAgain = false;

  private boolean isToReplace;
  private boolean promptOnReplace;
  private boolean isToReplaceAll;
  private String  ReplaceWith;

  public void find () {
    FindDialog Finder = new FindDialog();
    Finder.setVisible (true);
    if (!Finder.isConfirmed())
      return;
    setData (Finder);
    isToReplace = false;
    findAgain();
  }

  public void replace () {
    ReplaceDialog Replacer = new ReplaceDialog();
    Replacer.setVisible (true);
    if (!Replacer.isConfirmed())
      return;
    setData (Replacer);
    isToReplace = true;
    promptOnReplace = Replacer.isPromptOnReplace();
    isToReplaceAll  = Replacer.isToReplaceAll();
    ReplaceWith     = Replacer.getReplaceWith();
    findAgain();
  }

  private void setData (FindDialog Finder) {
    FindWhat    = Finder.getTextToFind();
    TheTextArea = ((EditFile)Globals.TextFiles.getActiveTextFile()).getSourceArea().getTextArea();

    if (Finder.isGlobal()) {
      FindWhere = TheTextArea.getText();
      if (Finder.isFromCursor()) {
        FindFrom = TheTextArea.getCaretPosition();
        FindWhere = FindWhere.substring (FindFrom, FindWhere.length());
      }
    } else {
      FindWhere = TheTextArea.getSelectedText();
      FindFrom = TheTextArea.getSelectionStart();
    }

    if (!Finder.isCaseSensitive()) {
      FindWhere = FindWhere.toLowerCase();
      FindWhat  = FindWhat .toLowerCase();
    }

    WholeWordsOnly = Finder.isWholeWordsOnly();
    isForward      = Finder.isForward();

    Start = FindFrom;
    FindFrom = 0;
    OkToFindAgain = true;
  }

  public void findAgain () {
    if (!OkToFindAgain) {
      Misc.messageBox (new String [] {"OK"}, new String [] {"Unable to find the specified text."}, 0);
      return;
    }

    int FoundIndex = 0;
    boolean Found = false;

    while (!Found) {
      if (isForward)
        FoundIndex = FindWhere.indexOf (FindWhat);
      else
        FoundIndex = FindWhere.lastIndexOf (FindWhat);

      if (FoundIndex < 0)
        break;

      if (WholeWordsOnly) {
        if ((FoundIndex > 0) &&
            (Character.isLetter(FindWhere.charAt(FoundIndex - 1)))) {
          FindFrom = FoundIndex + FindWhat.length();
          if (FindFrom > FindWhere.length() - FindWhat.length())
            break;
          continue;
        }
        if ((FoundIndex + FindWhat.length() < FindWhere.length()) &&
            (Character.isLetter(FindWhere.charAt(FoundIndex + FindWhat.length())))) {
          FindFrom = FoundIndex + FindWhat.length();
          if (FindFrom > FindWhere.length() - FindWhat.length())
            break;
          continue;
        }
        Found = true;
      } else
        Found = true;
    }

    if (Found) {
      TheTextArea.select   (Start + FoundIndex, Start + FoundIndex + FindWhat.length());
      if (isForward) {
        FindWhere = FindWhere.substring (FoundIndex + FindWhat.length(), FindWhere.length());
        Start += FoundIndex + FindWhat.length();
      } else {
        FindWhere = FindWhere.substring (0, FoundIndex);
      }
      FindFrom  = 0;
      if (isToReplace) {
        boolean OKToReplace = true;
        if (promptOnReplace) {
           int PromptResult =  Misc.messageBox (new String [] {"Yes", "No", "All"}, new String [] {"Searched text found. Replace?"}, 1);
           switch (PromptResult) {
             case 0: OKToReplace = true;
             break;
             case 1: OKToReplace = false;
             break;
             case 2: OKToReplace = true;
                     isToReplaceAll = true;
                     promptOnReplace = false;
             break;
           }
        }
        if (OKToReplace) {
          TheTextArea.replaceSelection (ReplaceWith);
          if (isForward)
            Start += (ReplaceWith.length() - FindWhat.length());
        }
        if (isToReplaceAll)
          findAgain();
      }
    } else {
      OkToFindAgain = false;
      if (isToReplace)
        Misc.messageBox (new String [] {"OK"}, new String [] {"All occuraneces of the search string replaced."}, 0);
      else
        Misc.messageBox (new String [] {"OK"}, new String [] {"Unable to find the specified text."}, 0);
    }
  }
}



/**
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class NotDirectory implements FilenameFilter {
  public boolean accept (File TheDir, String FileName) {
    File NewFile = new File (TheDir.getAbsolutePath() + Globals.FileSeparator + FileName);
    return !NewFile.isDirectory();
  }
}

/**
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class Directory implements FilenameFilter {
  public boolean accept (File TheDir, String FileName) {
    File NewFile = new File (TheDir.getAbsolutePath() + Globals.FileSeparator + FileName);
    return NewFile.isDirectory();
  }
}

/**
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class PackageEntry implements FilenameFilter {
  public boolean accept (File TheDir, String FileName) {
    return (FileName.endsWith(Globals.JavaFileExt) ||
            FileName.endsWith(Globals.PizzaFileExt)||
            FileName.endsWith(Globals.ClassFileExt));
  }
}


class DebugRunner extends java.lang.Thread {
  private String [] PrgArgs;
  private RemoteDebugger TheRD;
  public DebugRunner (RemoteDebugger RD, String [] args) {
    PrgArgs = args;
    TheRD = RD;
  }
  public void run () {
    try {
      TheRD.run(PrgArgs.length, PrgArgs);
    } catch (Exception e) {
      System.out.println ("Error during RemoteDebugger.run().");
    }
  }
}

class Runner extends java.lang.Thread {
  private String [] PrgArgs;
  public Runner (String [] args) {
    PrgArgs = args;
  }
  public void run () {
    try {
      Runtime.getRuntime().exec (PrgArgs);
    } catch (IOException e) {
      System.out.println ("I/O Error during exec(...).");
    }
  }
}

/**
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class AddToJarDialog extends JDialog implements ActionListener, ChangeListener {
/*
  -v  generate verbose output on standard error
  -m  include manifest information from specified manifest file
  -0  store only; use no ZIP compression
  -M  Do not create a manifest file for the entries
*/

  private static final String OK_CMD     = "Store";
  private static final String CANCEL_CMD = "Cancel";
  private static final String CHOOSE_FILE_CMD     = "ChooseArchive";
  private static final String CHOOSE_MANIFEST_CMD = "ChooseManifest";

  private JPanel MainPanel         = new JPanel ();
  private JPanel ButtonsPanel      = new JPanel ();
  private BPanel FileNamePanel     = new BPanel ("Archive");
  private BPanel ManifestNamePanel = new BPanel ("Mainfest");
  private BPanel OptionsPanel      = new BPanel ("Options");
  private Button OKButton          = new Button (OK_CMD);
  private Button CancelButton      = new Button (CANCEL_CMD);

  private JButton ChooseFileButton     = new JButton ("Browse");
  private JButton ChooseManifestButton = new JButton ("Browse");

  private JCheckBox Verbose        = new JCheckBox ("Generate verbose output");
  private JCheckBox Manifest       = new JCheckBox ("Include manifest information from specified manifest file");
  private JCheckBox StoreOnly      = new JCheckBox ("Store only; use no ZIP compression",            true);
  private JCheckBox NoManifest     = new JCheckBox ("Do not create a manifest file for the entries", true);

  private JTextField FileName      = new JTextField();
  private JTextField ManifestName  = new JTextField();

  public AddToJarDialog () {
    super(Globals.IdeMainFrame, "Store project in archive");
    ButtonsPanel.setLayout (new GridLayout (1, 2, 5, 5));
    ButtonsPanel.add (OKButton);
    ButtonsPanel.add (CancelButton);

    FileNamePanel.setLayout (new BorderLayout (5, 5));
    FileNamePanel.add (FileName,         BorderLayout.CENTER);
    FileNamePanel.add (ChooseFileButton, BorderLayout.EAST);

    ManifestNamePanel.setLayout (new BorderLayout (5, 5));
    ManifestNamePanel.add (ManifestName,         BorderLayout.CENTER);
    ManifestNamePanel.add (ChooseManifestButton, BorderLayout.EAST);

    OptionsPanel.setLayout (new GridLayout (4, 1, 1, 1));
    OptionsPanel.add (Verbose);
    OptionsPanel.add (Manifest);
    OptionsPanel.add (StoreOnly);
    OptionsPanel.add (NoManifest);

    Manifest.addChangeListener (this);

    ChooseFileButton    .setActionCommand(CHOOSE_FILE_CMD);
    ChooseManifestButton.setActionCommand(CHOOSE_MANIFEST_CMD);

    MainPanel.setLayout (new BorderLayout());
    MainPanel.add (FileNamePanel,     BorderLayout.NORTH);
    MainPanel.add (ManifestNamePanel, BorderLayout.CENTER);
    MainPanel.add (OptionsPanel,      BorderLayout.SOUTH);

    getContentPane().setLayout (new BorderLayout());
    getContentPane().add (ButtonsPanel, BorderLayout.SOUTH);
    getContentPane().add (MainPanel,    BorderLayout.NORTH);

    pack();
    Misc.hardCenter (this);

    OKButton.            addActionListener (this);
    CancelButton.        addActionListener (this);
    ChooseFileButton.    addActionListener (this);
    ChooseManifestButton.addActionListener (this);
    stateChanged (null);
  }

  public void stateChanged (ChangeEvent e) {
    boolean State = Manifest.isSelected();
    ManifestName.setEditable         (State);
    ManifestName.setEnabled          (State);
    ChooseManifestButton.setEnabled  (State);
    ManifestName.repaint();
  }

  private void chooseFile (JTextField GUI) {
    FileDialog FD = new FileDialog (Globals.IdeMainFrame);
    FD.setVisible (true);
    if (FD.getDirectory() == null)
      return;
    GUI.setText (FD.getDirectory() + FD.getFile());
  }

  private void getPackageRootEntries(String TheDirName, Vector Params) {
    File TheDir = new File (TheDirName);
    int i;

    String [] PackItems = TheDir.list (new PackageEntry());
    String [] SubDirs  = TheDir.list (new Directory());
    if (PackItems != null)
      for (i = 0; i < PackItems.length; i ++)
        Params.addElement (TheDir.getAbsolutePath() + Globals.FileSeparator + PackItems[i]);
    if (SubDirs != null)
      for (i = 0; i < SubDirs.length; i ++)
        getPackageRootEntries (TheDir.getAbsolutePath() + Globals.FileSeparator + SubDirs[i], Params);
  }

  private void getFilesFromProject (Vector Params) {
    int i;
    synchronized (Globals.PItems) {
      int Count = Globals.PItems.getCount();
      for (i = 0; i < Count; i++) {
        PacketItemI CurrentItem = Globals.PItems.getPacketItem (i);
        if ((CurrentItem == null) || (CurrentItem.isStable()))
          continue;

        switch (Globals.PItems.getPacketItemKind(i)) {
          case BrowseCollItem.ikFile:
          case BrowseCollItem.ikDirectoryRoot:
            Params.addElement (CurrentItem.getFullDosPath());
          break;
          case BrowseCollItem.ikPackagesRoot:
            getPackageRootEntries(CurrentItem.getFullDosPath(), Params);
          break;
          case BrowseCollItem.ikPackage:
          case BrowseCollItem.ikDirectory:
            File TheDir = new File (CurrentItem.getFullDosPath());
            String [] DirItems;
            if (BrowseCollItem.ikPackage == CurrentItem.getPacketItemKind())
              DirItems = TheDir.list (new PackageEntry());
            else
              DirItems = TheDir.list (new NotDirectory());
            int j;
            for (j = 0; j < DirItems.length; j++)
              Params.addElement (TheDir.getAbsolutePath() + Globals.FileSeparator + DirItems[j]);
          break;
        }
      }
    }
  }

  private String [] getJARArgs() {
    String FirstArg = "-cf";
    Vector Params = new Vector();
    if (Manifest.isSelected ())
      FirstArg+= "m";
    if (NoManifest.isSelected ())
      FirstArg+= "M";
    if (Verbose.isSelected ())
      FirstArg+= "v";
    if (StoreOnly.isSelected ())
      FirstArg+= "0";

    Params.addElement (FirstArg);
    if (Manifest.isSelected ())
      Params.addElement (ManifestName.getText());

    Params.addElement (FileName.getText());

    getFilesFromProject (Params);

    String [] Ret = new String [Params.size()];
    Params.copyInto(Ret);
    return Ret;
  }

  public void actionPerformed (ActionEvent e) {
    String CMD = e.getActionCommand ();
    if (CMD.compareTo (OK_CMD) == 0) {
      if ((FileName.getText() == null) || (FileName.getText().length() == 0)) {
        Misc.messageBox (new String[] {"OK"}, new String [] {"Please choose the archive file name."}, 0);
        return;
      }
      if (Manifest.isSelected() && ((FileName.getText() == null) || (FileName.getText().length() == 0))) {
        Misc.messageBox (new String[] {"OK"}, new String [] {"Please choose the manifest file name."}, 0);
        return;
      }
      String [] Args = getJARArgs();
      sun.tools.jar.Main Archivator = new sun.tools.jar.Main (System.out, System.out, "");
      setCursor (new Cursor (Cursor.WAIT_CURSOR));
      if (Archivator.run (Args))
        Misc.messageBox (new String [] {"OK"}, new String [] {"Project successfully stored in", FileName.getText()}, 0);
      else
        Misc.messageBox (new String [] {"OK"}, new String [] {"Failed storing the project in", FileName.getText()}, 0);
      setVisible (false);
      dispose();
    }

    if (CMD.compareTo (CANCEL_CMD) == 0) {
      setVisible (false);
      dispose();
    }

    if (CMD.compareTo (CHOOSE_FILE_CMD) == 0)
      chooseFile (FileName);

    if (CMD.compareTo (CHOOSE_MANIFEST_CMD) == 0)
      chooseFile (ManifestName);
  }
}

/**
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class MySM extends java.lang.SecurityManager {
  private boolean canExit = false;
  public MySM () {}
  public void canExit (boolean b) {canExit = b;}
  public void checkExit(int i) {
    if (!canExit)
      throw new java.lang.SecurityException ();
  }
  public void checkCreateClassLoader() { }
  public void checkAccess(Thread g) { }
  public void checkAccess(ThreadGroup g) { }
  public void checkExec(String cmd) { }
  public void checkLink(String lib) { }
  public void checkRead(FileDescriptor fd) { }
  public void checkRead(String file) { }
  public void checkRead(String file, Object context) { }
  public void checkWrite(FileDescriptor fd) { }
  public void checkWrite(String file) { }
  public void checkDelete(String file) { }
  public void checkConnect(String host, int port) { }
  public void checkConnect(String host, int port, Object context) { }
  public void checkListen(int port) { }
  public void checkAccept(String host, int port) { }
  public void checkMulticast(InetAddress maddr) { }
  public void checkMulticast(InetAddress maddr, byte ttl) { }
  public void checkPropertiesAccess() { }
  public void checkPropertyAccess(String key) { }
  public void checkPropertyAccess(String key, String def) { }
  public boolean checkTopLevelWindow(Object window) { return true; }
  public void checkPrintJobAccess() { }
  public void checkSystemClipboardAccess() { }
  public void checkAwtEventQueueAccess() { }
  public void checkPackageAccess(String pkg) { }
  public void checkPackageDefinition(String pkg) { }
  public void checkSetFactory() { }
  public void checkMemberAccess(Class clazz, int which) { }
  public void checkSecurityAccess (String provider) { }
}

/**
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
public class MainPanel extends BPanel implements DebuggerCallback, OptionsSetable, CheckEnDisI {
  private boolean OutputVisible;
  private boolean DebuggerVisible;
  private Editor Editor          = new Editor();
  private Output Output          = null;
  private Frame ParentFrame      = null;
  private MySplitPane MainSplit  = null;
  private MySplitPane SubSplit   = null;
  private int OldLoc             = 0;
  private int MainOldLoc         = 0;
  private JPanel RightPanel      = new JPanel();

  private JPanel ProjLeftPanel       = new JPanel();
  private JPanel FormLeftPanel       = new JPanel();
  private JPanel BeanLeftPanel       = new JPanel();
  private JPanel DebugLeftPanel      = new JPanel();

  private JPanel MainPanel       = new JPanel();
  private JTabbedPane MainTabs   = new JTabbedPane();
  private ProjectPanel ProjPanel = new ProjectPanel();
  private Finder TheFinder       = new Finder();

  public MainPanel (Frame ParentFrame) {
    super();
    this.ParentFrame = ParentFrame;
    init();
    initEvents();
  }

  private void initEvents () {
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgCanCloseID, SystemEventAdapter.BEFORE_CALL_MASK, this, "canCloseIDE");

    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanUndo,      this, "canUndo");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanRedo,      this, "canRedo");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanCut,       this, "canCut");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanCopy,      this, "canCopy");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanPaste,     this, "canPaste");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanDelete,    this, "canDelete");

    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanFind,      this, "canFind");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanReplace,   this, "canReplace");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanSelectAll, this, "canSelectAll");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanSearchAgain,    this, "canSearchAgain");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanGoToLineNumber, this, "canGotoLineNumber");

    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiEditCut,              this, "doCut");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiEditUndo,             this, "doUndo");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiEditRedo,             this, "doRedo");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiEditCopy,             this, "doCopy");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiSearchFind,           this, "doFind");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiEditPaste,            this, "doPaste");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiEditDelete,           this, "doDelete");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiSearchBrowse,         this, "doBrowse");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiSearchReplace,        this, "doReplace");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiEditSelectAll,        this, "doSelectAll");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiSearchFindAgain,      this, "doFindAgain");
    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiSearchGoToLineNumber, this, "doGoToLineNumber");

    Globals.IEngine.AddMainMenuAdapter (IdeMainMenu.msgmiProjectStore,         this, "JARProject");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgActiveFileSet, this, "SetMeActive");
  }

  public void JARProject (IdeSuperEvent AEvent) {
    if (!isThereSomtingToProcess ())
      return;
    AddToJarDialog Archivator = new AddToJarDialog();
    Archivator.setVisible (true);
  }

  public void getCurrentOptions (IdeSuperEvent AEvent) {
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    MainPanelOptions TheOptions = (MainPanelOptions) (AOptions.getOptions(MainPanelOptions.MainPanelOptionsName));
    if (TheOptions != null) {
      if (DebuggerVisible != TheOptions.isDebuggerVisible())
        softTriggerDebugger();
      if (OutputVisible   != TheOptions.isOutputVisible())
        softTriggerOutput();
      MainSplit.setDividerLocation (TheOptions.getVSpliterLoc());
      SubSplit .setDividerLocation (TheOptions.getHSpliterLoc());

      MainOldLoc = TheOptions.getVSpliterLoc();
      OldLoc     = TheOptions.getHSpliterLoc();
    }
  }

  public void setNewOptions () {
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    MainPanelOptions TheOptions = (MainPanelOptions) (AOptions.getOptions(MainPanelOptions.MainPanelOptionsName));
    TheOptions.setVSpliterLoc    (MainSplit.getDividerLocation());
    TheOptions.setHSpliterLoc    (SubSplit .getDividerLocation());
    TheOptions.setDebuggerVisible (DebuggerVisible);
    TheOptions.setOutputVisible   (OutputVisible);
    AOptions.setOptions(TheOptions);
  }

  public void triggerOutput () {
    softTriggerOutput();
    setNewOptions();
  }

  public void softTriggerOutput () {
    if (RightPanel.getComponent(0).equals(SubSplit)) {
      OldLoc = SubSplit.getDividerLocation();
      RightPanel.remove            (SubSplit);
      RightPanel.add               (Editor);
    } else {
      RightPanel.remove            (Editor);
      RightPanel.add               (SubSplit);
      SubSplit .setTopComponent    (Editor);
      SubSplit .setDividerLocation (OldLoc);
    }
    RightPanel.invalidate();
    RightPanel.validate();
    OutputVisible = !OutputVisible;
  }

  public void triggerDebugger () {
    softTriggerDebugger();
    setNewOptions();
  }

  public void softTriggerDebugger () {
    if (MainPanel.getComponent(0).equals(MainSplit)) {
      MainOldLoc = MainSplit.getDividerLocation();
      MainPanel.remove            (MainSplit);
      MainPanel.add               (RightPanel);
    } else {
      MainPanel.remove            (RightPanel);
      MainPanel.add               (MainSplit);
      MainSplit.setRightComponent (RightPanel);
      MainSplit.setDividerLocation(MainOldLoc);
    }
    MainPanel.invalidate();
    MainPanel.validate();
    DebuggerVisible = !DebuggerVisible;
  }

  public void init() {
    
    Output = new Output();
    Output.run();
    SubSplit  = new MySplitPane (MySplitPane.VERTICAL_SPLIT, this);
    MainSplit = new MySplitPane (MySplitPane.HORIZONTAL_SPLIT, this);
    SubSplit .setUI (new MySplitPaneUI(this));
    MainSplit.setUI (new MySplitPaneUI(this));
    SubSplit .setDividerSize (3);
    MainSplit.setDividerSize (3);

    SubSplit .setTopComponent    (Editor);
    SubSplit .setBottomComponent (Output);

    RightPanel.setLayout (new BorderLayout());
    RightPanel.add       (SubSplit, BorderLayout.CENTER);

    MainSplit.setRightComponent  (RightPanel);
    MainSplit.setLeftComponent   (MainTabs);

    ProjLeftPanel.setLayout  (new BorderLayout());
    FormLeftPanel.setLayout  (new BorderLayout());
    BeanLeftPanel.setLayout  (new BorderLayout());
    DebugLeftPanel.setLayout (new BorderLayout());

    MainTabs.addTab ("Project",       ProjLeftPanel);
    MainTabs.addTab ("Forms",         FormLeftPanel);
    MainTabs.addTab ("Beans",         BeanLeftPanel);
    MainTabs.addTab ("Debugger",      DebugLeftPanel);

    ProjLeftPanel.add (ProjPanel, BorderLayout.CENTER);

    MainPanel.setLayout  (new BorderLayout());
    MainPanel.add        (MainSplit, BorderLayout.CENTER);

    setLayout (new BorderLayout());
    add       (MainPanel);
    DebuggerVisible = true;
    OutputVisible   = true;
    OptionsEventAdapter OptionsAdapter = null;
    try {
      OptionsAdapter = new OptionsEventAdapter(MainPanelOptions.MainPanelOptionsName, OptionsEvent.msgOptionsRefresh,
                                             IdeEventListenerI.CALL_MASK, this, "getCurrentOptions");
    } catch (NoSuchMethodException e) {
      Debug.Assert (e, this, "MainPanel : No such Method.");
    }
    Globals.EventsEngine.AddEventListener(OptionsAdapter);
    Globals.AddDoAfterProjectManEAdapter (ProjectManEvent.msgOpenProject,   this, "projectOpened");
    Globals.AddDoAfterProjectManEAdapter (ProjectManEvent.msgCreateProject, this, "projectOpened");
    getCurrentOptions(null);

    IdeMenuItem AMenuItem;
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiRunRun);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiRunDebug);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiBuildRebuildProject);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiBuildRebuild);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiBuildMake);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiBuildMakeProject);
    AMenuItem.setCheckStatusListener (this);

    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiProjectClose);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiProjectSave);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiProjectSaveAs);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiProjectProperties);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiProjectRename);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiProjectStore);
    AMenuItem.setCheckStatusListener (this);
    AMenuItem = (IdeMenuItem) Globals.MainMenu.FindMenu (IdeMainMenu.msgmiProjectDocumentation);
    AMenuItem.setCheckStatusListener (this);
  }

  public boolean HaveToEnableMenuItem (IdeMenuItem TheItem) {
    return Globals.PManager.isOpenedProject();
  }

  public void projectOpened (IdeSuperEvent e) {
    if (MainTabs.getSelectedIndex () != 0)
      MainTabs.setSelectedIndex (0);
  }

  private boolean isToCompile (PacketItemI PISourceFile) {
    PathsOptions POpts = (PathsOptions) Globals.Options.getOptions (PathsOptions.PathsOptionsName);
    String ClassOutPath = POpts.getClassOutPath();
    if (ClassOutPath.length() == 0)
      ClassOutPath = Globals.HomeDirectory;
    else
      ClassOutPath += Globals.FileSeparator;

    String ClassFileName = PISourceFile.getRelativeDosPath();
    if ((ClassFileName == null) || (ClassFileName.length() ==0))
      return true;
    ClassFileName = ClassFileName.substring (0, ClassFileName.lastIndexOf('.'));
    ClassFileName = ClassOutPath + ClassFileName + Globals.ClassFileExt;
    File ClassFile  = new File (ClassFileName);
    File SourceFile = new File (PISourceFile.getFullDosPath());
    if (!ClassFile.exists())
      return true;

    EditFile SourceEditied = (EditFile) Globals.TextFiles.getTextEditor (PISourceFile.getFullDosPath());
    if (SourceEditied != null)
      while (!SourceEditied.saveIt())
        if (Misc.messageBox(new String[] {"Retry", "Cancel"}, new String [] {"Unable to save file ",
                                                                              PISourceFile.getFullDosPath()}, 1) != 0)
          return false;

    if (ClassFile.lastModified() <= SourceFile.lastModified())
      return true;
    return false;
  }

  private Vector getSourceFilesToCompile () {
    Vector Ret = getAllSourcePItemsToCompile ();
    int i;
    int Count = Ret.size();
    for (i = 0; i < Count; i++) {
      PacketItemI CurrentItem = (PacketItemI)Ret.elementAt(i);
      if (isToCompile (CurrentItem))
        Ret.setElementAt (CurrentItem.getFullDosPath(), i);
      else {
        Ret.removeElement (CurrentItem);
        i--;Count--;
      }
    }
    return Ret;
  }

  private Vector getAllSourceFilesToCompile () {
    Vector Ret = getAllSourcePItemsToCompile ();
    int i;
    int Count = Ret.size();
    for (i = 0; i < Count; i++)
      Ret.setElementAt (((PacketItemI) Ret.elementAt(i)).getFullDosPath(), i);
    return Ret;
  }

  private Vector getAllSourcePItemsToCompile () {
    int i;
    int Count = Globals.PItems.getCount();
    Vector Ret = new Vector();
    for (i = 0; i < Count; i++)
      extractSourceFilesNames (Globals.PItems.getPacketItemKind(i), Globals.PItems.getPacketItem(i), Ret);
    return Ret;
  }

  private void extractSorceFilesNames (PacketItemI TheItem, Vector Storage) {
    int i;
    int Count = TheItem.getPacketItemsCount();
    for (i = 0; i < Count; i++) {
      PacketItemI CurrentItem = TheItem.getPacketItemAt(i);
      if ((CurrentItem != null) && (CurrentItem.getPacketItemKind() == PacketItemI.cpikSource)
                                && (!CurrentItem.isStable()))
        Storage.addElement (CurrentItem);
    }
  }

  private void extractSorceFilesNamesR (PacketItemI TheItem, Vector Storage) {
    int i;
    int Count = TheItem.getPacketItemsCount();
    for (i = 0; i < Count; i++) {
      PacketItemI CurrentItem = TheItem.getPacketItemAt(i);
      if (CurrentItem != null) {
        if (CurrentItem.getPacketItemKind() == PacketItemI.cpikSource)
          Storage.addElement (CurrentItem);
        if (CurrentItem.getPacketItemKind() == PacketItemI.cpikPacket)
          extractSorceFilesNamesR (CurrentItem, Storage);
      }
    }
  }

  private void extractSourceFilesNames (int ItemKind, PacketItemI TheItem, Vector Storage) {
    if ((TheItem == null) || (TheItem.isStable()))
      return;

    if (TheItem.getPacketItemKind() == PacketItemI.cpikSource)     {
      Storage.addElement (TheItem);
      return;
    }

    if (TheItem.getPacketItemKind() == PacketItemI.cpikPacket) {
      if (ItemKind == BrowseCollItem.ikPackage)
        extractSorceFilesNames (TheItem, Storage);
      if (ItemKind == BrowseCollItem.ikPackagesRoot)
        extractSorceFilesNamesR (TheItem, Storage);
    }
  }

  public void javaCompile (String [] Files) {
    if (Files.length == 0)
      return;

    int i;
    String [] Options;
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    CompilerOptions      JavaCompOptions =  (CompilerOptions) (AOptions.getOptions(CompilerOptions.CompilerManOptionsName));

    Options = JavaCompOptions.getOptions();
    String [] Params = new String [Options.length + Files.length];

    for (i = 0; i < Options.length; i++)
      Params[i] = Options[i];
    for (i = 0; i < Files.length ; i++)
      Params[i+Options.length] = Files[i];
    sun.tools.javac.Main JavaCompiler = new sun.tools.javac.Main(System.out, "Javac");
    if (JavaCompiler.compile (Params))
      System.out.println ("Compilation successfull.");
    else
      System.out.println ("Compilation failed.");
  }

  public void pizzaCompile (String [] Files) {
    if (Files.length == 0)
      return;
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    PizzaCompilerOptions PizzaCompOptions = (PizzaCompilerOptions) (AOptions.getOptions(PizzaCompilerOptions.PizzaCompilerOptionsName));
    pizza.compiler.Main PizzaCompiler = new pizza.compiler.Main();
    PizzaCompOptions.initCompiler (PizzaCompiler);
    PizzaCompiler.compile (Files);
  }

  private boolean isThereActiveFile () {
    if (Globals.TextFiles.getActiveFile() == null) {
      Misc.messageBox (new String[] {"OK"}, new String [] {"No active file to compile!"}, 0);
      return false;
    }
    return true;
  }

  public void doCompile (Vector Files) {
    int i;
    int Count = Files.size();
    System.out.println ("About to compile " + Files.size() + " file(s) :");
    for (i = 0; i < Count; i++)
       System.out.println (Files.elementAt(i));

    for (i = 0; i < Count; i++) {
      String CurrentFile = (String) Files.elementAt(i);
      EditFile CurrentSource = (EditFile) Globals.TextFiles.getTextEditor (CurrentFile);
      if (CurrentSource ==  null)
        continue;

      while (!CurrentSource.saveIt()) {
        int DoWhat = Misc.messageBox(new String[] {"Retry", "Cancel"},
                                     new String [] {"Unable to save file ",
                                                     CurrentFile},
                                     1);
        if (DoWhat != 0) {
          Misc.messageBox (new String [] {"OK"}, new String [] {"File "+CurrentFile, "is excluded from compilation."}, 0);
          Files.removeElement (CurrentFile);
        }
      }
    }

    OptionsMngr AOptions = Globals.IEngine.getOptions();
    ProjectOptions ProjOptions = (ProjectOptions) (AOptions.getOptions(ProjectOptions.cDefaultName));

    String [] SFiles = new String [Files.size()];
    Files.copyInto(SFiles);
    if (ProjOptions.isUsePizzaCompiler())
      pizzaCompile (SFiles);
    else
      javaCompile (SFiles);
  }

  public void compile (boolean Rebuild) {
    if (!isThereActiveFile ())
      return;
    String FileName = Globals.TextFiles.getActiveFile();
    if (!Rebuild) {
      PacketItemI TheItem = Globals.RootPaths.findPacketItemI (FileName);
      if (TheItem == null) {
        Misc.messageBox (new String[] {"OK"}, new String [] {"File " + FileName, "unreachable from system paths!"}, 0);
        return;
      }
      if (!isToCompile (TheItem)) {
        Misc.messageBox (new String[] {"OK"}, new String [] {"File " + FileName, "needs no compilation."}, 0);
        return;
      }
    }
    Vector TheFile = new Vector();
    TheFile.addElement (FileName);
    doCompile(TheFile);
  }

  public void compileAll (boolean Rebuild) {
    Vector Sources;
    if (Rebuild) {
      Sources = getAllSourceFilesToCompile();
    } else {
	Sources = getSourceFilesToCompile();
    }

    int FileCount = Sources.size();
    int i;
    System.out.println ("Compiling...  Please Wait...");
    doCompile(Sources);
    System.out.println ("Compilation Ended.");
  }

  public void addSubPackages (PacketItemI RootPackage, Vector Data) {
    Data.addElement (RootPackage.getDotName());
    int Count = RootPackage.getPacketItemsCount();
    for (int i = 0; i < Count; i++) {
      PacketItemI CurrentItem = RootPackage.getPacketItemAt(i);
      if (CurrentItem.getPacketItemKind() == PacketItemI.cpikPacket)
        addSubPackages (CurrentItem, Data);
    }
  }

  public String [] getProjectItemsForJavDOC(String[] JavaDocOpts) {
    Vector Ret = new Vector();
    int i;
    int PItemsCount = Globals.PItems.getCount();

    for (i = 0; i < JavaDocOpts.length; i++)
      Ret.addElement (JavaDocOpts[i]);

    for (i = 0; i < PItemsCount; i++)
      switch (Globals.PItems.getPacketItemKind(i)) {
        case BrowseCollItem.ikFile:
          if (Globals.PItems.getPacketItem(i).getPacketItemKind() == PacketItemI.cpikSource)
            Ret.addElement (Globals.PItems.getPacketItem(i).getFullDosPath());
        break;
        case BrowseCollItem.ikPackage:
          if (Globals.PItems.getPacketItem(i) == null)
            Misc.messageBox (new String [] {"OK"}, new String [] {"Package " + Globals.PItems.getString(i),
                                                                  "is unreachable from system paths.",
                                                                  "Documentation for " + Globals.PItems.getString(i) + " will NOT be generated!"}, 0);
          else
            Ret.addElement (Globals.PItems.getPacketItem(i).getDotName());
        break;
        case BrowseCollItem.ikPackagesRoot :
          if (Globals.PItems.getPacketItem(i) == null)
            Misc.messageBox (new String [] {"OK"}, new String [] {"Packages root " + Globals.PItems.getString(i),
                                                                  "is unreachable from system paths.",
                                                                  "Documentation for " + Globals.PItems.getString(i) + " will NOT be generated!"}, 0);
          else
            addSubPackages (Globals.PItems.getPacketItem(i), Ret);
        break;
      }
    String [] RetArgs = new String [Ret.size()];
    Ret.copyInto (RetArgs);
    return RetArgs;
  }

  private boolean isThereSomtingToProcess () {
    int i;
    int Count = Globals.PItems.getCount();
    for (i = 0; i < Count; i++)
      if (Globals.PItems.getPacketItem(i) != null)
        break;

    if (i == Count) {
      Misc.messageBox (new String [] {"OK"}, new String [] {"No project entries to process."}, 0);
      return false;
    }
    return true;
  }

  public synchronized void genDoc () {
    PathsOptions POpts = (PathsOptions) Globals.Options.getOptions (PathsOptions.PathsOptionsName);
    String JDOCOut = POpts.getJavaDOCOutPath();

    if (!isThereSomtingToProcess ())
      return;

    if ((JDOCOut == null) || (JDOCOut.length() == 0)) {
      Misc.messageBox (new String [] {"OK"}, new String [] {"Please supply a value for \"JavaDOC Output Path\" project property!"}, 0);
      return;
    }

    String [] JavaDocArgs  = getProjectItemsForJavDOC (((JavaDOCOptions) Globals.Options.getOptions (JavaDOCOptions.JavaDOCOptionsName)).getOptions());
    sun.tools.javadoc.Main Generator = new sun.tools.javadoc.Main();
    System.gc();
    try {
      if (System.getSecurityManager() == null)
        System.setSecurityManager (new MySM());
      ((MySM)System.getSecurityManager()).canExit(false);
      Generator.main (JavaDocArgs);
    } catch (Exception e) {
      ((MySM)System.getSecurityManager()).canExit(true);
    }

  }

  public synchronized void runProgram () {
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    ProjectOptions POpts = (ProjectOptions) AOptions.getOptions (ProjectOptions.cDefaultName);
    if (POpts.getRunApplication() == 0)
      runApplet();
    if (POpts.getRunApplication() == 1)
      runApplication();
  }

  public void runApplet () {
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    String RunnableHTML  = ((ProjectOptions) (AOptions.getOptions(ProjectOptions.cDefaultName))).getRunnableHTML();
    if ((RunnableHTML == null) || (RunnableHTML.length() == 0)) {
      Misc.messageBox (new String [] {"OK"}, new String [] {"Please Enter a HTML File Name in the Project Properties."}, 0);
      return;
    }
    VMOptions TheOptions = (VMOptions) (AOptions.getOptions(VMOptions.VMOptionsName));
    String [] Options    = TheOptions.getOptions();
    String [] AllOptions = new String [Options.length + 1];
    int i;
    for (i = 0; i < Options.length; i++)
      AllOptions[i] = Options[i];

    AllOptions[AllOptions.length - 3] = "sun.applet.AppletViewer";
    AllOptions[AllOptions.length - 2] = RunnableHTML;
    AllOptions[AllOptions.length - 1] = Options[Options.length-1];

    try {
      System.out.println ("Starting " + RunnableHTML);
      Runner TheRunner = new Runner (AllOptions);
      TheRunner.start();
      System.out.println (RunnableHTML + " started.");
    } catch (Exception e) {
      System.out.println ("Unable to execute " + RunnableHTML);
    }
  }

  public void runApplication () {
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    String RunnableClass = ((ProjectOptions) (AOptions.getOptions(ProjectOptions.cDefaultName))).getRunnable();

    if ((RunnableClass == null) || (RunnableClass.length() == 0)) {
      Misc.messageBox (new String [] {"OK"}, new String [] {"Please Enter a Runnable Class Name in the Project Properties."}, 0);
      return;
    }

    VMOptions TheOptions = (VMOptions) (AOptions.getOptions(VMOptions.VMOptionsName));
    String [] Options    = TheOptions.getOptions();
    String [] Params     = Misc.getParamsArray(TheOptions.getParams());
    String [] AllOptions = new String [Options.length + Params.length + 1];
    int i,j;
    for (i = 0; i < Options.length; i++)
      AllOptions[i] = Options[i];
    AllOptions[i] = RunnableClass;
    i++;
    for (j = 0; i < AllOptions.length; i++,j++)
      AllOptions[i] = Params[j];

    try {
      System.out.println ("Starting " + RunnableClass);
      Runner TheRunner = new Runner (AllOptions);
      TheRunner.start();
      System.out.println (RunnableClass + " started.");
    } catch (Exception e) {
      System.out.println ("Unable to execute " + RunnableClass);
    }
  }

  public synchronized void debugProgram () {
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    ProjectOptions POpts = (ProjectOptions) AOptions.getOptions (ProjectOptions.cDefaultName);
    if (POpts.getRunApplication() == 0)
      debugApplet();
    if (POpts.getRunApplication() == 1)
      debugApplication();
  }

  public void debugApplet () {
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    VMOptions TheOptions = (VMOptions) (AOptions.getOptions(VMOptions.VMOptionsName));
    String RunnableHTML  = ((ProjectOptions) (AOptions.getOptions(ProjectOptions.cDefaultName))).getRunnableHTML();
    if ((RunnableHTML == null) || (RunnableHTML.length() == 0)) {
      Misc.messageBox (new String [] {"OK"}, new String [] {"Please Enter a HTML File Name in the Project Properties."}, 0);
      return;
    }

    try {
      RemoteDebugger RD = new RemoteDebugger (TheOptions.getDebugOptions(), this, true);
      System.out.println ("Starting " + RunnableHTML);
      String[] Params = Misc.getParamsArray(TheOptions.getParams());
      String[] DebuggerParams = new String [Params.length + 2];
      for (int i = 0; i < Params.length; i++)
         DebuggerParams[i+2] = Params[i];
      DebuggerParams[0] = "sun.applet.AppletViewer";
      DebuggerParams[1] = RunnableHTML;
      DebugRunner TheRunner = new DebugRunner (RD, DebuggerParams);
      TheRunner.start();
      System.out.println (RunnableHTML + " started.");
    } catch (Exception e) {
      System.out.println ("Unable to execute " + RunnableHTML);
    }
  }

  public void debugApplication () {
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    VMOptions TheOptions = (VMOptions) (AOptions.getOptions(VMOptions.VMOptionsName));
    String RunnableClass = ((ProjectOptions) (AOptions.getOptions(ProjectOptions.cDefaultName))).getRunnable();
    if ((RunnableClass == null) || (RunnableClass.length() == 0)) {
      Misc.messageBox (new String [] {"OK"}, new String [] {"Please Enter a Runnable Class Name in the Project Properties."}, 0);
      return;
    }

    try {
      RemoteDebugger RD = new RemoteDebugger (TheOptions.getDebugOptions(), this, true);
      System.out.println ("Starting " + RunnableClass);
      String[] Params = Misc.getParamsArray(TheOptions.getParams());
      String[] DebuggerParams = new String [Params.length + 1];
      for (int i = 0; i < Params.length; i++)
         DebuggerParams[i+1] = Params[i];
      DebuggerParams[0] = RunnableClass;
      DebugRunner TheRunner = new DebugRunner (RD, DebuggerParams);
      TheRunner.start();
      System.out.println (RunnableClass + " started.");
    } catch (Exception e) {
      System.out.println ("Unable to execute " + RunnableClass);
    }
  }

  public void quitEvent () {
  }
  public void threadDeathEvent (RemoteThread T) {
  }
  public void exceptionEvent (RemoteThread T, String S) {
  }
  public void breakpointEvent (RemoteThread T) {
  }
  public void printToConsole (String S) {
    System.out.println (S);
  }

  public void setResult (IdeSuperEvent AEvent, int Result) {
    ((SystemEvent)AEvent).Result = Result;
  }

  public void canUndo (IdeSuperEvent AEvent) {
    setResult(AEvent, 0);
  }
  public void canRedo (IdeSuperEvent AEvent) {
    setResult(AEvent, 0);
  }

  public void canCut (IdeSuperEvent AEvent) {
    if (!ifReadOnlyThenForbid (AEvent))
      return;
    canCopy (AEvent);
  }
  public void canCopy (IdeSuperEvent AEvent) {
    if (Editor.getSourceArea() == null) {
      setResult (AEvent, 0);
      return;
    }
    if  (Editor.getSourceArea().getTextArea().getSelectionStart() !=
         Editor.getSourceArea().getTextArea().getSelectionEnd())
      setResult(AEvent, 1);
    else
      setResult(AEvent, 0);
  }

  private boolean ifReadOnlyThenForbid (IdeSuperEvent AEvent) {
    EditFile TheEditFile = (EditFile)Globals.TextFiles.getActiveTextFile();
    if ((TheEditFile == null) || (TheEditFile.isReadOnly())) {
      setResult (AEvent, 0);
      return false;
    }
    else {
      setResult (AEvent, 1);
      return true;
    }
  }

  public void canPaste (IdeSuperEvent AEvent) {
    if (!ifReadOnlyThenForbid (AEvent))
      return;
    if (getToolkit().getSystemClipboard().getContents(this) != null)
      setResult(AEvent, 1);
    else
      setResult(AEvent, 0);
  }

  public void canDelete (IdeSuperEvent AEvent) {
    canCut (AEvent);
  }

  public void canSelectAll (IdeSuperEvent AEvent) {
    if (Editor.getSourceArea() == null) {
      setResult (AEvent, 0);
      return;
    }
    int TextLen      = Editor.getSourceArea().getTextArea().getText().length();
    String Selection = Editor.getSourceArea().getTextArea().getSelectedText();
    int SelectionLen;
    if (Selection == null)
      SelectionLen = 0;
    else
      SelectionLen = Selection.length();

    if ((Editor.getSourceArea().getTextArea().getText() != null) &&
        (TextLen != 0) &&
        (TextLen != SelectionLen))
      setResult(AEvent, 1);
    else
      setResult(AEvent, 0);
  }

  public void canFind (IdeSuperEvent AEvent) {
    if (Editor.getSourceArea() == null) {
      setResult (AEvent, 0);
      return;
    }
    int TextLen      = Editor.getSourceArea().getTextArea().getText().length();
    if ((Editor.getSourceArea().getTextArea().getText() != null) &&
        (TextLen != 0))
      setResult(AEvent, 1);
    else
      setResult(AEvent, 0);
  }

  public void canReplace (IdeSuperEvent AEvent) {
    if (!ifReadOnlyThenForbid (AEvent))
      return;
    canFind (AEvent);
  }

  public void canSearchAgain (IdeSuperEvent AEvent) {
    canFind (AEvent);
  }

  public void canGotoLineNumber (IdeSuperEvent AEvent) {
    canFind (AEvent);
  }

  public void canCloseIDE (IdeSuperEvent AEvent) {
    MainMenuEvent ClosePrjEvent = new MainMenuEvent (IdeMainMenu.msgmiFileCloseAll, this);
    Globals.EventsEngine.SendEvent (ClosePrjEvent);
    ((SystemEvent) AEvent).Result = ClosePrjEvent.Result;
  }

  public void doCut (IdeSuperEvent AEvent) {
    Editor.getSourceArea().getTextArea().cut();
  }
  public void doUndo (IdeSuperEvent AEvent) {
  }

  public void doRedo (IdeSuperEvent AEvent) {
  }
  public void doCopy (IdeSuperEvent AEvent) {
    Editor.getSourceArea().getTextArea().copy();
  }

  public void doFind (IdeSuperEvent AEvent) {
    TheFinder.find();
  }

  public void doPaste (IdeSuperEvent AEvent) {
    Editor.getSourceArea().getTextArea().paste();
  }

  public void doDelete (IdeSuperEvent AEvent) {
    JTextArea TheText = Editor.getSourceArea().getTextArea();
    TheText.replaceSelection ("");
  }

  public void doBrowse (IdeSuperEvent AEvent) {
    Misc.messageBox (new String[] {"OK"}, new String [] {"Sorry, feature not implemented yet."}, 0);
  }

  public void doReplace (IdeSuperEvent AEvent) {
    TheFinder.replace();
  }

  public void doSelectAll (IdeSuperEvent AEvent) {
    Editor.getSourceArea().getTextArea().selectAll();
  }

  public void doFindAgain (IdeSuperEvent AEvent) {
    TheFinder.findAgain();
  }
  public void doGoToLineNumber (IdeSuperEvent AEvent) {
    GotoLineNumberDialog TheDialog = new GotoLineNumberDialog();
    TheDialog.setVisible (true);
    if (TheDialog.isConfirmed()) {
      JTextArea Source = Editor.getSourceArea().getTextArea();
      try {
        Source.setCaretPosition (Source.getLineStartOffset (TheDialog.getLineNumber()));
        Source.requestFocus();
      }catch (Exception e) {
        Misc.messageBox (new String [] {"OK"}, new String [] {"No such line"}, 0);
      }
    }
  }

  public void SetMeActive (IdeSuperEvent AEvent) {
    EditFile TheFile = (EditFile) Globals.TextFiles.getActiveTextFile();
    Editor.setActiveEditFile (TheFile);
  }

  public String getSelectedProjectItemName () {
    return ProjPanel.getSelectedProjectItemName ();
  }

  public int getSelectedProjectItemKind () {
    return ProjPanel.getSelectedProjectItemKind ();
  }

  public void focusItem (String FullDosPath) {
    ProjPanel.focusItem (FullDosPath);
  }

  public void focusItem (PacketItemI PItem) {
    ProjPanel.focusItem (PItem.getFullDosPath());
  }

  public void clearOutput() {
    this.Output.clear();
  }
}
