/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.system.ideengine;

import java.io.*;

import java.awt.*;
import java.awt.image.*;
import java.lang.*;
import java.util.EventObject;

import org.freebuilder.system.projectman.*;
import org.freebuilder.system.debug.*;
import org.freebuilder.system.dpaths.*;
import org.freebuilder.system.classes.*;
import org.freebuilder.system.packages.*;
import org.freebuilder.system.classes.events.*;
import org.freebuilder.system.classes.options.*;
import org.freebuilder.system.classes.events.engine.*;
import org.freebuilder.system.classes.options.manager.*;

import org.freebuilder.gui.xysaveframe.XyFrameOptions;
import org.freebuilder.gui.IdeTitleFrame;
import org.freebuilder.gui.idemenu.*;
import org.freebuilder.gui.config.*;
import org.freebuilder.gui.beans.*;
import org.freebuilder.gui.ideoptions.*;
import org.freebuilder.gui.editors.*;
import org.freebuilder.wizards.*;


import org.freebuilder.gui.ide.*;

/** The class <code>IdeEngine</code> Every listener which wants to handle events
  * for options chainge of some tool have to descent from this class.
  *
  * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
  * @version 0.7, 02-12-97 */

public class IdeEngine implements CreateDOptionsListener {
  private MainPanel        MainPanel;

  public IdeEngine() throws IdeException {
    try {
      new Globals();
      Globals.ResourceMngr = new ResMan(); // Let's create Resource manager

      Globals.EventsEngine = new IdeEventsEngine();
      Globals.Options      = new OptionsMngr(Globals.cDefaultProjectFileName, this);
      Globals.MainMenu     = new IdeMainMenu();

      Globals.IEngine      = this;

      Globals.RootPaths    = new RootPathsColl();
      Globals.TextFiles    = new TextFilesManager();

      Globals.PManager     = new ProjectManager();
      Globals.PItems       = new ProjectItems();
      InitIdeEngine();

      InitTitleFrame();
      InitEventListeners();
      return;
    }

    catch(NoSuchMethodException e) {
      Debug.Assert(e, null,"Application : NoSuchMethodException");
    }
    catch(SecurityException e) {
      Debug.Assert(e, null,"Application : SecurityException");
    }
    catch(Exception e) {
      Debug.Assert(e, null,"Application : SecurityException");
    }

    throw new IdeException();
  }

  public void Done() {
    Globals.EventsEngine.EnableEvents(false);
    Globals.GlobalsRef = null;
  }

  public void WakeUp() {
    Globals.Options.EnableSaveOptions(true);
    Globals.Options.EnableEvents     (true);
    Globals.EventsEngine.EnableEvents(true);
    Globals.PManager.EnableSendEvents(true);
    Globals.MainMenu.UpdateMenu();
    PostStartUpMessage();
  }

  private void PostStartUpMessage() {
    PostSystemEvent(SystemEvent.msgOnStartUp);
  }

  private void PostSystemEvent(int EventNumber) {
    Globals.EventsEngine.PostEvent(new SystemEvent(EventNumber, this));
  }

  private void InitIdeEngine() throws NoSuchMethodException, SecurityException {
    SystemEventAdapter SystemAdapter;

    SystemAdapter = new SystemEventAdapter(SystemEvent.msgOnStartUp,   IdeEventListenerI.AFTER_CALL_MASK, this,"AfterOnStartUp");
    Globals.EventsEngine.AddEventListener(SystemAdapter);

    SystemAdapter = new SystemEventAdapter(SystemEvent.msgOpenProject, IdeEventListenerI.AFTER_CALL_MASK, this,"OnOpenProject");
    Globals.EventsEngine.AddEventListener(SystemAdapter);
  }

  //=====================================================================================
  void InitTitleFrame() {
    boolean   packFrame  = false;

    Globals.IdeMainFrame = new IdeTitleFrame(XyFrameOptions.cDefaultName);
//    Globals.Environment  = new NewEnv (Globals.IdeMainFrame);

    Globals.IdeMainFrame.setBackground (Color.lightGray);
    MainPanel = new MainPanel (Globals.IdeMainFrame);
    Globals.IdeMainFrame.getContentPane().add (MainPanel, BorderLayout.CENTER);

    if (packFrame) Globals.IdeMainFrame.pack();
    else Globals.IdeMainFrame.validate();

    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    Dimension frameSize = Globals.IdeMainFrame.getSize();
    if (frameSize.height > screenSize.height) frameSize.height = screenSize.height;
    if (frameSize.width > screenSize.width)   frameSize.width = screenSize.width;
    Globals.IdeMainFrame.setLocation((screenSize.width - frameSize.width) / 2, (screenSize.height - frameSize.height) / 2);
    if (Globals.ShowLogoAtStartUp)
      Globals.IdeMainFrame.showAboutWindow ();
  }
  //=====================================================================================

  public  void AfterOnStartUp(IdeSuperEvent AEvent) { // Will be called when receive msgOnStartUp
    String      ProjectName;
    SystemEvent Event = null;
    if (Globals.Options.ShallOpenProjectFile()) {
      ProjectName = Globals.Options.getLastProjectFileName();
      Event       = new SystemEvent(SystemEvent.msgOpenProject, this, ProjectName);
      Globals.EventsEngine.SendEvent(Event);
      if (Event.Result != 0) return; // Project is opened successfully.
    }
    Globals.Options.Refresh();
  }


  public void OnOpenProject(IdeSuperEvent AEvent) { // Will be called when receive msgOnStartUp
    SystemEvent Event           = (SystemEvent)AEvent;
    String      ProjectFileName = Event.StrValue;
    if (OpenProject(ProjectFileName))
      Event.Result = 1;
  }

  public IdeEventsEngine getEventsEngine() { return Globals.EventsEngine; }
  public OptionsMngr     getOptions     () { return Globals.Options;      }
  public IdeMainMenu     getMainMenu    () { return Globals.MainMenu;     }

  /** This method is called every time the DefaultOptions.CreateDefaultConfig() is called.
    * Add to the AOptionsColl all the objects which holds some kind of options which will
    * be saved is some configuration. */
  public void CreateDefaultConfig(OptionsColl AOptionsColl) {
    XyFrameOptions       ThisFrameOptions  = null;
    PaletteOptions       PManager          = null;
    CompilerOptions      CManager          = null;
    VMOptions            VMOpts            = null;
    MainPanelOptions     MainPanelOpts     = null;
    JavaDOCOptions       JavaDOCOpts       = null;
    ProjectOptions       LProjectOptions   = null;
    PathsOptions         LPathsOptions      = null;
    PizzaCompilerOptions PizzaCompilerOpts = null;
    EditorOptions        EditorOpts        = null;

    try {
      ThisFrameOptions = new XyFrameOptions(XyFrameOptions.cDefaultName);
      PManager         = new PaletteOptions();
      CManager         = new CompilerOptions();
      VMOpts           = new VMOptions();
      MainPanelOpts    = new MainPanelOptions();
      JavaDOCOpts      = new JavaDOCOptions();
      PizzaCompilerOpts= new PizzaCompilerOptions();
      LPathsOptions    = new PathsOptions();
      LProjectOptions  = new ProjectOptions();
      EditorOpts       = new EditorOptions();


      PManager        .setDefaults();
      CManager        .setDefaults();
      VMOpts          .setDefaults();
      MainPanelOpts   .setDefaults();
      JavaDOCOpts     .setDefaults();
      LProjectOptions .setDefaults();
      LPathsOptions   .setDefaults();
      PizzaCompilerOpts.setDefaults();
      EditorOpts      .setDefaults();

      AOptionsColl.addOptions(ThisFrameOptions);
      AOptionsColl.addOptions(PManager);
      AOptionsColl.addOptions(CManager);
      AOptionsColl.addOptions(VMOpts);
      AOptionsColl.addOptions(MainPanelOpts);
      AOptionsColl.addOptions(JavaDOCOpts);
      AOptionsColl.addOptions(LPathsOptions);
      AOptionsColl.addOptions(LProjectOptions);
      AOptionsColl.addOptions(PizzaCompilerOpts);
      AOptionsColl.addOptions(EditorOpts);
    }
    catch (DuplicateOptionNameException e) {
      if (Debug.isDebug) {
        Debug.Assert(e, this,"DuplicateOptionNameException");
      }
    }
  }

  public MainMenuEventAdapter AddMainMenuAdapter (int AEventNumber, Object AMethodRef, String AMethodName) {
    MainMenuEventAdapter MMenuAdapter = null;
    try {
      MMenuAdapter = new MainMenuEventAdapter(AEventNumber, AMethodRef, AMethodName);
      Globals.EventsEngine.AddEventListener(MMenuAdapter);
    }
    catch (Exception e) {
      if (Debug.isDebug) Debug.Assert(e,this);
    }
    return MMenuAdapter;
  }

  public SystemEventAdapter AddSystemEventAdapter(int AEventNumber, int CallMask,
                                                  Object AMethodRef, String AMethodName) {
    SystemEventAdapter SystemEAdapter = null;
    try {
      SystemEAdapter = new SystemEventAdapter(AEventNumber, CallMask, AMethodRef, AMethodName);
      Globals.EventsEngine.AddEventListener(SystemEAdapter);
    }
    catch (Exception e) {
      if (Debug.isDebug) Debug.Assert(e,this);
    }
    return SystemEAdapter;
  }

  public SystemEventAdapter AddSystemEventAdapter(int AEventNumber, Object AMethodRef, String AMethodName) {
    return AddSystemEventAdapter (AEventNumber, SystemEventAdapter.CALL_MASK, AMethodRef, AMethodName);
  }

  public void clearOutput() {
    MainPanel.clearOutput();
  }

  //=====================================================================================
  // Event handlers
  //=====================================================================================

  public void OnExit(IdeSuperEvent AEvent) {
    SystemEvent ExitQueryEvent = new SystemEvent(SystemEvent.msgCanCloseID, this);
    Globals.EventsEngine.SendEvent(ExitQueryEvent);
    if (ExitQueryEvent.Result != 0) return; // if somebody changes the result it means that Application can not be closed
    Done();
    System.exit(0);
  }

  public void OnShowIDEOptions(IdeSuperEvent AEvent) {
    IDEConfigDialog IDEConfigDlg = new IDEConfigDialog();
    IDEConfigDlg.setVisible(true);
  }

  public void OnViewOutput (IdeSuperEvent AEvent) {
    MainPanel.triggerOutput();
  }

  public void OnViewDebugger (IdeSuperEvent AEvent) {
    MainPanel.triggerDebugger();
  }

  public void OnShowPaletteOptions(IdeSuperEvent AEvent) {
    PaletteConfigDialog PConfigDlg = new PaletteConfigDialog(Globals.IdeMainFrame, "Configure beans palette", true);
    PConfigDlg.setVisible(true);
  }

  public void OnFileCloseAll (IdeSuperEvent AEvent) {
    StringList ModifiedFiles = Globals.TextFiles.getModifiedFiles();
    if (ModifiedFiles.getCount() == 0) {
      Globals.TextFiles.closeAllFiles();
//      Globals.PManager.removeAllItems();
    } else {
      ConfirmCloseAllDialog Confirm = new ConfirmCloseAllDialog(ModifiedFiles);
      Confirm.setVisible (true);
      if (Confirm.isConfirmed()) {
        StringList FilesToSave = Confirm.getSelectedFiles();
        int i;
        int Len = FilesToSave.getCount();
        for (i = 0; i < Len; i++)
          checkAndSave ((TextEditorI) FilesToSave.getObject(i), AEvent);
        Globals.TextFiles.closeAllFiles();
//        Globals.PManager.removeAllItems();
      } else
        ((MainMenuEvent) AEvent).Result = -1;
    }
  }

  public void OnFileClose (IdeSuperEvent AEvent) {
    ConfirmCloseDialog ConfirmDialog;
    if (Globals.TextFiles.getActiveTextFile().isModified()) {
      ConfirmDialog = new ConfirmCloseDialog (Globals.TextFiles.getActiveFile());
      ConfirmDialog.setVisible(true);
      switch (ConfirmDialog.getResult()) {
        case ConfirmDialog.SAVE_RESULT:
          OnFileSave (AEvent);
        case ConfirmDialog.DONTSAVE_RESULT:
          Globals.TextFiles.closeActiveFile();
        break;
        case ConfirmDialog.DONTCLOSE_RESULT:
        return;
      }
    } else
      Globals.TextFiles.closeActiveFile();
  }

  private void checkAndSave (TextEditorI TheFile, IdeSuperEvent AEvent) {
    if (TheFile.exists()) {
      if (Globals.TextFiles.saveFile (TheFile))
        ((MainMenuEvent)AEvent).Result = 1;
      else
        ((MainMenuEvent)AEvent).Result = 0;
    } else
      if (!TheFile.equals(Globals.TextFiles.getActiveFile())) {
        Globals.TextFiles.setActiveFile(TheFile.getFullDosPath());
        OnFileSaveAs (AEvent);
      }
  }

  public void OnFileSave (IdeSuperEvent AEvent) {
    if (Globals.TextFiles.getActiveTextFile() == null)
      return;
    if (!Globals.TextFiles.getActiveTextFile().isModified()) {
      ((MainMenuEvent)AEvent).Result = 1;
      return;
    }
    checkAndSave (Globals.TextFiles.getActiveTextFile(), AEvent);
  }

  public void OnFileSaveAll (IdeSuperEvent AEvent) {
    StringList ModifiedFiles = Globals.TextFiles.getModifiedFiles();
    if (ModifiedFiles.getCount() != 0) {
      ConfirmCloseAllDialog Confirm = new ConfirmCloseAllDialog(ModifiedFiles);
      Confirm.setVisible (true);
      if (Confirm.isConfirmed()) {
        StringList FilesToSave = Confirm.getSelectedFiles();
        int i;
        int Len = FilesToSave.getCount();
        for (i = 0; i < Len; i++)
          checkAndSave ((TextEditorI) FilesToSave.getObject(i), AEvent);
      }
    }
  }

  public void OnFileSaveAs (IdeSuperEvent AEvent) {
    FileDialog SaveDialog = new FileDialog (Globals.IdeMainFrame, "", FileDialog.SAVE);
    SaveDialog.setVisible (true);
    if (SaveDialog.getDirectory() != null)
      if (Globals.TextFiles.saveFileAs(Globals.TextFiles.getActiveFile(), SaveDialog.getDirectory() +
                                                          SaveDialog.getFile()))
        ((MainMenuEvent)AEvent).Result = 1;
      else
        ((MainMenuEvent)AEvent).Result = 0;
  }

  public void OnFileNew (IdeSuperEvent AEvent) {
    Globals.TextFiles.newFile();
    Globals.PManager.addFile (Globals.TextFiles.getActiveFile());
  }

  private boolean isToExcludeItemFromDir (PacketItemI CurrentPItem) {
    if (CurrentPItem == null)
      return false;
    return !((CurrentPItem.getPacketItemKind() == PacketItemI.cpikSource) ||
             (CurrentPItem.getPacketItemKind() == PacketItemI.cpikClass)  ||
             (CurrentPItem.getPacketItemKind() == PacketItemI.cpikPacket));
  }

  private boolean isToBeViewed (String FileName) {
     String TheName = FileName.toLowerCase();
     return  (TheName.endsWith (Globals.JavaFileExt) ||
                  TheName.endsWith (Globals.PizzaFileExt) ||
                  TheName.endsWith (Globals.HTMLFileExt) ||
                  TheName.endsWith (Globals.ClassFileExt));
  }
  private void checkAndExcludeDir (PacketItemI TheDirRoot) {
    int i;
    int Count = Globals.PItems.getCount();
    for (i = 0; i < Count; i++) {
      PacketItemI CurrentPItem  = Globals.PItems.getPacketItem (i);
      if (!isToExcludeItemFromDir(CurrentPItem))
        continue;

      if ((TheDirRoot.MatchPath (CurrentPItem.getFullDosPath())) &&
          (Globals.PItems.getPacketItemKind(i) == BrowseCollItem.ikDirectory))
        continue;

      CurrentPItem = TheDirRoot.findPacketItem (CurrentPItem.getFullDosPath());
      if ((CurrentPItem != null) && (CurrentPItem.getPacketItemKind() != PacketItemI.cpikDirectory))
        Globals.PManager.RemoveBrowseItem (CurrentPItem.getFullDosPath(), Globals.PItems.getPacketItemKind(i));
    }
  }

  private void checkAndExcludeDirRoot (PacketItemI TheDirRoot) {
    int i;
    int Count = Globals.PItems.getCount();
    for (i = 0; i < Count; i++) {
      PacketItemI CurrentPItem  = Globals.PItems.getPacketItem (i);
      if (!isToExcludeItemFromDir(CurrentPItem))
        continue;

      if ((TheDirRoot.MatchPath (CurrentPItem.getFullDosPath())) &&
          (Globals.PItems.getPacketItemKind(i) == BrowseCollItem.ikDirectoryRoot))
        continue;

      if (TheDirRoot.findPacketItemR(CurrentPItem.getFullDosPath()) != null)
        Globals.PManager.RemoveBrowseItem (CurrentPItem.getFullDosPath(), Globals.PItems.getPacketItemKind(i));
    }
  }

  private boolean isDirectoryInProject (String ADirName) {
    int i;
    int Count = Globals.PItems.getCount();
    for (i = 0; i < Count; i++) {
      PacketItemI CurrentPItem  = Globals.PItems.getPacketItem (i);
      int CurrentPItemKind      = Globals.PItems.getPacketItemKind (i);

      if ((CurrentPItemKind == BrowseCollItem.ikDirectory) &&
          (CurrentPItem.MatchPath (ADirName)))
        return true;

      if ((CurrentPItemKind == BrowseCollItem.ikDirectoryRoot) &&
          (CurrentPItem.findPacketItemR (ADirName) != null))
        return true;
    }
    return false;
  }

  private boolean isDirRootInProject (String ADirName) {
    int i;
    int Count = Globals.PItems.getCount();
    for (i = 0; i < Count; i++) {
      PacketItemI CurrentPItem  = Globals.PItems.getPacketItem (i);
      int CurrentPItemKind      = Globals.PItems.getPacketItemKind (i);

      if ((CurrentPItemKind == BrowseCollItem.ikDirectoryRoot) &&
          (CurrentPItem.findPacketItemR (ADirName) != null))
        return true;
    }
    return false;
  }

  private void checkAndExcludePack (PacketItemI APackage) {
    int i;
    int Count = Globals.PItems.getCount();
    for (i = 0; i < Count; i++) {
      PacketItemI CurrentPItem  = Globals.PItems.getPacketItem (i);
      if (CurrentPItem == null)
        continue;
      if ((CurrentPItem.getPacketItemKind() != PacketItemI.cpikSource) &&
          (CurrentPItem.getPacketItemKind() != PacketItemI.cpikClass))
        continue;
      if (APackage.findPacketItem (CurrentPItem.getFullDosPath()) != null)
        Globals.PManager.RemoveBrowseItem (CurrentPItem.getFullDosPath(), Globals.PItems.getPacketItemKind(i));
    }
  }

  private void checkAndExcludePackRoot (PacketItemI APackage) {
    int i;
    int Count = Globals.PItems.getCount();
    for (i = 0; i < Count; i++) {
      PacketItemI CurrentPItem  = Globals.PItems.getPacketItem (i);
      if (CurrentPItem == null)
        continue;
      if ((CurrentPItem.getDotName().compareTo (APackage.getDotName()) == 0) &&
          (Globals.PItems.getPacketItemKind (i) == BrowseCollItem.ikPackagesRoot))
        continue;
      if (((CurrentPItem.getPacketItemKind() == PacketItemI.cpikSource) ||
           (CurrentPItem.getPacketItemKind() == PacketItemI.cpikClass)) &&
           (APackage.findPacketItemR (CurrentPItem.getFullDosPath()) != null))
        Globals.PManager.RemoveBrowseItem (CurrentPItem.getFullDosPath(), Globals.PItems.getPacketItemKind(i));

      if (CurrentPItem.getPacketItemKind() != PacketItemI.cpikPacket)
        continue;

      if (APackage.getPacketItemByDotName (CurrentPItem.getDotName()) != null)
        Globals.PManager.RemoveBrowseItem (CurrentPItem.getDotName(), Globals.PItems.getPacketItemKind(i));
    }
  }

  private boolean isPackageInProject (PacketItemI APackage, int PIKind) {
    int i;
    int Count = Globals.PItems.getCount();
    for (i = 0; i < Count; i++) {
      PacketItemI CurrentPItem  = Globals.PItems.getPacketItem (i);
      if (CurrentPItem == null)
        continue;

      if ((Globals.PItems.getPacketItemKind (i) == PIKind) &&
          (CurrentPItem.getDotName().compareTo (APackage.getDotName()) == 0))
        return true;

      if ((Globals.PItems.getPacketItemKind (i) == BrowseCollItem.ikPackagesRoot) &&
          (CurrentPItem.getPacketItemByDotName(APackage.getDotName()) != null))
        return true;
    }
    return false;
  }

  private boolean isFileInDirs (String TheFileName) {
    int i;
    int Count = Globals.PItems.getCount();

    for (i = 0; i < Count; i++) {
      PacketItemI CurrentPItem = Globals.PItems.getPacketItem (i);
      int Kind                 = Globals.PItems.getPacketItemKind (i);
      if (CurrentPItem == null)
        continue;
      if (Kind == BrowseCollItem.ikDirectory) {
        if (CurrentPItem.findPacketItem (TheFileName) != null)
          return true;
      }
      if (Kind == BrowseCollItem.ikDirectoryRoot) {
        if (CurrentPItem.findPacketItemR (TheFileName) != null)
          return true;
      }
    }
    return false;
  }

  private boolean isFileInPackages (String TheFileName) {
    int i;
    int Count = Globals.PItems.getCount();

    for (i = 0; i < Count; i++) {
      PacketItemI CurrentPItem = Globals.PItems.getPacketItem (i);
      int Kind                 = Globals.PItems.getPacketItemKind (i);
      if (CurrentPItem == null)
        continue;
      if (Kind == BrowseCollItem.ikPackage) {
        if (CurrentPItem.findPacketItem (TheFileName) != null)
          return true;
      }
      if (Kind == BrowseCollItem.ikPackagesRoot) {
        if (CurrentPItem.findPacketItemR (TheFileName) != null)
          return true;
      }
    }
    return false;
  }

  private boolean isToCheckInPackages (String TheFileName) {
    int ExtIndex = TheFileName.lastIndexOf (".");
    if (ExtIndex < 0)
      return false;
    String FileExt  = TheFileName.substring (ExtIndex, TheFileName.length()).toLowerCase();
    return ((FileExt.compareTo (Globals.JavaFileExt)  == 0) ||
            (FileExt.compareTo (Globals.PizzaFileExt) == 0) ||
            (FileExt.compareTo (Globals.ClassFileExt) == 0));
  }

  private boolean isFileInProject (String TheFileName) {
    if (isToCheckInPackages (TheFileName))
      return isFileInPackages (TheFileName);
    return isFileInDirs     (TheFileName);
  }

  public void doOpenFile (String TheFileName) {
    if (!isFileInProject(TheFileName))
      Globals.PManager .addFile  (TheFileName);
    if (isToBeViewed(TheFileName))
      Globals.TextFiles.openFile (TheFileName);
  }


  public void OnFileOpen (IdeSuperEvent AEvent) {
    String     LFileName;
    FileDialog FD = new FileDialog (Globals.IdeMainFrame);
    FD.setFile ("*"+Globals.JavaFileExt+";*"+Globals.PizzaFileExt);
    FD.setVisible (true);
    LFileName = FD.getFile();
    if (LFileName != null)
      doOpenFile (FD.getDirectory() + FD.getFile());
  }

  // Methods for managing Project Manager
  // ---------------------------------------------------------------------------------------
  public void OnProjectNew(IdeSuperEvent AEvent) {
    Globals.PManager.CreateNewProject();
  }

  public boolean OpenProject(String AProjectFileName) {
    return Globals.PManager.OpenProject(AProjectFileName);
  }

  protected String getProjectFilesMask() {
    return "*" + ProjectManager.cProjectFileExt;
  }

  public void OnProjectOpen(IdeSuperEvent AEvent) {
    String     LFileName;
    String     Directory;
    String     FName;

    FileDialog FD = new FileDialog (Globals.IdeMainFrame);
    FD.setFile (getProjectFilesMask()); FD.setVisible (true);
    Directory = FD.getDirectory();
    FName     = FD.getFile();
    if ((FName == null) || (Directory == null)) {
      MainMenuEvent TheEvent = (MainMenuEvent) AEvent;
      TheEvent.Result = -1;
      return;
    }
    LFileName = Directory + FName;
    OpenProject(LFileName);
  }

  public void OnProjectClose(IdeSuperEvent AEvent) {
    ((MainMenuEvent) AEvent).Result = Globals.PManager.CloseProjectAndRefresh() ? 0 : -1;
  }

  public void OnProjectRemoveFromProject (IdeSuperEvent AEvent) {
    Globals.PManager.RemoveBrowseItem (MainPanel.getSelectedProjectItemName(),
                                       MainPanel.getSelectedProjectItemKind());
    Globals.TextFiles.closeActiveFile();
  }

  public void OnProjectSave(IdeSuperEvent AEvent) {
    Globals.PManager.SaveProject();
  }

  public boolean OnProjectSaveAs(IdeSuperEvent AEvent) {
    String     LFileName;
    FileDialog FD = new FileDialog (Globals.IdeMainFrame,"", FileDialog.SAVE);
    FD.setFile (getProjectFilesMask()); FD.setVisible (true);
    LFileName = FD.getFile(); if (LFileName == null) return false;
    if (LFileName.length() == 0) return false;
    return Globals.PManager.SaveProjectAs(FD.getDirectory() + LFileName);
  }

  public void OnProjectRename (IdeSuperEvent AEvent) {
    String OldProjectName = Globals.getProjectName();
    File   PrFile = null;
    if (! OnProjectSaveAs(AEvent)) return;
    try {
      PrFile = new File(OldProjectName);
      PrFile.delete();
      Globals.Options.afterRenameProject(OldProjectName);
    }
    catch(Exception e) {
      Debug.Assert(e, this,"Error Deleting Old Project file");
    }
  }
  public void OnProjectProperties (IdeSuperEvent AEvent) {
    ProjectConfigDialog PrjConfigurator = new ProjectConfigDialog();
    PrjConfigurator.setVisible (true);
  }

  public void OnProjectJavaDOC (IdeSuperEvent AEvent) {
    MainPanel.genDoc();
  }
  // ---------------------------------------------------------------------------------------

  private String getDir () {
    DirectoryChooser DirGetter = new DirectoryChooser(Globals.IdeMainFrame);
    DirGetter.setVisible (true);
    return DirGetter.getDirectory();
  }

  private String getPackage () {
    DirectoryChooser DirGetter = new PackageChooser();
    DirGetter.setVisible (true);
    return DirGetter.getDirectory();
  }

  public void OnFileOpenPackage (IdeSuperEvent AEvent) {
    String TheNewPackage = getPackage();
    if (TheNewPackage != null) {
      PacketItemI NewPII = Globals.RootPaths.getPacketItemByDotName (TheNewPackage);
      if (NewPII == null)
        Misc.messageBox (new String[] {"OK"}, new String [] {"Unable to load package", TheNewPackage + ".", "Unreachable from system paths"}, 0);
      else
        if (NewPII.getPacketItemKind() == PacketItemI.cpikPacket) {
          if (!isPackageInProject (NewPII, BrowseCollItem.ikPackage)) {
            Globals.PManager.addPackage (NewPII.getDotName());
            checkAndExcludePack (NewPII);
          } else
            MainPanel.focusItem(NewPII);
        } else
          doOpenFile (NewPII.getFullDosPath());
    }
  }

  public void OnFileOpenPackageTree (IdeSuperEvent AEvent) {
    String TheNewPackageRoot = getPackage();
    if (TheNewPackageRoot != null) {
      PacketItemI NewPII = Globals.RootPaths.getPacketItemByDotName (TheNewPackageRoot);
      if (NewPII == null)
        Misc.messageBox (new String[] {"OK"}, new String [] {"Unable to load package", TheNewPackageRoot + ".", "Unreachable from system paths"}, 0);
      else
        if (NewPII.getPacketItemKind() == PacketItemI.cpikPacket)
          if (!isPackageInProject (NewPII, BrowseCollItem.ikPackagesRoot)) {
            Globals.PManager.addPackageRoot (NewPII.getDotName());
            checkAndExcludePackRoot (NewPII);
          } else
            MainPanel.focusItem(NewPII);
        else
          doOpenFile (NewPII.getFullDosPath());
    }
  }

  public void OnFileOpenDirectory (IdeSuperEvent AEvent) {
    String TheNewDirectory = getDir();
    if (TheNewDirectory != null) {
      if (!isDirectoryInProject (TheNewDirectory)) {
        Globals.PManager.addDirectory (TheNewDirectory);
        PacketItemI TheDirAdded = PCreateEntry.CreateDirEntry (TheNewDirectory);
        if (TheDirAdded != null)
          checkAndExcludeDir (TheDirAdded);
        else
          Misc.messageBox (new String [] {"OK"}, new String [] {"Warning", "The Directory does NOT exist !!!"}, 0);
      } else
        MainPanel.focusItem(TheNewDirectory);
    }
  }

  public void OnFileOpenDirTree (IdeSuperEvent AEvent) {
    String TheNewDirRoot = getDir();
    if (TheNewDirRoot != null) {
      if (!isDirRootInProject (TheNewDirRoot)) {
        Globals.PManager.addDirectoryRoot (TheNewDirRoot);
        PacketItemI TheDirAdded = PCreateEntry.CreateDirEntry (TheNewDirRoot);
        if (TheDirAdded != null)
          checkAndExcludeDirRoot (TheDirAdded);
        else
          Misc.messageBox (new String [] {"OK"}, new String [] {"Warning", "The Directory does NOT exist !!!"}, 0);
      } else
        MainPanel.focusItem(TheNewDirRoot);
    }
  }

  public void OnMake (IdeSuperEvent AEvent) {
    MainPanel.compile (false);
  }

  public void OnRebuild (IdeSuperEvent AEvent) {
    MainPanel.compile (true);
  }

  public void OnRebuildAll (IdeSuperEvent AEvent) {
    MainPanel.compileAll (true);
  }

  public void OnMakeAll (IdeSuperEvent AEvent) {
    MainPanel.compileAll (false);
  }

  public void OnRun (IdeSuperEvent AEvent) {
    MainPanel.runProgram();
  }
  public void OnDebug (IdeSuperEvent AEvent) {
    MainPanel.debugProgram();
  }

  private boolean forceProjectOpenCreate () {
    if (!Globals.PManager.isOpenedProject()) {
      Misc.messageBox (new String [] {"OK"}, new String [] {"Please open a project.", "(or create new one.)"}, 0);
      MainMenuEvent TheEvent = new MainMenuEvent (IdeMainMenu.msgmiProjectOpenCreate, this);
      Globals.EventsEngine.SendEvent (TheEvent);
      if (TheEvent.Result != 0)
        return false;
    }
    return true;
  }

  public void OnApplicationWizard (IdeSuperEvent AEvent) {
    if (!forceProjectOpenCreate ())
      return;
    ApplicationWizard AW = new ApplicationWizard();
    AW.setVisible (true);
    if (AW.isConfirmed())
      AW.createFile();
    AW.dispose();
  }

  public void OnAppletWizard (IdeSuperEvent AEvent) {
    if (!forceProjectOpenCreate ())
      return;
    AppletWizard AW = new AppletWizard();
    AW.setVisible (true);
    if (AW.isConfirmed())
      AW.createFile();
    AW.dispose();
  }
  //=====================================================================================
  public void InitEventListeners() {
    AddMainMenuAdapter(IdeMainMenu.msgmiFileClose,               this,"OnFileClose");
    AddMainMenuAdapter(IdeMainMenu.msgmiFileOpen,                this,"OnFileOpen");

    AddMainMenuAdapter(IdeMainMenu.msgmiOpenPackage,             this,"OnFileOpenPackage");
    AddMainMenuAdapter(IdeMainMenu.msgmiOpenPackageTree,         this,"OnFileOpenPackageTree");
    AddMainMenuAdapter(IdeMainMenu.msgmiOpenDirectory,           this,"OnFileOpenDirectory");
    AddMainMenuAdapter(IdeMainMenu.msgmiOpenDirTree,             this,"OnFileOpenDirTree");

    AddMainMenuAdapter(IdeMainMenu.msgmiFileSave,                this,"OnFileSave");
    AddMainMenuAdapter(IdeMainMenu.msgmiFileSaveAll,             this,"OnFileSaveAll");

    AddMainMenuAdapter(IdeMainMenu.msgmiProjectNew,              this,"OnProjectNew");
    AddMainMenuAdapter(IdeMainMenu.msgmiProjectOpenCreate,       this,"OnProjectOpen");
    AddMainMenuAdapter(IdeMainMenu.msgmiProjectRemoveFromProject,this,"OnProjectRemoveFromProject");
    AddMainMenuAdapter(IdeMainMenu.msgmiProjectClose,            this,"OnProjectClose");
    AddMainMenuAdapter(IdeMainMenu.msgmiProjectSave,             this,"OnProjectSave");
    AddMainMenuAdapter(IdeMainMenu.msgmiProjectSaveAs,           this,"OnProjectSaveAs");
    AddMainMenuAdapter(IdeMainMenu.msgmiProjectRename,           this,"OnProjectRename");
    AddMainMenuAdapter(IdeMainMenu.msgmiProjectProperties,       this,"OnProjectProperties");
    AddMainMenuAdapter(IdeMainMenu.msgmiProjectDocumentation,    this,"OnProjectJavaDOC");

    AddMainMenuAdapter(IdeMainMenu.msgmiFileSaveAs,              this,"OnFileSaveAs");
    AddMainMenuAdapter(IdeMainMenu.msgmiFileCloseAll,            this,"OnFileCloseAll");
    AddMainMenuAdapter(IdeMainMenu.msgmiFileNew,                 this,"OnFileNew");
    AddMainMenuAdapter(IdeMainMenu.msgmiBuildRebuildProject,     this,"OnRebuildAll");
    AddMainMenuAdapter(IdeMainMenu.msgmiBuildMakeProject,        this,"OnMakeAll");
    AddMainMenuAdapter(IdeMainMenu.msgmiBuildMake,               this,"OnMake");
    AddMainMenuAdapter(IdeMainMenu.msgmiBuildRebuild,            this,"OnRebuild");
    AddMainMenuAdapter(IdeMainMenu.msgmiRunRun,                  this,"OnRun");
    AddMainMenuAdapter(IdeMainMenu.msgmiRunDebug,                this,"OnDebug");

    AddMainMenuAdapter(IdeMainMenu.msgmiViewOutput,              this,"OnViewOutput");
    AddMainMenuAdapter(IdeMainMenu.msgmiViewDebugger,            this,"OnViewDebugger");

    AddMainMenuAdapter(IdeMainMenu.msgmiWizardsAppletWizard,     this,"OnAppletWizard");
    AddMainMenuAdapter(IdeMainMenu.msgmiWizardsApplicationWizard,this,"OnApplicationWizard");

    AddMainMenuAdapter(IdeMainMenu.msgmiFileExit,                this,"OnExit");
    AddMainMenuAdapter(IdeMainMenu.msgmiToolsConfigurePalette,   this,"OnShowPaletteOptions");
    AddMainMenuAdapter(IdeMainMenu.msgmiToolsEnviromentOptions,  this,"OnShowIDEOptions");
  }
}
