/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.system.classes;

import java.io.*;
import java.lang.*;
import java.util.*;
import org.freebuilder.system.debug.*;

/** <code>Strings</code> is an abstract class for creating a collection of
  * String & Object pairs. You can use it like Collection of Strings, or of
  * Objects, or both together. Strings class defines the interface functions
  * and main behaviour for some of them.
  * It implements Serializable, but does not override writeObject() or
  * readObject() methods. So it is your responsobilities to do that it it
  * is neccessary.
  * This object is some Java variant of the Borland's Delphi TStrings class.
  * Many of methods defined in TStrings are translated from Object Pascal.
  *
  * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
  * @version 0.7, 12/12/97
  */

    
abstract public class Strings implements Cloneable, Serializable {
  protected int FUpdateCount;

  protected Strings() {
    FUpdateCount = 0;
  }

  protected void CheckBounds(int Index) throws ArrayIndexOutOfBoundsException {
    if (Index >= getCount()) throw new ArrayIndexOutOfBoundsException(Index + " >= " + getCount());
    if (Index < 0)           throw new ArrayIndexOutOfBoundsException(Index + " < 0 ");
  }

  
  abstract protected int GetCapacity();

  /** Adds a string to the collection. Returns an Index of the string in it. */
  public int  Add (final String S) {
    return Add(S,null);
  }

  /** Adds a string and Object to the collection. Returns an Index of the
    * string in it. */
  public int  Add (final String S, Object AObject) { // AddObject
    int Result = getCount();
    Insert(Result,S,AObject);
    return Result;
  }

  /** Adds Collection of Strings & Objects to this one */
  public void Add (final Strings AStrings) {        // AddStrings
    if (AStrings == null) return;

    int Cnt, Cntr;

    Cnt = AStrings.getCount(); if (Cnt == 0) return;
    BeginUpdate();
    try {
      for (Cntr = 0; Cntr < Cnt; Cntr ++) {
        Add(AStrings.getString(Cntr), AStrings.getObject(Cntr));
      }
    }
    finally {
      EndUpdate();
    }
  }

  /** Clears the Collection & Assign the ones from the other one */
  public void Assign(Strings Source) {
    BeginUpdate();
    try {
      Clear();
      Add(Source);
    }
    finally {
      EndUpdate();
    }
  }

  /** This function is called by many methods like : add, remove, insert ..
    * before do any changes.
    * This function can be called by other object too, to say that it will make
    * some changes with the Strings
    * IF BeforeUpdate is called, AfterUpdate must be called after the changes.
    * It's better to say BeforeUpdate(); try { .... } finally { EndUpdate() }
    * If exceptions is throwed when BeforeUpdate is executed then EndUpdate()
    * must Not be called.
    */
  public void BeginUpdate() {
    if (FUpdateCount == 0) SetUpdateState(true);
    FUpdateCount++;
  }

  /** This function is called by many methods like : add, remove, insert .. after the changes have been done. */
  public void EndUpdate  () {
    FUpdateCount--;
    if (FUpdateCount == 0) SetUpdateState(false);
  }

  /** You can override this method to call some functions every time the object has changed and FUpdateCount == 0 */
  protected void   SetUpdateState(boolean Updating) {};

  /** Exchange two items */
  public void Exchange   (int Index1, int Index2) {
    Object TempObject;
    String TempString;

    BeginUpdate();
    try {
      TempString = getString(Index1);
      TempObject = getObject(Index1);
      setString(Index1, getString(Index2));
      setObject(Index1, getObject(Index2));
      setString(Index2, TempString);
      setObject(Index2, TempObject);
    }
    finally {
      EndUpdate();
    }
  }

  /** Moves this item to the new position, and remove the old one */
  public void Move(int CurIndex, int NewIndex) {
    Object TempObject;
    String TempString;

    if (CurIndex == NewIndex) return;
    BeginUpdate();
    try {
      TempString = getString(CurIndex);
      TempObject = getObject(CurIndex);
      Delete(CurIndex);
      Insert(NewIndex, TempString, TempObject);
    }
    finally {
      EndUpdate();
    }
  }

  /** Compares only the strings. You can override this method to compare whole Objects */
  public synchronized boolean Equals  (Strings AStrings) {
    int i, Count;

    Count = getCount(); if (Count != AStrings.getCount()) return false;
    for (i = 0; i < Count - 1; i++) {
      if (getString(i).compareTo(AStrings.getString(i)) != 0)
        return false;
    }
    return true;
  }

  /** Returns index of the string in collection (0 .. getCount() - 1), or -1 if not found */
  public synchronized int  IndexOf    (final String S) {
    int Result = -1;         if (S == null) return -1;
    int To     = getCount(); if (To <=   0) return -1;

    for (Result = 0; Result < To; Result++) {
      if (S.compareTo(getString(Result)) == 0)
        return Result;
    }
    return -1;
  }

  /** Returns index of the AObject in collection (0 .. getCount() - 1), or -1 if not found */
  public synchronized int  IndexOf    (Object AObject) {  // IndexOfObject
    int Result = -1;
    int To     = getCount();

    for (Result = 0; Result < To; Result++) {
      if (getObject(Result) == AObject)
        return Result;
    }
    return -1;
  }

  /** Insert a string in Index position */
  public void Insert      (int Index, final String S) {
    Insert(Index,S,null);
  }

  public void LoadFromFile(final String FileName) {
    FileInputStream InpStrm = null;
    try {
      InpStrm = new FileInputStream(FileName);
      LoadFromStream(InpStrm);
    }
    catch (FileNotFoundException e) {
      if (Debug.isDebug) {
        Debug.Assert(this, "Can not find file : " + FileName);
      }
    }
    catch (IOException e) {
      if (Debug.isDebug) {
        Debug.Assert(this, "Error reading from file : " + FileName);
      }
    }
  }

  public void SaveToFile  (final String FileName) {
    FileOutputStream OutStrm = null;
    try {
      OutStrm = new FileOutputStream(FileName);
      SaveToStream(OutStrm);
    }
    catch (IOException e) {
      if (Debug.isDebug) {
        Debug.Assert(this, "Error writing to file : " + FileName);
      }
    }
  }

  synchronized public void LoadFromStream(InputStream  Stream) throws IOException {
    String CrntLine;

    InputStreamReader InpStrR   = new InputStreamReader(Stream);
    BufferedReader    InpStream = new BufferedReader   (InpStrR);

    BeginUpdate();
    Clear();
    try {
      while ((CrntLine = InpStream.readLine()) != null) {
        Add(CrntLine);
      }
    }
    finally {
      EndUpdate();
    }
  }

  synchronized public void SaveToStream  (OutputStream Stream) throws IOException {
    int    Cntr, To;
    String Str;

    To = getCount(); if (To == 0) return;

    OutputStreamWriter OutStrW   = new OutputStreamWriter(Stream);
    BufferedWriter     OutStream = new BufferedWriter(OutStrW);

    try {
      for (Cntr = 0; Cntr < To; Cntr++) {
        Str = getString(Cntr);
        OutStream.write(Str,0,Str.length());
        OutStream.newLine();
      }
    }
    finally {
      OutStream.close();
    }
  }

  public void TestShowStrings() {
    for (int i = 0; i < getCount(); i++)
      System.out.println(i + " " + getString(i));
  }

  abstract public String getString(int Index);
  abstract public void   setString(int Index, final String S);
  /** returns Object at Index, or null */
  abstract public Object getObject (int Index); 
  abstract public void   setObject(int Index, Object AObject);
  abstract public int    getCount();
  abstract public void   Clear      ();
  abstract public void   Delete(int Index);
  abstract public void   Delete(String ItemName);
  abstract public void   Delete(Object Item);
  abstract public void   Insert(int Index, final String S, Object AObject);

//  abstract public int    IndexOfName(final String Name);
//  abstract public String getNames  (int Index);
//  abstract public String getValues (int Index);
}

