/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.gui.editors;

import java.io.*;
import org.freebuilder.gui.ide.*;
import org.freebuilder.gui.beans.*;
import java.util.zip.*;
import org.freebuilder.gui.idemenu.*;
import org.freebuilder.gui.config.*;
import org.freebuilder.gui.ideoptions.*;
import org.freebuilder.tools.editor.*;
import org.freebuilder.system.classes.*;
import org.freebuilder.system.packages.*;
import org.freebuilder.system.ideengine.*;
import org.freebuilder.system.classes.events.*;
import org.freebuilder.system.classes.events.engine.*;

import com.sun.java.swing.event.*;

/** This class is a simple class loader - the class files are loaded
    from the class files, reachable from the project paths - Class Path,
    Class Output Path, etc., and if not found, the file is tried to be loaded
    calling Class.forName() function.
  * @author	<a href=mailto:wreck@nat.bg> Peter Radkov </a>
    @version 0.7, 07-01-98
 */
class AClassLoader extends ClassLoader {
  public AClassLoader () {}

  public Class loadClass (InputStream Data, String Name, boolean resolve, int Size) {
    Class Ret = null;
    if (Name.compareTo ("java.lang.Object") == 0) {
      try {
        Ret = Class.forName (Name);
        if (resolve)
          resolveClass (Ret);
        return Ret;
      } catch (Exception e) {return null;}
    }

    if (Name != null)
     Ret = findLoadedClass (Name);

    if (Ret != null) {
      if (resolve)
        resolveClass (Ret);
      return Ret;
    }
    try {
      byte [] ClassData = new byte [Size];
      Data.read (ClassData, 0, Size);
      Ret = defineClass (Name, ClassData, 0, Size);
      if (resolve)
        resolveClass (Ret);
      return Ret;
    } catch (Exception e) {
      Misc.messageBox (new String [] {"OK"}, new String [] {"Failed loading " + Name+ "."},0);
      return null;
    }
  }

  public Class loadClass (String Name, boolean resolve) {
    Class Ret;

    if (Name.compareTo ("java.lang.Object") == 0)
      try {
        Ret = Class.forName (Name);
        if (resolve)
          resolveClass (Ret);
        return Ret;
      } catch (Exception e) {
        Misc.messageBox (new String [] {"OK"}, new String [] {"Exception occured while trying to load class " + Name + ".", "Browsing failed."},0);
        return null;
      }
    else
      Ret = findLoadedClass (Name);

    if (Ret != null) {
      if (resolve)
        resolveClass (Ret);
      return Ret;
    }

    PacketItemI TheClass = Globals.RootPaths.getPacketItemByDotName (Name+Globals.ClassFileExt);

    if (TheClass == null)
      try {
        Ret = Class.forName (Name);
      } catch (Exception e) {
        Misc.messageBox (new String [] {"OK"}, new String [] {"Exception occured while trying to load class " + Name + " from system classes.", "Browsing failed."},0);
        return null;
      }

    if (Ret == null)
    try {
      if (TheClass instanceof PZipFileEntry) {
        PZipFileEntry TheFile = (PZipFileEntry) TheClass;
        ZipFile TheData       = new ZipFile (TheFile.getRootFullPath());
        ZipEntry TheEntry     = TheData.getEntry(TheFile.getFileNameInZip());
        Ret = loadClass (TheData.getInputStream (TheEntry), Name, resolve, (int) TheEntry.getSize());
      } else {
        File F = new File (TheClass.getFullDosPath());
        FileInputStream Data = new FileInputStream (F);
        Ret = loadClass (Data, Name, resolve, (int) F.length());
      }
    } catch (Exception e) {
      Misc.messageBox (new String [] {"OK"}, new String [] {"Exception occured while Free Builder was trying to load class " + Name + ".", "Browsing failed."},0);
      return null;
    }

    if (resolve)
      resolveClass (Ret);

    return Ret;
  }
}

/** Class <code>TextFilesManager</code> keeps track of all opened project files.
  *  The files are NOT kept open. This class contains a (FullDosName, EditFile)
  *  pairs.
  *  @author  Peter Radkov  <wreck@nat.bg>
  *  @author	George Petkov <pgeorge@mail.techno-link.com>,<gpetkov@hotmail.com>, <gpetkov@usa.net>
  *  @version 0.7, 12/12/97
  */

public class TextFilesManager extends StringList implements FileManagerI, StringsListener {
  TextEditorI ActiveFile = null;
  private String ActiveFileName;
  static int FileCount  = 0;

  public TextFilesManager () {
    AddChangeListener(this);

    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanSave,         this, "canSave");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanClose,        this, "canClose");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanSaveAs,       this, "canSaveAs");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanSaveAll,      this, "canSaveAll");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanCloseAll,     this, "canCloseAll");
    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgMenuQuery_CanSaveProject,  this, "canSaveProject");

    Globals.IEngine.AddSystemEventAdapter (SystemEvent.msgSetActiveFile, this, "SetMeActive");
  }

  public TextEditorI getTextEditor(int Index) {
    if ((Index < 0) || (Index > getCount() - 1))
      return null;
    return (TextEditorI) getObject (Index);
  }

  public TextEditorI getTextEditor(String FName) {
    if (FName == null)
      return null;
    return (TextEditorI) getObject (FName);
  }

  protected String getUntitledFileName() {
    FileCount++;
    return Globals.getProjectPath() + Globals.FileSeparator + "Untitled " + FileCount + Globals.JavaFileExt;
  }

  public boolean newFile() {
    File F;
    String NewName;
    do {
      NewName = getUntitledFileName();
      F = new File (NewName);
    } while (F.exists());
    return openFile(NewName);
  }

  private String getClassAsString (Class TheClass) {
    return (new ClassBrowser()).showClass (TheClass);
  }

  private EditFile extractClassInfo (PacketItemI AClass, InputStream TheData, long Size) {
    String ClassName = AClass.getDotName();
    if (ClassName.endsWith(Globals.ClassFileExt))
      ClassName = ClassName.substring (0, ClassName.length() - Globals.ClassFileExt.length());

    if ((ClassName == null) || (ClassName.length() == 0)) {
      Misc.messageBox (new String [] {"OK"},new String [] {"Unable to browse " + AClass.getFileName()+"!", "(Unreachable form system paths)"}, 0);
      return null;
    }

    AClassLoader Loader = new AClassLoader ();
    Class TheClass = Loader.loadClass (TheData, ClassName, false, (int)Size);
    String TheStrData = getClassAsString (TheClass);
    return new EditFile (new StringReader (TheStrData),
                         AClass.getFullDosPath(), true, TheStrData.length());
  }

  private void doOpen (EditFile NewFile) {
    if (NewFile != null) {
      if (IndexOf (NewFile.getFullDosPath()) == -1)
        Add (NewFile.getFullDosPath(), NewFile);
      setActiveFile (NewFile.getFullDosPath());
    }
  }

  public void openFile(String FullDosPath, Reader Data, int Size) {
    EditFile ToOpen = new EditFile (Data, FullDosPath, false, Size);
    ToOpen.setModified (true);
    doOpen (ToOpen);
  }

  public boolean openFile(String FName) {
    boolean IsOK = false;
    EditFile NewFile;
    BeginUpdate();
    try {
      PacketItemI TheItem = Globals.RootPaths.findPacketItemI (FName);
      if (TheItem == null)
        TheItem = PCreateEntry.CreateEntry(FName);

      if (TheItem instanceof PZipFileEntry) {
        PZipFileEntry TheFile = (PZipFileEntry) TheItem;
        ZipFile TheData       = new ZipFile (TheFile.getRootFullPath());
        ZipEntry TheEntry     = TheData.getEntry(TheFile.getFileNameInZip());

        if (TheFile.getPacketItemKind() == PacketItemI.cpikClass)
          NewFile = extractClassInfo (TheItem, TheData.getInputStream (TheEntry), TheEntry.getSize());
        else
          NewFile               = new EditFile (new InputStreamReader(TheData.getInputStream (TheEntry)),
                                                TheItem.getFullDosPath(),
                                                true, TheEntry.getSize());
      }
      else {
        File F = new File (FName);
        if (F.exists())
          if (TheItem.getPacketItemKind() == PacketItemI.cpikClass)
            NewFile = extractClassInfo (TheItem, new FileInputStream (FName), F.length());
          else
            NewFile = new EditFile (new InputStreamReader(new FileInputStream (FName)),
                                    FName, false, F.length());
        else
          NewFile = new EditFile (null, FName, false, 0);
      }

      doOpen (NewFile);
      IsOK = true;
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    finally {
      EndUpdate();
      return IsOK;
    }
  }
  /** Closes the specified file. The file is simply deleted from the "opened" files list. */
  public boolean closeFile(String FName) {
    BeginUpdate ();
    try {
      int Index = IndexOf(getActiveTextFile());
      Delete (Index);
      int Count = getCount();
      if (Count > 0)
        if (Index < Count-1)
          setActiveFile (getString (Index));
        else
          setActiveFile (getString (Count-1));
      else
        setActiveFile (null);
    } finally {
      EndUpdate();
    }
    return true;
  }

  /** Closes the active file. The file is simply deleted from the "opened" files list. */
  public boolean closeActiveFile() {
    if (getActiveFile() != null)
      return closeFile (getActiveFile());
    else
      return false;
  }

  public boolean saveFile(String FileName) {
    return getTextEditor (FileName).saveIt();
  }

  public boolean saveFile(TextEditorI TheFile) {
    return TheFile.saveIt();
  }

  public boolean saveActiveFile () {
    if (ActiveFile != null)
      return ((EditFile)ActiveFile).saveIt();
    else
      return false;
  }

  public boolean saveAllFiles() {
    int i;
    boolean Ret = true;
    for (i = 0; i < getCount(); i++) {
      EditFile TheFile = (EditFile) getObject (i);
      Ret &= TheFile.saveIt();
    }
    return Ret;
  }

  public boolean isModified(String FName) {
    return getTextEditor(FName).isModified();
  }

  public StringList getModifiedFiles() {
    StringList Ret = new StringList();

    for (int i = 0; i < getCount(); i++)
      if (isModified (getString(i)))
        Ret.Add (getString(i), getObject (i));
    return Ret;
  }

  public boolean closeAllFiles() {
    BeginUpdate();
    try {
      int Count = getCount()-1;
      while (Count >= 0) {
        Delete (Count);
        Count --;
      }
      setActiveFile(null);
    } finally {
      EndUpdate();
      return true;
    }
  }

  public boolean saveFileAs(String OldFileName, String NewFileName) {
    EditFile TheFile;
    int PrevIndex;
    BeginUpdate();
    try {
      PrevIndex = IndexOf (OldFileName);
      TheFile = (EditFile) getObject (PrevIndex);
      TheFile.setFullDosPath(NewFileName);
      TheFile.saveFile();
      setString (PrevIndex, NewFileName);
    }
    finally {
      EndUpdate();
    }
    return TheFile.saveIt();
  }

  public void setActiveFile(String FName) {
    if (ActiveFileName == FName)
      return;
    BeginUpdate();
    try {
      ActiveFile = getTextEditor(FName);
      ActiveFileName = FName;
    }
    finally {
      EndUpdate();
    }
  }

  // Begin of functions from interface StringsListener
  public void BeforeChange(Strings Sender) {}

  public void AfterChange (Strings Sender) {
    Globals.EventsEngine.SendEvent (new SystemEvent   (SystemEvent.msgActiveFileSet, this, ActiveFileName));
    Globals.EventsEngine.PostEvent (new MainMenuEvent (IdeMainMenu.msgmiFileSave,      MainMenuEvent.msubcRefresh, this));
    Globals.EventsEngine.PostEvent (new MainMenuEvent (IdeMainMenu.msgmiFileClose,     MainMenuEvent.msubcRefresh, this));
    Globals.EventsEngine.PostEvent (new MainMenuEvent (IdeMainMenu.msgmiFileCloseAll,  MainMenuEvent.msubcRefresh, this));
    Globals.EventsEngine.PostEvent (new MainMenuEvent (IdeMainMenu.msgmiSearchFind,           MainMenuEvent.msubcRefresh, this));
    Globals.EventsEngine.PostEvent (new MainMenuEvent (IdeMainMenu.msgmiSearchReplace,        MainMenuEvent.msubcRefresh, this));
    Globals.EventsEngine.PostEvent (new MainMenuEvent (IdeMainMenu.msgmiSearchFindAgain,      MainMenuEvent.msubcRefresh, this));
    Globals.EventsEngine.PostEvent (new MainMenuEvent (IdeMainMenu.msgmiSearchGoToLineNumber, MainMenuEvent.msubcRefresh, this));
  }
  // End   of functions from interface StringsListener


  public String getActiveFile () {
    if (getActiveTextFile() == null)
      return null;
    return ActiveFile.getFullDosPath();
  }

  public TextEditorI getActiveTextFile() {
    return ActiveFile;
  }

  private void RegisterAllListeners() {}
  public void UnRegisterAllListeners() {}

//********************  Adapter Functions *************************


  private void setResult (IdeSuperEvent AEvent, int Result) {
    ((SystemEvent)AEvent).Result = Result;
  }

  public void SetMeActive (IdeSuperEvent AEvent) {
    String TheFileName = ((SystemEvent)AEvent).StrValue;
    if (IndexOf(TheFileName) >=0)
      setActiveFile(TheFileName);
    else {
      if (TheFileName.toLowerCase().endsWith(Globals.JavaFileExt)  ||
          TheFileName.toLowerCase().endsWith(Globals.HTMLFileExt)  ||
          TheFileName.toLowerCase().endsWith(Globals.HTMFileExt)   ||
          TheFileName.toLowerCase().endsWith(Globals.PizzaFileExt) ||
          TheFileName.toLowerCase().endsWith(Globals.ClassFileExt))
        openFile (TheFileName);
      else
        setActiveFile(null);
    }
  }

  public void canSave (IdeSuperEvent AEvent) {
    TextEditorI TheFile = getActiveTextFile();
    if (TheFile != null) {
      if (TheFile.isModified())
        setResult (AEvent, 1);
      else
        setResult (AEvent, 0);
    } else
      setResult (AEvent, 0);
  }

  public void canClose (IdeSuperEvent AEvent) {
    if (getActiveTextFile() != null)
      setResult (AEvent, 1);
    else
      setResult (AEvent, 0);
  }

  public void canSaveAs (IdeSuperEvent AEvent) {
    canClose (AEvent);
  }

  public void canSaveAll (IdeSuperEvent AEvent) {
    if (getModifiedFiles().getCount() > 0)
      setResult (AEvent, 1);
    else
      setResult (AEvent, 0);
  }

  public void canCloseAll (IdeSuperEvent AEvent) {
    if (getCount() > 0)
      setResult (AEvent, 1);
    else
      setResult (AEvent, 0);
  }

  public void canSaveProject (IdeSuperEvent AEvent) {
    setResult (AEvent, 0);
  }
}

