/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.system.classes.options;

import java.io.*;
import java.lang.*;
import org.freebuilder.system.debug.*;
import org.freebuilder.system.classes.*;
import org.freebuilder.system.classes.events.engine.IdeEventsEngine;

/** The class <code>OptionsColl</code> is a collection of all the options which will be saved and restored.
  * If you want some options to be saved whith project or default configuration, have to add here.
  * Options are saved on every change of any of them. Only cloned options are returned.
  *
  * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
  * @version 0.7, 02-12-97 */

public class OptionsColl extends StringList implements StringsListener {
  private transient IdeEventsEngine EventsEngine  = null; // To post a message to.
  private transient boolean         IsSaveEnabled = false;
  private transient boolean         EventsEnabled = false;
  private transient boolean         FIsLoaded     = false;

  private String ConfigFileName = "";

  public OptionsColl() {
    AddChangeListener(this);
  }

  public OptionsColl(IdeEventsEngine EventsEngine, String AConfigFileName) {
    this();
    this.EventsEngine  = EventsEngine;
    setConfigFileName(AConfigFileName);
  }

  /** ConfigFileName consist of path and file name. Path is relative to user home directory */
  public void setConfigFileName(String FileName) {
    ConfigFileName = FileName;
  }

  private OptionsSuperI getOrigOptions(int OptionsIndex) throws ArrayIndexOutOfBoundsException {
    return (OptionsSuperI)getObject(OptionsIndex);
  }

  /** Posts a message to the Option's listener to notify Tool which options have been changed. */
  protected void RefreshOption(OptionsSuperI NewOptions, int EventNumber) {
    OptionsPostEvent(NewOptions.PrepareEvent(EventNumber));
  }

  /** Sends a message to the Option's listener to notify Tool which options have been changed. */
  public void SendOptionsRefresh(OptionsSuperI NewOptions) {
    if (NewOptions == null) return;
    OptionsPostEvent(NewOptions.PrepareEvent(OptionsEvent.msgOptionsRefresh));
  }

  /** Enables sending and posting event to the EventsEngine */
  public synchronized void EnableEvents(boolean Enable) {
    if (EventsEnabled == Enable) return;
    EventsEnabled = Enable;
  }

  public boolean IsEventsEnabled() { return EventsEnabled; }
  /** Is it enabled to save options */
  public boolean IsSaveEnabled  () { return IsSaveEnabled; }
  /** Are the options loaded from configuration file ? */
  public boolean IsLoaded       () { return FIsLoaded;     }

  /** Returns full path and the name of configuration file. */
  public String getConfigFileName() { return ConfigFileName; }

  /** This method is called by some Options object to post a message */
  void OptionsPostEvent(OptionsEvent Event) {
    if ((EventsEngine != null) & (IsEventsEnabled()))
      EventsEngine.PostEvent(Event);
  }

  /** Enables or Disable saving the configuration when some of them has changed, added, removed .. */
  public void EnableSaveOptions(boolean Enable) {
    if (IsSaveEnabled == Enable) return;
    IsSaveEnabled = Enable;

    if (IsSaveEnabled)
      RefreshOptions();
  }

  /** This function is called just befor some changes are beeing performed */
  public void BeforeChange(Strings Sender) { }  // Not used here

  /** This function is called after all the changes have been performed and saves the configuration */
  public void AfterChange (Strings Sender) {
    if (! IsSaveEnabled) return;
    Save();
  }

  /** Write the collection of Options objects to file */
  public synchronized void Save() {
    String FileName = getConfigFileName();
    try {
      FileOutputStream   FileOutStrm   = new FileOutputStream(FileName);
      ObjectOutputStream ObjectOutStrm = new ObjectOutputStream(FileOutStrm);
      ObjectOutStrm.writeObject(this);
      ObjectOutStrm.flush();
      ObjectOutStrm.close();
    }
    catch (IOException e) {
      if (Debug.isDebug) {
        Debug.Assert(e,this,"Error writing configuration file : " + FileName);
      }
    }
  }

  /** This method is like a constructor for this object when loading from the file */
  public static synchronized OptionsColl Load(IdeEventsEngine EventsEngine, String AConfigFileName) {
    OptionsColl Result = null;
    String      PathFileName;
    try {
      PathFileName = AConfigFileName;
      FileInputStream   FileInpStrm   = new FileInputStream(PathFileName);
      ObjectInputStream ObjectInpStrm = new ObjectInputStream(FileInpStrm);
      Result = (OptionsColl)ObjectInpStrm.readObject();
      ObjectInpStrm.close();

      Result.setConfigFileName(AConfigFileName);
      Result.EventsEngine = EventsEngine;
      Result.FIsLoaded = true;
      return Result;
    }
    catch (IOException e) {
      // Debug.Assert(e,null,"Error reading configuration file : " + AConfigFileName);
    }
    catch (ClassNotFoundException e) {
      Debug.Assert(e,null,"Class OptionsColl not found while reading from file : " + AConfigFileName);
    }
   return null;
  }

  /** This method tries to load OptionsCollection with AConfigFileName. if does not succeed
    * create a default Options collection */
  public static OptionsColl LoadOrDefault(IdeEventsEngine EventsEngine, String AConfigFileName) {
    OptionsColl NewColl = Load(EventsEngine, AConfigFileName);
    if (NewColl != null) return NewColl;

    NewColl = new OptionsColl(EventsEngine, AConfigFileName);

    NewColl.BeginUpdate();
    try {
      // NewColl.CreateDefaultConfig();
      return NewColl;
    }
    finally {
      NewColl.EndUpdate();
    }
  }

  /** Updates only the options which names are in both collections */
  public void UpdateOptionsFrom(OptionsColl OtherOptionsColl) {
    int Cntr, To, OtherCollIndex;
    String CrntOptiosName;

    if (OtherOptionsColl == null) return;

    BeginUpdate();
    try {
      To = getCount();
      for (Cntr = 0; Cntr < To; Cntr++) {
        CrntOptiosName = getString(Cntr);
        OtherCollIndex = OtherOptionsColl.IndexOf(CrntOptiosName);
        if (OtherCollIndex >= 0)
          setOptions(CrntOptiosName, OtherOptionsColl.getOptions(OtherCollIndex));
      }
    }
    finally {
      EndUpdate();
    }
  }

  /** Makes all the Options objects in collection to post a message to their listeners */
  public void RefreshOptions() {
    int Cntr;
    OptionsSuperI Options;
    try {
      for (Cntr = 0; Cntr < getCount(); Cntr++) {
        Options = getOptions(Cntr);
        SendOptionsRefresh(Options);
        // RefreshOption(Options,OptionsEvent.msgOptionsRefresh);
      }
    }
    catch (ArrayIndexOutOfBoundsException e) {
      if (Debug.isDebug) {
        Debug.Assert(e,this, "IndexOutOfBounds");
      }
    }
  }

  /** Returns clonned object with OptionsName if found. Otherwhise returns null */
  public OptionsSuperI getOptions(String OptionsName) {
    int Index = IndexOf(OptionsName);

    if (Index < 0) return null;
    try {
      return getOptions(Index);
    }
    catch (ArrayIndexOutOfBoundsException e) { // This should never happened !
      if (Debug.isDebug) Debug.Assert(e,this);
    };
    return null;
  }

  /** Returns clonned object at OptionsIndex. */
  public OptionsSuperI getOptions(int OptionsIndex) throws ArrayIndexOutOfBoundsException {
    OptionsSuperI OldOptions, NewOptions;

    OldOptions = getOrigOptions(OptionsIndex);
    try {
      NewOptions = OldOptions.Clone();
    }
    catch(CloneNotSupportedException e) {
      if (Debug.isDebug) {
        Debug.Assert(e,this, "The big problem. StringList.IndexOf does not work properly ");
      }
      return null;
    }
    return NewOptions;
  }

  /** If there is an object with such a name, and the new options are diffrent than the options included
    * in collection, the new object replaces the old one, OptionsColl is saved, and Event is possed to the listeners. */
  public void setOptions(String OptionsName, OptionsSuperI NewOptions) { // Changed
    int           OIndex  = IndexOf       (OptionsName);
    OptionsSuperI Options = null;
    try {
      Options = getOrigOptions(OIndex);
    }
    catch(ArrayIndexOutOfBoundsException e) {
      Options = null;
    }
    
    if ((Options == null) | (NewOptions == null) | (OIndex < 0)) return; // skip the bullshits
    if (! Options.IsDifferent(NewOptions))                       return; // if the new options are the same, what to do ?

    setObject(OIndex,NewOptions);    // This line will call Save() (via AfterChange())
    RefreshOption(NewOptions,OptionsEvent.msgOptionsChanged); // Post a message
  }

  public void setOptions(OptionsSuperI NewOptions) {
    if (NewOptions == null) return;
    setOptions(NewOptions.getName(),NewOptions);
  }

  public boolean IsOptions (String OptionsName) {
    return IndexOf(OptionsName) >= 0;
  }

  public boolean IsOptions (OptionsSuperI NewOptions) {
    if (NewOptions == null) return false;
    return IsOptions(NewOptions.getName());
  }

  /** Add the NewOptions object to the collection, save collection (if IsSaveEnabled) and Post a message
    * IF there is an Object with NewOptions.getName() name than the DuplicateOptionNameException will be thowned. */
  public boolean addOptions(OptionsSuperI NewOptions) throws DuplicateOptionNameException {
    String NewOptionsName;

    if (NewOptions == null) return false;
    NewOptionsName = NewOptions.getName();
    if (IndexOf(NewOptionsName) >= 0)
      throw new DuplicateOptionNameException("Duplicate Options name : " + NewOptionsName);

    Add(NewOptionsName, NewOptions);
    RefreshOption(NewOptions, OptionsEvent.msgOptionsChanged); // Post a message
    return true;
  }

  private boolean IsListenerFor(OptionsSuperI AOptions) {
    if ((AOptions == null) | (EventsEngine == null))return false;
    return EventsEngine.IsListenerFor(AOptions.PrepareEvent(OptionsEvent.msgOptionsChanged));
  }

  //   ,   .        ,  
  //   .   .
  public boolean removeOptions(String OptionsName) {
    OptionsSuperI AOptions;
    int Index = IndexOf(OptionsName); if (Index < 0) return true;
    AOptions = getOrigOptions(Index);
    return removeOptions(AOptions);
  }

  public boolean removeOptions(OptionsSuperI AOptions) {
    if (AOptions == null) return true;
    if (Debug.isDebug) {
      Debug.Assert(!IsListenerFor(AOptions),"Remove options object before removing it's listener");
    }
    Delete(AOptions); return true;
  }

  /** Creates the collection with all the options which must have by default ( for Compiiler, Editor .. )
    * Override this method to make a default options collection. */
  public void CreateDefaultConfig() {
  }

  /** Restore default options for whole the collection */
  public void RestoreDefaults() {
    OptionsSuperI AOptions;
    int           Cntr;
    for (Cntr = 0; Cntr < getCount(); Cntr++) {
      AOptions = getOrigOptions(Cntr);
      if (AOptions != null)
        AOptions.setDefaults();
    }
    Save();
    RefreshOptions();
  }

  /** for testing */
  public void PrintOptions() {
    OptionsSuperI AOptions;
    int           Cntr;
    for (Cntr = 0; Cntr < getCount(); Cntr++) {
      AOptions = getOrigOptions(Cntr);
      if (AOptions != null)
        System.out.println(AOptions.getName());
    }
  }
}
