/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.gui.config;

import java.util.zip.*;
import java.util.*;
import java.awt.*;
import java.io.*;
import java.awt.event.*;

import com.sun.java.swing.*;
import com.sun.java.swing.event.*;
import com.sun.java.swing.plaf.basic.*;

import org.freebuilder.gui.beans.*;
import org.freebuilder.gui.ide.*;
import org.freebuilder.gui.ideoptions.*;
import org.freebuilder.gui.ideoptions.abstr.*;

import org.freebuilder.system.classes.options.manager.OptionsMngr;
import org.freebuilder.system.ideengine.*;

/**
  * @author	<a href=mailto:wreck@nat.bg> Peter Radkov </a>
    @version 0.7, 07-01-98
*/
class BeanListCellRenderer extends BasicListCellRenderer {
  public Component getListCellRendererComponent(JList list,
                                                Object value,
                                                int index,
                                                boolean isSelected,
                                                boolean cellHasFocus) {
    Component Ret = super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
    ABean TheBean = (ABean) value;
    setName (TheBean.getName());
    setIcon (new ImageIcon (TheBean.getIcon()));
    return Ret;
  }

}

/**
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class WindowCloser implements WindowListener {
  public static WindowCloser Closer = null;

  public WindowCloser () {
    Closer = this;
  }

  public void close (Window W) {
    W.addWindowListener (this);
  }

  public void windowClosing (WindowEvent e) {
    ((Window)e.getSource()).dispose();
  }

  public void windowClosed (WindowEvent e) {}
  public void windowOpened (WindowEvent e) {}
  public void windowActivated (WindowEvent e) {}
  public void windowDeactivated (WindowEvent e) {}
  public void windowIconified (WindowEvent e) {}
  public void windowDeiconified (WindowEvent e) {}

}

/**
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class ErrorLoadingBeanDialog extends JDialog implements ActionListener {
  private JButton OK_Button = new JButton ("OK");
  private Label Message = new Label ();

  public ErrorLoadingBeanDialog (Frame Parent, String BeanName) {
    super (Parent, "Error!", true);
    WindowCloser.Closer.close(this);

    BPanel MainPanel = new BPanel();
    String MessageText = "Error Loading Bean " + BeanName + " !";

    MainPanel.setInner (BPanel.NONE);
    MainPanel.setOuter (BPanel.NONE);
    MainPanel.setBorderWidth (5);

    MainPanel.setLayout (new BorderLayout());
    MainPanel.add (OK_Button, BorderLayout.SOUTH);
    Message.setText(MessageText);
    MainPanel.add (Message, BorderLayout.WEST);

    OK_Button.addActionListener (this);
    getContentPane().add (MainPanel);
    setSize (Misc.calcStringWidth (MessageText, null) + 30,100);
    setResizable (false);
    Misc.hardCenter(this);
  }

  public void actionPerformed (ActionEvent e) {
    setVisible (false);
  }
}


//------------------------------------------------------------------------------
// Begin of FocusRouter
//------------------------------------------------------------------------------

/**
  This class keeps track of which object have last
  gained focus - either the page list, or the named icons list.
  Used by PaletteConfigDialog.
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class FocusRouter implements FocusListener {
  /** The PaletteConfigDialog, which keeps track of focused objects.*/
  private PaletteConfigDialog Boss;

  /** The last focused object. */
  private Object Source;

  /**
    Constructs the object.
    @param Source The initially set focused object.
    @param Boss The main dialog.
  */
  public FocusRouter (Object Source, PaletteConfigDialog Boss) {
    this.Boss = Boss;
    this.Source = Source;
  }

  /** Does nothing. */
  public void focusLost (FocusEvent e) {}

  /**
    Updates the Source field, and updates the buttons of
    PaletteConfigDialog.
  */
  public void focusGained (FocusEvent e) {
    Object NewSource = e.getSource();
    if (!NewSource.equals(Source)) {
      Source = NewSource;
      Boss.updateButtons ();
    }
  }

  /** Returns the last object, which had gained focus. */
  public Object getSource () {
    return Source;
  }
}

//------------------------------------------------------------------------------
// End of FocusRouter
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Begin of BeanConfig
//------------------------------------------------------------------------------

/**
  This class represents the dialog, which is shown, when the
  user wants to configure a bean's properties.
  Used by PaletteConfigDialog.
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/

class BeanConfig extends JDialog implements ActionListener {
  /** The main dialog. */
  private PaletteConfigDialog Boss;

  /** The parent frame. */
  private Frame BigBoss;

  /** The TextField, containing the name of the bean. */
  private JTextField NameField;

  /** The TextField, containing the name of the icon file. */
  private JTextField IconField;

  /** The Browse command.*/
  private static final String BROWSE_CMD  = "Browse";

  /** The OK command.*/
  private static final String OK_CMD      = "OK";

  /** The Cancel command.*/
  private static final String CANCEL_CMD  = "Cancel";

  /** If the user exited, pressing OK or Cancel. */
  public boolean OK = false;

  /** The bean to be configured. */
  public ABean B;

  /** The new name of the bean. */
  public String NewName = "";

  /** The new name of the icon file. */
  public String NewIcon = "";

  /**
    Construcs the dialog.
    @param Boss The main frame.
    @param SmallBoss The main dialog.
    @param B The bean to be configured.
  */
  public BeanConfig (Frame Boss, PaletteConfigDialog SmallBoss, ABean B) {
    super (Boss, "Bean configuration", true);
    WindowCloser.Closer.close(this);

    int BoxWidth = 400;
    int BoxHeight = 200;
    BPanel MainPanel = new BPanel ("Bean Properties:");
    BPanel NamePanel = new BPanel ("Bean Name:");
    BPanel IconPanel = new BPanel ("Bean Icon:");
    BPanel ButtonsPanel = new BPanel ();
    JButton BrowseButton = new JButton (BROWSE_CMD);
    Button OKButton      = new Button (OK_CMD);
    Button CancelButton  = new Button (CANCEL_CMD);

    setSize (BoxWidth, BoxHeight);
    Misc.hardCenter (this);

    this.Boss = SmallBoss;
    this.BigBoss = Boss;
    this.B = B;
    BrowseButton.addActionListener (this);
    OKButton.addActionListener (this);
    CancelButton.addActionListener (this);

    MainPanel.setLayout (new BorderLayout (5, 5));

    NameField = new JTextField (B.getName());
    NamePanel.setLayout (new BorderLayout());
    NamePanel.add (NameField, BorderLayout.CENTER);
    MainPanel.add (NamePanel,BorderLayout.NORTH);

    IconField = new JTextField (B.getIconFile());
    IconPanel.setLayout (new BorderLayout(10, 10));
    IconPanel.add (IconField, BorderLayout.CENTER);
    IconPanel.add (BrowseButton, BorderLayout.EAST);
    MainPanel.add (IconPanel,BorderLayout.SOUTH);

    ButtonsPanel.setLayout (new GridLayout (1, 1, 5, 5));
    ButtonsPanel.add (OKButton);
    ButtonsPanel.add (CancelButton);

    getContentPane().setLayout (new BorderLayout());
    getContentPane().add (MainPanel, BorderLayout.NORTH);
    getContentPane().add (ButtonsPanel, BorderLayout.SOUTH);
  }

  /** Called when one of the buttons is pessed.*/
  public void actionPerformed (ActionEvent e) {
    String CMD = e.getActionCommand ();

    if (CMD == BROWSE_CMD) {
      FileDialog LoadIcon = new FileDialog (BigBoss);
      LoadIcon.setFile ("*.gif");
      LoadIcon.setVisible (true);
      if ((LoadIcon.getDirectory() != null) && (LoadIcon.getFile() != null))
        IconField.setText(LoadIcon.getDirectory() + LoadIcon.getFile());
      toFront();
      return;
    }

    OK = ( CMD == OK_CMD);
    NewName = NameField.getText();
    NewIcon = IconField.getText();
    Boss.configBean (this);
  }
}

//------------------------------------------------------------------------------
// End of BeanConfig
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Begin of RemoveConfirm
//------------------------------------------------------------------------------

/**
  This class represens the dialog, shown before the user removes
  a bean or a page.
  Used by PaletteConfigDialog.
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class RemoveConfirm extends JDialog implements ActionListener {

  /** If the user exited, pressing OK or Cancel. */
  public boolean OK = false;
  /** If the user wishes to remove a page or a bean.*/
  private boolean RemovingBean;

  /** The main dialog. */
  private PaletteConfigDialog Boss;

  /** The OK command. */
  private final String OK_CMD      = "OK";

  /** The Cancel command. */
  private final String CANCEL_CMD  = "Cancel";

  /** The Cancel button. */
  private Button CancelButton      = new Button (CANCEL_CMD);

  /** The OK button. */
  private Button OKButton          = new Button (OK_CMD);

  /**
    Constructs the dialog.
    @param Boss The owner frame.
    @param SmallBoss The main dialog.
    @param Name The name of the object being removed.
    @param ABean If the object being removed is a bean or a page.
  */
  public RemoveConfirm (Frame Boss, PaletteConfigDialog SmallBoss,
                        String Name, boolean ABean) {
    super(Boss, true);
    WindowCloser.Closer.close(this);

    int BoxWidth = 300;
    int BoxHeight = 150;
    int NameWidth = Misc.calcStringWidth (Name, null) + 50;

    BPanel ButtonsPanel = new BPanel();
    BPanel MainPanel    = new BPanel();
    BPanel MessagePanel = new BPanel("About to remove " + (ABean ? "bean:":"page:"));

    RemovingBean = ABean;
    MainPanel.setLayout (new BorderLayout (5, 5));
    MainPanel.setBorderWidth (3);
    MainPanel.setInner (BPanel.NONE);
    MainPanel.setOuter (BPanel.NONE);

    if (!RemovingBean)
      setTitle ("Removing Page");
    else
      setTitle ("Removing Bean");

    this.Boss = SmallBoss;

    MessagePanel.add (new  Label (Name));
    MainPanel.add (MessagePanel, BorderLayout.CENTER);

    ButtonsPanel.setLayout (new GridLayout (1, 2, 7, 7));
    ButtonsPanel.add (OKButton);
    ButtonsPanel.add (CancelButton);

    OKButton.addActionListener (this);
    CancelButton.addActionListener (this);

    MainPanel.add (ButtonsPanel, BorderLayout.SOUTH);
    getContentPane().setLayout (new BorderLayout());
    getContentPane().add (MainPanel, BorderLayout.CENTER);
    setSize (BoxWidth > NameWidth ? BoxWidth : NameWidth,
             BoxHeight);
    Misc.hardCenter(this);
  }

  /** Called when a button is pressed. */
  public void actionPerformed (ActionEvent e) {
    String CMD = e.getActionCommand();
    if (CMD == OK_CMD)
      OK = true;
    else
      OK = false;

    if (RemovingBean)
      Boss.removeBean (this);
    else
      Boss.removePage (this);
  }
}

//------------------------------------------------------------------------------
// Begin of RemoveConfirm
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Begin of SetPage
//------------------------------------------------------------------------------

/**
  This class represents the dialog, shown when the user wishes
  to add or configure a page.
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class SetPage extends JDialog implements ActionListener {

  /** The TextField, containg the name of the page. */
  private JTextField NewPage;

  /** The main dialog. */
  private PaletteConfigDialog Boss;

  /** If the user exited, pressing OK or Cancel. */
  public boolean OK = false;

  /** The new name of the page. */
  public String NewPageName;

  /** The page, being configured, or null if a new page is being created. */
  public PalettePage ThePage;

  /** The OK command. */
  private final String OK_CMD      = "OK";

  /** The Cancel command. */
  private final String CANCEL_CMD  = "Cancel";

  /**
    Construcs the dialog.
    @param Boss The owner frame.
    @param SmallBoss The main dialog.
    @param PP The page, being configured, or null if new page is
           being created.
  */
  public SetPage (Frame Boss, PaletteConfigDialog SmallBoss, PalettePage PP) {
    super(Boss);
    WindowCloser.Closer.close(this);

    int BoxWidth = 300;
    int BoxHeight = 150;
    BPanel ConfigPanel  = new BPanel("Page Name:");
    BPanel MainPanel    = new BPanel();
    BPanel ButtonsPanel = new BPanel();
    Button OKButton     = new Button (OK_CMD);
    Button CancelButton = new Button (CANCEL_CMD);

    if (PP == null)
      setTitle ("New Page");
    else
      setTitle("Page Configuration");

    MainPanel.setLayout (new BorderLayout (5, 5));
    MainPanel.setInner (BPanel.NONE);
    MainPanel.setOuter (BPanel.NONE);
    MainPanel.setBorderWidth (3);

    setModal (true);
    getContentPane().setLayout (new BorderLayout());

    this.Boss = SmallBoss;

    if (PP == null)
      NewPage = new JTextField ("New Page");
    else
      NewPage = new JTextField (PP.getName());

    ConfigPanel.setLayout (new BorderLayout());
    ConfigPanel.add (NewPage, BorderLayout.CENTER);

    ButtonsPanel.setLayout (new GridLayout (1, 2, 7, 7));
    ButtonsPanel.add (OKButton);
    ButtonsPanel.add (CancelButton);

    OKButton.addActionListener (this);
    CancelButton.addActionListener (this);

    MainPanel.add (ConfigPanel, BorderLayout.NORTH);
    MainPanel.add (ButtonsPanel, BorderLayout.SOUTH);

    getContentPane().add (MainPanel, BorderLayout.CENTER);
    setSize (BoxWidth, BoxHeight);
    Misc.hardCenter (this);
    ThePage = PP;
  }

  /**
    Called, when a button is pressed.
  */
  public void actionPerformed (ActionEvent e) {
    String CMD = e.getActionCommand();
    if (CMD == OK_CMD)
      OK = true;
    else
      OK = false;

    NewPageName = NewPage.getText();
    Boss.pageSet (this);
  }
}

//------------------------------------------------------------------------------
// End of SetPage
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Begin of PaletteConfigDialog
//------------------------------------------------------------------------------

/**
  This class represents the main dialog, which configures
  the palette.
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
public class PaletteConfigDialog extends JDialog
                                 implements ListSelectionListener,
                                            ItemListener,
                                            ActionListener,
                                            WindowListener {
  /** The JTabbedPane shownig the pages. */
  private JTabbedPane TSP = new JTabbedPane ();

  /** The owner frame. */
  private Frame ParentFrame;

  /** The palette to be configured. */
  private PaletteOptions PM       = null;

  /** The list, showing the beans in the selected page. */
  private JList NIL   = null;

  /** The list, showing the pages in the palette. */
  private JList PageList          = new JList();
  private JComboBox ArchPageChoice= new JComboBox();
  private JComboBox PackPageChoice= new JComboBox();

  /**
    The object, which keeps track if the PageList or NIL is
    currently focused, so the dialog could know what the
    buttons actually do.
  */
  private FocusRouter FR      = new FocusRouter (PageList, this);

  private static final String HELP_CMD          = "Help";
  private static final String OK_CMD            = "OK";
  private static final String CANCEL_CMD        = "Cancel";
  private static final String ADD_CMD           = "Add";
  private static final String REMOVE_CMD        = "Remove";
  private static final String PROPERTIES_CMD    = "Properties";
  private static final String MOVE_UP_CMD       = "Move Up";
  private static final String MOVE_DOWN_CMD     = "Move Down";
  private static final String BROWSE_ARCH_CMD   = "BrowseArch";
  private static final String INSTALL_ARCH_CMD  = "InstallFromArch";
  private static final String INVERT_ARCH_CMD   = "InvertArchSelection";
  private static final String BROWSE_PACK_CMD   = "BrowsePack";
  private static final String INSTALL_PACK_CMD  = "InstallFromPackage";
  private static final String INVERT_PACK_CMD   = "InvertPackSelection";
  private static final String BROWSE_LABEL      = "Browse";
  private static final String INSTALL_LABEL     = "Install";
  private static final String INVERT_LABEL      = "Invert Selection";
  private static final String BEANINFO_ONLY_CMD = "Only display classes that have BeanInfo";

  private Button OKButton         = new Button (OK_CMD);
  private Button CancelButton     = new Button (CANCEL_CMD);
  private Button HelpButton       = new Button (HELP_CMD);
  private JButton AddButton        = new JButton (ADD_CMD);
  private JButton RemoveButton     = new JButton (REMOVE_CMD);
  private JButton PropertiesButton = new JButton (PROPERTIES_CMD);
  private JButton MoveUpButton     = new JButton (MOVE_UP_CMD);
  private JButton MoveDownButton   = new JButton (MOVE_DOWN_CMD);

  private JButton InvertArchButton        = new JButton (INVERT_LABEL);
  private JButton BrowseForArchButton     = new JButton (BROWSE_LABEL);
  private JButton InstallFromArchButton   = new JButton (INSTALL_LABEL);
  private JTextField ZipFileName          = new JTextField ();
  private JScrollPane ArchBeanScroll      = new JScrollPane ();
  private JList ArchBeanList              = new JList ();
  private JCheckBox ArchBeanInfoBeansOnly = new JCheckBox (BEANINFO_ONLY_CMD);

  private JButton InvertPackButton        = new JButton (INVERT_LABEL);
  private JButton BrowseForPackButton     = new JButton (BROWSE_LABEL);
  private JButton InstallFromPackButton   = new JButton (INSTALL_LABEL);
  private JTextField PackageName          = new JTextField ();
  private JScrollPane PackBeanScroll      = new JScrollPane ();
  private JList PackBeanList              = new JList ();
  private JCheckBox PackBeanInfoBeansOnly = new JCheckBox (BEANINFO_ONLY_CMD);

  private BPanel BeansArchPanel = new BPanel ("JavaBeans found in archive:");
  private BPanel BeansPackPanel = new BPanel ("JavaBeans found in package:");

  private BPanel PagesMainPanel   = new BPanel ();
  private BPanel AddArchMainPanel = new BPanel ();
  private BPanel AddPackMainPanel = new BPanel ();

  private BPanel PagesPanel       = new BPanel ("Pages");
  private BPanel BeansPanel       = new BPanel ("Components:");

  private JScrollPane SP          = new JScrollPane ();

  /**
    Construcs the dialog.
    @param frame The owner frame.
    @param title The title of the dialog.
    @param modal If the dialog should be modal or not.
  */
  public PaletteConfigDialog (Frame frame, String title, boolean modal) {
    super(frame, title, modal);
    ParentFrame = frame;
    init();
    getCurrentOptions();
  }

  public void valueChanged (ListSelectionEvent e) {
    if (e.getSource().equals(ArchBeanList)) {
      if ((ArchBeanList.getSelectedIndex() >= 0) ||
          (ArchBeanList.getSelectedIndices().length > 0))
        InstallFromArchButton.setEnabled(true);
      else
        InstallFromArchButton.setEnabled(false);
    }

    if (e.getSource().equals(PackBeanList)) {
      if ((PackBeanList.getSelectedIndex() >= 0) ||
          (PackBeanList.getSelectedIndices().length > 0))
        InstallFromPackButton.setEnabled(true);
      else
        InstallFromPackButton.setEnabled(false);
    }

    if (e.getSource().equals(PageList)) {
      processPageSelectionChange (PageList.getSelectedIndex());
      updateButtons ();
    }

    if (e.getSource().equals(NIL)) {
      updateButtons ();
    }
  }

  private void setSource (PalettePage PP) {
    if (PP != null)
      NIL.setListData     (PP.getItems());
    else
      NIL.setListData     (new Object[0]);
    SP.invalidate       ();
    SP.validate         ();
    SP.doLayout         ();
    NIL.repaint         ();
  }

  /**
    This method is called when the user selects a page from
    the list. It updates the list of beans.
  */
  private void processPageSelectionChange (int i) {
    PalettePage P = PM.getPage (i);
    setSource (P);
  }

  private void movePage (int DeltaY) {
    int PalettePageNo = PageList.getSelectedIndex();
    int i;
    PM.movePage (PalettePageNo, PalettePageNo + DeltaY);

    PageList.setListData(PM.getPages());
    PageList.setSelectedIndex (PalettePageNo + DeltaY);

    ArchPageChoice.removeAllItems ();
    PackPageChoice.removeAllItems ();
    for (i = 0; i < PageList.getModel().getSize(); i++) {
      ArchPageChoice.addItem (PM.getPage(i));
      PackPageChoice.addItem (PM.getPage(i));
    }
    updateButtons();
  }

  /**
    This method is called when the user presses Move Up or
    Move Down button of the dialog, while the bean list is
    focused. It preforms the actual reorder of the beans in
    the selected page of the palette.
  */
  private void moveBean (int DeltaY) {
    PalettePage PP = PM.getPage(PageList.getSelectedIndex());
    int BeanPos = NIL.getSelectedIndex();
    PP.moveBean (BeanPos, BeanPos + DeltaY);
    setSource (PP);
    NIL.setSelectedIndex (BeanPos + DeltaY);
  }


  private void extractClassFileNames (File ThePackage, Vector Names, Vector BeanInfos) {
    String [] FileNames = ThePackage.list();
    int i;
    if (FileNames == null)
      return;

    for (i = 0; i < FileNames.length; i++) {
      File F = new File (ThePackage + FileNames[i]);
      if (F.isFile()) {
        if (F.getName().endsWith (".class")) {
          String FullName = ThePackage + F.getName();
          FullName = FullName.substring (0, FullName.length() - ".class".length());
          FullName = FullName.substring (3, FullName.length());
          FullName = FullName.replace ('\\', '.');
          Names.addElement (FullName);
          if (FullName.endsWith ("BeanInfo"))
            BeanInfos.addElement (FullName);
        }
      } else
        extractClassFileNames (new File (ThePackage + FileNames[i] + "\\"), Names, BeanInfos);
    }
  }

  private void reorderPackBeanList () {
    File ThePackage = new File (PackageName.getText());
    Vector BeanNames = new Vector();
    Vector ClassNames = new Vector();
    int NamesCount;
    int i;

    extractClassFileNames (ThePackage, BeanNames, ClassNames);

    NamesCount = BeanNames.size();

    PackBeanList.setCursor (new Cursor(Cursor.WAIT_CURSOR));

    if (PackBeanInfoBeansOnly.isSelected()) {
      NamesCount = BeanNames.size();
      for (i = 0; i < NamesCount; i++) {
        String CurrentName = (String)BeanNames.elementAt(i);
        CurrentName = CurrentName.substring (CurrentName.lastIndexOf('.') + 1,
                                 CurrentName.length());
        if (ClassNames.indexOf (CurrentName + "BeanInfo") == -1) {
          BeanNames.removeElementAt(i);
          i--;NamesCount--;
        }
      }
    }

    PackBeanList.setListData (BeanNames);
    PackBeanList.setCursor (new Cursor(Cursor.DEFAULT_CURSOR));
    InstallFromPackButton.setEnabled (false);
    if (PackBeanList.getModel().getSize() > 0)
      InvertPackButton.setEnabled (true);
    else
      InvertPackButton.setEnabled (false);

    PackBeanScroll.invalidate();
    PackBeanScroll.validate();
  }

  private void reorderArchBeanList () {
    Enumeration Entries = null;
    Vector BeanNames = new Vector();
    Vector ClassNames = new Vector();
    int i;
    int NamesCount;

    ArchBeanList.setCursor (new Cursor(Cursor.WAIT_CURSOR));

    try {
      ZipFile TheZip = new ZipFile (ZipFileName.getText());
      Entries = TheZip.entries();
      for (;Entries.hasMoreElements();) {
        String S = Entries.nextElement().toString();
        if (!S.endsWith(".class"))
          continue;
        S = S.replace('/', '.');
        S = S.substring (0, S.length() - ".class".length());

        if (ArchBeanInfoBeansOnly.isSelected()) {
          if (S.endsWith("BeanInfo")) {
            String ClassName = S.substring (S.lastIndexOf('.') + 1, S.length());
            ClassNames.addElement (ClassName);
          }
          BeanNames.addElement (S);
        }
        else
          BeanNames.addElement (S);
      }
      ArchBeanList.setListData (BeanNames);
    } catch (IOException ee) {
      InstallFromArchButton.setEnabled(false);
      ArchBeanList.setCursor (new Cursor(Cursor.DEFAULT_CURSOR));
      if (ArchBeanList.getModel().getSize() > 0)
        InvertArchButton.setEnabled (true);
      else
        InvertArchButton.setEnabled (false);
      ArchBeanScroll.invalidate();
      ArchBeanScroll.validate();
      return;
    }


    if (ArchBeanInfoBeansOnly.isSelected()) {
      NamesCount = BeanNames.size();
      for (i = 0; i < NamesCount; i++) {
        String CurrentName = (String)BeanNames.elementAt(i);
        CurrentName = CurrentName.substring (CurrentName.lastIndexOf('.') + 1,
                                 CurrentName.length());
        if (ClassNames.indexOf (CurrentName + "BeanInfo") == -1) {
          BeanNames.removeElementAt(i);
          i--;NamesCount--;
        }
      }
    }

    ArchBeanList.setListData (BeanNames);
    ArchBeanList.setCursor (new Cursor(Cursor.DEFAULT_CURSOR));

    InstallFromArchButton.setEnabled(false);
    if (ArchBeanList.getModel().getSize() > 0)
      InvertArchButton.setEnabled (true);
    else
      InvertArchButton.setEnabled (false);
    ArchBeanScroll.invalidate();
    ArchBeanScroll.validate();
  }


  public PaletteOptions getPalette () {
    return PM;
  }

  /** Sets the palette to be configured. */
  public void setPalette (PaletteOptions PM) {
    int i,j;
    this.PM = PM;
    j = PM.getPageCount();

    ArchPageChoice.removeAllItems();
    PackPageChoice.removeAllItems();

    for (i = 0; i < j; i++) {
      ArchPageChoice.addItem (PM.getPage(i).getName());
      PackPageChoice.addItem (PM.getPage(i).getName());
    }
    PageList.setListData (PM.getPages());

    if (j > 0) {
      PageList.setSelectedIndex (0);
      processPageSelectionChange(0);
    } else
      setSource (null);
  }

  /** Resizes and centers the dialog, and then shows it. */
  public void setVisible (boolean Visible) {
    if (Visible) {
      setSize (550, 400);
      Misc.hardCenter (this);
    }
    super.setVisible (Visible);
  }


  /** Initializes the dialog. */
  public void init() {
    BPanel ButtonsPanel       = new BPanel();

    BPanel ConfigButtonsPanel = new BPanel ();
    BPanel ConfigPanel        = new BPanel ();

    BPanel SourceArchPanel    = new BPanel ("Select a .JAR or .ZIP file:");
    BPanel DestArchPanel      = new BPanel ("Add to page:");
    BPanel SouthArchPanel     = new BPanel ();

    BPanel SourcePackPanel    = new BPanel ("Select a .CLASS file from package:");
    BPanel DestPackPanel      = new BPanel ("Add to page:");
    BPanel SouthPackPanel     = new BPanel ();

    new WindowCloser();
    WindowCloser.Closer.close(this);

    setResizable (true);
    getContentPane().setLayout (new BorderLayout ());

    ButtonsPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
    ButtonsPanel.add (OKButton);
    ButtonsPanel.add (CancelButton);
    ButtonsPanel.add (HelpButton);

    getContentPane().add (ButtonsPanel,BorderLayout.SOUTH);

    NIL = new JList ();
    NIL.setCellRenderer (new BeanListCellRenderer());
    NIL.addFocusListener (FR);
    NIL.addListSelectionListener (this);
    NIL.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);

    SP = new JScrollPane ();
    SP.setBackground (Color.gray);
    SP.getViewport().setView(NIL);
    BeansPanel.setLayout (new BorderLayout ());
    BeansPanel.add (SP, BorderLayout.CENTER);

    PageList.addListSelectionListener (this);
    PageList.addFocusListener (FR);
    PageList.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);


    PagesPanel.setLayout (new BorderLayout ());
    PagesPanel.add (PageList, BorderLayout.CENTER);

    ConfigPanel.setLayout (new BorderLayout (5, 5));
    ConfigPanel.setInner  (BPanel.NONE);
    ConfigPanel.setOuter  (BPanel.NONE);
    ConfigPanel.add (BeansPanel, BorderLayout.CENTER);
    ConfigPanel.add (PagesPanel, BorderLayout.WEST);

    ConfigButtonsPanel.setLayout(new GridLayout(1, 1, 5, 5));

    OKButton.addActionListener (this);
    CancelButton.addActionListener (this);
    HelpButton.addActionListener (this);

    AddButton.addActionListener (this);
    RemoveButton.addActionListener (this);
    PropertiesButton.addActionListener (this);
    MoveUpButton.addActionListener (this);
    MoveDownButton.addActionListener (this);

    ConfigButtonsPanel.add (AddButton);
    ConfigButtonsPanel.add (RemoveButton);
    ConfigButtonsPanel.add (PropertiesButton);
    ConfigButtonsPanel.add (MoveUpButton);
    ConfigButtonsPanel.add (MoveDownButton);

    PagesMainPanel.setLayout (new BorderLayout(5, 5));
    PagesMainPanel.setInner (BPanel.NONE);
    PagesMainPanel.setOuter (BPanel.NONE);
    PagesMainPanel.add (ConfigButtonsPanel, BorderLayout.SOUTH);
    PagesMainPanel.add (ConfigPanel, BorderLayout.CENTER);

    TSP.addTab ("Pages", PagesMainPanel);

    AddArchMainPanel.setInner(BPanel.NONE);
    AddArchMainPanel.setOuter(BPanel.NONE);
    AddArchMainPanel.setLayout (new BorderLayout(5, 5));

    SourceArchPanel.setLayout (new BorderLayout(10, 10));
    SourceArchPanel.add (ZipFileName, BorderLayout.CENTER);
    SourceArchPanel.add (BrowseForArchButton, BorderLayout.EAST);
    AddArchMainPanel.add (SourceArchPanel, BorderLayout.NORTH);

    BeansArchPanel.setLayout (new BorderLayout(5, 5));
    ArchBeanScroll.getViewport().setView(ArchBeanList);
    BeansArchPanel.add (ArchBeanScroll, BorderLayout.CENTER);
    BeansArchPanel.add (InvertArchButton , BorderLayout.SOUTH);
    AddArchMainPanel.add (BeansArchPanel, BorderLayout.CENTER);

    SouthArchPanel.setLayout (new BorderLayout(5, 5));
    SouthArchPanel.setInner (BPanel.NONE);
    SouthArchPanel.setOuter (BPanel.NONE);

    DestArchPanel.setLayout (new BorderLayout(10, 10));
    DestArchPanel.add (InstallFromArchButton , BorderLayout.EAST);
    DestArchPanel.add (ArchPageChoice , BorderLayout.CENTER);

    SouthArchPanel.add (ArchBeanInfoBeansOnly , BorderLayout.SOUTH);
    SouthArchPanel.add (DestArchPanel , BorderLayout.CENTER);

    AddArchMainPanel.add (SouthArchPanel, BorderLayout.SOUTH);

    BrowseForArchButton.addActionListener   (this);
    InstallFromArchButton.addActionListener (this);
    InvertArchButton.addActionListener      (this);
    BrowseForArchButton.setActionCommand   (BROWSE_ARCH_CMD);
    InstallFromArchButton.setActionCommand (INSTALL_ARCH_CMD);
    InvertArchButton.setActionCommand      (INVERT_ARCH_CMD);

    ArchBeanList.addListSelectionListener (this);
    ArchBeanList.setSelectionMode         (ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    ArchBeanInfoBeansOnly.addItemListener (this);
    InstallFromArchButton.setEnabled      (false);
    InvertArchButton.setEnabled           (false);

    TSP.addTab ("Add from Archive", AddArchMainPanel);

    AddPackMainPanel.setInner(BPanel.NONE);
    AddPackMainPanel.setOuter(BPanel.NONE);
    AddPackMainPanel.setLayout (new BorderLayout(5, 5));

    SourcePackPanel.setLayout (new BorderLayout(10, 10));
    SourcePackPanel.add (PackageName, BorderLayout.CENTER);
    SourcePackPanel.add (BrowseForPackButton, BorderLayout.EAST);
    AddPackMainPanel.add (SourcePackPanel, BorderLayout.NORTH);

    BeansPackPanel.setLayout(new BorderLayout(5, 5));
    PackBeanScroll.getViewport().setView (PackBeanList);
    BeansPackPanel.add (PackBeanScroll, BorderLayout.CENTER);
    BeansPackPanel.add (InvertPackButton , BorderLayout.SOUTH);
    AddPackMainPanel.add (BeansPackPanel, BorderLayout.CENTER);

    SouthPackPanel.setLayout (new BorderLayout(5, 5));
    SouthPackPanel.setInner (BPanel.NONE);
    SouthPackPanel.setOuter (BPanel.NONE);

    DestPackPanel.setLayout (new BorderLayout(10, 10));
    DestPackPanel.add (InstallFromPackButton , BorderLayout.EAST);
    DestPackPanel.add (PackPageChoice , BorderLayout.CENTER);

    SouthPackPanel.add (PackBeanInfoBeansOnly , BorderLayout.SOUTH);
    SouthPackPanel.add (DestPackPanel , BorderLayout.CENTER);

    AddPackMainPanel.add (SouthPackPanel, BorderLayout.SOUTH);

    BrowseForPackButton.addActionListener (this);
    InstallFromPackButton.addActionListener (this);
    InvertPackButton.addActionListener (this);
    BrowseForPackButton.setActionCommand (BROWSE_PACK_CMD);
    InstallFromPackButton.setActionCommand (INSTALL_PACK_CMD);
    InvertPackButton.setActionCommand (INVERT_PACK_CMD);

    PackBeanList.addListSelectionListener (this);
    PackBeanList.setSelectionMode (ListSelectionModel.MULTIPLE_INTERVAL_SELECTION );
    PackBeanInfoBeansOnly.addItemListener (this);
    InstallFromPackButton.setEnabled(false);
    InvertPackButton.setEnabled(false);

    TSP.addTab ("Add from Package",AddPackMainPanel);
    TSP.setSelectedIndex(0);

    getContentPane().add(TSP, BorderLayout.CENTER);
    addWindowListener (this);
  }

  public void getCurrentOptions() {
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    PaletteOptions  POptions = (PaletteOptions)(AOptions.getOptions(PaletteOptions.PaletteManOptionsName));
    if (POptions != null)
      setPalette (POptions);
  }

  public void setNewOptions() {
    PaletteOptions  POptions = getPalette();
    OptionsMngr AOptions = Globals.IEngine.getOptions();
    AOptions.setOptions(POptions);
  }


  /**
    This method is called by the RemoveConfirm dialog on exit,
    if the user wishes to remove a bean. It removes or not the
    selected bean, depending on how the user exited the
    RemoveConfirm dialog.
  */
  public void removeBean (RemoveConfirm RP) {
    if (RP.OK) {
      int SelectedBeanNo = NIL.getSelectedIndex();
      PalettePage PP = PM.getPage (PageList.getSelectedIndex());
      PP.removeBean (SelectedBeanNo);
      setSource (PP);
      NIL.setSelectedIndex (SelectedBeanNo);
    }
    updateButtons();
    RP.dispose();
    toFront();
  }

  /**
    This method is called by the RemoveConfirm dialog on exit,
    if the user wishes to remove a page. It removes or not the
    selected page, depending on how the user exited the
    RemoveConfirm dialog.
  */
  public void removePage (RemoveConfirm RP) {
    if (RP.OK) {
      PM.removePage (PageList.getSelectedIndex());
      setPalette (PM);
    }
    RP.dispose();
    toFront();
  }

  /**
    This method is called by the SetPage dialog on exit.
    Configures the selected page, or creates a new one, if
    the user exited the SetPage dialog with the OK button.
    If the user exited the SetPage dialog with the Cancel button,
    does nothing.
  */
  public void pageSet (SetPage NewSettings) {
    if (!NewSettings.OK) {
      NewSettings.dispose();
      toFront();
      return;
    }

    if (NewSettings.ThePage == null) {
      PalettePage PP = new PalettePage (PM);
      PP.setName (NewSettings.NewPageName);
      PM.addPage (PP);
    } else {
      PalettePage PP = PM.getPage(PageList.getSelectedIndex());
      PP.setName (NewSettings.NewPageName);
    }

    setPalette (PM);
    NewSettings.dispose();
    toFront();
  }

  /**
    This method is called by the BeanConfig dialog on exit.
    It configures the selected bean or does nothing, depending
    on how the user exited the dialog.
  */
  public void configBean (BeanConfig CB) {
    if (CB.OK) {
      CB.B.setName (CB.NewName);
      if (CB.NewIcon != null) {
        CB.B.setIcon (CB.NewIcon);
      }
    }
    CB.dispose();
    setSource (PM.getPage(PageList.getSelectedIndex()));
    toFront();
  }

  /**
    Updates the buttons of the dialog.
    This method is called after every change of selection -
    either in the page list or in the bean list.
    The buttons are updated, according to the currently
    selected list and item.
  */
  public void updateButtons () {
    if (FR.getSource().equals (PageList)) {
      int Selected = PageList.getSelectedIndex();
      int Last = PageList.getModel().getSize() - 1;

      AddButton.setEnabled (true);

      if (Selected <= 0)
        MoveUpButton.setEnabled (false);
      else
        MoveUpButton.setEnabled (true);

      if ((Last == -1) || (Selected == -1) || (Selected == Last))
        MoveDownButton.setEnabled (false);
      else
        MoveDownButton.setEnabled (true);

      if ((Last == -1) || (Selected == -1)) {
        PropertiesButton.setEnabled (false);
        RemoveButton.setEnabled (false);
      }
      else {
        PropertiesButton.setEnabled (true);
        RemoveButton.setEnabled (true);
      }
    }
    else {
      int Selected = NIL.getSelectedIndex();
      int Last = NIL.getModel().getSize() - 1;

      AddButton.setEnabled (false);

      if ((Selected <= 0)|| (Last < 0))
        MoveUpButton.setEnabled (false);
      else
        MoveUpButton.setEnabled (true);

      if ((Selected == Last) || (Selected < 0) || (Last < 0))
        MoveDownButton.setEnabled (false);
      else
        MoveDownButton.setEnabled (true);

      if ((Last == -1) || (Selected == -1)) {
        PropertiesButton.setEnabled (false);
        RemoveButton.setEnabled (false);
      }
      else {
        PropertiesButton.setEnabled (true);
        RemoveButton.setEnabled (true);
      }
    }
  }

  public void doLayout () {
    super.doLayout ();
    OKButton.requestFocus();
  }

//=============== WindoListener implementation ===================
  public void windowActivated   (WindowEvent e) {
    OKButton.requestFocus();
  }
  public void windowDeactivated (WindowEvent e) {}
  public void windowIconified   (WindowEvent e) {}
  public void windowDeiconified (WindowEvent e) {}
  public void windowClosed      (WindowEvent e) {}
  public void windowClosing     (WindowEvent e) {}
  public void windowOpened      (WindowEvent e) {}
//=============== ActionListener implementation ==================
  /**
    This method is called when the user presses one of
    the buttons of the dialog. It resolves the command
    and decides what action to be preformed.
  */
  public void actionPerformed (ActionEvent e) {
    String CMD = e.getActionCommand();

    if (CMD == OK_CMD){
      setNewOptions();
      dispose();
    }

    if (CMD == CANCEL_CMD)
      dispose();

    if (CMD == HELP_CMD) {}

    if (CMD == ADD_CMD) {
      SetPage Add= new SetPage (ParentFrame, this, null);
      Add.setVisible (true);
    }

    if (CMD == REMOVE_CMD) {
      if (FR.getSource().equals(PageList)) {
        String CurrentPageName = PM.getPage (PageList.getSelectedIndex()).getName();
        RemoveConfirm Remove = new RemoveConfirm (ParentFrame, this, CurrentPageName, false);
        Remove.setVisible (true);
      }
      else {
        PalettePage PP = PM.getPage (PageList.getSelectedIndex());
        String CurrentBeanName = PP.getBean (NIL.getSelectedIndex()).getName();
        RemoveConfirm Remove = new RemoveConfirm (ParentFrame, this, CurrentBeanName, true);
        Remove.setVisible (true);
      }
    }

    if (CMD == PROPERTIES_CMD) {
      if (FR.getSource().equals(PageList)) {
        PalettePage PP = PM.getPage(PageList.getSelectedIndex());
        SetPage Add= new SetPage (ParentFrame, this, PP);
        Add.setVisible (true);
      }
      else {
        int SelectedPage = PageList.getSelectedIndex();
        int SelectedBean = NIL.getSelectedIndex();
        ABean B = PM.getPage(SelectedPage).getBean(SelectedBean);
        BeanConfig BC = new BeanConfig (ParentFrame, this, B);
        BC.setVisible (true);
      }
    }

    if (CMD == MOVE_UP_CMD) {
      if (FR.getSource().equals(PageList))
        movePage (-1);
      else
        moveBean (-1);
    }

    if (CMD == MOVE_DOWN_CMD) {
      if (FR.getSource().equals(PageList))
        movePage (+1);
      else
        moveBean (+1);
    }

    if ((CMD == INSTALL_ARCH_CMD) ||(CMD == INSTALL_PACK_CMD)) {
      PalettePage PP;
      ABean NewBean;
      String BeanFile = "";
      int [] Indexes;
      int i,j;
      JList ActiveList;
      JComboBox ActiveChoice;

      if (CMD == INSTALL_ARCH_CMD) {
        ActiveList = ArchBeanList;
        ActiveChoice = ArchPageChoice;
      } else {
        ActiveList = PackBeanList;
        ActiveChoice = PackPageChoice;
      }

      Indexes = ActiveList.getSelectedIndices();
      PP = PM.getPage(ActiveChoice.getSelectedIndex());

      if (Indexes.length + PP.getBeansCount() > PaletteOptions.MAX_PAGE_SIZE) {
        Misc.messageBox (new String [] {"OK"}, new String [] {"Too Many Beans In Page (Maximum Beans In Page is " + PaletteOptions.MAX_PAGE_SIZE + ")"}, 0);
        return;
      }

      try {
        for (i = 0; i < Indexes.length; i++) {
          BeanFile = (String) ActiveList.getModel().getElementAt (Indexes[i]);
          NewBean = new ABean (PP, BeanFile,
          Globals.IconsDir + "Beans\\DefaultBeanIcon.gif", true);
          PP.addBean (NewBean);
        }

        ActiveChoice.setVisible (false);
        i = ArchPageChoice.getSelectedIndex();
        j = PackPageChoice.getSelectedIndex();
        setPalette (PM);
        ArchPageChoice.setSelectedIndex(i);
        PackPageChoice.setSelectedIndex(j);
        ActiveChoice.setVisible (true);
      } catch (Exception ee) {
        ErrorLoadingBeanDialog Error = new ErrorLoadingBeanDialog(ParentFrame, BeanFile);
        Error.setVisible (true);
        Error.dispose();
      }
    }

    if ((CMD == INVERT_PACK_CMD) || (CMD == INVERT_ARCH_CMD)){
      JList TheList = null;
      JButton TheInstallButton = null;
      Container TheContainer;
      int ItemsCount;
      int i;

      if (CMD == INVERT_PACK_CMD) {
        TheList = PackBeanList;
        TheInstallButton = InstallFromPackButton;
      } else {
        TheList = ArchBeanList;
        TheInstallButton = InstallFromArchButton;
      }

      ItemsCount = TheList.getModel().getSize();
      TheContainer = TheList.getParent();

      TheContainer.setCursor (new Cursor(Cursor.WAIT_CURSOR));

      int [] SelectedItems    = TheList.getSelectedIndices();
      int ListSize = TheList.getModel().getSize();
      int [] NotSelectedItems = new int [ListSize - SelectedItems.length];
      int j,k;

      if (SelectedItems.length == 0)
        for (i = 0; i < ListSize; i++)
          NotSelectedItems [i] = i;
      else
        for (i = 0, j = 0, k = 0; i < ListSize; i++)
          if ((j == SelectedItems.length) || (SelectedItems[j] != i)) {
            NotSelectedItems [k] = i;
            k++;
          } else
            j++;

      TheList.clearSelection();
      TheList.setSelectedIndices(NotSelectedItems);

      if ((TheList.getSelectedIndex() != -1) ||
         (TheList.getSelectedIndices().length > 0))
        TheInstallButton.setEnabled(true);
      else
        TheInstallButton.setEnabled(false);

      TheContainer.setCursor (new Cursor(Cursor.DEFAULT_CURSOR));
    }


    if ((CMD == BROWSE_ARCH_CMD) || (CMD == BROWSE_PACK_CMD)){
      FileDialog Load = new FileDialog(ParentFrame);;
      if (CMD == BROWSE_ARCH_CMD)
        Load.setFile ("*.jar;*.zip");
      else
        Load.setFile ("*.class");

      Load.setVisible (true);
      if ((Load.getDirectory() != null) && (Load.getFile() != null)) {
        if (CMD == BROWSE_ARCH_CMD) {
          ZipFileName.setText (Load.getDirectory() + Load.getFile());
          reorderArchBeanList();
        } else {
          PackageName.setText (Load.getDirectory());
          reorderPackBeanList();
        }
      }
      Load.dispose();
    }
  }
//===================== ItemListener implementation ==============
  /**
    Thic method is called when the user selects a page or a bean
    from the list. It updates the buttons of the dialog and the
    list of beans, if the user have selected a page.
  */

  public void itemStateChanged (ItemEvent e) {
    if (e.getSource().equals(ArchBeanInfoBeansOnly))
      reorderArchBeanList();

    if (e.getSource().equals(PackBeanInfoBeansOnly)) {
      reorderPackBeanList();
    }
  }
}

//------------------------------------------------------------------------------
// End of PaletteConfigDialog
//------------------------------------------------------------------------------
