/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.gui.beans;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import org.freebuilder.gui.ideoptions.*;

/** This is a class that represents a speed button.
  * @author	<a href=mailto:wreck@nat.bg> Peter Radkov </a>
    @version 0.7, 07-01-98
*/

public class SpeedButton extends IconCanvas {
  private static final int PRESSED = 1000;
  private static final int ENTERED = 1001;
  private static final int EXITED  = 1002;
  public static final String PRESSED_CMD = "SPEED_BUTTON_PRESSED";

  private boolean Released       = true;
  private boolean CanBePressed   = true;
  private boolean AlwaysOn       = false;
  private Vector  Listeners     = new Vector();

  /**
    Initializes the object.
  */
  public SpeedButton (Image I, int No, int States) {
    super (I, No, States);
  }

  /**
    Initializes the object.
  */
  public SpeedButton (Image I, int No) {
    super (I, No, 3);
  }

  /**
    Notifies the listeners that an action occured.
  */
  private void notifyListeners (int WhatHappened) {
    int ListenersCount = Listeners.size();
    int i;
    if (Disabled)
      return;

    for (i = 0; i < ListenersCount; i ++) {
      SpeedListener CurrentListener = (SpeedListener) Listeners.elementAt (i);
      switch (WhatHappened) {
        case ENTERED: CurrentListener.speedEntered (this);
          break;
        case EXITED: CurrentListener.speedExited (this);
          break;
        case PRESSED:
          ActionEvent NewEvent = new ActionEvent (this, No, PRESSED_CMD);
          CurrentListener.actionPerformed (NewEvent);
          break;
      }
    }
  }

  /**
    Adds an action listener to the list of listeners
    for notification.
  */
  public void addActionListener (SpeedListener NewListener) {
    Listeners.addElement (NewListener);
  }
  /**
    Removes an action listener from the list of listeners
    for notification.
  */
  public void removeActionListener (SpeedListener OldListener) {
    Listeners.removeElement (OldListener);
  }

  public void mouseReleased (MouseEvent e) {
    super.mouseReleased (e);
    if (!CanBePressed)
      deselect();
    Released = true;
    if (IsMouseInside)
      notifyListeners (PRESSED);
  }

  /**
    Indicates if the button can remain pressed.
  */
  public void setAlwaysOn (boolean AlwaysOn) {
    this.AlwaysOn = AlwaysOn;
    if (isShowing())
      repaint();
  }

  public void canBePressed (boolean CanIt) {
    CanBePressed = CanIt;
  }

  public void mouseEntered (MouseEvent e) {
    super.mouseEntered (e);
    if ((!Disabled) && (States > 2))
      RealImagePos = (IconWidth - 4) * 2;
    if (!Released)
      select();
    else
      if (isShowing())
        repaint();
    notifyListeners (ENTERED);
  }

  public void mouseExited (MouseEvent e) {
    super.mouseExited (e);
    if (!Disabled)
      RealImagePos = 0;
    if (Selected) {
      if (!CanBePressed)
        deselect();
    } else
      if (isShowing())
        repaint();
    notifyListeners (EXITED);
  }

  public void mouseDragged (MouseEvent e) {
    super.mouseDragged(e);

    if (Selected) {
      Rectangle R = getBounds();
      Point P = getLocationOnScreen();
      int X = e.getX();
      int Y = e.getY();

      R.x = P.x;
      R.y = P.y;

      if ((!CanBePressed)
       &&((X < R.x) || (X > (R.x + R.width)) ||
          (Y < R.y) || (Y > (R.y + R.height))))
        deselect();
    }
  }

  public void mousePressed (MouseEvent e) {
    super.mousePressed (e);
    Released = false;
    select();
  }

  /**
    Paints the button.
  */
  public void paint (Graphics g) {
    if (Selected || IsMouseInside || AlwaysOn)
      super.paint(g);
    else {
      g.setClip (2, 2, IconWidth - 4, IconHeight - 4);
      g.drawImage (Bean, 2 - RealImagePos, 2, (IconWidth- 4) * States, IconHeight - 4, this);
    }
  }
}

